/****************************************************************************
 File: calceng.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.4  $
   $Author:   kcatlin  $
     $Date:   31 Jan 1992 08:08:00  $

 This file contains the class definition and methods for clsCalcEngine, a
 calculator "engine".
****************************************************************************/

													#ifndef CALCENG_INCLUDED
#include <calceng.h>
													#endif
													#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
													#endif
													#ifndef DEBUG_INCLUDED
#include <debug.h>
													#endif
													#ifndef FS_INCLUDED
#include <fs.h>
													#endif
													#ifndef OS_INCLUDED
#include <os.h>
													#endif
													#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
													#endif

#include <cengmeth.h>			// method function prototypes generated by MT
#include <math.h>
#include <string.h>
#include <stdlib.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Local Types and Constants                                              *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct CALC_ENGINE_DATA {

	double				xValue;
	double				yValue;
	CALC_ENGINE_KEY		pendingOp;
	BUFFER				keysSeen;
	BOOLEAN				numberEntered;
	BOOLEAN				calcError;

} CALC_ENGINE_DATA,
  * P_CALC_ENGINE_DATA,
  * * PP_CALC_ENGINE_DATA;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Local Functions and Macros                                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 CalcEngineEvalBinary
*/
STATIC STATUS
CalcEngineEvalBinary (
	double 				opnd1,
	CALC_ENGINE_KEY 	op,
	double 				opnd2,
	double *			pResult)
{
	STATUS				s = stsOK;

	switch (op) {
		case divide:
			if (opnd2 == 0.0) {
				s = stsCalcEngineComputeError;
			} else {
				*pResult = opnd1 / opnd2;
			}
			break;

		case multiply:
			*pResult = opnd1 * opnd2;
			break;

		case plus:
			*pResult = opnd1 + opnd2;
			break;

		case minus:
			*pResult = opnd1 - opnd2;
			break;

		default:
			Dbg(Debugf("CalcEngineEvalBinary: unknown operator %ld",(U32)op);)
			s = stsCalcEngineUnknownOperator;
			break;
	}
	return s;

} /* CalcEngineEvalBinary */


/****************************************************************************
 CalcEngineEvalUnary
*/
STATIC STATUS
CalcEngineEvalUnary (
	double 				opnd1,
	CALC_ENGINE_KEY 	op,
	double *			pResult,
	P_CALC_ENGINE_DATA	pInst)
{
	STATUS				s = stsOK;

	switch (op) {
		case squareRoot:
			if (opnd1 < 0.0) {
				s = stsCalcEngineComputeError;
			} else {
				*pResult = sqrt(opnd1);
			}
			break;

		case changeSign:
			*pResult = -opnd1;
			break;

		case clearAll:
			pInst->xValue = 0.0;
			pInst->yValue = 0.0;
			pInst->pendingOp = nop;
			pInst->numberEntered = false;
			pInst->calcError = false;
			*pResult = 0.0;
			break;

		case clear:
			*pResult = 0.0;
			break;

		default:
			s = stsCalcEngineUnknownOperator;
			break;
	}

	return s;

} /* CalcEngineEvalUnary */


/****************************************************************************
 CalcEngineSaveValue
*/
STATIC void
CalcEngineSaveValue (
	double					value,
	P_CALC_ENGINE_TOKEN 	pToken,
	P_CALC_ENGINE_DATA 		pInst)
{
	if (pInst->pendingOp == nop) {
		pInst->xValue = value;
	}
	pInst->yValue = value;
	gcvt(value, 10, pToken->buf);

} /* CalcEngineSaveValue */


/****************************************************************************
 CalcEngineUpdateToken
*/
STATIC void
CalcEngineUpdateToken (
	P_CALC_ENGINE_TOKEN	pToken,
	P_CALC_ENGINE_DATA 	pInst)
{
	pToken->value = atof(pInst->keysSeen);
	strcpy(pToken->buf, pInst->keysSeen);

} /* CalcEngineUpdateToken */


/****************************************************************************
 CalcEngineAppendCharToToken
*/
STATIC void
CalcEngineAppendCharToToken (
	CHAR 					c,
	P_CALC_ENGINE_TOKEN 	pToken,
	P_CALC_ENGINE_DATA 		pInst)
{
	U16						len;
	
	len = strlen(pInst->keysSeen);

	if (len >= maxDigits-1) {
		return;
	}
	
	pInst->keysSeen[len] = c;
	pInst->keysSeen[len+1] = '\0';

	CalcEngineUpdateToken(pToken, pInst);

} /* CalcEngineAppendCharToToken */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Message Handlers                                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 CalcEngineInit
	
 Note: Will likely leak storage under error conditions.
*/

MsgHandlerWithTypes(CalcEngineInit, P_CALC_ENGINE_NEW, PP_CALC_ENGINE_DATA)
{
	P_CALC_ENGINE_DATA	pInst;
	STATUS				s;

	//
	//	Allocate storage for instance information.
	//	The only instance data in the object is a pointer to this storage.
	//
	StsRet( OSHeapBlockAlloc(osProcessHeapId, SizeOf(CALC_ENGINE_DATA), \
			&pInst), s);

	//
	//	Initialize the instance data.
	//
	pInst->xValue = 0.0;
	pInst->yValue = 0.0;
	pInst->pendingOp = nop;
	pInst->numberEntered = false;
	pInst->calcError = false;
	pInst->keysSeen[0] = '\0';

	//
	//	Save the instance data in the object.
	//
	StsRet(ObjectWrite(self, ctx, &pInst), s);

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CalcEngineInitialize */


/****************************************************************************
 CalcEngineFree
*/

MsgHandlerWithTypes(CalcEngineFree, P_ARGS, PP_CALC_ENGINE_DATA)
{
	if (*pData != pNull) {
		StsWarn(OSHeapBlockFree(*pData));
	}

	return stsOK;
	MsgHandlerParametersNoWarning;

}  /* CalcEngineFree */


/****************************************************************************
 CalcEngineDump
*/

MsgHandlerWithTypes(CalcEngineDump, P_ARGS, PP_CALC_ENGINE_DATA)
{
	P_CALC_ENGINE_DATA	pInst = *pData;	// Reduce dereferences.

	Debugf("clsCalcEngine Dump: xValue=%f yValue=%f pendingOp=%ld",
			(U32)pInst->xValue, 
			(U32)pInst->yValue, 
			(U32)pInst->pendingOp);
	Debugf("                    keys=<%s> numberEncountered=%s calcError=%s",
			pInst->keysSeen,
			pInst->numberEntered ? "TRUE" : "FALSE",
			pInst->calcError ? "TRUE" : "FALSE");

	return stsOK;
	MsgHandlerParametersNoWarning;

}  /* CalcEngineDump */


/****************************************************************************
 CalcEngineRestore

 Note: Will likely leak storage under error conditions.
*/

MsgHandlerWithTypes(CalcEngineRestore, P_OBJ_RESTORE, PP_CALC_ENGINE_DATA)
{
	STREAM_READ_WRITE	srw;
	STATUS				s;
	P_CALC_ENGINE_DATA	pInst;

	//
	//	Allocate the instance data storage.
	//
	StsRet( OSHeapBlockAlloc(osProcessHeapId, SizeOf(CALC_ENGINE_DATA), \
			&pInst), s);
	StsRet( ObjectWrite(self, ctx, &pInst), s);

	//
	//	Read in the instance data.
	//
	srw.numBytes = SizeOf(CALC_ENGINE_DATA);
	srw.pBuf = (P_U8)pInst;
	ObjCallRet(msgStreamRead, pArgs->file, &srw, s);

	return stsOK;
	MsgHandlerParametersNoWarning;
	
} /* CalcEngineRestore */


/****************************************************************************
 CalcEngineSave
*/

MsgHandlerWithTypes(CalcEngineSave, P_OBJ_SAVE, PP_CALC_ENGINE_DATA)
{
	STREAM_READ_WRITE	srw;
	STATUS				s;

	//
	//	Write out our instance data.
	//
	srw.numBytes = SizeOf(CALC_ENGINE_DATA);
	srw.pBuf = (P_U8)*pData;
	ObjCallRet(msgStreamWrite, pArgs->file, &srw, s);

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CalcEngineSave */


/****************************************************************************
 CalcEngineEnterNumber
	
 Store number.
*/

MsgHandlerWithTypes(CalcEngineEnterNumber, P_CALC_ENGINE_TOKEN, \
		PP_CALC_ENGINE_DATA)
{
	CalcEngineSaveValue(pArgs->value, pArgs, *pData);
	(*pData)->numberEntered = true;

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CalcEngineEnterNumber */


/****************************************************************************
 CalcEngineEnterOperator
	
 Enter an operator.
*/

MsgHandlerWithTypes(CalcEngineEnterOperator, P_CALC_ENGINE_TOKEN, \
		PP_CALC_ENGINE_DATA)
{
	P_CALC_ENGINE_DATA	pInst = *pData;	// Reduce dereferences.
	double				result;
	STATUS				s;

	if (pArgs->key == nop) {

		// Nothing to do!

	} else if (pArgs->key == equals) {

		if (pInst->pendingOp != nop) {
			if ((s = CalcEngineEvalBinary(pInst->xValue,
					pInst->pendingOp, pInst->yValue, &result)) < stsOK) {
				Dbg(if (s != stsCalcEngineComputeError) StsWarn(s);)
				return s;
			}
			pInst->pendingOp = nop;
			CalcEngineSaveValue(result, pArgs, pInst);
		}

	} else if (pArgs->key >= unaryOps) {

		if ((s = CalcEngineEvalUnary(pInst->yValue, pArgs->key, &result,
				pInst)) < stsOK) {
			Dbg(if (s != stsCalcEngineComputeError) StsWarn(s);)
			return s;
		}	
		CalcEngineSaveValue(result, pArgs, pInst);

	} else {

		//
		// Execute any outstanding binary operations.  Then remember
		// the new operation.
		//
		if ((pInst->pendingOp != nop) AND (pInst->numberEntered)) {
			if ((s = CalcEngineEvalBinary(pInst->xValue, pInst->pendingOp,
					pInst->yValue, &result)) < stsOK) {
				Dbg(if (s != stsCalcEngineComputeError) StsWarn(s);)
				return s;
			}	
			pInst->pendingOp = nop;
			CalcEngineSaveValue(result, pArgs, pInst);
		}
		pInst->pendingOp = pArgs->key;
		pInst->numberEntered = false;

	}

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CalcEngineEnterOperator */


/****************************************************************************
 CalcEngineProcessKey

 Process single keys enter by user one at a time.
*/

MsgHandlerWithTypes(CalcEngineProcessKey, P_CALC_ENGINE_TOKEN, \
		PP_CALC_ENGINE_DATA)
{
	STATUS	s;

	switch (pArgs->key) {

		case zero: case one: case two: case three: case four:
		case five: case six: case seven: case eight: case nine:
		case decimalPoint:

			// 
			// If we get a second decimal point, throw it away.
			// Otherwise, append the character to the current token
			// and update the display.
			//
			if ((pArgs->key == decimalPoint) AND
					(strchr((*pData)->keysSeen, '.') != NULL)) {
				return stsOK;
			}
			CalcEngineAppendCharToToken((CHAR)pArgs->key, pArgs, *pData);
			break;

		case plus:
		case minus:
		case multiply:
		case divide:
		case squareRoot:
		case changeSign:
		case equals:

			//
			// If the previous token was a number, enter it.
			// Then enter the operator.
			//
			if ((*pData)->keysSeen[0] != '\0') {
				CalcEngineUpdateToken(pArgs, *pData);
				ObjCallJmp(msgCalcEngineEnterNumber, self, pArgs, s, Error);
				(*pData)->keysSeen[0] = '\0';
			}
			if ((s = ObjectCall(msgCalcEngineEnterOperator, self, pArgs))
					< stsOK) {
				Dbg(if (s != stsCalcEngineComputeError) StsWarn(s);)
				goto Error;
			}
			break;

		case clearAll:
		case clear:
			(*pData)->keysSeen[0] = '\0';
			if ((s = ObjectCall(msgCalcEngineEnterOperator, self, pArgs))
					< stsOK) {
				Dbg(if (s != stsCalcEngineComputeError) StsWarn(s);)
				goto Error;
			}
			break;

		default:
			s = stsCalcEngineUnknownKey;
			goto Error;
	}

	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	CalcEngineSaveValue(0.0, pArgs, *pData);
	strcpy(pArgs->buf, "Error");
	(*pData)->calcError = true;
	return s;

} /* CalcEngineProcessKey */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Global Functions                                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 DllMain

 Create the calculator engine class.  Called automatically by the PenPoint
 loader.
*/

STATUS EXPORTED 
DllMain (void)
{
	CLASS_NEW		new;
	STATUS			s;

	ObjCallWarn(msgNewDefaults, clsClass, &new);
	new.object.uid			= clsCalcEngine;
	new.cls.pMsg			= clsCalcEngineTable;
	new.cls.ancestor		= clsObject;
	new.cls.size			= SizeOf(P_CALC_ENGINE_DATA);
	new.cls.newArgsSize		= SizeOf(CALC_ENGINE_NEW);
	ObjCallRet(msgNew, clsClass, &new, s);

	return stsOK;

} /* DllMain */




