/****************************************************************************
 File: milsvc0.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.0  $
   $Author:   jgarrett  $
     $Date:   12 Mar 1992 12:33:00  $

 This file contains the ring 0 code for clsTestMILService.
****************************************************************************/

#include <clsmgr.h>
#include <debug.h>
#include <list.h>
#include <servmgr.h>
#include <service.h>
#include <stream.h>
#include <drvmil.h>
#include <os.h>
#include <string.h>
#include <dvparall.h>
#include <milsvc.h>
#include <milsvc0.h>


/****************************************************************************
	pRBInit, pRBPrn, pRBSet, pRBStatus, and pRBAttach

	Shorthand version of corresponding variable on right

	pRBInit   -- used in printer initialization routine
	pRBPrn    -- used in print routine
	pRBSet    -- used in get/set device parameters routines
	pRBStatus -- used in the get printer status routine
	pRBAttach -- used in the printer attachment routines
****************************************************************************/

#define	pRBInit		((P_MIL_RB_PUB_PARALLEL_INIT_PRINTER)(pInst->pRBMisc))
#define pRBPrn		((P_MIL_RB_PUB_PARALLEL_WRITE)(pInst->pRBPrint))
#define pRBSet		((P_MIL_RB_PUB_PARALLEL_DEVICE_PARAMETERS)(pInst->pRBMisc))
#define pRBStatus	((P_MIL_RB_PUB_PARALLEL_DEVICE_PARAMETERS)(pInst->pRBMisc))
#define pRBAttach	((P_MIL_RB_PUB_BASE_ATTACHMENT_CONT)(pInst->pRBConnect))


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcGetReqBlocks(P_INSTANCE_DATA pInst)

	Allocates ring0 system resources.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcGetReqBlocks(P_INSTANCE_DATA pInst)
{
	STATUS	s;


	/*
	 * Our instance data is used by the attachment callback
	 * procedure.  Locking it in memory insures that it will
	 * be in memory when we need it.  We aren't required to do
	 * this since the attachment routine is a timed not an interrupt
	 * function.  Otherwise page faults can occur to bring pages
	 * swapped to disk back into memory.
	 */
	OSMemLock(pInst, SizeOf(INSTANCE_DATA));

	// Allocate a request block for all the single stage mil requests
	StsRet(DrvMILRequestBlockCreate(pInst->logicalId,
										(PP_UNKNOWN)(&(pInst->pRBMisc))), s);

	// Allocate a request to perform printing
	StsRet(DrvMILRequestBlockCreate(pInst->logicalId,
										(PP_UNKNOWN)(&(pInst->pRBPrint))), s);

	// Initialize the unit field in the request block to our unit number
	pInst->pRBPrint->unit = pInst->unit;
	pInst->pRBMisc->unit = pInst->unit;

	/*
	 * Allocate a 512 byte buffer (size arbitrary).
	 * This buffer must be allocated/locked such that it
	 * in memory at all times.  The printing mil request
	 * is a stage on interrupt function; therefore, the
	 * buffer to be printed must be in memory at all times.
	 * A page fault to bring in a non-memory resident page
	 * cannot occur within an interrupt routine.
	 */
	StsRet(OSHeapBlockAlloc(osProcessSharedHeapId, 512,
												&(pInst->printBuffer)), s);
//	OSMemLock(pInst->printBuffer, 512);

	return stsOK;

}	// TestMILSvcGetReqBlocks


/****************************************************************************
	STATUS EXPORTED0 TestMILSvc0Destroy(P_INSTANCE_DATA pInst)

	Free all ring 0 resources used by this service
****************************************************************************/

STATUS EXPORTED0 TestMILSvc0Destroy(P_INSTANCE_DATA	pInst)
{
	BOOLEAN		enable;
	LIST		conflictGroupItems;
	STATUS		s;
	U16			numConflictItems;

	// Unlock memory used by this service
	OSMemUnlock(pInst, SizeOf(INSTANCE_DATA));

	// Free request blocks
	StsRet(DrvMILRequestBlockFree((P_UNKNOWN)(pInst->pRBMisc)), s);
	StsRet(DrvMILRequestBlockFree((P_UNKNOWN)(pInst->pRBPrint)), s);

	// Free our print buffer
	if (pInst->printBuffer != 0)
	{
//		OSMemUnlock(pInst->printBuffer, 512);
		StsRet(OSHeapBlockFree(pInst->printBuffer), s);
	}

	// If we are the last mil service on our
	// conflict group, disable our interrupt.
	ObjCallRet(msgIMGetList, pInst->conflictGroup, &conflictGroupItems, s);
	ObjCallRet(msgListNumItems, conflictGroupItems,	&numConflictItems, s);
	if (numConflictItems == 1)
	{
		enable = false;
		OSIntMask(pInst->pRBMisc->logicalId, &enable);
	}
	return ObjectCall(msgDestroy, conflictGroupItems, pNull);

}	/* TestMILSvc0Destroy */


/****************************************************************************
	STATUS EXPORTED TestMILSvcSetInterrupt(P_INSTANCE_DATA pInst)

	Enables parallel port interrupt
****************************************************************************/

STATUS EXPORTED0 TestMILSvcSetInterrupt(P_INSTANCE_DATA pInst)
{
	BOOLEAN		enable;

	enable = true;
	return OSIntMask(pInst->pRBMisc->logicalId, &enable);

}	/* TestMILSvcSetInterrupt */


/****************************************************************************
	void EXPORTED0 TestMILSvcAttachmentCallBack(P_MIL_COMMON_DATA	pCommonData,
							P_MIL_RB_PUB_BASE_ATTACHMENT_CONT	pRB)

	The connection detection call back routine.  This routine is called
	by the mil connection detection function whenever a printer is
	attached or detached.
****************************************************************************/

void EXPORTED0 TestMILSvcAttachmentCallBack(P_MIL_COMMON_DATA		pCommonData,
									P_MIL_RB_PUB_BASE_ATTACHMENT_CONT	pRB)
{
	P_INSTANCE_DATA			pInst;
	SVC_GET_SET_CONNECTED	svcGetSetConnected;

	
	Unused(pCommonData);


	// Get our instance data pointer from the request block
	pInst = (P_INSTANCE_DATA)(pRB->callerDataU32A);

	if (pRB->event == milDevAttachedEvent)
	{
		// We're attached
		pInst->connected = true;
		Dbg(Debugf("TEST_MIL_SVC0: printer connected to device %d", pRB->logicalId);)
	}
	else if (pRB->event == milDevDetachedEvent)
	{
		// We're detached
		pInst->connected = false;
		Dbg(Debugf("TEST_MIL_SVC0: printer not connected to device %d", pRB->logicalId);)
	}

	// Send message to ourself indicating new connection state
	svcGetSetConnected.connected = pInst->connected;
	ObjectPost(msgSvcSetConnected, pInst->self, &svcGetSetConnected,
											SizeOf(SVC_GET_SET_CONNECTED));

}	/* TestMILSvcAttachmentCallBack */


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcStartConnectionDetection(P_INSTANCE_DATA pInst)

	Start the parallel mil devices connection function.  The connection
	function is a continuous staged on time function.  When the connection
	function runs, it checks for a printer connected to the parallel port.
	When a printer is attached or detached, the attachment call back
	procedure is called with the attachment state.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcStartConnectionDetection(P_INSTANCE_DATA pInst)
{
	STATUS	s;

	// Initially indicate we are disconnected
	pInst->connected = false;

	/*
	 * Allocate a request block for the attachment function
	 * We allocate the request block here since the cancel
	 * request frees the request block
	 */
	StsRet(DrvMILRequestBlockCreate(pInst->logicalId,
									(PP_UNKNOWN)(&(pInst->pRBConnect))), s);

	/* Fill in the appropriate fields of the request block
	 *		our unit number
	 *		function code for attachment
	 *		user data (our instance data)
	 *		the call back function to use
	 */
							
	pRBAttach->unit = pInst->unit;
	pRBAttach->functionCode = milBaseAttachmentCont;
	pRBAttach->callerDataU32A = (U32)pInst;
	pRBAttach->pAsyncEventFunc =
							(P_MIL_ASYNC_EVENT_FUNC)TestMILSvcAttachmentCallBack;

	// Start the attachment function
	return DrvMILRequest(pInst->pRBConnect);

}	/* TestMILSvcStartConnectionDetection */


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcStopConnectionDetection(P_INSTANCE_DATA pInst)

	Terminates the connection detection mil request.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcStopConnectionDetection(P_INSTANCE_DATA pInst)
{
	return DrvMILCancelRequest(pInst->pRBConnect);

}	/* TestMILSvcStopConnectionDetection */


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcGetMetrics(P_TEST_MIL_SVC_METRICS	pArgs,
									P_INSTANCE_DATA		pInst)

	Get the mil device parameters portion of testMILSvc metrics
****************************************************************************/

STATUS EXPORTED0 TestMILSvcGetMetrics(P_TEST_MIL_SVC_METRICS	pArgs,
									P_INSTANCE_DATA	pInst)
{
	STATUS	s;

	/*
	 * You must reset the request block before
	 * reusing an existing request block.
	 */
	DrvMILRequestBlockReset(pInst->pRBMisc);

	// Request the device parameters from the parallel port mil device
	pRBSet->functionCode = milParallelGetDevParameters;
	StsRet(DrvMILRequest(pInst->pRBMisc), s);

	// Copy parameters to metrics structure
	pArgs->devFlags = pRBSet->parallelDevParms.parallelDevFlags;
	pArgs->unitFlags = pRBSet->parallelUnitParms.parallelUnitFlags;
	pArgs->initDelay = pRBSet->parallelTimeIntervals.initDelay;
	pArgs->interruptTimeOut = pRBSet->parallelTimeIntervals.interruptTimeOut;

	return stsOK;

}	/* TestMILSvcGetMetrics */


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcSetMetrics(P_TEST_MIL_SVC_METRICS	pArgs,
									P_INSTANCE_DATA		pInst)

	Set the mil device parameters from the testMILSvc metrics
****************************************************************************/

STATUS EXPORTED0 TestMILSvcSetMetrics(P_TEST_MIL_SVC_METRICS	pArgs,
									P_INSTANCE_DATA	pInst)
{
	STATUS			s;


	// interrupt time out and init delay cannot be zero
	if (pArgs->interruptTimeOut == 0 || pArgs->initDelay == 0)
	{
		return stsBadParam;
	}
	else
	{
		/*
		 * You must reset the request block before
		 * reusing an existing request block.
		 */
		DrvMILRequestBlockReset(pInst->pRBMisc);

		// Fill in the proper fields in the request block
		pRBSet->functionCode = milParallelSetDevParameters;
		pRBSet->parallelDevParms.parallelDevFlags = pArgs->devFlags;
		pRBSet->parallelUnitParms.parallelUnitFlags = pArgs->unitFlags;
		pRBSet->parallelTimeIntervals.initDelay = pArgs->initDelay;
		pRBSet->parallelTimeIntervals.interruptTimeOut =
													pArgs->interruptTimeOut;

		// Perform the request to set the parameters
		if ((s = DrvMILRequest(pInst->pRBMisc)) == stsOK)
		{
			// If successful copy new metrics to instance data
			memcpy(&(pInst->testMILSvcMetrics), pArgs, SizeOf(TEST_MIL_SVC_METRICS));
		}
		else
		{
			// Otherwise get old metrics from mil device
			TestMILSvcGetMetrics(&(pInst->testMILSvcMetrics), pInst);
		}
		return s;
	}

}	/* TestMILSvcSetMetrics */


/****************************************************************************
	STATUS EXPORTED TestMILSvcInitialize(P_INSTANCE_DATA pInst)

	Have the parallel printer mil device initialize the printer by
	asserting the initialize hardware signal to the printer.  The
	initialize signal is asserted for initDelay microseconds.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcInitialize(P_INSTANCE_DATA pInst)
{
	/*
	 * You must reset the request block before
	 * reusing an existing request block.
	 */
	DrvMILRequestBlockReset(pInst->pRBMisc);

	// Indicate what function is requested and call the mil
	pRBInit->functionCode = milParallelInitPrinter;
	DrvMILRequest(pInst->pRBMisc);

	return (pInst->pRBMisc)->status;

}	/* TestMILSvcInitialize */


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcGetStatus(P_TEST_MIL_SVC_STATUS		portStatus,
									P_INSTANCE_DATA		pInst)

	Get the status of the printer connected to the parallel port.  All
	parallel printer functions return the printer status.  We use the
	get device function since it gets the status without affecting
	the printer.  The get printer status returns the contents of the
	parallel port status register.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcGetStatus(P_TEST_MIL_SVC_STATUS		portStatus,
								P_INSTANCE_DATA		pInst)
{
	STATUS	s;

	/*
	 * You must reset the request block before
	 * reusing an existing request block.
	 */
	DrvMILRequestBlockReset(pInst->pRBMisc);

	// Indicate what function is requested and call the mil
	pRBStatus->functionCode = milParallelGetDevParameters;
	StsRet(DrvMILRequest(pInst->pRBMisc), s);

	// Get the printer status from the request block
	portStatus->testMILSvcStatus = pRBStatus->parallelUnitParms.parallelStatus;
	return stsOK;

}	/* TestMILSvcGetStatus */


#ifdef DEBUG

/****************************************************************************
	The following functions are not necessary for printing.  They are
	included to display the buffer contents when the buffer is not
	printed.  Only present in debug version.
****************************************************************************/

U8 LOCAL HexToAscii(U8 num)
{
	U8	ascii;

	ascii = num + '0';
	if (ascii > '9')
	{
		ascii = (num - 10) + 'a';
	}
	return ascii;

}	/* HexToAscii */

#define HighU4(x)		(((x) >> 4) & 0x0f)
#define LowU4(x)		((x) & 0x0f)


void LOCAL DisplayBuffer(P_U8 pBuffer, U16 bufferSize)
{
	U16		count;
	U16		i;
	U8		outbuf[81];

	for (count = 0, i = 0; count < bufferSize; count++)
	{
		outbuf[i++] = HexToAscii(HighU4(pBuffer[count]));
		outbuf[i++] = HexToAscii(LowU4(pBuffer[count]));
		outbuf[i++] = ' ';
		if (i > 77)
		{
			outbuf[i] = '\0';
			Debugf(outbuf);
			i = 0;
		}
	}
	if (i)
	{
		outbuf[i] = '\0';
		Debugf(outbuf);
	}

}	/* DisplayBuffer */

#endif


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcPrint(P_STREAM_READ_WRITE	pStream,
								P_INSTANCE_DATA		pInst)

	Sends the contents of the buffer specified in the pStream structure
	to the printer to be printed.  The buffer is copied to our local
	buffer before printing.  It is copied to our buffer to insure that
	the data is available when the interrupt routine is executed.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcPrint(P_STREAM_READ_WRITE	pStream,
							P_INSTANCE_DATA		pInst)
{
	U16		count;
	U16		i, j;


	Dbg(Debugf("TEST_MIL_SVC0: Printing");)

	// initialize the count we have printed variable to zero
	count = 0;
	do
	{
		/*
		 * copy 512 bytes of the client buffer
		 * or the entire client buffer to our
		 * local buffer
		 */
		j = count;
		for (i = 0; i < 512 && j < pStream->numBytes; i++, j++)
		{
			pInst->printBuffer[i] = ((P_U8)(pStream->pBuf))[j];
		}

		/*
		 * You must reset the request block before
		 * reusing an existing request block.
		 */
		DrvMILRequestBlockReset(pInst->pRBPrint);

		/*
		 * Initialize request block parameters
		 *		function to perform
		 *		pointer to our buffer
		 *		indicate we are giving
		 *			the mil device a pointer
		 *			to a buffer
		 *		the size of the buffer
		 */
		pRBPrn->functionCode = milParallelWrite;
		pRBPrn->data.pBuffer = pInst->printBuffer;
		pRBPrn->dataIsBuffer = false;
		pRBPrn->bufferSize = i;

		// indicate we are inside the mil request
		pInst->printInProgress = true;

		// call the mil
		StsWarn(DrvMILRequest(pInst->pRBPrint));

		// has the printing been cancelled?
		if (pInst->printInProgress == false)
		{
			// yes - exit
			break;
		}

		// indicate we are not in the mil
		pInst->printInProgress = false;

		// update count of total characters printed
		count += pRBPrn->countPrinted;

		// continue until an error or the entire buffer is printed
	} while (pRBPrn->countPrinted == pRBPrn->bufferSize &&
												count < pStream->numBytes);

	// indicate number of characters printed
	pStream->count = count;

#ifdef	DEBUG
	// if an error occurred display buffer being printed
	if (pRBPrn->status != stsOK)
	{
		Debugf("count printed is %d -- count requested %d",
				pRBPrn->countPrinted,
				pRBPrn->bufferSize);

		Dbg(DisplayBuffer(pInst->printBuffer, pRBPrn->bufferSize);)

		Debugf("pStream->count is %d -- pStream->numBytes is %d",
				pStream->count,
				pStream->numBytes);
	}
#endif

	return pRBPrn->status;

}	/* TestMILSvcPrint */


/****************************************************************************
	STATUS EXPORTED0 TestMILSvcCancelPrint(P_INSTANCE_DATA pInst)

	Cancel print request.
****************************************************************************/

STATUS EXPORTED0 TestMILSvcCancelPrint(P_INSTANCE_DATA pInst)
{
	STATUS	s;

	// If print mil reques in progress, cancel request
	if (pInst->printInProgress != false)
	{
		// call mil with request block to be cancelled
		StsRet(DrvMILCancelRequest(pInst->pRBPrint), s);

		// indicate request has been cancelled
		pInst->printInProgress = false;
	}
	return stsOK;

}	/* TestMILSvcCancelPrint */
