/****************************************************************************
 File: tttutil.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.5  $
 $Author:   kcatlin  $
 $Date:   17 Mar 1992 14:08:12  $

 This file contains the implementation of miscellaneous utility routines
 used by TttApp.  The interfaces to these routines are in tttpriv.h.
****************************************************************************/

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef SWIN_INCLUDED
#include <swin.h>
#endif

#ifndef MENU_INCLUDED
#include <menu.h>
#endif

#ifndef APP_INCLUDED
#include <app.h>
#endif

#ifndef FRAME_INCLUDED
#include <frame.h>
#endif

#ifndef TTTPRIV_INCLUDED
#include <tttpriv.h>
#endif

#ifndef SWIN_INCLUDED
#include <swin.h>
#endif

#ifndef VIEW_INCLUDED
#include <view.h>
#endif

#ifndef NOTE_INCLUDED
#include <note.h>
#endif

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef SEL_INCLUDED
#include <sel.h>
#endif

#ifndef APPTAG_INCLUDED
#include <apptag.h>
#endif

#include <string.h>
#include <stdio.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Utility Routines							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	TttUtilStrForSquareValue
****************************************************************************/

//
// Output is nicer if all strings have the same length.  Don't worry
// about the "Unknown" string -- it should never be output anyhow.
//
static const P_STRING valueStrings[] = {
	"_", "X", "O", "Unknown"};

P_STRING PASCAL
TttUtilStrForSquareValue(
	TTT_SQUARE_VALUE	v)
{
	if (v == tttBlank) {
		return valueStrings[0];
	} else if (v == tttX) {
		return valueStrings[1];
	} else if (v == tttO) {
		return valueStrings[2];
	} else {
		return valueStrings[3];
	}
} /* TttUtilStrForSquareValue */


/****************************************************************************
	TttUtilSquareValueForChar
****************************************************************************/
TTT_SQUARE_VALUE PASCAL
TttUtilSquareValueForChar(
	char	ch)
{
	return ((TTT_SQUARE_VALUE) ch);

} /* TttUtilSquareValueForChar */


/****************************************************************************
	TttUtilInsertUnique

	Utility routine that inserts and element into an array if
	it's not already in the array.  Assumes enough space in the array.
****************************************************************************/
#define DbgTttUtilInsertUnique(x) \
	TttDbgHelper("TttUtilInsertUnique",tttUtilDbgSet,0x1,x)

STATIC void PASCAL
TttUtilInsertUnique(
	P_U32		pValues,
	P_U16		pCount, 	// In: number of elements in pValues
							// Out: new number of elements in pValues
	U32			new)
{
	U16			i;

	DbgTttUtilInsertUnique(("*pCount=%ld new=%ld=0x%lx", (U32)(*pCount), \
			(U32)new, (U32)new))
	for (i=0; i<*pCount; i++) {
		if (pValues[i] == new) {
			DbgTttUtilInsertUnique(("found it at %ld; returning",(U32)i))
			return;
		}
	}
	pValues[*pCount] = new;
	*pCount = *pCount + 1;
	DbgTttUtilInsertUnique(("didn't find it; new count=%ld",(U32)(*pCount)))
} /* TttUtilInsertUnique */


/****************************************************************************
	TttUtilCreateScrollWin

	This is in a utility routine rather than in the caller because
	including swin.h brings in too many symbols.

	The example of scrolling used here is slightly artificial.
	This tells the scroll window that it can expand the Tic-Tac-Toe view
	beyond its desired size, but should not contract it.  Thus if the
	scroll win shrinks, the TttView will be scrollable.

	When an application does scrolling, the view should get the
	messages.  So we set the client window appropriately now.
****************************************************************************/
STATUS PASCAL
TttUtilCreateScrollWin(
	OBJECT		clientWin,
	P_OBJECT	pScrollWin)

{
	SCROLL_WIN_NEW	scrollWinNew;
	STATUS			s;

	ObjCallJmp(msgNewDefaults, clsScrollWin, &scrollWinNew, s, Error);
	scrollWinNew.scrollWin.clientWin					= clientWin;
	scrollWinNew.scrollWin.style.expandChildWidth		= true;
	scrollWinNew.scrollWin.style.expandChildHeight		= true;
	ObjCallJmp(msgNew, clsScrollWin, &scrollWinNew, s, Error);
	*pScrollWin = scrollWinNew.object.uid;
	return stsOK;

Error:
	return s;
} /* TttUtilCreateScrollWin */


/****************************************************************************
	TttUtilCreateMenu

	This is in a utility routine rather than in the caller because
	including menu.h brings in too many symbols.
****************************************************************************/
STATUS PASCAL
TttUtilCreateMenu(
	OBJECT			parent,
	OBJECT			client,
	P_UNKNOWN		pEntries,	// really  P_TK_TABLE_ENTRY pEntries
	P_OBJECT		pMenu)
{
	MENU_NEW		mn;
	STATUS			s;

	ObjCallJmp(msgNewDefaults, clsMenu, &mn, s, Error);
	mn.win.parent = parent;
	mn.tkTable.client = client;
	mn.tkTable.pEntries = (P_TK_TABLE_ENTRY)pEntries;
	ObjCallJmp(msgNew, clsMenu, &mn, s, Error);
	*pMenu = mn.object.uid;
	return stsOK;

Error:
	return s;
} /* TttUtilCreateMenu */


/****************************************************************************
	TttUtilAdjustMenu

	"Adjusts" the menu by removing the items that this app
	does not support.

	Each menu that has an item removed from it must be layed out
	and "break adjusted."  The latter involves adjusting the border edges
	and margins for a menu with lines dividing it into sections.
	But because the Standard Application Menus can change, we don't want
	to compile in knowledge of which menus these are.  (The menu item's
	containing menu is 	easy to find -- it's just the menu item's
	parent window.)

	The obvious approach is to simply lay out and break adjust
	the item's menu after removing the item.  Unfortunately
	this potentially results in laying out and break
	adjusting the same menu several times, since several of
	the items could be removed from the same menu.

	Our solution is to keep an array of all unique parents seen.
	The array is known to be no longer than than the array of
	disableTags, so space allocation is easy. TttUtilUniqueSort
	is used to do the unique insertion.  Finally, we run over
	the unique array and do the necessary operations on the
	menus.
****************************************************************************/
#define DbgTttUtilAdjustMenu(x) \
	TttDbgHelper("TttUtilAdjustMenu",tttUtilDbgSet,0x2,x)

//
// Tags which correspond to the menu items that this application
// will not implement.
//
static const TAG disableTags[] = {
	tagAppMenuSearch,
	tagAppMenuSpell};

#define N_DISABLE_TAGS (SizeOf(disableTags) / SizeOf(disableTags[0]))


STATUS PASCAL
TttUtilAdjustMenu(
	OBJECT		menuBar)
{
	WIN			parentMenus [N_DISABLE_TAGS];	// There are at most
												// one menu per tag.
	U16			parentMenuCount;
	WIN_METRICS	wm;
	U16			i;
	OBJECT		o;
	STATUS		s;

	DbgTttUtilAdjustMenu((""))

	memset(parentMenus, 0, sizeof(parentMenus));
	parentMenuCount = 0;

	for (i=0; i<N_DISABLE_TAGS; i++) {
		DbgTttUtilAdjustMenu(("i=%ld t=0x%lx",(U32)i,(U32)(disableTags[i])))
		if ((o = (OBJECT)ObjCallWarn(msgWinFindTag, menuBar,
				(P_ARGS)(disableTags[i]))) != objNull) {
			ObjCallJmp(msgWinGetMetrics, o, &wm, s, Error);
			TttUtilInsertUnique((P_U32)parentMenus, &parentMenuCount,
					(U32)(wm.parent));
			ObjCallWarn(msgDestroy, o, pNull);
			DbgTttUtilAdjustMenu(("destroyed it; parent=0x%lx",wm.parent))
		} else {
			DbgTttUtilAdjustMenu(("didn't find tag!"))
		}
	}

	//
	// Adjust the breaks and re-layout each affected menu
	//
	for (i=0; i<parentMenuCount; i++) {
		DbgTttUtilAdjustMenu(("i=%ld parent=0x%lx",(U32)i, parentMenus[i]))
		// pArgs of true tells menu to layout self.
		ObjCallJmp(msgMenuAdjustSections, parentMenus[i], (P_ARGS)true, \
				s, Error);
	}

	DbgTttUtilAdjustMenu(("returns stsOK"))
	return stsOK;

Error:
	DbgTttUtilAdjustMenu(("Error; returns 0x%lx",s))
	return s;
} /* TttUtilAdjustMenu */


/****************************************************************************
	TttUtillWrite
****************************************************************************/
STATUS PASCAL
TttUtilWrite(
	OBJECT				file,
	U32					numBytes,
	P_UNKNOWN			pBuf)
{
	STREAM_READ_WRITE	write;

	write.numBytes = numBytes;
	write.pBuf = pBuf;
	return ObjCallWarn(msgStreamWrite, file, &write);
} /* TttUtilWrite */


/****************************************************************************
	TttUtilWriteVersion

	Optimization Note: This could put in-line or converted to a macro.
****************************************************************************/
STATUS PASCAL
TttUtilWriteVersion(
	OBJECT		file,
	TTT_VERSION	version)
{
	return TttUtilWrite(file, SizeOf(version), &version);
} /* TttUtilWriteVersion */


/****************************************************************************
	TttUtilRead
****************************************************************************/
STATUS PASCAL
TttUtilRead(
	OBJECT		file,
	U32			numBytes,
	P_UNKNOWN	pBuf)
{
	STREAM_READ_WRITE	read;

	read.numBytes = numBytes;
	read.pBuf = pBuf;
	return ObjCallWarn(msgStreamRead, file, &read);
} /* TttUtilRead */


/****************************************************************************
	TttUtilReadVersion
****************************************************************************/
#define DbgTttUtilReadVersion(x) \
	TttDbgHelper("TttUtilReadVersion",tttUtilDbgSet,0x4,x)

STATUS PASCAL
TttUtilReadVersion(
	OBJECT				file,
	TTT_VERSION			minVersion,
	TTT_VERSION			maxVersion,
	P_TTT_VERSION		pVersion)
{
	STATUS				s;

	DbgTttUtilReadVersion(("min=%ld max=%ld",(U32)minVersion, (U32)maxVersion))
	StsJmp(TttUtilRead(file, SizeOf(*pVersion), pVersion), s, Error);
	if ((*pVersion < minVersion) OR (*pVersion > maxVersion)) {
		DbgTttUtilReadVersion(("version mismatch; v=%ld",(U32)(*pVersion)))
		s = stsIncompatibleVersion;
		goto Error;
	}
	DbgTttUtilReadVersion(("version=%ld; return stsOK",(U32)(*pVersion)))
	return stsOK;

Error:
	DbgTttUtilReadVersion(("Error; return 0x%lx",s))
	return s;
} /* TttUtilReadVersion */


/****************************************************************************
	TttUtilGetComponents

	Note: this is an internal utility routine.  Therefore it does not
	check carefully for null values.
****************************************************************************/
STATUS PASCAL
TttUtilGetComponents(
	OBJECT	  		app,
	U16		  		getFlags,
	P_OBJECT	 	pScrollWin,
	P_OBJECT	 	pView,
	P_OBJECT	 	pDataObject)
{
	OBJECT	  		view;
	APP_METRICS		am;
	OBJECT	  		clientWin;
	STATUS	  		s;

	//
	// Get the scrollWin regardless of the getFlags because we need
	// the scrollWin to get anything else and we assume the getFlags
	// aren't empty.
	//
	ObjCallJmp(msgAppGetMetrics, app, &am, s, Error);
	ObjCallJmp(msgFrameGetClientWin, am.mainWin, &clientWin, s, Error);
	if (FlagOn(tttGetScrollWin, getFlags)) {
		*pScrollWin = clientWin;
	}

	//
	// Do we need anything else?
	//
	if (FlagOn(tttGetView, getFlags) OR FlagOn(tttGetDataObject, getFlags)) {

		//
		// Get the view regardless of the getFlags because we need
		// the view to get either the view or the dataObject.
		// 
		ObjCallJmp(msgScrollWinGetClientWin, clientWin, &view, s, Error);
		if (FlagOn(tttGetView, getFlags)) {
			*pView = view;
		}

		if (FlagOn(tttGetDataObject, getFlags)) {
			ObjCallJmp(msgViewGetDataObject, view, pDataObject, s, Error);
		}
	}

	return stsOK;

Error:
	return s;
} /* TttUtilGetComponents */


/****************************************************************************
	TttUtilInitTextOutput
****************************************************************************/
void PASCAL
TttUtilInitTextOutput(
	P_SYSDC_TEXT_OUTPUT	p,
	U16			 		align,
	P_U8		 		buf)
{
	memset(p, 0, sizeof(*p));
	p->spaceChar = (U16)' ';
	p->stop = maxS32;
	p->alignChr = align;
	p->pText = buf;
	if (buf) {
		p->lenText = strlen(buf);
	}
} /* TttUtilInitTextOutput */

