/****************************************************************************
 File: tttview.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.10  $
 $Author:   kcatlin  $
 $Date:   06 Feb 1992 13:32:10  $

 This file contains the implementation of clsTttView.
****************************************************************************/

#ifndef APPTAG_INCLUDED
#include <apptag.h>
#endif

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

#ifndef CONTROL_INCLUDED
#include <control.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#ifndef XGESTURE_INCLUDED
#include <xgesture.h>
#endif

#ifndef PEN_INCLUDED
#include <pen.h>
#endif

#ifndef STDIO_INCLUDED
#include <stdio.h>
#endif

#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
#endif

#ifndef TTTDATA_INCLUDED
#include <tttdata.h>
#endif

#ifndef TTTVIEW_INCLUDED
#include <tttview.h>
#endif

#ifndef TTTPRIV_INCLUDED
#include <tttpriv.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef PREFS_INCLUDED
#include <prefs.h>
#endif

#ifndef SEL_INCLUDED
#include <sel.h>
#endif

#ifndef KEY_INCLUDED
#include <key.h>
#endif

#include <string.h>

#include <methods.h>


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Defines, Types, Globals, Etc				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


//
// CURRENT_VERSION is the file format version written by this implementation.
// MIN_VERSION is the minimum file format version readable by this 
// implementation.  MAX_VERSION is the maximum file format version readable
// by this implementation.
//
#define CURRENT_VERSION 0
#define MIN_VERSION		0
#define MAX_VERSION		0

//
// The desired size for the tic-tac-toe view
//
#define desiredWidth 	200
#define desiredHeight	200

typedef struct TTT_VIEW_FILED_0 {

	U32	lineThickness;

} TTT_VIEW_FILED_0, * P_TTT_VIEW_FILED_0;


//
// This struct defines the positions and sizes of all of the interesting
// pieces of the window.
//
// It is a relatively large structure, but since it is only used as a 
// stack variable, its size isn't of much concern.
//
typedef struct {

	U32		normalBoxWidth;
	U32		normalBoxHeight;
	RECT32	vertLines[2];
	RECT32	horizLines[2];
	RECT32	r[3][3];
	SCALE	scale;


} TTT_VIEW_SIZES, * P_TTT_VIEW_SIZES;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Utility Routines							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	TttViewFiledData0FromInstData

	Computes filed data from instance data.
****************************************************************************/
STATIC void PASCAL
TttViewFiledData0FromInstData(
	P_TTT_VIEW_INST		pInst,
	P_TTT_VIEW_FILED_0	pFiled)
{
	pFiled->lineThickness = pInst->metrics.lineThickness;
} /* TttViewFiledData0FromInstData */


/****************************************************************************
	TttViewInstDataFromFiledData0

	Computes instance data from filed data.
****************************************************************************/
STATIC void PASCAL
TttViewInstDataFromFiledData0(
	P_TTT_VIEW_FILED_0	pFiled,
	P_TTT_VIEW_INST		pInst)
{
	pInst->metrics.lineThickness = pFiled->lineThickness;
} /* TttViewInstDataFromFiledData0 */


/****************************************************************************
	TttViewNeedRepaint

	Marks for repaint.
****************************************************************************/
STATIC STATUS PASCAL
TttViewNeedRepaint(
	OBJECT	self)
{
	return ObjCallWarn(msgWinDirtyRect, self, pNull);
} /* TttViewNeedRepaint */


/****************************************************************************
	TttViewCreateDC
	
	Constructs and initializes dc.

	This routine uses the system font as of the time the dc is
	created.  No effort is made to track changes to the system font.
****************************************************************************/
#define DbgTttViewCreateDC(x) \
	TttDbgHelper("TttViewCreateDC",tttViewDbgSet,0x1,x)

STATIC STATUS PASCAL
TttViewCreateDC(
	OBJECT			self,
	P_OBJECT   		pDC)
{
	SYSDC_NEW 			dcNew;
	RES_READ_DATA		resRead;
	PREF_SYSTEM_FONT	font;
	STATUS				s;

	DbgTttViewCreateDC(("self=0x%lx",self))

	//
	// Initialize for error recovery
	//
	dcNew.object.uid = objNull;
	*pDC = objNull;

	//
	// Create the dc
	//
	ObjCallJmp(msgNewDefaults, clsSysDrwCtx, &dcNew, s, Error);
	ObjCallJmp(msgNew, clsSysDrwCtx, &dcNew, s, Error);

	// 
	// Set the dc's font to the current system font.
	//
	resRead.resId = prSystemFont;
	resRead.heap = Nil(OS_HEAP_ID);
	resRead.pData = &font;
	resRead.length = SizeOf(font);
	ObjCallJmp(msgResReadData, theSystemPreferences, &resRead, s, Error);
	ObjCallJmp(msgDcOpenFont, dcNew.object.uid, &(font.spec), s, Error);

	//
	// Bind dc to self
	//
	ObjCallJmp(msgDcSetWindow, dcNew.object.uid, (P_ARGS)self, s, Error);

	*pDC = dcNew.object.uid;

	DbgTttViewCreateDC(("return stsOK"))
	return stsOK;

Error:
	if (dcNew.object.uid) {
		ObjCallWarn(msgDestroy, dcNew.object.uid, pNull);
	}
	DbgTttViewCreateDC(("Error; returns 0x%lx",s))
	return s;
} /* TttViewCreateDC */


/****************************************************************************
	TttViewGestureSetSquare
	
	Handles all gestures that set the value
	of a single square.
****************************************************************************/
#define DbgTttViewGestureSetSquare(x) \
	TttDbgHelper("TttViewGestureSetSquare",tttViewDbgSet,0x2,x)

STATIC STATUS PASCAL
TttViewGestureSetSquare(
	VIEW					self,
	P_GWIN_GESTURE			pGesture,
	TTT_SQUARE_VALUE		value)		// Either tttX or tttO
{
	TTT_DATA_SET_SQUARE		set;
	OBJECT					dataObject;
	WIN_METRICS				wm;
	STATUS					s;

	DbgTttViewGestureSetSquare(("hot=[%ld %ld]",pGesture->hotPoint))

	//
	// Compute row and col
	//
	ObjCallJmp(msgWinGetMetrics, self, &wm, s, Error);
	if (pGesture->hotPoint.x < (wm.bounds.size.w / 3)) {
		set.col = 0;
	} else if (pGesture->hotPoint.x < (2 * (wm.bounds.size.w / 3))) {
		set.col = 1;
	} else {
		set.col = 2;
	}
	if (pGesture->hotPoint.y < (wm.bounds.size.h / 3)) {
		set.row = 0;
	} else if (pGesture->hotPoint.y < (2 * (wm.bounds.size.h / 3))) {
		set.row = 1;
	} else {
		set.row = 2;
	}

	//
	// Set new square value.
	//
	set.value = value;
	ObjCallJmp(msgViewGetDataObject, self, &dataObject, s, Error);
	ObjCallJmp(msgTttDataSetSquare, dataObject, &set, s, Error);

	DbgTttViewGestureSetSquare(("return stsOK"))
	return stsOK;

Error:
	DbgTttViewGestureSetSquare(("Error; returns 0x%lx",s))
	return s;
} /* TttViewGestureSetSquare */


/****************************************************************************
	TttViewInitAndRestoreCommon
	
	Has code common to Init and Restore
****************************************************************************/
#define DbgTttViewInitAndRestoreCommon(x) \
	TttDbgHelper("TttViewInitAndRestoreCommon",tttViewDbgSet,0x4,x)

STATIC STATUS PASCAL
TttViewInitAndRestoreCommon(
	VIEW	   		self,
	P_TTT_VIEW_INST	pInst)
{
	STATUS			s;

	DbgTttViewInitAndRestoreCommon(("self=0x%lx",self))

	//
	// Initialize for Error Recovery
	//
	pInst->dc = objNull;

	//
	// Recreate the dc
	//
	StsJmp(TttViewCreateDC(self, &(pInst->dc)), s, Error);

	DbgTttViewInitAndRestoreCommon(("return stsOK"))
	return stsOK;

Error:
	DbgTttViewInitAndRestoreCommon(("Error; returns 0x%lx",s))
	return s;
} /* TttViewInitAndRestoreCommon */


/****************************************************************************
	TttViewSingleKey

	Utility routine to handle single key.
	
	Actions:
		* x sets upper left cell to X
		* y sets upper left cell to Y
		* space sets upper left cell to Blank
****************************************************************************/
#define DbgTttViewSingleKey(x) \
	TttDbgHelper("TttViewSingleKey",tttViewDbgSet,0x8,x)

STATIC STATUS PASCAL
TttViewSingleKey(
	OBJECT				self,
	U16					keyCode)
{
	TTT_DATA_SET_SQUARE	set;
	OBJECT	   			dataObject;
	BOOLEAN	   			inputIgnored;
	STATUS	   			s;

	DbgTttViewSingleKey(("keyCode=%ld=%c", (U32)keyCode, (U8)keyCode))

	ObjCallJmp(msgViewGetDataObject, self, &dataObject, s, Error);
	inputIgnored = TRUE;
	if ((keyCode == 'x') OR (keyCode == 'X')) {
		inputIgnored = FALSE;
		set.value = tttX;
	} else if ((keyCode == 'o') OR (keyCode == 'O')) {
		inputIgnored = FALSE;
		set.value = tttO;
	} else if (keyCode == ' ') {
		inputIgnored = FALSE;
		set.value = tttBlank;
	}

	if (inputIgnored) {
		s = stsInputIgnored;
	} else {
		set.row = 2;
		set.col = 0;
		ObjCallJmp(msgTttDataSetSquare, dataObject, &set, s, Error);
		s = stsInputTerminate;
	}

	DbgTttViewSingleKey(("return 0x%lx",(U32)s))
	return s;

Error:
	DbgTttViewSingleKey(("Error; return 0x%lx",s))
	return s;
} /* TttViewSingleKey */


/****************************************************************************
	TttViewKeyInput

	Utility routine that handles all clsKey input events.
	Assumes ancestor is not interested in keyboard input.

	Note that one and only one of msgKeyMulti and msgKeyChar should be
	handled.
****************************************************************************/
#define DbgTttViewKeyInput(x) \
	TttDbgHelper("TttViewKeyInput",tttViewDbgSet,0x10,x)

STATIC STATUS PASCAL
TttViewKeyInput(
	OBJECT			self, 
	P_INPUT_EVENT	pArgs)
{
	STATUS			s;

	DbgTttViewKeyInput(("self=0x%lx",self))

	ASSERT((ClsNum(pArgs->devCode) == ClsNum(clsKey)), \
			"KeyInput gets wrong cls");

	if (MsgNum(pArgs->devCode) == MsgNum(msgKeyMulti)) {

		U16 i;
		U16 j;
		P_KEY_DATA pKeyData = (P_KEY_DATA)(pArgs->eventData);
		for (i=0; i < pKeyData->repeatCount; i++) {
			for (j=0; j < pKeyData->multi[i].repeatCount; j++) {
				s = TttViewSingleKey(self, pKeyData->multi[i].keyCode);
				if (s < stsOK) {
					break;
				}
			}
			if (s < stsOK) {
				break;
			}
		}

	} else {
		s = stsInputIgnored;
	}

	DbgTttViewKeyInput(("return 0x%lx",s))
	return s;

} /* TttViewKeyInput */

/****************************************************************************
	TttViewComputeSizes
****************************************************************************/
#define DbgTttViewComputeSizes(x) \
	TttDbgHelper("TttViewComputeSizes",tttViewDbgSet,0x40,x)

#define FONT_SIZE_FUDGE 5

STATIC void PASCAL
TttViewComputeSizes(
	P_TTT_VIEW_SIZES	p,
	PP_TTT_VIEW_INST	pData,
	P_RECT32			pBounds)	// window bounds
{
	U32					thickness;
	S16					t;

	DbgTttViewComputeSizes(("bounds=[%ld %ld %ld %ld]", *pBounds))

	thickness = Max(1L, (*pData)->metrics.lineThickness);
	p->normalBoxWidth = Max(1L, (pBounds->size.w - (2 * thickness)) / 3);
	p->normalBoxHeight = Max(1L, (pBounds->size.h - (2 * thickness)) / 3);

	//
	// x and width of horiztonal stripes
	//
	p->horizLines[0].origin.x = 
	p->horizLines[1].origin.x = 0;
	p->horizLines[0].size.w = 
	p->horizLines[1].size.w = Max(1L, pBounds->size.w);

	//
	// y and height of vertical stripes
	//
	p->vertLines[0].origin.y = 
	p->vertLines[1].origin.y = 0;
	p->vertLines[0].size.h = 
	p->vertLines[1].size.h = Max(1L, pBounds->size.h);

	//
	// x and width of left column.
	//
	p->r[0][0].origin.x =
	p->r[1][0].origin.x =
	p->r[2][0].origin.x = 0;
	p->r[0][0].size.w =
	p->r[1][0].size.w =
	p->r[2][0].size.w = p->normalBoxWidth;

	//
	// x and width of left vertical stripe.
	//
	p->vertLines[0].origin.x = p->r[0][0].size.w;
	p->vertLines[0].size.w = thickness;

	//
	// x and width of middle column.
	//
	p->r[0][1].origin.x =
	p->r[1][1].origin.x =
	p->r[2][1].origin.x = p->vertLines[0].origin.x + p->vertLines[0].size.w;
	p->r[0][1].size.w =
	p->r[1][1].size.w =
	p->r[2][1].size.w = p->normalBoxWidth;

	//
	// x and width of right vertical stripe.
	//
	p->vertLines[1].origin.x = p->r[0][1].origin.x + p->r[0][1].size.w;
	p->vertLines[1].size.w = thickness;

	//
	// x and width of right column.  Accumlate all extra width here.
	//
	p->r[0][2].origin.x =
	p->r[1][2].origin.x =
	p->r[2][2].origin.x = p->vertLines[1].origin.x + p->vertLines[1].size.w;
	p->r[0][2].size.w =
	p->r[1][2].size.w =
	p->r[2][2].size.w = Max(1L, (pBounds->size.w -
			(p->vertLines[0].size.w + p->vertLines[1].size.w +
			p->r[0][0].size.w + p->r[0][1].size.w)));

	//
	// y and height of bottom row.
	//
	p->r[0][0].origin.y = 
	p->r[0][1].origin.y = 
	p->r[0][2].origin.y = 0;
	p->r[0][0].size.h = 
	p->r[0][1].size.h = 
	p->r[0][2].size.h = p->normalBoxHeight;

	//
	// y and height of bottom horizontal stripe.
	//
	p->horizLines[0].origin.y = p->r[0][0].size.h;
	p->horizLines[0].size.h = thickness;

	//
	// y and height of middle row.
	//
	p->r[1][0].origin.y = 
	p->r[1][1].origin.y = 
	p->r[1][2].origin.y = p->horizLines[0].origin.y + p->horizLines[0].size.h;
	p->r[1][0].size.h = 
	p->r[1][1].size.h = 
	p->r[1][2].size.h = p->normalBoxHeight;

	//
	// y and height of top horizontal stripe.
	//
	p->horizLines[1].origin.y = p->r[1][0].origin.y + p->r[1][0].size.h;
	p->horizLines[1].size.h = thickness;

	//
	// y and height of top row.  Accumulate all extra height here.
	//
	p->r[2][0].origin.y = 
	p->r[2][1].origin.y = 
	p->r[2][2].origin.y = p->horizLines[1].origin.y + p->horizLines[1].size.h;
	p->r[2][0].size.h = 
	p->r[2][1].size.h = 
	p->r[2][2].size.h = Max(1L, (pBounds->size.h -
			(p->horizLines[0].size.h + p->horizLines[1].size.h +
			p->r[0][0].size.h + p->r[1][0].size.h)));

	//
	// Compute font scale info.
	//
	if ((p->normalBoxWidth - FONT_SIZE_FUDGE) > 0) {
		t = (S16)(p->normalBoxWidth - FONT_SIZE_FUDGE);
	} else {
		t = 0;
	}
	p->scale.x = FxMakeFixed(t, 0);
	if ((p->normalBoxHeight - FONT_SIZE_FUDGE) > 0) {
		t = (S16)(p->normalBoxHeight - FONT_SIZE_FUDGE);
	} else {
		t = 0;
	}
	p->scale.y = FxMakeFixed(t, 0);

	DbgTttViewComputeSizes(("nBW=%ld nBH=%ld",p->normalBoxWidth, p->normalBoxHeight))
	DbgTttViewComputeSizes(("vert [%ld %ld %ld %ld]  [%ld %ld %ld %ld]",
			p->vertLines[0], p->vertLines[1]));
	DbgTttViewComputeSizes(("horiz [%ld %ld %ld %ld]  [%ld %ld %ld %ld]",
			p->horizLines[0], p->horizLines[1]));
	DbgTttViewComputeSizes((
			"r0 [%ld %ld %ld %ld]  [%ld %ld %ld %ld]  [%ld %ld %ld %ld]",
			p->r[0][0], p->r[0][1], p->r[0][2]));
	DbgTttViewComputeSizes((
			"r1 [%ld %ld %ld %ld]  [%ld %ld %ld %ld]  [%ld %ld %ld %ld]",
			p->r[1][0], p->r[1][1], p->r[1][2]));
	DbgTttViewComputeSizes((
			"r2 [%ld %ld %ld %ld]  [%ld %ld %ld %ld]  [%ld %ld %ld %ld]",
			p->r[2][0], p->r[2][1], p->r[2][2]));
} /* TttViewComputeSizes */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Message Handlers							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	TttViewNewDefaults
	
	Respond to msgNewDefaults.
****************************************************************************/
#define DbgTttViewNewDefaults(x) \
	TttDbgHelper("TttViewNewDefaults",tttViewDbgSet,0x80,x)

MsgHandlerWithTypes(TttViewNewDefaults, P_TTT_VIEW_NEW, PP_TTT_VIEW_INST)
{
	DbgTttViewNewDefaults(("self=0x%lx",self))

	pArgs->win.flags.input |= inputHoldTimeout;
	pArgs->gWin.helpId = tagTttQHelpForView;
	pArgs->embeddedWin.style.moveable = true;
	pArgs->embeddedWin.style.copyable = true;
	pArgs->view.createDataObject = true;
	pArgs->tttView.lineThickness = 5L;
	pArgs->tttView.spare1 = 0;
	pArgs->tttView.spare2 = 0;

	DbgTttViewNewDefaults(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttViewNewDefaults */


/****************************************************************************
	TttViewInit
	
	Initialize instance data of new object.

	Note: clsmgr has already initialized instance data to zeros.
****************************************************************************/
#define DbgTttViewInit(x) \
	TttDbgHelper("TttViewInit",tttViewDbgSet,0x100,x)

MsgHandlerWithTypes(TttViewInit, P_TTT_VIEW_NEW, PP_TTT_VIEW_INST)
{
	P_TTT_VIEW_INST	pInst;
	TTT_DATA_NEW  	tttDataNew;
	STATUS 		  	s;
	BOOLEAN		  	responsibleForDataObject;

	DbgTttViewInit(("self=0x%lx",self))

	//
	// Initialize for error recovery
	//
	pInst = pNull;
	tttDataNew.object.uid = objNull;
	responsibleForDataObject = false;

	//
	// Allocate instance data and initialize those parts of it
	// that come from pArgs.
	//
	StsJmp(OSHeapBlockAlloc(osProcessHeapId, SizeOf(*pInst), &pInst), \
			s, Error);
	pInst->metrics.lineThickness = pArgs->tttView.lineThickness;

	//
	// Create the data object, if appropriate.
	//
	if ((pArgs->view.dataObject == Nil(OBJECT)) AND
			(pArgs->view.createDataObject)) {
		ObjCallJmp(msgNewDefaults, clsTttData, &tttDataNew, s, Error);
		ObjCallJmp(msgNew, clsTttData, &tttDataNew, s, Error);
		responsibleForDataObject = true;
		pArgs->view.createDataObject = false;
		pArgs->view.dataObject = tttDataNew.object.uid;
	}

	//
	// Now let ancestor finish initializing self.
	// clsView will make self an observer of the data object.
	//
	ObjCallAncestorCtxJmp(ctx, s, Error);
	responsibleForDataObject = false;

	//
	// Use the utility routine to handle things common to Init and Restore.
	//
	StsJmp(TttViewInitAndRestoreCommon(self, pInst), s, Error);

	ObjectWrite(self, ctx, &pInst);
	DbgTttViewInit(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (responsibleForDataObject AND tttDataNew.object.uid) {
		ObjCallWarn(msgDestroy, tttDataNew.object.uid, pNull);
		// clsView will notice that the data object has been destroyed
		// and will update its instance data, so no need to ObjectWrite.
	}
	if (pInst) {
		OSHeapBlockFree(pInst);
	}
	DbgTttViewInit(("Error; returns 0x%lx",s))
	return s;
} /* TttViewInit */


/****************************************************************************
	TttViewFree
	
	Respond to msgFree.

	Note:  Always return stsOK, even if a problem occurs.  This is
	(1) because there's nothing useful to do if a problem occurs anyhow
	and (2) because the ancestor is called after this function if and
	only if stsOK is returned, and it's important that the ancestor
	get called.
****************************************************************************/
#define DbgTttViewFree(x) \
	TttDbgHelper("TttViewFree",tttViewDbgSet,0x200,x)

MsgHandlerWithTypes(TttViewFree, P_ARGS, PP_TTT_VIEW_INST)
{
	OBJECT	dataObject;

	DbgTttViewFree(("self=0x%lx",self))

	if ((*pData)->dc) {
		ObjCallWarn(msgDestroy, (*pData)->dc, pNull);
	}

	if (ObjCallWarn(msgViewGetDataObject, self, &dataObject) >= stsOK) {
		if (dataObject) {
			ObjCallWarn(msgViewSetDataObject, self, objNull);
			ObjCallWarn(msgDestroy, dataObject, pNull);
		}
	}

	OSHeapBlockFree(*pData);

	DbgTttViewFree(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttViewFree */


/****************************************************************************
	TttViewSave
	
	Save self to a file.
****************************************************************************/
#define DbgTttViewSave(x) \
	TttDbgHelper("TttViewSave",tttViewDbgSet,0x400,x)

MsgHandlerWithTypes(TttViewSave, P_OBJ_SAVE, PP_TTT_VIEW_INST)
{
	TTT_VIEW_FILED_0	filed;
	STATUS 				s;

	DbgTttViewSave(("self=0x%lx",self))

	StsJmp(TttUtilWriteVersion(pArgs->file, CURRENT_VERSION), s, Error);
	TttViewFiledData0FromInstData(*pData, &filed);
	StsJmp(TttUtilWrite(pArgs->file, SizeOf(filed), &filed), s, Error);

	DbgTttViewSave(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewSave(("Error; return 0x%lx",s))
	return s;
} /* TttViewSave */


/****************************************************************************
	TttViewRestore
	
	Restore self from a file.

	Note: clsmgr has already initialized instance data to zeros.
****************************************************************************/
#define DbgTttViewRestore(x) \
	TttDbgHelper("TttViewRestore",tttViewDbgSet,0x800,x)

MsgHandlerWithTypes(TttViewRestore, P_OBJ_RESTORE, PP_TTT_VIEW_INST)
{
	P_TTT_VIEW_INST		pInst;
	TTT_VIEW_FILED_0  	filed;
	TTT_VERSION		  	version;
	STATUS 			  	s;

	DbgTttViewRestore(("self=0x%lx",self))

	//
	// Initialize for error recovery.
	//
	pInst = pNull;

	//
	// Read version, then read filed data.  (Currently there's only
	// only one legitimate file format, so no checking of the version
	// need be done.)
	//
	// The allocate instance data and convert filed data.
	//
	StsJmp(TttUtilReadVersion(pArgs->file, MIN_VERSION, MAX_VERSION, \
			&version), s, Error);
	StsJmp(TttUtilRead(pArgs->file, SizeOf(filed), &filed), s, Error);
	StsJmp(OSHeapBlockAlloc(osProcessHeapId, SizeOf(*pInst), &pInst), \
			s, Error);
	TttViewInstDataFromFiledData0(&filed, pInst);

	//
	// Use the utility routine to handle things common to Init and Restore.
	//
	StsJmp(TttViewInitAndRestoreCommon(self, pInst), s, Error);

	ObjectWrite(self, ctx, &pInst);
	DbgTttViewRestore(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (pInst) {
		OSHeapBlockFree(pInst);
	}
	DbgTttViewRestore(("Error; return 0x%lx",s))
	return s;
} /* TttViewRestore */


/****************************************************************************
	TttViewDump
	
	Respond to msgDump.
****************************************************************************/

#ifdef DEBUG

MsgHandlerWithTypes(TttViewDump, P_ARGS, PP_TTT_VIEW_INST)
{
	Debugf("TttViewDump: dc=0x%lx lineThickness=%ld",
			(*pData)->dc,(U32)((*pData)->metrics.lineThickness));
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttViewDump */

#endif // DEBUG


/****************************************************************************
	TttViewDataChanged
	
	Respond to changes in viewed object.
****************************************************************************/
#define DbgTttViewDataChanged(x) \
	TttDbgHelper("TttViewDataChanged",tttViewDbgSet,0x1000,x)

MsgHandlerWithTypes(TttViewDataChanged, P_TTT_DATA_CHANGED, PP_TTT_VIEW_INST)
{
	STATUS s;

	DbgTttViewDataChanged(("self=0x%lx pArgs=0x%lx",self,pArgs))

	if (pArgs == pNull) {
		ObjCallJmp(msgWinDirtyRect, self, pNull, s, Error);
	} else {
		WIN_METRICS		wm;
		TTT_VIEW_SIZES sizes;
		DbgTttViewDataChanged(("row=%ld col=%ld",(U32)(pArgs->row), \
				(U32)(pArgs->col)))
		ObjCallJmp(msgWinGetMetrics, (*pData)->dc, &wm, s, Error);
		TttViewComputeSizes(&sizes, pData, &(wm.bounds));
		ObjCallJmp(msgWinDirtyRect, (*pData)->dc, \
				&(sizes.r[pArgs->row][pArgs->col]), s, Error);
	}

	DbgTttViewDataChanged(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewDataChanged(("Error; return 0x%lx",s))
	return s;
} /* TttViewDataChanged */


/****************************************************************************
	TttViewRepaint
	
	Respond to msgRepaint.

	This handler demonstrates how to do "smart repainting." It asks the
	window manager for the "dirty" rectangle, and then only repaints those
	objects that intersect with the rectangle. 

	Smart repainting should be used by applications that have expensive
	repainting procedures. Although Tic-Tac-Toe's repainting is not even
	close to being prohibitive, it uses smart repainting for the sake
	of demonstrating how to do it.

	Instead of using smart repainting, many applications simply redraw
	their entire window. For an example of this approach, take a look
	at Hello World (Custom Window).
****************************************************************************/
#define DbgTttViewRepaint(x) \
	TttDbgHelper("TttViewRepaint",tttViewDbgSet,0x2000,x)

MsgHandlerWithTypes(TttViewRepaint, P_ARGS, PP_TTT_VIEW_INST)
{
	TTT_VIEW_SIZES		sizes;
	SYSDC_TEXT_OUTPUT	tx;
	XY32			  	sizeX;
	XY32			  	sizeO;
	OBJECT				dataObject;
	BOOLEAN				endRepaintNeeded;
	WIN_METRICS			wm;
	RECT32				dirtyRect;
	TTT_DATA_METRICS	dm;
	U16					row;
	U16					col;
	U16					i;
	STATUS				s;
	BOOLEAN 			drawIt;

	DbgTttViewRepaint(("self=0x%lx",self))

	//
	// General initialization and intialization for error recovery.
	// Also collect miscellaneous info needed to paint.
	//
	endRepaintNeeded = false;
	ObjCallJmp(msgViewGetDataObject, self, &dataObject, s, Error);
	ObjCallJmp(msgTttDataGetMetrics, dataObject, &dm, s, Error);
	ObjCallJmp(msgWinGetMetrics, (*pData)->dc, &wm, s, Error);
	TttViewComputeSizes(&sizes, pData, &(wm.bounds));

	//
	// Must do msgWinBeginRepaint before any painting starts, and
	// to get dirtyRect.
	//
	ObjCallJmp(msgWinBeginRepaint, (*pData)->dc, &dirtyRect, s, Error);
	endRepaintNeeded = true;

	// ImagePoint ROUNDS from LWC to LUC, but DrawRectangle TRUNCATES.
	// Therefore, increase the size of the dirtyRect so as to be sure to
	// cover all the pixels that need to be painted.
	// Another solution would be to use finer LUC than points.
	dirtyRect.origin.x--;
	dirtyRect.origin.y--;
	dirtyRect.size.w += 2;
	dirtyRect.size.h += 2;

	//
	// Fill the dirty rect with the appropriate background.  If we hold the
	// selection, the appropriate background is grey, otherwise it is white.
	//
	s = ObjectCall(msgSelIsSelected, self, pNull);
	if (s == stsOK) {
		DbgTttViewRepaint(("self is selected"))
		ObjectCall(msgDcSetBackgroundRGB, (*pData)->dc, \
				(P_ARGS)sysDcRGBGray33);
	} else {
		DbgTttViewRepaint(("self is not selected"))
		ObjectCall(msgDcSetBackgroundRGB, (*pData)->dc, \
				(P_ARGS)sysDcRGBWhite);
	} 
	
	ObjectCall(msgDcSetFillPat, (*pData)->dc, (P_ARGS)sysDcPatBackground);
	ObjectCall(msgDcSetLineThickness, (*pData)->dc, (P_ARGS)0L);
	ObjectCall(msgDcDrawRectangle, (*pData)->dc, &dirtyRect);
	ObjectCall(msgDcSetFillPat, (*pData)->dc, (P_ARGS)sysDcPatForeground);

	//
	// Paint the vertical lines
	//
	for (i=0; i<2; i++) {
		if (Rect32sIntersect(&dirtyRect, &(sizes.vertLines[i]))) {
			DbgTttViewRepaint(("vertical i=%ld; overlap",(U32)i))
			ObjectCall(msgDcDrawRectangle, (*pData)->dc, \
					&(sizes.vertLines[i]));
		} else {
			DbgTttViewRepaint(("vertical i=%ld; no overlap",(U32)i))
		}
	}

	//
	// Paint the horizontal lines
	//
	for (i=0; i<2; i++) {
		if (Rect32sIntersect(&dirtyRect, &(sizes.horizLines[i]))) {
			DbgTttViewRepaint(("horizontal i=%ld; overlap",(U32)i))
			ObjectCall(msgDcDrawRectangle, (*pData)->dc, \
					&(sizes.horizLines[i]));
		} else {
			DbgTttViewRepaint(("horizontal i=%ld; no overlap",(U32)i))
		}
	}

	//
	// Scale the font to the box size.
	//
	// Note: This could be done once when the window size
	// changes rather than each time the window is painted.
	//
	ObjCallJmp(msgDcIdentityFont, (*pData)->dc, pNull, s, Error);
	ObjCallJmp(msgDcScaleFont, (*pData)->dc, &(sizes.scale), s, Error);

	//
	// Measure X and O in the font.
	//
	TttUtilInitTextOutput(&tx, sysDcAlignChrTop, pNull);
	tx.pText = "X";
	tx.lenText = strlen(tx.pText);
	ObjectCall(msgDcMeasureText, (*pData)->dc, (P_ARGS)&tx);
	sizeX = tx.cp;
	DbgTttViewRepaint(("measure X=[%ld %ld]",sizeX.x, sizeX.y))
	TttUtilInitTextOutput(&tx, sysDcAlignChrTop, pNull);
	tx.pText = "O";
	tx.lenText = strlen(tx.pText);
	ObjectCall(msgDcMeasureText, (*pData)->dc, (P_ARGS)&tx);
	sizeO = tx.cp;
	DbgTttViewRepaint(("measure O=[%ld %ld]",sizeO.x, sizeO.y))

	//
	// Paint the cells.
	//
	for (row=0; row<3; row++) {
		for (col=0; col<3; col++) {
			if (Rect32sIntersect(&dirtyRect, &(sizes.r[row][col]))) {
				DbgTttViewRepaint(("row=%ld col=%ld; overlap", \
						(U32)row, (U32)col));
				if (dm.squares[row][col] == tttX) {
					drawIt = TRUE;
					tx.pText = "X";
					tx.lenText = strlen(tx.pText);
					tx.cp.x = sizes.r[row][col].origin.x +
							((sizes.r[row][col].size.w - sizeX.x) / 2);
					tx.cp.y = sizes.r[row][col].origin.y + sizeX.y +
							((sizes.r[row][col].size.h - sizeX.y) / 2);
				} else if (dm.squares[row][col] == tttO) {
					drawIt = TRUE;
					tx.pText = "O";
					tx.lenText = strlen(tx.pText);
					tx.cp.x = sizes.r[row][col].origin.x +
							((sizes.r[row][col].size.w - sizeO.x) / 2);
					tx.cp.y = sizes.r[row][col].origin.y + sizeO.y +
							((sizes.r[row][col].size.h - sizeO.y) / 2);
				} else {
					DbgTttViewRepaint(("blank cell"))
					drawIt = FALSE;
				}

				if (drawIt) {
					ObjCallJmp(msgDcDrawText, (*pData)->dc, &tx, s, Error);
				}
			} else {
				DbgTttViewRepaint(("row=%ld col=%ld; no overlap", \
						(U32)row, (U32)col));
			}
		}
	}

	//
	// Balance the msgWinBeginRepaint
	//
	ObjCallWarn(msgWinEndRepaint, (*pData)->dc, pNull);

	DbgTttViewRepaint(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	if (endRepaintNeeded) {
		ObjCallWarn(msgWinEndRepaint, (*pData)->dc, pNull);
	}
	DbgTttViewRepaint(("Error; return 0x%lx",s))
	return s;
} /* TttViewRepaint */


/****************************************************************************
	TttViewGetDesiredSize
	
	Respond to msgGetDesiredSize.

	The desired size is an appropriate minimum size for the drawing.
****************************************************************************/
#define DbgTttViewGetDesiredSize(x) \
	TttDbgHelper("TttViewGetDesiredSize",tttViewDbgSet,0x2000,x)

MsgHandlerArgType(TttViewGetDesiredSize, P_WIN_METRICS)
{
	pArgs->bounds.size.w	= desiredWidth;
	pArgs->bounds.size.h	= desiredHeight;

	DbgTttViewGetDesiredSize(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttViewGetDesiredSize */

/****************************************************************************
	TttViewGesture
	
	Let ancestor handle unrecognized gestures.
****************************************************************************/
#define DbgTttViewGesture(x) \
	TttDbgHelper("TttViewGesture",tttViewDbgSet,0x8000,x)

MsgHandlerWithTypes(TttViewGesture, P_GWIN_GESTURE, PP_TTT_VIEW_INST)
{
	STATUS		s;
//	OBJECT		owner;

#ifdef DEBUG
	{
		P_CLS_SYMBUF	mb;
		DbgTttViewGesture(("self=0x%lx msg=0x%lx %s", self, pArgs->msg,
				ClsMsgToString(pArgs->msg,mb)))
	}
#endif // DEBUG

	switch(ClsNum(pArgs->msg)) {

		case ClsNum(clsXGesture):

			switch(MsgNum(pArgs->msg)) {

				case MsgNum(xgs1Tap):
					ObjCallJmp(msgTttViewToggleSel, self, pNull, s, Error);
					break;

				case MsgNum(xgsCross):
					StsJmp(TttViewGestureSetSquare(self, pArgs, tttX), s, Error);
					break;

				case MsgNum(xgsCircle):
					StsJmp(TttViewGestureSetSquare(self, pArgs, tttO), s, Error);
					break;

				case MsgNum(xgsPigtailVert):
				case MsgNum(xgsPigtailHorz):
					StsJmp(TttViewGestureSetSquare(self, pArgs, tttBlank), \
							s, Error);
					break;

				case MsgNum(xgsCheck):
				case MsgNum(xgsUGesture):
					// Make sure there is a selection.
					s = ObjectCall(msgSelIsSelected, self, pNull);
					if (s == stsNoMatch) {
						ObjCallJmp(msgTttViewTakeSel, self, pNull, s, Error);
						ObjCallJmp(msgWinUpdate, self, pNull, s, Error);
					}
					// Then call the ancestor.
					ObjCallAncestorCtxJmp(ctx, s, Error);
					break;

				default:
					DbgTttViewGesture(("Letting ancestor handle gesture"))
					return ObjCallAncestorCtxWarn(ctx);
			}
			break;

		default:
			DbgTttViewGesture(("Letting ancestor handle gesture"))
			return ObjCallAncestorCtxWarn(ctx);
	}

	DbgTttViewGesture(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewGesture(("Error; return 0x%lx",s))
	return s;
} /* TttViewGesture */

/****************************************************************************
	TttViewSelYield
	
	 msgSelYield from selection manager.
****************************************************************************/
#define DbgTttViewSelYield(x) \
	TttDbgHelper("TttViewSelYield",tttViewDbgSet,0x10000,x)

MsgHandlerWithTypes(TttViewSelYield, P_ARGS, PP_TTT_VIEW_INST)
{
	STATUS		s;

	DbgTttViewSelYield(("self=0x%lx",self))

	StsJmp(TttViewNeedRepaint(self), s, Error);

	DbgTttViewSelYield(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewSelYield(("Error; return 0x%lx",s))
	return s;
} /* TttViewSelYield */


/****************************************************************************
	TttViewSelDelete

	In this particular application, deleting is a poorly defined concept.
	Rather than do nothing, though, we clear the board.
****************************************************************************/
#define DbgTttViewSelDelete(x) \
	TttDbgHelper("TttViewSelDelete",tttViewDbgSet,0x80000,x)

MsgHandlerWithTypes(TttViewSelDelete, P_ARGS, PP_TTT_VIEW_INST)
{
	TTT_DATA_METRICS	dm;
	OBJECT				dataObject;
	U16					row;
	U16					col;
	STATUS				s;

	DbgTttViewSelDelete((""))

	ObjCallJmp(msgViewGetDataObject, self, &dataObject, s, Error);
	ObjCallJmp(msgTttDataGetMetrics, dataObject, &dm, s, Error);
	for (row=0; row<3; row++) {
		for (col=0; col<3; col++) {
			dm.squares[row][col] = tttBlank;
		}
	}
	dm.undoTag = tagTttDataUndoDelete;
	ObjCallJmp(msgTttDataSetMetrics, dataObject, &dm, s, Error);

	DbgTttViewSelDelete(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewSelDelete(("Error; return 0x%lx",s))
	return s;
} /* TttViewSelDelete */


/****************************************************************************
	TttViewGetMetrics
****************************************************************************/
#define DbgTttViewGetMetrics(x) \
	TttDbgHelper("TttViewGetMetrics",tttViewDbgSet,0x100000,x)

MsgHandlerWithTypes(TttViewGetMetrics, P_TTT_VIEW_METRICS, PP_TTT_VIEW_INST)
{
	DbgTttViewGetMetrics(("self=0x%lx",self))

	*pArgs = (*pData)->metrics;

	DbgTttViewGetMetrics(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttViewGetMetrics */


/****************************************************************************
	TttViewSetMetrics
****************************************************************************/
#define DbgTttViewSetMetrics(x) \
	TttDbgHelper("TttViewSetMetrics",tttViewDbgSet,0x200000,x)

MsgHandlerWithTypes(TttViewSetMetrics, P_TTT_VIEW_METRICS, PP_TTT_VIEW_INST)
{
	DbgTttViewSetMetrics(("self=0x%lx",self))

	(*pData)->metrics = *pArgs;
	TttViewNeedRepaint(self);

	DbgTttViewSetMetrics(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* TttViewSetMetrics */


/****************************************************************************
	TttViewToggleSel

	 msgTttViewToggleSel
****************************************************************************/
#define DbgTttViewToggleSel(x) \
	TttDbgHelper("TttViewToggleSel",tttViewDbgSet,0x400000,x)

MsgHandlerWithTypes(TttViewToggleSel, P_ARGS, PP_TTT_VIEW_INST)
{
	STATUS		s;

	DbgTttViewToggleSel(("self=0x%lx",self))

	s = ObjectCall(msgSelIsSelected, self, pNull);
	if (s == stsOK) {
		DbgTttViewToggleSel(("View is selected; deselect it"))
		ObjCallJmp(msgSelSetOwner, theSelectionManager, pNull, s, Error);
		if (self == InputGetTarget()) {
			StsJmp(InputSetTarget(objNull, inputAllRealEventsFlags), s, Error);
		}
	} else {
		DbgTttViewToggleSel(("View is not selected; select it"))
		ObjCallJmp(msgSelSelect, self, pNull, s, Error);
	}

	StsJmp(TttViewNeedRepaint(self), s, Error);

	DbgTttViewToggleSel(("return stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewToggleSel(("Error; return 0x%lx",s))
	return s;
} /* TttViewToggleSel */


/****************************************************************************
	TttViewTakeSel
****************************************************************************/
#define DbgTttViewTakeSel(x) \
	TttDbgHelper("TttViewTakeSel",tttViewDbgSet,0x800000,x)

MsgHandlerWithTypes(TttViewTakeSel, P_ARGS, PP_TTT_VIEW_INST)
{
	STATUS	s;

	DbgTttViewTakeSel(("self=0x%lx",self))

	s = ObjectCall(msgSelIsSelected, self, pNull);
	if (s == stsNoMatch) {
		DbgTttViewTakeSel(("self is not selected; taking"))
		ObjCallJmp(msgSelSelect, self, pNull, s, Error);
		StsJmp(TttViewNeedRepaint(self), s, Error);
	} else {
		DbgTttViewTakeSel(("self is already selected; doing nothing"))
	}

	DbgTttViewTakeSel(("returns stsOK"))
	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	DbgTttViewTakeSel(("Error; return 0x%lx",s))
	return s;

} /* TttViewTakeSel */


/****************************************************************************
	TttViewInputEvent

	 msgInputEvent.
****************************************************************************/
#define DbgTttViewInputEvent(x) \
	TttDbgHelper("TttViewInputEvent",tttViewDbgSet,0x1000000,x)

MsgHandlerWithTypes(TttViewInputEvent, P_INPUT_EVENT, PP_TTT_VIEW_INST)
{
	STATUS		s;

	switch (ClsNum(pArgs->devCode)) {

		case ClsNum(clsKey):
			s = TttViewKeyInput(self, pArgs);
			break;

		default:
			s = ObjectCallAncestorCtx(ctx);
			break;

	}

	return s;
	MsgHandlerParametersNoWarning;
} /* TttViewInputEvent */


/****************************************************************************
	TttViewSelSelect

	 msgSelSelect.
****************************************************************************/
MsgHandler(TttViewSelSelect)
{
	STATUS	s;

	// 
	// If the view is not selected, force it to repaint.
	// (This code is needed for the move/copy protocol; otherwise,
	// tapping or press-tapping on a ttt board does not highlight
	// the board's selection properly).
	//
	
	s = ObjectCall(msgSelIsSelected, self, pNull);

	if (s == stsNoMatch) {
		StsWarn(TttViewNeedRepaint(self));	
	}

	return stsOK;
	MsgHandlerParametersNoWarning;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Installation 								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	ClsTttViewInit
	
	Install the class.
****************************************************************************/

STATUS PASCAL
ClsTttViewInit (void)
{
	CLASS_NEW		new;
	STATUS			s;

	ObjCallJmp(msgNewDefaults, clsClass, &new, s, Error);
	new.object.uid			= clsTttView;
	new.object.key			= 0;
	new.cls.pMsg			= clsTttViewTable;
	new.cls.ancestor		= clsView;
	new.cls.size			= SizeOf(P_TTT_VIEW_INST);
	new.cls.newArgsSize	= SizeOf(TTT_VIEW_NEW);
	ObjCallJmp(msgNew, clsClass, &new, s, Error);

	return stsOK;

Error:
	return s;
} /* ClsTttViewInit */
