/*
 *      (c) Copyright 1989 Sun Microsystems, Inc. Sun design patents
 *      pending in the U.S. and foreign countries. See LEGAL_NOTICE
 *      file for terms of the license.
 *
 *	Written for Sun Microsystems by Crucible, Santa Cruz, CA.
 */

static	char	sccsid[] = "@(#)events.c 50.5 90/12/12 Crucible";

#ifdef OW_I18N
#include "i18n.h"
#endif
#include <stdio.h>

#include <sys/types.h>
#include <sys/time.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef OW_I18N
#include "ollocale.h"
#endif
#include "olwm.h"
#include "win.h"
#include "globals.h"
#include "events.h"
#include "list.h"
#include "mem.h"

/* ===== externs ========================================================== */

extern int	DeadChildren;
extern void	ReapChildren();

/* ===== globals ========================================================== */

XModifierKeymap *ModMap = NULL;
Time LastEventTime = (Time) 0;

/* ===== private data ===================================================== */

static int (*interposer)() = NULL;
static void *closure;
static List *deferredEventListHead = NULL_LIST;
static List *deferredEventListTail = NULL_LIST;

static struct timeval timeoutNext = 0;
static TimeoutFunc timeoutFunc = NULL;
static void *timeoutClosure = NULL;

/* ===== private functions ================================================ */

/*
 * lookupWindow
 * 
 * Given an event, look up the WinInfo corresponding to the window field of 
 * the event.  For events that have no window field, return NULL.
 */
WinGeneric *
lookupWindow( event )
    XEvent *event;
{
    switch ( event->xany.type ) {
    case KeymapNotify:
    case MappingNotify:
	/* these events have no window field */
	return NULL;
    default:
	return WIGetInfo( event->xany.window );
    }
}


/*
 * saveTimestamp
 *
 * Given an event, update LastEventTime with its timestamp.  For events that 
 * have no timestamp, do nothing.
 */
void
saveTimestamp( event )
    XEvent *event;
{
    switch ( event->xany.type ) {
    case KeyPress:
    case KeyRelease:
	LastEventTime = event->xkey.time;
	break;
    case ButtonPress:
    case ButtonRelease:
	LastEventTime = event->xbutton.time;
	break;
    case MotionNotify:
	LastEventTime = event->xmotion.time;
	break;
    case EnterNotify:
    case LeaveNotify:
	LastEventTime = event->xcrossing.time;
	break;
    case PropertyNotify:
	LastEventTime = event->xproperty.time;
	break;
    case SelectionClear:
	LastEventTime = event->xselectionclear.time;
	break;
    case SelectionRequest:
	/*
	 * The timestamp in SelectionRequest events comes from other clients; 
	 * it's not generated by the server.  It's thus not clear whether we 
	 * should believe any timestamp in one of these events.
	 */
	/* LastEventTime = event->xselectionrequest.time; */
	break;
    case SelectionNotify:
	/*
	 * Ignore SelectionNotify events generated by other clients.
	 * Save only the timestamps from server-generated events.
	 */
	if (!event->xany.send_event)
	    LastEventTime = event->xselection.time;
	break;
    default:
	break;
    }
}

static void
dispatchEvent(dpy, event, winInfo)
Display *dpy;
XEvent *event;
WinGeneric *winInfo;
{
	EvFunc evFunc;

	if (GRV.PrintAll)
	    DebugEvent(event, "Dispatch - debug");

	saveTimestamp( event );

	if (winInfo == NULL)
	{
		/* REMIND should we do anything with these events? */
	    	if (GRV.PrintOrphans)
	    	    DebugEvent(event, "Orphaned (null window)");
		return;
	}

	if (GRV.PrintAll)
	    DebugWindow(winInfo);

#ifdef DEBUG
        if (winInfo->class == NULL)
	{
	    ErrorGeneral("Window instance with NULL class pointer");
	}

	if (winInfo->core.kind != winInfo->class->core.kind)
	    ErrorWarning("Warning:  window with kind different than class kind");
#endif 

	evFunc = WinClass(winInfo)->core.xevents[event->xany.type];
	if (evFunc != NULL)
	{
	    (*evFunc)(dpy, event, winInfo);
	}
	else
	    if (GRV.PrintOrphans)
		DebugEvent(event, "Orphaned");
}

static int
dispatchInterposer(dpy, event)
Display *dpy;
XEvent *event;
{
	int disposition;
	WinGeneric *winInfo;
	XEvent *temp;

	if (GRV.PrintAll)
	    DebugEvent(event, "Dispatch - interposer");

	saveTimestamp( event );
	winInfo = lookupWindow(event);
	disposition = (*interposer)(dpy, event, winInfo, closure);

	switch (disposition) {
	case DISPOSE_DISPATCH:
	    dispatchEvent(dpy, event, winInfo);
	    break;
	case DISPOSE_DEFER:
	    temp = MemNew(XEvent);
	    *temp = *event;
	    if (deferredEventListHead == NULL_LIST) {
		deferredEventListHead = ListCons(temp, NULL_LIST);
		deferredEventListTail = deferredEventListHead;
	    } else {
		deferredEventListTail->next = ListCons(temp, NULL_LIST);
		deferredEventListTail = deferredEventListTail->next;
	    }
	    break;
	case DISPOSE_USED:
	    /* do nothing */
	    break;
	}
}

static void
doTimeout()
{
	TimeoutFunc f;
	void *closure;

	f = timeoutFunc;
	closure = timeoutClosure;
	TimeoutCancel();
	(*f)(closure);
}

static void
nextEventOrTimeout(dpy, event)
Display *dpy;
XEvent *event;
{
	int fd = ConnectionNumber(dpy);
	struct timeval polltime;
	fd_set rdset, wrset, xset;
	int ready = -1;

	while (XPending(dpy) == 0 && ready <= 0 && timeoutFunc != NULL) {
	    gettimeofday(&polltime,NULL);
	    if ((timeoutFunc != NULL) &&
		((polltime.tv_sec > timeoutNext.tv_sec) ||
		 ((polltime.tv_sec == timeoutNext.tv_sec) && 
		  (polltime.tv_usec >= timeoutNext.tv_usec))))
	    {
		doTimeout();
		continue;
	    }

	    polltime.tv_sec = timeoutNext.tv_sec - polltime.tv_sec;
	    polltime.tv_usec = 0;

	    FD_ZERO(&rdset);
	    FD_SET(fd,&rdset);
	    FD_ZERO(&wrset);
	    FD_ZERO(&xset);
	    FD_SET(fd,&xset);

	    ready = select(fd+1,&rdset,&wrset,&xset,&polltime);

	    gettimeofday(&polltime,NULL);
	    if ((timeoutFunc != NULL) &&
		((polltime.tv_sec > timeoutNext.tv_sec) ||
		 ((polltime.tv_sec == timeoutNext.tv_sec) && 
		  (polltime.tv_usec >= timeoutNext.tv_usec))))
	    {
		doTimeout();
	    }
	}

	XNextEvent(dpy, event);
}


void
handleMappingNotify(dpy, e)
    Display *dpy;
    XEvent *e;
{
    void UpdateModifierMap();

    switch (e->xmapping.request) {
    case MappingModifier:
	UpdateModifierMap(dpy);
	break;
    case MappingKeyboard:
	XRefreshKeyboardMapping(e);
	break;
    case MappingPointer:
	/* REMIND - handle mouse button changes */
	break;
    }
}


/* ===== public functions ================================================= */


/*
 * EventLoop
 * 
 * The main event loop.  Reads events from the wire and dispatches them.
 */
void
EventLoop( dpy )
    Display *dpy;
{
	XEvent		event;

	for (;;) {
		if (timeoutFunc == NULL)
		{
		    XNextEvent( dpy, &event );
		} 
		else
		{
		    nextEventOrTimeout(dpy, &event);
		}

		if (DeadChildren > 0)
			ReapChildren();

		/*
		 * Discard user events that have the Synthetic bit set.
		 *
		 * All device events (mouse and keyboard events) have types
		 * that fall in between KeyPress and MotionNotify.
		 */
		if (event.xany.send_event
			&& event.type <= MotionNotify
			&& event.type >= KeyPress)
		    continue;


		/*
		 * Handle MappingNotify events.  These events don't have 
		 * window field, so they can't be dispatched normally.
		 */
		if (event.xany.type == MappingNotify) {
		    handleMappingNotify(dpy, &event);
		    continue;
		}

		/*
		 * If there is an event interposer, and it returns True, that 
		 * means it has successfully processed the event.  We continue 
		 * around the loop instead of processing the event normally.
		 */
		if ( interposer != NULL)
		    dispatchInterposer(dpy, &event);
		else
		    dispatchEvent(dpy, &event, lookupWindow(&event));
	}

	/*NOTREACHED*/
}

int
PropagateEventToParent(dpy,event,win)
Display *dpy;
XEvent *event;
WinGeneric *win;
{
	dispatchEvent(dpy,event,win->core.parent);
}


void
UpdateModifierMap(dpy)
    Display *dpy;
{
    if (ModMap != NULL)
	XFreeModifiermap(ModMap);
    ModMap = XGetModifierMapping(dpy);
}


/*
 * FindModifierMask
 *
 * Given a keycode, look in the modifier mapping table to see if this keycode 
 * is a modifier.  If it is, return the modifier mask bit for this key; 
 * otherwise, return zero.
 */
unsigned int
FindModifierMask(kc)
{
    int i, j;
    KeyCode *mapentry;

    if (ModMap == NULL)
	return 0;

    mapentry = ModMap->modifiermap;
    for (i=0; i<8; ++i) {
	for (j=0; j<(ModMap->max_keypermod); ++j) {
	    if (kc == *mapentry)
		return 1 <<
		    ( (mapentry - ModMap->modifiermap) / 
		      ModMap->max_keypermod );
	    ++mapentry;
	}
    }
    return 0;
}


static void *
redispatchEvent(e)
    XEvent *e;
{
    dispatchEvent(e->xany.display, e, lookupWindow(e));
    MemFree(e);
}


/***************************************************************************
* Interposer functions
***************************************************************************/

/*
 * InstallInterposer
 *
 * Install an event interposition function.  Nested interposition is illegal.
 */
void
InstallInterposer( func, cl )
    Bool (*func)();
    void *cl;
{
    if ( interposer != NULL ) {
#ifdef OW_I18N
	fputs( gettext("olwm: warning, nested event interposer!\n"), stderr );
#else
	fputs( "olwm: warning, nested event interposer!\n", stderr );
#endif
#ifdef DEBUG
	abort();
#endif /* DEBUG */
    }
    interposer = func;
    closure = cl;
}

/*
 * UninstallInterposer
 *
 * Uninstall an event interposition function.
 */
void
UninstallInterposer()
{
    if ( interposer == NULL ) {
#ifdef DEBUG
	fputs( "olwm: warning, no interposer to uninstall.\n", stderr );
#endif /* DEBUG */
    }
    interposer = (int (*)()) NULL;

    if (deferredEventListHead != NULL) {
	ListApply(deferredEventListHead, redispatchEvent, 0);
	ListDestroy(deferredEventListHead);
	deferredEventListHead = deferredEventListTail = NULL_LIST;
    }
}

/***************************************************************************
* Timeout functions
***************************************************************************/

/* TimeoutRequest(t,f,c) -- request that a timeout be generated t seconds
 *	in the future; when the timeout occurs, the function f is called
 *	with the closure c.
 */
void
TimeoutRequest(t,f,c)
int t;
TimeoutFunc f;
void *c;
{
#ifdef DEBUG
	if (timeoutFunc != NULL)
	{
	    ErrorWarning("Timeout being set while another timeout current");
	}
#endif

	gettimeofday(&timeoutNext,NULL);
	timeoutNext.tv_sec += t;
	timeoutFunc = f;
	timeoutClosure = c;
}

/* TimeoutCancel() -- cancel an outstanding timeout.
 */
void 
TimeoutCancel()
{
	timeoutNext.tv_sec = 0;
	timeoutNext.tv_usec = 0;
	timeoutFunc = NULL;
	timeoutClosure = NULL;
}

/***************************************************************************
* Mouse button interpretation
***************************************************************************/

/* the default button mapping */
int     ButtonMap[] = {
                -1,             /* Button Zero */
                MB_SELECT,      /* Button1 */
                MB_ADJUST,      /* Button2 */
                MB_MENU,        /* Button3 */
                -1,             /* Button4 */
                -1,             /* Button5 */
                };


/* MouseButton  -- convert a button press to a logical button
 */
/*ARGSUSED*/    /* dpy arg will be used when multiple Displays supported */
MouseButton(dpy, event)
Display *dpy;
XEvent  *event;
{
        return ButtonMap[event->xbutton.button];
}

/* GetSelectButtonMapping -- return the current MB_SELECT button
 *
 *      Currently unused.
 */
/*ARGSUSED*/    /* dpy arg will be used when multiple Displays supported */
GetSelectButtonMapping(dpy)
Display *dpy;
{
        int             i;
        static int      button = -1;
        static int      found = False;

        if (!found)
        {
                for (i=Button1; i<Button5; i++)
                        if (ButtonMap[i] == MB_SELECT)
                                button = i;

                /* if the user has not specified a Select button,
                 * then use Button1 as a backup.
                 */
                if (button == -1)
                        button = Button1;
        }

        return button;
}

