/*
 * Copyright (c) 1991 Sun Microsystems
 * All rights reserved.
 */

#ident	"@(#)msgfmt.c	1.4	91/09/18 SMI"

#include <string.h>
#include <locale.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/param.h>
#include <signal.h>

#define	DOMAIN_TOKEN	"domain"	/* domain token in po file  */
#define	MSGID_TOKEN	"msgid"		/* msg id token in po file  */
#define	MSGSTR_TOKEN	"msgstr"	/* target str token in po file */
#define	DEFAULT_DOMAIN	"messages"	/* implementation dependent */
					/* default domain name	    */

#define	MAXDOMAIN		257	/* size of domain name string */
#define	MAX_VALUE_LEN		2047	/* size of msg id and target str */

#define	LEAF_INDICATOR	-99		/* Must match with gettext.h */

struct msg_struct {
	char	*msgid;		/* msg id string */
	char	*msgstr;	/* msg target string */
	int	msgid_offset;	/* msg id offset in mo file */
	int	msgstr_offset;	/* msg target string offset in mo file */
	struct  msg_struct *next;	/* next node */
};

struct  domain_struct {
	char			*domain;	/* domain name */
	struct msg_struct	*first_elem;	/* head of msg link list */
	struct msg_struct	*current_elem;	/* most recently used msg */
	struct domain_struct	*next;		/* next domain node */
};

#ifdef __STDC__
static void	read_psffm(char *);
static void	handle_duplicate_msgid (char *);
static void	sortit (char *, char *);
static char	*consume_whitespace(char *);
static char	expand_meta(char **);
static struct domain_struct	*find_domain_node (char *);
static void	insert_message (struct domain_struct *, char *, char *);
static void	output_all_mo_files();
static void	output_one_mo_file(struct domain_struct *);
static void	check_next_char (char *, int);
#else
static void	read_psffm();
static void	handle_duplicate_msgid ();
static void	sortit ();
static char	*consume_whitespace();
static char	expand_meta();
static struct domain_struct	*find_domain_node ();
static void	insert_message ();
static void	output_all_mo_files();
static void	output_one_mo_file();
static void	check_next_char ();
#endif

#ifdef DEBUG
static void	printlist();
#endif

#define	USAGE	"usage: msgfmt [-v] [-o filename] [files ... ]\n"

extern	char *optarg;
extern	int optind;
extern	int opterr;

char	linebuf[MAX_VALUE_LEN];
char	current_domain[MAXDOMAIN];
char	*outfile = NULL;
int	linenum;	/* The line number in the file */
int	msgid_linenum;	/* The last msgid token line number */
int	msgstr_linenum;	/* The last msgstr token line number */

int	oflag = 0;
int	verbose = 0;

struct domain_struct *first_domain = NULL;
struct domain_struct *last_used_domain = NULL;

/*
 * msgfmt - Generate binary tree for runtime gettext() using psffm: "Portable
 * Source File Format for Messages" file template. This file may have
 * previously been generated by the xgettext filter for c source files.
 */

main(argc, argv)
	int	argc;
	char	*argv[];
{
	int	errflg = 0;
	int	c;

	opterr = 0;
	while ((c = getopt(argc, argv, "vo:")) != -1) {
		switch (c) {
		case 'o':
			oflag = 1;
			outfile = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			errflg = 1;
		}
	} /* while */

	if (errflg) {
		fprintf(stderr, "%s\n", USAGE);
		exit(2);
	}

	/* If no .po files specified, then error */
	if (optind >= argc) {
		fprintf(stderr, "%s\n", USAGE);
		exit(2);

	}

	/*
	 * read all portable object files specified in command arguments.
	 */
	for (; optind < argc; optind++) {

		if (verbose)
			fprintf(stderr, "Processing file \"%s\"...\n",
				argv[optind]);

		read_psffm(argv[optind]);

	} /* for */

	output_all_mo_files();

#ifdef DEBUG
	printlist();
#endif

	exit(0);

} /* main */



/*
 * read_psffm - read in "psffm" format file, check syntax, printing error
 * messages as needed, output binary tree to file <domain>
 */

static void
read_psffm(filename)
	char	*filename;
{
	FILE		*fp;
	static char	msgfile[MAXPATHLEN];
	char		*linebufptr;
	char		msgid[MAX_VALUE_LEN + 1];
	char		msgstr[MAX_VALUE_LEN + 1];
	char		*bufptr = 0;
	int		whitespace = 0;
	int		quotefound = 0;	/* double quote was seen */
	int		inmsgid = 0;	/* indicates "msgid" was seen */
	int		inmsgstr = 0;	/* indicates "msgstr" was seen */
	char		c;
	char		token_found;	/* Boolean value */

	/*
	 * For each po file to be read,
	 * 1) set domain to default and
	 * 2) set linenumer to 0.
	 */
	strcpy (current_domain, DEFAULT_DOMAIN);
	linenum = 0;

	if ((fp = fopen(filename, "r")) == NULL) {
		perror(filename);
		exit(2);
	}

	while (1) {
		if (fgets(linebuf, sizeof (linebuf), fp) == NULL) {
			if (ferror(fp)) {
				perror(filename);
				exit(2);
			}
			break; /* End of File. */
		}
		linenum++;

		linebufptr = linebuf;
		whitespace = 0;

		switch (*linebufptr) {
			case '#':	/* comment    */
			case '\n':	/* empty line */
				continue;
				break;

			case '\"': /* multiple lines of msgid and msgstr */
			case ' ':
			case '\t':
				/* whitespace should be handled correctly */
				whitespace = 1;
				break;
		}

		/*
		 * Process MSGID Tokens.
		 */
		token_found = !strncmp (MSGID_TOKEN, linebuf,
						sizeof (MSGID_TOKEN)-1);
		if (token_found || (whitespace && inmsgid)) {

			if (token_found)
				check_next_char(linebuf, sizeof (MSGID_TOKEN));

			if (inmsgid && !whitespace) {
				fprintf(stderr,
			"Warning, Consecutive MSGID tokens encountered ");
				fprintf(stderr,
			"at line number: %d, ignored.\n", msgid_linenum);
				continue;
			}
			if (inmsgstr) {
				sortit(msgid, msgstr);
				memset(msgid, 0, sizeof (msgid));
				memset(msgstr, 0, sizeof (msgstr));
			}

			if (inmsgid) {
				/* multiple lines of msgid */
				linebufptr = consume_whitespace(linebuf);
			} else {
				/*
				 * The first line of msgid.
				 * Save linenum of msgid to be used when
				 * printing warning or error message.
				 */
				msgid_linenum = linenum;
				linebufptr = consume_whitespace
					(linebuf +
					sizeof (MSGID_TOKEN) - 1);
				bufptr = msgid;
			}

			inmsgid = 1;
			inmsgstr = 0;
			goto load_buffer;
		}

		/*
		 * Process MSGSTR Tokens.
		 */
		token_found = !strncmp (MSGSTR_TOKEN, linebuf,
						sizeof (MSGSTR_TOKEN)-1);
		if (token_found || (whitespace && inmsgstr)) {

			if (token_found)
				check_next_char(linebuf, sizeof (MSGSTR_TOKEN));


			if (inmsgstr && !whitespace) {
				fprintf(stderr,
			"Warning, Consecutive MSGSTR tokens encountered ");
				fprintf(stderr,
			"at line number: %d, ignored.\n", msgstr_linenum);
				continue;
			}
			if (inmsgstr) {
				/* multiple lines of msgstr */
				linebufptr = consume_whitespace(linebuf);
			} else {
				/*
				 * The first line of msgstr.
				 * Save linenum of msgid to be used when
				 * printing warning or error message.
				 */
				msgstr_linenum = linenum;
				linebufptr = consume_whitespace
					(linebuf +
					sizeof (MSGSTR_TOKEN) - 1);
				bufptr = msgstr;
			}

			inmsgstr = 1;
			inmsgid = 0;
			goto load_buffer;
		}

		/*
		 * Process DOMAIN Tokens.
		 * Add message id and message string to sorted list
		 * if msgstr was processed last time.
		 */
		token_found = !strncmp (DOMAIN_TOKEN, linebuf,
						sizeof (DOMAIN_TOKEN)-1);
		if (token_found) {
			check_next_char(linebuf, sizeof (DOMAIN_TOKEN));

			/*
			 * process msgid and msgstr pair for previous domain
			 */
			if (inmsgstr) {
				sortit(msgid, msgstr);
			}

			/* refresh msgid and msgstr buffer */
			if (inmsgstr || inmsgid) {
				memset(msgid, 0, sizeof (msgid));
				memset(msgstr, 0, sizeof (msgstr));
			}
			inmsgid = 0;
			inmsgstr = 0;

			linebufptr = consume_whitespace
					(linebuf +
					sizeof (DOMAIN_TOKEN) - 1);
			memset (current_domain, 0, sizeof (current_domain));
			bufptr = current_domain;
		} /* if */

load_buffer:
		/*
		 * Now, fill up the buffer pointed by bufptr.
		 * At this point bufptr should point to one of
		 * msgid, msgptr, or current_domain.
		 * Otherwise, the entire line is ignored.
		 */

		if (!bufptr) {
			fprintf(stderr, "Warning, Syntax at line number: %d, ",
				linenum);
			fprintf(stderr, " line ignored\n");
			continue;
		}

		if (*linebufptr++ != '\"') {
			fprintf(stderr, "Warning, Syntax at line number: %d, ",
				linenum);
			fprintf(stderr, "Missing \", ignored\n");
			--linebufptr;
		}
		quotefound = 0;

		while (c = *linebufptr++) {
			switch (c) {
			case '\n':
				if (!quotefound) {
					fprintf(stderr,
					"Warning, Syntax at line number: %d, ",
						linenum);
					fprintf(stderr,
					"Missing \" at EOL, ignored\n");
				}
				break;

			case '\"':
				quotefound = 1;
				break;

			case '\\':
				*bufptr++ = expand_meta(&linebufptr);
				break;

			default:
				*bufptr++ = c;
			} /* switch */
		} /* while */

		strcpy(msgfile, current_domain);
		strcat(msgfile, ".mo");
	} /* while */

	if (inmsgstr) {
		sortit(msgid, msgstr);
	}
	if (inmsgstr || inmsgid) {
		memset(msgid, 0, sizeof (msgid));
		memset(msgstr, 0, sizeof (msgstr));
	}

	fclose (fp);
	return;

} /* read_psffm */


/*
 * Remove prepending white spaces and tabs.
 */
static char *
consume_whitespace (buf)
	char	*buf;
{
	char	*bufptr = buf;
	char	c;

	while (c = *bufptr++) {
		switch (c) {
		case ' ':
		case '\t':
			continue;
		default:
			return (bufptr - 1);
		}
	}

	return (bufptr);

} /* consume_white_space */


/*
 * handle escape sequences.
 */
static char
expand_meta(buf)
	char	**buf;
{
	char	c = **buf;
	char	n;

	switch (c) {
	case '"':
		(*buf)++;
		return ('\"');
	case '\\':
		(*buf)++;
		return ('\\');
	case 'b':
		(*buf)++;
		return ('\b');
	case 'f':
		(*buf)++;
		return ('\f');
	case 'n':
		(*buf)++;
		return ('\n');
	case 'r':
		(*buf)++;
		return ('\r');
	case 't':
		(*buf)++;
		return ('\t');
	case 'v':
		(*buf)++;
		return ('\v');
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
		/*
		 * This case handles \ddd where ddd is octal number.
		 * There could be one, two, or three octal numbers.
		 */
		(*buf)++;
		n = c - '0';
		c = **buf;
		if (c >= '0' && c <= '7') {
			(*buf)++;
			n = 8*n + (c-'0');
			c = **buf;
			if (c >= '0' && c <= '7') {
				(*buf)++;
				n = 8*n + (c-'0');
			}
		}
		return (n);
		break;
	default:
		c = **buf;
		(*buf)++;
		return (c);
	}
} /* expand_meta */


/*
 * Print Warning message for duplicate message ID
 */
static void
handle_duplicate_msgid (msgid)
	char	*msgid;
{
	/*
	 * Duplicate MSG-ID warning is printed only if user
	 * specifies -v optiotn.
	 */

	if (!verbose)
		return;

	fprintf(stderr, "Warning, Duplicate id \"%s\" at line number: %d, ",
			msgid, msgid_linenum);
	fprintf(stderr, " line ignored\n");

	return;

} /* handle_duplicate_msgid */


/*
 * Check if the next character is possible valid character.
 * Otherwise, it is an error and exit.
 */
static void
check_next_char (linebuf, len)
	char	*linebuf;
	int	len;
{
	if ((linebuf[len - 1] == ' ') ||
		(linebuf[len - 1] == '\t') ||
		(linebuf[len - 1] == '\n') ||
		(linebuf[len - 1] == 0)) {

		return;
	}

	fprintf(stderr,
		"Error, No space after directive at line number %d.\n",
		linenum);
	fprintf(stderr, "Exiting...\n");
	exit(2);

} /* check_next_char */

/*
 * Finds the head of the current domain linked list and
 * call insert_message() to insert msgid and msgstr pair
 * to the linked list.
 */
static void
sortit(msgid, msgstr)
	char	*msgid;
	char 	*msgstr;
{
	int			i, j, lessthan;
	struct domain_struct	*dom;

#ifdef DEBUG
	fprintf(stderr,
		"==> sortit(), domain=<%s> msgid=<%s> msgstr=<%s>\n",
		current_domain, msgid, msgstr);
#endif

	/*
	 * If "-o filename" is specified, then all "domain" directive
	 * are ignored and, all messages will be stored in domain
	 * whose name is filename.
	 */
	if (oflag) {
		dom = find_domain_node (outfile);
	} else {
		dom = find_domain_node (current_domain);
	}

	insert_message (dom, msgid, msgstr);

	return;
}

/*
 * This routine inserts message in the current domain message list.
 * It is inserted in ascending order.
 */
static void
insert_message (dom, msgid, msgstr)
	struct domain_struct	*dom;
	char			*msgid;
	char			*msgstr;
{
	struct msg_struct	*p1 = NULL;
	struct msg_struct	*node, *prev_node;
	char			b;

	/*
	 * Find the optimal starting search position.
	 * The starting search position is either the first node
	 * or the current_elem of domain.
	 * The current_elem is the pointer to the node which
	 * is most recently accessed in domain.
	 */
	if (dom->current_elem != NULL) {
		b = strcmp(msgid, dom->current_elem->msgid);
		if (b == 0) {
			handle_duplicate_msgid (msgid);
			return;
		} else if (b > 0) { /* to implement descending order */
			p1 = dom->first_elem;
		} else {
			p1 = dom->current_elem;
		}
	} else {
		p1 = dom->first_elem;
	}

	/*
	 * search msgid insert position in the list
	 * Search starts from the node pointed by p1.
	 */
	prev_node = NULL;
	while (p1) {
		b = strcmp(msgid, p1->msgid);
		if (b == 0) {
			handle_duplicate_msgid (msgid);
			return;
		} else if (b < 0) {  /* to implement descending order */
			/* move to the next node */
			prev_node = p1;
			p1 = p1->next;
		} else {
			/* insert a new msg node */
			node = (struct msg_struct *) malloc
				(sizeof (struct msg_struct));
			node->next = p1;
			node->msgid  = strdup (msgid);
			node->msgstr = strdup (msgstr);
			if (prev_node) {
				prev_node->next = node;
			} else {
				dom->first_elem = node;
			}
			dom->current_elem = node;
			return;
		}
	} /* while */

	/*
	 * msgid is smaller than any of msgid in the list or
	 * list is empty.
	 * Therefore, append it.
	 */
	node = (struct msg_struct *) malloc (sizeof (struct msg_struct));
	node->next = NULL;
	node->msgid  = strdup (msgid);
	node->msgstr = strdup (msgstr);
	if (prev_node) {
		prev_node->next = node;
	} else {
		dom->first_elem = node;
	}
	dom->current_elem = node;

	return;

} /* insert_message */


/*
 * This routine will find head of the linked list for the given
 * domain_name. This looks up cache entry first and if cache misses,
 * scans the list.
 * If not found, then create a new node.
 */
static struct domain_struct *
find_domain_node (domain_name)
	char *domain_name;
{
	struct domain_struct	*p1 = NULL;
	struct domain_struct	*node;
	struct domain_struct	*prev_node;
	char			b;


	/* for perfomance, check cache 'last_used_domain' */
	if (last_used_domain) {
		b = strcmp(domain_name, last_used_domain->domain);
		if (b == 0) {
			return (last_used_domain);
		} else if (b < 0) {
			p1 = first_domain;
		} else {
			p1 = last_used_domain;
		}
	} else {
		p1 = first_domain;
	}

	prev_node = NULL;
	while (p1) {
		b = strcmp(domain_name, p1->domain);
		if (b == 0) {
			/* node found */
			last_used_domain = p1;
			return (p1);
		} else if (b > 0) {
			/* move to the next node */
			prev_node = p1;
			p1 = p1->next;
		} else {
			/* insert a new domain node */
			node = (struct domain_struct *) malloc
				(sizeof (struct domain_struct));
			node->next = p1;
			node->domain = strdup (domain_name);
			node->first_elem = NULL;
			node->current_elem = NULL;
			if (prev_node) {
				/* insert the node in the middle */
				prev_node->next = node;
			} else {
				/* node inserted is the smallest */
				first_domain = node;
			}
			last_used_domain = node;
			return (node);
		}
	} /* while */

	/*
	 * domain_name is larger than any of domain name in the list or
	 * list is empty.
	 */
	node = (struct domain_struct *)
		malloc (sizeof (struct domain_struct));
	node->next = NULL;
	node->domain = strdup (domain_name);
	node->first_elem = NULL;
	node->current_elem = NULL;
	if (prev_node) {
		/* domain list is not empty */
		prev_node->next = node;
	} else {
		/* domain list is empty */
		first_domain = node;
	}
	last_used_domain = node;

	return (node);

} /* find_domain_node */


/*
 * binary_compute() is used for pre-computing a binary search.
 */
static int
binary_compute(i, j, more, less)
	int	i, j;
	int	more[], less[];
{
	int	k;

	if (i > j) {
		return (LEAF_INDICATOR);
	}
	k = (i + j) / 2;

	less[k] = binary_compute(i, k - 1, more, less);
	more[k] = binary_compute(k + 1, j, more, less);

	return (k);

} /* binary_compute */


/*
 * Write all domain data to file.
 * Each domain will create one file.
 */
static void
output_all_mo_files()
{
	struct domain_struct 	*p;

	p = first_domain;
	while (p) {
		/*
		 * generate message object file only if there is
		 * at least one element.
		 */
		if (p->first_elem) {
			output_one_mo_file (p);
		}
		p = p->next;
	}
	return;

} /* output_all_mo_files */


/*
 * Write one domain data list to file.
 */
static void
output_one_mo_file(dom)
	struct domain_struct	*dom;
{
	FILE	*fp;
	struct msg_struct	*p;
	int	message_count;
	int	string_count_msgid = 0;
	int	string_count_msg = 0;
	int	msgid_index = 0;
	int	msgstr_index = 0;
	int	*less, *more;
	int	i;
	char	fname [MAXDOMAIN];

	if (!dom || !dom->first_elem)
		return;

	/*
	 * If -o flag is specified, then file name is used as domain name.
	 * If not, ".mo" is appended to the domain name.
	 */
	strcpy(fname, dom->domain);
	if (!oflag) {
		strcat (fname, ".mo");
	}
	fp = fopen (fname, "w");
	if (fp == NULL) {
		perror("cannot open file");
		exit(2);
	}

	/* compute offsets and counts */
	message_count = 0;
	p = dom->first_elem;
	while (p) {
		p->msgid_offset = msgid_index;
		p->msgstr_offset = msgstr_index;
		msgid_index += strlen(p->msgid) + 1;
		msgstr_index += strlen(p->msgstr) + 1;
		message_count++;
		p = p->next;
	}

	/*
	 * Fill up less and more entries to be used for binary search.
	 */
	string_count_msgid = msgid_index;
	string_count_msg = msgstr_index;
	if ((less = (int *) calloc(message_count, sizeof (int))) == NULL) {
		perror("msgfmt output_one_mo_file()");
		exit(8);
	}
	if ((more = (int *) calloc(message_count, sizeof (int))) == NULL) {
		perror("msgfmt output_one_mo_file()");
		exit(8);
	}

	(void) binary_compute(0, message_count - 1, more, less);

#ifdef DEBUG
	{
	int i;
	for (i = 0; i < message_count; i++) {
		fprintf (stderr, "  less[%2d]=%2d, more[%2d]=%2d\n",
				i, less[i], i, more[i]);
	}
	}
#endif

	/*
	 * write out the message object file.
	 * The middle one is the first message to check by gettext().
	 */
	i = (message_count - 1) / 2;
	(void) fwrite(&i, sizeof (int), 1, fp);
	(void) fwrite(&message_count, sizeof (int), 1, fp);
	(void) fwrite(&string_count_msgid, sizeof (int), 1, fp);
	(void) fwrite(&string_count_msg, sizeof (int), 1, fp);
	i = sizeof (struct msg_struct) * message_count;
	(void) fwrite(&i, sizeof (int), 1, fp);

	/* march through linked list and write out all nodes. */
	i = 0;
	p = dom->first_elem;
	while (p) {	/* put out message struct */
		(void) fwrite(&less[i], sizeof (int), 1, fp);
		(void) fwrite(&more[i], sizeof (int), 1, fp);
		(void) fwrite(&p->msgid_offset, sizeof (int), 1, fp);
		(void) fwrite(&p->msgstr_offset, sizeof (int), 1, fp);
		i++;
		p = p->next;
	}

	/* put out message id strings */
	p = dom->first_elem;
	while (p) {
		(void) fwrite(p->msgid, strlen(p->msgid)+1, 1, fp);
		p = p->next;
	}

	/* put out message strings */
	p = dom->first_elem;
	while (p) {
		(void) fwrite(p->msgstr, strlen(p->msgstr)+1, 1, fp);
		p = p->next;
	}

	fclose(fp);

	return;

} /* output_one_mo_file */



/*
 * This is debug function. Not compiled in the final executable.
 */
#ifdef DEBUG
static void
printlist()
{
	struct domain_struct	*p;
	struct msg_struct	*m;

	fprintf(stderr, "\n=== Printing contents of all domains ===\n");
	p = first_domain;
	while (p) {
		fprintf(stderr, "domain name = <%s>\n", p->domain);
		m = p->first_elem;
		while (m) {
			fprintf(stderr, "   msgid=<%s>, msgstr=<%s>\n",
					m->msgid, m->msgstr);
			m = m->next;
		}
		p = p->next;
	}
} /* printlist */
#endif DEBUG
