/*
 * $XConsortium$
 *
 * Copyright 1991 MIPS Computer Systems, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of MIPS not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  MIPS makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * MIPS DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL MIPS
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ident	"$Header: mipsGenFb.c,v 1.4 92/08/17 16:22:31 dd Exp $"

/*
 * Color frame buffer support
 *
 * Uses frame buffer device (e.g. /dev/fb0).  This is available in RISC/os
 * 5.0B and later, for the RC4230 and related systems.
 */

#if MIPS_RISCOS_VERSION >= 5

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/fb_ioctl.h>	/* RISC/os 5.0B and later only */

#include <X.h>
#include <Xproto.h>
#include <misc.h>
#include <cursorstr.h>
#include <input.h>
#include <scrnintstr.h>
#include <servermd.h>

#include "mipsFb.h"

extern caddr_t mmap();
extern char *mipsMapit();

static int BlankFb();
static int WriteCMapFb();
static void CloseFb();

#ifdef FB_FORMAT_CURSOR		/* RISC/os 5.01 and later only */
#define	CURSOR_SUPPORT
#endif

#ifdef CURSOR_SUPPORT
static Bool RealizeCursorFb();
static int SetCursorFb();
static int MoveCursorFb();
static int RecolorCursorFb();
#endif /* CURSOR_SUPPORT */

Bool
mipsMapFb(pm)
	MipsScreenPtr pm;
{
	int fd = pm->fd;
	struct fb_type fb_type;
	struct fb_mapinfo fb_mapinfo;

	static int save_fd[MAXSCREENS];

	/*
	 * We don't really want to close the frame buffer device
	 * because the terminal emulator screen will appear momentarily.
	 * It's easier to dup here than to figure out how to avoid
	 * reopening it (but stupid).
	 */
	if (save_fd[pm->unit] <= 0)
		save_fd[pm->unit] = dup(fd) + 1;

	if (ioctl(fd, FB_GET_TYPE, &fb_type) < 0)
		return FALSE;

	pm->scr_width = fb_type.width;
	pm->scr_height = fb_type.height;

	fb_mapinfo.depth = 0;
	if (ioctl(fd, FB_GET_MAPINFO, &fb_mapinfo) < 0)
		return FALSE;

	pm->depth = fb_mapinfo.depth;
	pm->bitsPerPixel = fb_mapinfo.bits_per_pixel;
	pm->fb_width = fb_mapinfo.pixels_per_line;

	{
		caddr_t addr;
		off_t off = fb_mapinfo.mmap_offset;
		size_t size = fb_mapinfo.mmap_size;

		if ((int) off & 0xff)
			addr = (caddr_t) mipsMapit((char *) 0,
				(int) off, (int) size);
		else if ((addr = mmap((caddr_t) 0, size,
				PROT_READ | PROT_WRITE, MAP_SHARED,
				fd, off)) != (caddr_t) -1)
			pm->fb_mmap_size = (int) size;
		else
			addr = 0;

		if (addr == 0)
			return FALSE;

		pm->fbnorm = (unsigned char *) addr;
		pm->fbnocache = (unsigned char *) addr;
	}

	pm->dpi = mipsMonitorDPI(pm);

	pm->Blank = (void (*)()) BlankFb;
	pm->WriteCMap = (void (*)()) WriteCMapFb;
	pm->Close = CloseFb;

	if (!(pm->option & MIPS_SCR_NOHW)) {
#ifdef CURSOR_SUPPORT
		struct fb_cursor_xy cursize;

		if (ioctl(fd, FB_GET_CURSOR_SIZE, &cursize) >= 0) {
			pm->cap = MIPS_SCR_CURSOR;
			pm->RealizeCursor = RealizeCursorFb;
			pm->SetCursor = (void (*) ()) SetCursorFb;
			pm->MoveCursor = (void (*) ()) MoveCursorFb;
			pm->RecolorCursor = (void (*) ()) RecolorCursorFb;
		}
#endif /* CURSOR_SUPPORT */

#ifdef FB_REMAP
		if (pm->fb_mmap_size) {
			struct fb_map fb_map;

			fb_map.addr = (caddr_t) pm->fbnorm;
			fb_map.size = pm->fb_mmap_size;
			if (ioctl(fd, FB_REMAP, &fb_map) < 0)
				Error("mipsMapFb -- FB_REMAP");
		}
#endif /* FB_REMAP */
	}

	return TRUE;
}

static int
BlankFb(pm, on)
	MipsScreenPtr pm;
	Bool on;
{
	int video = (on != SCREEN_SAVER_ON);

	return ioctl(pm->fd, FB_SET_VIDEO, &video);
}

static int
WriteCMapFb(pm, index, count, rgb)
	MipsScreenPtr pm;
	int index;
	int count;
	unsigned char *rgb;
{
	struct fb_colormap fb_colormap;

	fb_colormap.map = 0;
	fb_colormap.index = index;
	fb_colormap.count = count;
	fb_colormap.rgb = rgb;

	return ioctl(pm->fd, FB_SET_COLORMAP, &fb_colormap);
}

static void
CloseFb(pm)
	MipsScreenPtr pm;
{
	caddr_t addr;
	size_t size;

#ifdef CURSOR_SUPPORT
	/* disable HW cursor */
	(void) SetCursorFb(pm, (CursorPtr) 0, (pointer) 0, 0, 0);
#endif /* CURSOR_SUPPORT */

	if (addr = (caddr_t) pm->fbnorm)
		if (size = (size_t) pm->fb_mmap_size) {
#ifdef FB_UNMAP
			struct fb_map fb_map;

			fb_map.addr = addr;
			fb_map.size = size;
			(void) ioctl(pm->fd, FB_UNMAP, &fb_map);
#endif /* FB_UNMAP */

			(void) munmap(addr, size);
		}
		else
			(void) shmdt((char *) addr);

	(void) close(pm->fd);
}

#ifdef CURSOR_SUPPORT

struct priv_shape {
	int shape_bytes;
	char shape_data[4];
};

static Bool
RealizeCursorFb(pm, pCurs, pPriv)
	MipsScreenPtr pm;
	CursorPtr pCurs;
	pointer *pPriv;
{
	CursorBitsPtr bits = pCurs->bits;
	struct priv_shape *priv_shape;
	struct fb_cursor_shape shape;

	shape.size.x = bits->width;
	shape.size.y = bits->height;

	/* XXX handle padding and bit order */
	shape.image = (unsigned int *) bits->source;
	shape.mask = (unsigned int *) bits->mask;
	shape.shape_data = 0;

	if (ioctl(pm->fd, FB_FORMAT_CURSOR, &shape) < 0)
		return FALSE;

	priv_shape = (struct priv_shape *)
		xalloc(sizeof (struct priv_shape) + shape.shape_bytes - 4);
	*pPriv = (pointer) priv_shape;
	if (priv_shape == 0)
		return FALSE;

	shape.shape_data = priv_shape->shape_data;
	if (ioctl(pm->fd, FB_FORMAT_CURSOR, &shape) < 0) {
		xfree(priv_shape);
		return FALSE;
	}
	priv_shape->shape_bytes = shape.shape_bytes;

	return TRUE;
}

static int
SetCursorFb(pm, pCurs, priv, x, y)
	MipsScreenPtr pm;
	CursorPtr pCurs;
	pointer priv;
	int x, y;
{
	struct fb_cursor curs;

	curs.set = FB_CUR_SETCUR;
	curs.enable = 0;

	if (pCurs) {
		curs.set = FB_CUR_SETCUR | FB_CUR_SETPOS |
			FB_CUR_SETHOT | FB_CUR_SETCOLORMAP | FB_CUR_SETSHAPE;

		curs.enable = 1;

		curs.rgb[0] = pCurs->backRed >> 8;
		curs.rgb[1] = pCurs->backGreen >> 8;
		curs.rgb[2] = pCurs->backBlue >> 8;
		curs.rgb[3] = pCurs->foreRed >> 8;
		curs.rgb[4] = pCurs->foreGreen >> 8;
		curs.rgb[5] = pCurs->foreBlue >> 8;

		curs.pos.x = x;
		curs.pos.y = y;

		curs.hot.x = pCurs->bits->xhot;
		curs.hot.y = pCurs->bits->yhot;

		curs.shape_bytes = ((struct priv_shape *) priv)->shape_bytes;
		curs.shape_data = ((struct priv_shape *) priv)->shape_data;
	}

	return ioctl(pm->fd, FB_SET_CURSOR, &curs);
}

static int
MoveCursorFb(pm, x, y)
	MipsScreenPtr pm;
	int x, y;
{
	struct fb_cursor_xy pos;

	pos.x = x;
	pos.y = y;

	return ioctl(pm->fd, FB_SET_CURSOR_POSITION, &pos);
}

static int
RecolorCursorFb(pm, pCurs)
	MipsScreenPtr pm;
	CursorPtr pCurs;
{
	struct fb_cursor curs;

	curs.set = FB_CUR_SETCOLORMAP;
	curs.rgb[0] = pCurs->backRed >> 8;
	curs.rgb[1] = pCurs->backGreen >> 8;
	curs.rgb[2] = pCurs->backBlue >> 8;
	curs.rgb[3] = pCurs->foreRed >> 8;
	curs.rgb[4] = pCurs->foreGreen >> 8;
	curs.rgb[5] = pCurs->foreBlue >> 8;

	return ioctl(pm->fd, FB_SET_CURSOR, &curs);
}
#endif /* CURSOR_SUPPORT */

#endif /* MIPS_RISCOS_VERSION >= 5 */
