@chapter(Structure)

There are two major points about the structure of the code. One has to do
with inter-layer communication and tasking; the other has to do with
modularity of the internet to network driver interface, and how new drivers
can be used easily.

@section(Upcalls and Tasking)
The PC code uses several ideas developed by Dave Clark about how to deal
with the asynchronous nature of the network interface. You can read about
them in @cite[ddc1]@cite[ddc2].

Let's look at what happens when a packet is received from the network. First,
the interrupt handler gets it into a packet buffer somewhere in main memory
and enqueues it on the queue associated with that network interface. Then
it wakes up the task associated with that interface. Later, when the task
runs, it will dequeue (atomically) the packet from its queue, and process it
(whatever that might involve). When it has decided that this is a valid
packet and a packet for the internet protocol, it will then @i(upcall)
@footnote[A minor digression on upcalls: people love to argue about what's
an upcall and what's a downcall, and that this person is really talking
about sideways calls. When I say @i(upcall) I mean a call from a layer of
code to one that is logically at a higher level than it, i.e.: a protocol
calling its client. When I say @i(downcall), I mean a call from a layer to
one which is logically at a lower level than it. Mike Greenwald suggests
that an upcall is from a server to a client, whereas a downcall is from a
client to a server. I think that I agree. This isn't the right place to
argue about it, though.] internet, passing the packet and the packet's
length. Otherwise, it discards it.

Internet will then do its own verification of the validity of the packet
(header checksums, destination address check, @i[et al]) and look in its
table of currently active protocols to see if anyone above it might be
interested in this packet. Suppose it's a UDP packet. Internet then upcalls
UDP with the packet, source host address, and length of internet data of the
packet. Otherwise, internet drops the packet.

UDP does similar processing. It checksums the whole packet and sees if
anyone is listening to the local port specified in the UDP header. Remember
that UDP only demultiplexes on the local port, so if the foreign port is
also of interest to the layer above UDP, it must check on that itself. UDP
upcalls the appropriate routine passing it the packet, length of UDP data,
foreign host and foreign port.

And now the packet has reached its final resting place (?).

Sending a packet is much easier. It consists of a sequence of @i(downcalls)
from one layer to the next one below it. Each layer fills in its own part of
the header and passes the packet to the next layer. After the packet has been
sent, it is @b(NOT) deallocated; the user program still has control of it.

@section(Network Interfaces and Drivers)

@begin(figure)
@begin(verbatim)
struct net {
	char	*n@us()name;	/* the net's name in ascii */
	int	(*n@us()init)();   /* the net initialization routine */
	int	(*n@us()send)();	/* the packet xmit routine */
 	int	(*n@us()open)();	/* the protocol open routine */
	int	(*n@us()close)();	/* the protocol close routine */
	task	*n@us()demux;   /* packet demultiplexing task to protocol */
	queue	*n@us()inputq;	/* the queue of received packets */
	unsigned n@us()initp1;	/* initialization parameter one */
	unsigned n@us()initp2;	/* "	     "	    two */
	int	n@us()stksiz;	/* net task initial stack size */
	int	n@us()lnh;	/* the net's local net header  size */
	int	n@us()lnt;	/* the net's local net trailer size */
	in@us()name	ip@us()addr;  /* the interface's internet address */
	int	(*n@us()stats)();	/* per net statistics */
	in@us()name n@us()defgw;      /* the default gateway for this net */
	struct custom *n@us()custom;	/* per-net custom structure */
	};

typedef struct net NET;
@end(verbatim)
@caption(The @i(net) structure)
@tag(net-struct)
@end(figure)

There are several things associated with a network interface driver: a task,
which is awakened when a packet is received; a queue, on which the packet is
enqueued; and lots of random constants such as the internet address of this
interface. This structure is presented in figure @ref(net-struct).

The queue is allocated by @i[Netinit()] but the task must be forked by the
net interface's initialization routine. The initialization routine can
indicate failure by setting its internet address to zero; @i[Netinit()] will
then abort.

@section(Packet Allocation)

A new type, @i[PACKET] is defined as a pointer to the @i[net@us2{}buf]
structure. The @i[net@us2{}buf] structure contains information about the
packet buffer and has a pointer to the buffer itself. Thus, to pass around
packets, declare a variable @i[p] as @i[PACKET p;] and then just pass around
@i(p). Each level has some macros to get at the packet buffer itself, so
you don't have to worry much about this once you get into internet.

Packet buffers are allocated at initialization time. They are all of the
same length. The length is given by the variable @i(LBUF) which can be
modified before calling @i[Netinit()] to change the length of packet
buffers. @i(NBUF) is the number of packet buffers which are allocated. There
is a queue called @i(freeq) which all currently free packet buffers are on.
There are three macros for enqueuing and dequeuing packets buffers:
@i[getfree(), a@us2{}getfree(),] and @i[putfree()]. @i[a@us2{}getfree()]
should be called at non-interrupt level and is atomic; @i[putfree()] is also
atomic. You don't want to use atomic routines at interrupt level since they
enable interrupts after they're done and you might not want this to happen.
So, to allocate a packet buffer at interrupt level (to DMA a packet into,
perhaps), use @i[getfree()].

The enqueuing routines don't check on some stupid things that people can do,
like enqueuing the same packet twice, or enqueuing a packet whose pointer
is 0. Doing these things are surefire ways to lose, so be careful about them.
Also, be sure to free up packets when you're done with them. One seemingly
good style for high level programs that can do this (like tftp or tcp) is to
allocate a single output packet and to reuse it.

@section(Initialization and Shutdown)

Most initialization is done explicitly, but certain steps of it are done
behind the scenes in @i[netcrt]. The hidden initialization includes copying
the custom structure into the data segment and initializing the standard I/O
library. Explicit initialization involves calling @i[NetInit()] and the
initialization routines for each protocol (like Internet, ICMP and UDP).
These initializations must be done before using the network. The tasking
and timer package is initialized by @i[NetInit()]. It also allocates the
packet buffers and calls the initializations routine for each net interface
that the code has been configured for. All @i[net] structures must be
initialized before calling @i[Netinit()]. Protocol layers should be
initialized in order of level; the following code would work:

@b[The @i{Init1()} function is gone. Net interface structures are now
completely initialized statically.]

@begin[verbatim]

	NetInit(stacksize);	/* initialize the underlying system */
	in@us()init();	/* initialize internet */
	IcmpInit();	/* initialize ICMP */
	UdpInit();	/* initialize UDP */
	LogInit();	/* initialize the logging code...*/

@end[verbatim]

Some of the names of initialization routines are inconsistent.

@section(Statistics)
	In the net structure, there is a pointer to a routine which yields
statistics for that interface. The field name for the pointer is
@i[n@us2()stats]. When called, this routine should be passed a file
descriptor (of type @i[FILE *]) as its argument. It will then dump
statistics about that interface (packets received, packets sent, net
errors, etc) to that stream. Quite often, the stream is @i(stdout).

	The routine @i[net@us2()stats()] calls the statistics routine for
each interface, and then calls the internet statistics routine,
@i[in@us()stats()]. It is of the same form as the net interface routines.
Eventually, it might call the protocols which are currently active for
statistics as well, requiring a new parameter to @i[in@us2()open()], but it
currently does nothing like that. UDP and TCP do, however, keep some
statistics.

@section(Debugging)

There is a global variable called @i(NDEBUG) which controls the level of
debugging messages that various pieces of code will emit. When it is 0, all
levels are off. Otherwise, various bits correspond with active levels. Bit
definitions are found in @i[<net.h>]. The levels are:

@begin(itemize)
BUGHALT@*
When a truly awful internal error is detected, print out an explanatory
message and cease program execution .

DUMP@*
When a malformed packet is received, display its contents. Some pieces of
code also check this to see if they should dump well formed received packets
or outgoing packets. There should probably be TXDUMP, BADRDUMP and RDUMP
modes.

INFOMSG@*
This tells you EVERYTHING that happens. Telnet when run in this
mode is completely unusable. It prints out messages when layers are
initialized, and when some tasks run, and when packets are received, and
when packet are transmitted, and when timeouts occur, and when errors occur,
and when errors don't occur, and whenever it seemed useful when the code was
written. This mode is useful to see where a buggy program stops if it is
really dying horribly.

NETERR and PROTERR@*
Sometimes I have trouble distinguishing between these
two. I think that when one is checked the other is almost always checked.
The idea was to print out errors that were probably hardware related or were
probably associated with the physical network when NETERR was active. When
violations of protocol or errors of a similar nature occurred, PROTERR was
to be checked. It's not always clear which is which (what kind of error is a
bad checksum?).

TRACE@*
...doesn't quite trace. It's supposed to trace packets across layer
boundaries inside the code, but it's not checked everywhere and so you
won't get messages from lots of the code when it gets a packet. It might
help though.

TMO@*
is the timeout flag. When this level is set, messages should be printed when
timeouts occur.

@end(itemize)

Overall, the debugging messages are incomplete. They occur mostly in places
where they had to be used for debugging, although occasionally their use
was planned.

@section[The Customizer]

@b[This section updates previous descriptions of the way the custom
structure was treated.]

The custom structure contains configuration information about the machine,
such as its internet address, its default gateway and hardware information
about the network interface. The custom structure is stored in a device
driver which programs read the information from. This device driver is
loaded into the operating system at boot time from a file named
@i[netdev.sys]. The name of the device is @i[netcust]. Conventions for
reading from @i[netdev.sys] and @i[netcust] are the same, and the program
@i[custom.com], which is used to edit the stored information, can be used to
modify either the version in a file on the disk, or, for temporary changes,
the in memory copy via @i[netcust].

When multiple network interfaces are supported, there will be a custom
structure for each interface.

The following code fragment, taken from @i[netinit.c], reads in the custom
structure.

@begin[verbatim]

struct custom custom;
int i;

	i = open("netcust", 2);
	if(i < 0) {
		printf("Couldn't read in the custom structure, so you \n");
		printf("probably didn't install netdev.sys correctly.\n");
		exit(1);
		}

	mkraw(i);
	read(i, &custom, sizeof(struct custom));
	close(i);

@end[verbatim]

Code which reads in the custom structure should verify that the structure
has been customized by looking at the date of the last customization, and
should verify that the version number is correct. The version number field
will always be at the same offset in the structure.

The field @i[c@us()otheruser] in the structure will be non-zero if a device
driver has been loaded which is using the net interface. The device driver
will have replaced @i[netdev.sys] and will have its own copy of the custom
structure internally. In this case any network interface driver should not
completely shutdown the interface when it exits, but should simply return
control of it to whatever was using the interface before the program ran.

@section[Bug Reports]

Bug reports are encouraged. If you are on the network, please report them by
sending mail to pc-ip-request@@mit-xx. Otherwise, try contacting people
listed in the user manual about bugs. Unfortunately, it may be a while
before we get to do anything about the bug.

Bug reports about the user manual and the programmer's manual are also
encouraged.
