@part[timers, root "progman"]

@chapter(Timers)

The timer code was originally written for the Unix tasking package by the
author of the Unix tasking package, Larry Allen. I ported it to the PC and
made modifications in how timers are referenced and allocated/freed. With
the exception of some changes to queue references to satisfy our compiler's
type checker, the code much is much the same as the original. It is strongly
tied in with the tasking code. Timers provide multiple programmable timers
for a program. When a timer expires, it calls a function with a specified
argument. Timers must be explicitly allocated and freed (via the
 @i[tm@us2()alloc()] and @i[tm@us2()free()] calls), and can be set, reset
and cleared. There is a timer task which handles the machine's single
physical timer via an almost Unix-like alarm mechanism@cite[unix-alarm].

@section(Include files and Libraries)

Anything which uses timers needs the @i[<timer.h>] file, which defines the
@i(timer) type as a pointer to a timer structure. In addition, it needs the
@i[<q.h>] file since timers use queues. The timer code is in the task library.

@section(Tracing and Debugging)

Timer tracing may be turned on by setting the variable @i[TIMERDEBUG] to
TRUE. The timer code will then print messages whenever a timer fires, or a
timer is set, reset or cleared, or when the timer task runs.

In addition, the @i[tm@us2()dump()] call may be used to display some
information about a specific timer and about the queue of active timers.

@section(User Timer Functions)

Descriptions of the timer functions available follow. None of the following
routines should be used by interrupt handlers.

@subsection{@i[tm@us2()alloc()]}

@begin(verbatim)

timer *tm@us()alloc()

@end(verbatim)

This function allocates a single timer and returns a pointer to it. If it
was not possible to allocate a timer, it returns 0. Allocating a timer
involves allocating storage, and the timer system manages a queue of free
timers to improve the efficiency of timer allocation. The allocated timer is
cleared and ready for use.

@subsection{@i[tm@us2()free()]}

@begin(verbatim)

tm@us()free(tm)
	timer *tm;

@end(verbatim)

This function frees a timer's storage, and clears it if the timer was
active. If the number of free timers is above the "high water mark" the
storage associated with the timer is @i[cfree()]ed; otherwise the timer is
enqueued on a queue of free timers managed by this function and
@i[tm@us2()alloc()].

Timers should not be referenced after they have been freed.

@subsection{@i[tm@us2()set()]}

@begin(verbatim)

tm@us()set(time, function, argument, tm)
	unsigned time;
	int (*function)();
	unsigned argument;
	timer *tm;

tm@us()tset(time, function, argument, tm)
	unsigned time;
	int (*function)();
	unsigned argument;
	timer *tm;

tm@us()mset(time, function, argument, tm)
	long time;
	int (*function)();
	unsigned argument;
	timer *tm;

@end(verbatim)

This call sets a previously allocated time, @i(tm), to fire in @i(time)
seconds. When the timer does fire, it will call the routine @i(function),
passing it the argument @i(argument). This routine may do what it wishes,
but should be careful about usage of stack space since it will be running on
the timer task's stack, which is normally 500 bytes@footnote[This note used
to say: "This stack should probably be doubled in size!" but tasking
statistics have shown that only around 200 bytes of it usually get used.]

Of course, the argument can actually be any word long value, and
@i(function) can ignore it altogether if it wishes.

The function @i[tm@us()mset()] sets in the timer in milliseconds and
@i[tm@us()tset()] sets the timer in clock ticks. The constant @c[tps],
defined in @i[<timer.h>], is the number of clock ticks per second.

@subsection{@i[tm@us2()reset()]}

@begin(verbatim)

int tm@us()reset(time, tm)
	unsigned time;
	timer *tm;

int tm@us()retset(time, tm)
	unsigned time;
	timer *tm;

int tm@us()remset(time, tm)
	long time;
	timer *tm;

@end(verbatim)

This function resets a timer @i(tm) to fire in @i(time) seconds (or
milliseconds or clock ticks). The timer must have previously been set
by @i[tm@us2()set()]. If the reset fails, FALSE is returned; otherwise
TRUE is returned.

@subsection{@i[tm@us2()clear()]}

@begin(verbatim)

int tm@us()clear(tm)
	timer *tm;

@end(verbatim)

This functions clears a timer, preventing it from firing altogether. The
timer must have previously been set by @i[tm@us2()set()]. If the clear fails,
FALSE is returned; otherwise TRUE is returned.

@subsection{@i[tm@us2()init()]}

@begin(verbatim)

tm@us()init()

@end(verbatim)

This calls initializes the timer system. It should be called before any
other function described in this section. It is automatically done by
@i[net@us2()init] but not by other functions.

@subsection{@i[tm@us2()dump()]}

@begin(verbatim)

tm@us()dump(tm)
	timer *tm;

@end(verbatim)

Prints out information about this timer and the queue of active timers. This
function is useful in debugging.


@section(Internal Timer Functions)

@begin(b)
The following functions are internal to the timer system only and should
not be called from outside of it. They are only included here for
thoroughness.
@end(b)


@subsection{@i[tm@us2()main()]}

@begin(verbatim)

tm@us()main()

@end(verbatim)

This function forms the body of the timer task. Whenever it runs, it checks
the queue of active timers to determine whether any have fired, and to
handle them if they have. Then it sets another alarm appropriately and
blocks again. It is awakened by the routine which handles the alarm signal,
@i[tm@us2()signal()]. You can think of this routine as multiplexing the
single physical timer into many logical ones if you wish.

@subsection{@i[tm@us2()signal()]}

@begin(verbatim)

tm@us()signal()

@end(verbatim)

This routine catches the alarm signal from the @i(crock) handling code. It
wakes up the timer task and arranges to have itself catch the next alarm
signal.

@b[The following routines are a kludge to interface to the BIOS clock, but
they need to be documented somewhere.]

@subsection[@i{crock@us()init()}]

@begin[verbatim]

crock@us()init()


long cticks;
unsigned @us()tcount;
extern int (*@us()alrm)();

@end[verbatim]

The @i[crock@us()init()] routine turns on the "real time crock". It installs
an interrupt handler to catch the interrupt that BIOS chains to the user,
and saves away the old contents of that interrupt vector.

The routine which gets called on every clock tick increments the global
variable @i[cticks]. It also decrements the variable @i[@us()tcount] if that
variable is not already zero. If the decrement takes @i[@us()tcount] to
zero, the function addressed by the variable @i[@us()alrm] is called.

Storage for these variables is declared @i[signal.c] in @i[-lpc].

@subsection{@i[crock@us()c()]}

@begin[verbatim]

crock@us()c()

@end[verbatim]

This routine deinstalls the chained clock interrupt handler and replaces it
with the saved away former handler. It is very important that this function
be called before a program which is using the crock exits.

@subsection[@i{alarm()}]

@begin[verbatim]

alarm(t)
	unsigned t;

@end[verbatim]

The @i[alarm()] function simply sets the clock count to @i[t] ticks.
The routine specified by @i[@us{}alrm] will be called when the count
reaches zero.
