Page	60,132
Title	-Video Drivers
;
;
;	Copyright (c) 1983
;	Zenith Data Systems
;	St. Joseph, Michigan
;
;	Copyright (c) 1985
;	Morrow Designs, Inc.
;	San Leandro, California
;
;	Last Update 1_Nov_85
;
;
;		Use, duplication or disclosure by the Government is
;	subject to restrictions as set forth in paragraph (b) (3) (B)
;	of the Rights in Technical Data and Computer Software clause
;	in DAR 7-104.9(a). Contractor/manufacturer is Zenith Data
;	Systems of Hilltop Road, St. Joseph, Michigan  49085.
;
;
%Out	Video.asm
page
;======================================================================
Monitor_Segment Segment Word Public
;==================================
;
Assume	cs:Monitor_Segment, ds:Rom_Data, es:Nothing, ss:Nothing

; Include Files
;--------------
;	Rom.lit
;	Intr.lit
;	Sys.ext
;	IO.lit
;	Video.lit
;	Video.ext

	.xlist
	include	../Rom/Rom.lit
	include	../Rom/Intr.lit
	include ../Rom/Sys.ext
	include	../IO/IO.lit
	include	Video.lit
	include	Video.ext
	.list

	extrn	DS_to_Rom_Data:Near
	extrn	Case:Near
	extrn	Get_Intr_Ptr:Near
	extrn	Beep:Near
	extrn	Read_Dip_Switches:Near
	extrn	Set_Monitor_Stack:Near
	extrn	Set_User_Stack:Near
	extrn	Dcr_Save:Byte

;*** START OF VIDEO PAGING CHANGE
	EXTRN	INIT_PAGE_MAP:NEAR, SWAP_PAGES:NEAR
;*** END OF VIDEO PAGING CHANGE


page
;----------------------------------------------------------------------
Video_Io_Interrupt Proc Far; (16_Jul_85)
;---------------------------------------
;	Video_IO_Interrupt is the entry point for all of the
; video Input/Output functions.  Called by performing an interrupt
; number 10H (16 decimal), it expects a function number in AH -
; this number is then used to call the appropriate function.
; The function calls supported by this interrupt are as follows:
;
; Function Code 0 - Set Video Mode
;    Input:
;	AL: New video mode, as follows:
;	    0 - 40 by 25 display in black and white
;	    1 - 40 by 25 display in color
;	    2 - 80 by 25 display in black and white
;	    3 - 80 by 25 display in color
;	    4 - 320 by 200 pixel graphics in color
;	    5 - 320 by 200 pixel graphics in black and white
;	    6 - 640 by 200 pixel graphics in black and white
;	    7 - 80 by 25 display on the monochrome board
;
; Function Code 1 - Set Cursor Style
;    Input:
;	CH: Starting scan line for cursor (from 0-31)
;	CL: Ending scan line for cursor (from 0-31)
;	    Note that CH must not be greater than CL!
;
; Function Code 2 - Set Cursor Position
;    Input:
;	DX: Alphanumeric cursor position (DH:row, and DL:column)
;	    The 'Home' position is the upper-left hand corner of the
;	    screen, and is defined as (0:0)
;	BH: Page number, as appopriate for the current video mode.
;	    See the Set Display Page function for more information
;
; Function Code 3 - Read Cursor Position
;    Input:
;	BH: Page number
;    Output:
;	DX: Cursor position (DH:row, and DL:column)
;	CX: Cursor style (CH:starting scan line, and CL:ending line)
;
; Function Code 4 - Read Light Pen Status
;    Output:
;	AH: Light pen status.  If AH is 0, then the light pen is
;	    either not pressed to the screen, or is not triggered.
;	    If AH = 1, then the following registers are valid:
;	DX: Position of light pen (DH:row, and DL:column) as
;	    an ALPHANUMERIC position.
;	CH: Vertical scan line number (from 0-199).
;	BX: Approximate pixel column (from 0-319, or 0-639, depending
;	    on the currently selected video mode).
;
; Function Code 5 - Select Display Page
;    Input:
;	AL: Page number to display.  For the 40 by 25 display modes
;	    the page may range from 0 to 7;  for 80 by 25 it may
;	    be 0-3, and only 0 is acceptable for the graphics modes.
;
; Function Code 6 - Scroll Currently Displayed Page Up
;    Input:
;	AL: Number of lines to scroll the selected window.  If
;	    AL is 0, the entire scroll window is erased.
;	CX: Upper-left hand scroll window position (CH:row, and
;	    CL:column).
;	DX: Lower-right hand scroll window position (DH:row, and
;	    DL:column).
;	BH: Attribute byte used for erased lines - ignored in
;	    graphics modes.
;
; Function Code 7 - Scroll Currently Displayed Page Down
;    Input:
;	AL: Number of lines to scroll.	If AL = 0, then the
;	    window is erased.
;	CX: Upper-left hand scroll window position (CH:row, and
;	    CL:column).
;	DX: Lower-right hand scroll window position (DH:row, and
;	    DL:column).
;	BH: Attribute byte for erased lines - ignored while in
;	    graphics modes.
;
; Function Code 8 - Read Character and Attribute at Cursor Position
;    Input:
;	BH: Page number.
;    Output:
;	AL: Ascii value of character read.
;	AH: Attribute byte for this character (except while graphics
;	    modes are in effect).
;
; Function Code 9 - Write Character and Attribute at Cursor Position
;    Input:
;	BH: Page number.
;	AL: Ascii value of the character to display.
;	BL: Attribute for this character (for alphanumeric modes).
;	CX: Repeat count (see the Display_Character routine for more
;	    information).
;
; Function Code 10 - Write Character ONLY at Current Cursor Position
;    Input:
;	BH: Page number.
;	AL: Ascii value of the character to display.
;	CX: Repeat count (see the Display_Character routine).
;
; Function Code 11 - Set Colors
;    Input:
;	BH: Color register selection.  If BH = 0, then the color value
;	    in BL is used for the background color (for alphanumeric
;	    modes) - legal values are from 0-15 (bits 0-3), with bit 4
;	    accepted as the high-intensity flag.  This may also be used 
;	    for medium-resolution graphics mode, except that the high-
;	    intensity flag is ignored.
;	    If BH = 1, then the value in BL will contain the 'pallette'
;	    used for medium-resolution graphics (320 by 200).
;	BL: Value to use for color register.  In medium-resolution
;	    graphics mode, two different pallettes are used:
;	    If BL=0, then:
;			Color 0 is the background color
;			Color 1 is green
;			Color 2 is red
;			Color 3 is yellow
;	    If BL=1, then:
;			Color 0 is the background color
;			Color 1 is cyan
;			Color 2 is magenta
;			Color 3 is white
;
; Function Code 12 - Write Graphics Pixel
;    Input:
;	DX: Vertical position (row number, from 0-199)
;	CX: Horizontal position (column, from 0-319, or 0-639)
;	AL: Color Value (0-1 for high-resolution graphics, or
;	    0-3 for medium-resolution graphics).  If the most-
;	    significant bit of AL is set, then the pixel value
;	    will be XORed with the existing screen contents.
;
; Function Code 13 - Read Graphics Pixel
;    Input:
;	DX: Vertical position (row number, from 0-199).
;	CX: Horizontal position (column, from 0-319, or 0-639).
;    Output:
;	AL: Pixel value at the specified location.
;
; Function Code 14 - Terminal Display
;    Input:
;	AL: Character to display / control character.  Carriage
;	    return, line feed, backspace and bell are handled
;	    as commands, not displayable characters.
;	BL: Foreground color (if in medium-resolution graphics mode).
;
; Function Code 15 - Determine Video Status
;    Output:
;	AL: Currently selected video mode (see Set Video Mode)
;	AH: Screen width in characters.
;	BH: Currently displayed page.
;
; Function Code 102 - Set Ultra-compatible display mode
;    Input:
;	AL: 00H to enable standard display, or 0FFH to enable
;	    ultra-compatible display.  In this mode, all low-level
;	    functions in the terminal display routines are vectored
;	    back through the video interrupt.
;
Public	Video_Io_Interrupt

	PushReg	<bp,ds>
	sti				;Interrupts:= Enabled
	cld				;Move_Direction:= Forward
	mov	Video_Active,True	;Video_Active:= True
	call	DS_to_Rom_Data		;DS:= Rom Data
	mov	bp,ax			;No, save the input parameters

	mov	Video_Segment,Screen_Segment ;Video_Segment:= Color Card Area
	mov	al,Byte Ptr Io_Config
	and	al,Video_Mode_Mask
	cmp	al,Video_Mode_Mono	;If (Monochrome Card Selected)
	jne	Vi1
	mov	Video_Segment,Mono_Segment ;	Video_Segment:= MonoChrome Card

Vi1:	mov	al,Byte Ptr Number_of_Columns
	cmp	al,Old_Columns		;If (Screen Width Changed)
	jne	Vi7			;	Change the page specifications

	mov	al,Video_Mode		;AL:= Current Video Mode
	cmp	al,Old_Video_Mode	;If (Video Mode has been Changed)
	jne	Vi5			;	Set up for the new mode

Vi2:	cmp	ah,14			;If ((Function eq Terminal_Display) And
	jne	Vi3
	cmp	Super_Compatible,False	;    (Super_Compatible Mode eq False))
	jne	Vi3

	and	ax,7			;	AX:= Truncated Byte Offset
	shl	ax,1			;	AX:= Word Offset
	xchg	bp,ax			;	BP:= Word Offset, AL:=char
	call	cs:Fast_Routine[bp]	;	Execute Fast Terminal Display
	jmp	Vi4			;	Return

Vi3:	cmp	ah,Vmax			;Else 	If (Index Not In Range)
	jae	Vi8			;		Check Extended Function

	mov	al,ah			;	(AL:= Video SubFunction Number)
	xor	ah,ah			;	AX:= Byte Offset
	shl	ax,1			;	AX:= Word Offset
	xchg	ax,bp			;	BP:= Word Offset, AL:= Char
	call	cs:Video_Routine[bp]	;	Execute Video SubFunction

Vi4:	mov	Video_Active,False	;Video_Active:= False
	PopReg	<ds,bp>
	iret				;Return


; Update video variables if the mode was POKEd into memory
;---------------------------------------------------------
;
Vi5:	mov	Old_Video_Mode,al	;Old_Video_Mode:= Current Video Mode
	call	Set_Video_Vars		;Initialize the other video variables
	jmp	Short Vi2		;Continue


; Handle POKEs into Numer_of_Columns
;------------------------------------
;
Vi7:	call	Set_Screen_Width	;Set the new screen width
	jmp	Short Vi1		;Continue


; Check for Extended Video Control Functions
;-------------------------------------------
;
Vi8:	cmp	ah,102			;If (Function eq Set_Super_Compatible)
	jne	Vi4
	mov	ax,bp			;       get back value to set from bp
	call	Set_Super_Compatible	;	Super_Compatible:= AL
	jmp	Vi4			;Continue


Video_Routine equ this Word
;--------------------------
;	-Jump table for video routines
;
	dw	Set_Video_Mode		;If AH = 0, Set video mode
	dw	Set_Cursor_Style	;If AH = 1, Set cursor style
	dw	Set_Cursor_Position	;If AH = 2, Set cursor position
	dw	Get_Cursor_Position	;If AH = 3, Get the cursor position
	dw	Get_LPen_Status		;If AH = 4, Read the light pen status
	dw	Set_Display_Page	;If AH = 5, Select a page for display
	dw	Scroll_Up		;If AH = 6, Scroll up page
	dw	Scroll_Down		;If AH = 7, Scroll down page
	dw	Read_Char		;If AH = 8, Read character at cursor
	dw	Disp_Char_Attr		;If AH = 9, Write char + attribute
	dw	Disp_Char		;IF AH = 10, Write character only
	dw	Set_Color		;If AH = 11, Set color register
	dw	Display_Pixel		;If AH = 12, Write graphics pixel
	dw	Read_Pixel		;If AH = 13, Read graphics pixel
	dw	Terminal_Display	;If AH = 14, Display as terminal
	dw	Get_Video_Status	;If AH = 15, Get the video status

Vmax	equ	($-Video_Routine)/2


Fast_Routine equ this Word
;-------------------------
;	-Special Fast Display Routines for Terminal-Display-Mode
;
	dw	Terminal_Display	;If Mode=0, use standard video routine
	dw	Terminal_Display	;If Mode=1, use standard video routine
	dw	Terminal_Display_8025	;If Mode=2, use fast 80x25 routine
	dw	Terminal_Display_8025	;If Mode=3, use fast 80x25 routine
	dw	Terminal_Display	;If Mode=4, use standard video routine
	dw	Terminal_Display	;If Mode=5, use standard video routine
	dw	Terminal_Display_Hres	;If Mode=6, use fast high-res routine
	dw	Terminal_Display_Mono	;If mode=7, use fast monochrome routine

Video_Io_Interrupt EndP


; Source File Includes
;---------------------
;
	include FastHRes.asm		;High-Resolution terminal mode display
	include Fast8025.asm		;Fast Text Mode Terminal Display
	include	DispWrt.asm		;Standard Write Char & Pixel
	include DispRd.asm		;Standard Read Char & Pixel
	include Scroll.asm		;Scroll Routines

page
;----------------------------------------------------------------------
Set_Super_Compatible Proc Near
;-----------------------------
;	1) Set_Super_Compatible is called to enable and disable the
;	   super-compatible display mode.
;	2) When this mode is enabled, all terminal display functions
;	   vector their low-level actions back through the video
;	   interrupt, permitting right-to-left display, etc.
;	3) Entry Register Values:
;		AL -> 0 for standard display, 0FFH for Super-Compatible display
;
	mov	Super_Compatible,al	;Set the new compatibility flag
	ret				;Return

Set_Super_Compatible EndP

page
;----------------------------------------------------------------------
Set_Video_Mode Proc Near; (9_Jul_85)
;-----------------------------------
;	1) Set_Video_Mode is called to set up the ROM for a new screen format.
;	2) All display memory is cleared, and the CRT controller is programmed
;	   for the new display format.  In addition, the screens are cleared,
;	   page 0 is displayed, and the ROM variables are set to a value
;	   appropriate for the new format.
;	3) Input Register Values:
;		AL -> Mode value
;
Public	Set_Video_Mode

	PushReg	<ax,bx,cx,dx,si,ds,es>
	call	DS_to_Rom_Data		;DS:= Rom Data Segment

	mov	ah,Byte Ptr IO_Config	;Get the current I/O configuration
	and	ah,Video_Mode_Mask	;Mask off the video select bits
	cmp	ah,Video_Mode_Mono	;Is the monochrome board active?
	jne	Svm0			;No, continue with the mode set

	mov	al,7			;Yes - force mono for compatibility(?)
Svm0:	call	Set_Video_Vars		;Initialize the video variables
;	call	Wait_Vertical_Sync	;Update screen hardware during VSYNC
	mov	al,Disable_Video_Color	;Get command for color video-disabled
	cmp	Video_Mode,7		;Is this monochrome?
	jne	Svm1			;No, continue

	mov	al,Disable_Video_Mono	;Yes - get monochrome video disable
Svm1:	mov	dx,Video_Base		;Point DX to the CRT mode-select port
	add	dx,Crt_Mode_Select
	out	dx,al			;Set the video-disabled mode
	mov	al,0			;Get the video hardware mode variable
	call	Out_Video_Mode		;Disable the CRT video
	call	Get_Video_Parms		;Point ES:SI to the video parameters
	mov	dx,Video_Base		;Point to the base reg of the CRT
	call	Set_Vid_Parms

	call	Clear_Video_Ram		;Erase all screens for this card
	mov	al,Current_Colors	;Get the current color values
	add	dx,Crt_Color_Select	;Point to the color selection port
	out	dx,al			;Set the default colors
;	call	Wait_Vertical_Sync	;Wait for vertical sync to occur
;	call	Wait_Vertical_Sync	;Wait for another sync pulse

	mov	al,0			;Get page number 0
	call	Set_Display_Page	;Display page 0
	mov	al,Video_HW_Mode	;Get the correct hardware mode
	add	dx,Crt_Mode_Select-Crt_Color_Select ;Point DX to mode select
	out	dx,al			;Enable the video hardware!

	PopReg	<es,ds,si,dx,cx,bx,ax>
	ret

Set_Video_Mode	EndP

page
;----------------------------------------------------------------------
Set_Vid_Parms Proc Near; (20_Jun_85)
;-----------------------------------
;	1) This routine writes the video parameters out to the controller chip.
;	2) Entry Register Values:
;		ES:SI -> Pointer to Base of Video Param
;
Public	Set_Vid_Parms

	mov	bl,0			;Keep register number in BL
	mov	cx,Size Vid_Parms	;Get the size of the video parameters
Svm2:	mov	al,bl			;Repeat	Get current register number
	out	dx,al			;	Send it to the CRT controller
	lods	Byte Ptr es:[si]	;	Get a video parameter

	inc	dx			;	Point to the parameter register
	out	dx,al			;	Output it to the CRT controller
	dec	dx			;	Point back to the control reg

	inc	bl			;	Point to the next register
	loop	Svm2			;Until (All parameters have been set)
	ret				;Return

Set_Vid_Parms	EndP

page
;**********************************************************************
; SET_VIDEO_VARS:
;
;	Set_Video_Vars is called when a new video mode has been
; programmed (via the Set_Video_Mode call), or when a mode change
; has been detected.  It initializes the global variables for a
; new video display format.
;
; Input:
;	AL: New mode value
;**********************************************************************
SET_VIDEO_VARS PROC NEAR
	PUBLIC	SET_VIDEO_VARS

	PUSHREG	<AX,BX,CX,SI,DI,ES>
	CMP	AL,7			;Is this a valid mode?
	JBE	SVV0			;Yes - change the variables to new values
	JMP	SVV6			;No, ignore it (must be 3rd party stuff)

SVV0:
;**	CMP	VIDEO_MODE,2		;Was the alpha cursor on?
;**	JB	SVV15
;**
;**	CMP	VIDEO_MODE,3
;**	JA	SVV15
;**
;**	CALL	SET_MONITOR_STACK	;Make sure local stack when cursor ram enabled
;**	CALL	ENABLE_CURSOR_RAM
;**	MOV	ES,VIDEO_SEGMENT	;Get the old cursor segment
;**	MOV	BX,CURSOR_PTR		;Get the old cursor offset
;**	MOV	BYTE PTR ES:[BX],0	;Turn off the old cursor
;**	CALL	DISABLE_CURSOR_RAM
;**	CALL	SET_USER_STACK		;Go back to user stack

SVV15:	CMP	AL,2			;Are we changing to mode 2 or 3?
	JB	SVV17

	CMP	AL,3
	JA	SVV17

	OR	CURSOR_FLAG,ALPHA_CURSOR_ON	;Yes, allow the cursor to display
SVV17:	MOV	VIDEO_MODE,AL		;Set the global display variable
	MOV	OLD_VIDEO_MODE,AL	;Set the 'old mode' to the new one
	MOV	BX,COLOR_CARD		;Set I/O addr as default to color card
	MOV	VIDEO_SEGMENT,SCREEN_SEGMENT ;Point to the color card RAM
	CMP	VIDEO_MODE,7		;Is this a monochrome set-mode?
	JNE	SVV1			;No, have correct base address

	MOV	BX,MONO_CARD		;Yes - point to the monochrome card
	MOV	VIDEO_SEGMENT,MONO_SEGMENT   ;Point to the mono card RAM
SVV1:	MOV	VIDEO_BASE,BX		;Update the global variable
	CBW				;Convert mode into a word
	MOV	BX,AX			;Copy mode into BX
	SHR	AX,1			;Turn mode into a page type
	CMP	VIDEO_MODE,7		;Is this a monochrome set mode?
	JNE	SVV2			;No, all set

	INC	AX			;Yes - use special vars for mono
SVV2:	MOV	CL,SIZE VID_VARS	;Get the size of the CRT variables
	MUL	CL			;Generate offset of the right set

	PUSH	DS			;Save the current data segment
	PUSH	CS			;Point DS to the ROM's code segment
	POP	DS

	push	ds
	call	DS_to_Rom_Data
	push	ds
	pop	es			;ES:= Rom Data
	pop	ds

	LEA	SI,ROM_VARS		;Point to the start of the ROM vars
	ADD	SI,AX			;Now point to the right set
	LODSB				;Read the number of pages
	MOV	NUMBER_OF_PAGES,AL	;Place it in the global variable
	LODSW				;Write the global page size
	MOV	ES: PAGE_SIZE,AX
	DEC	AX			;Make it into a page size mask
	MOV	TEXT_SCREEN_MASK,AX	;Save it in the global variable
	LODSW		 		;Update the page width
	MOV	ES: NUMBER_OF_COLUMNS,AX
	MOV	OLD_COLUMNS,AL		;...and save the 'old' copy for POKE checks
	LODSW				;Write the column increment
	MOV	COLUMN_INCREMENT,AX
	LODSW				;... and the row increment
	MOV	ROW_INCREMENT,AX
	LODSB				;Finally, get the color select
	MOV	ES: CURRENT_COLORS,AL
	POP	DS			;Restore the data segment

	MOV	AL,CS:CRT_MODE[BX]	;Get the right mode select
	MOV	VIDEO_HW_MODE,AL	;Place it in memory for updates
	MOV	AX,OFFSET SCREEN	;Point to the base of the screen RAM
	MOV	BX,0			;Use BX as word pointer to PAGE_BASE
	MOV	CL,NUMBER_OF_PAGES	;Get the number of valid pages
	MOV	CH,0

SVV3:	CMP	VIDEO_MODE,7		;Entering monochrome mode?
	JNE	SVV4			;No, set normal base addresses

	MOV	PAGE_START[8 * 2],0	;Yes - set monochrome page start
	MOV	PAGE_BASE[8 * 2],0	;Set monochrome page base
	JMP	SHORT SVV5		;Only set monochrome info!

SVV4:	MOV	PAGE_START[BX],AX	;Set a page's start address
	MOV	PAGE_BASE[BX],AX	;Set the page's current top-of-page
SVV5:	MOV	CURSOR_POSITION[BX],0	;Home this page's cursor
	ADD	AX,PAGE_SIZE		;Point to the next page
	ADD	BX,TYPE PAGE_BASE	;Point to the next PAGE_BASE
	LOOP	SVV3			;Set base addresses for all pages

;*** START OF VIDEO PAGING CHANGE

	CALL	INIT_PAGE_MAP		;Set the page map back to normal

;*** END OF VIDEO PAGING CHANGE


	MOV	AX,DEFAULT_CURSOR_STYLE	;Get the default cursor style
	MOV	CURSOR_STYLE,AX		;Initialize the current cursor style
	MOV	AL,NUMBER_OF_ROWS-1	;Get the number of the last line
	MOV	LAST_ROW,AL		;Set it up for global access
	MOV	VIDEO_ACTIVE,FALSE	;Flag that no scrolling is happening

SVV6:
;**	CMP	VIDEO_MODE,4		;Are we in graphics mode?
;**	JB	SVV7			;No, skip

;**	CMP	VIDEO_MODE,6
;**	JA	SVV7			;No, skip

;**	PUSH	DX
;**	MOV	AH,TRUE
;**	MOV	DX,CURSOR_POSITION[0]
;**	CALL	Set_Graphic_Cursor
;**	POP	DX

SVV7:	POPREG	<ES,DI,SI,CX,BX,AX>
	RET

SET_VIDEO_VARS ENDP

page
;----------------------------------------------------------------------
Set_Screen_Width Proc Near
;-------------------------
;	1) Set_Screen_Width is called to set up the new screen width for
;	   a program.
;	2) Input Register Values
;		AL -> Number of columns to be displayed.
;	3) Output Values:
;		Page_Start -------- 0
;		Old_Columns ------- New Screen Width (Columns)
;		Row_Increment ----- Columns * 2
;		Page_Size --------- Columns * 2 * Number_of_Rows
;		End_of_Page ------- Columns * 2 * Number_of_Rows - 1
;		Text_Screen_Mask -- Video Ram Size - 1
;		Super_Compatible -- True
;
	PushReg	<ax,cx,dx>
	mov	Page_Start[0],0		;Page_Start:= 0
	mov	Old_Columns,al		;Old_Columns:= New Screen Width
	mov	ah,0			;AX:= Characters per Row
	shl	ax,1			;(Include attributes in size)
	mov	Row_Increment,ax	;Row_Increment:= Bytes per Row

	mov	cx,Number_of_Rows
	mul	cx			;AX:= Number_of_Rows * Bytes per Row
	mov	Page_Size,ax		;Page_Size:= Number of Bytes per Page

	dec	ax			;Find address of last byte on page
	mov	End_of_Page,ax		;Save the end of page address

	mov	Text_Screen_Mask,Screen_Bytes-1 ;Use all of RAM for accesses
	mov	Super_Compatible,True	;Force use of slow video routines
	PopReg	<dx,cx,ax>
	ret

Set_Screen_Width EndP

page
;----------------------------------------------------------------------
Get_Video_Status Proc Near; (9_Jul_85)
;-------------------------------------
;	Get_Video_Status is used to read the current status of the
; video sub-system.  It returns the currently displayed page, CRT mode,
; screen width, and the current scrolling mode.
;
; Output:
;	AL: Currently selected video mode (see Set Video Mode)
;	AH: Screen width in characters.
;	BH: Currently displayed page.
;
Public	Get_Video_Status

	push	ds
	call	DS_to_Rom_Data
	mov	al,Video_Mode		;Get the current video mode
	mov	ah,Byte Ptr Number_of_Columns ;Get the page width
	mov	bh,Displayed_Page	;... and the displayed page
	pop	ds
	ret

Get_Video_Status EndP

page
;----------------------------------------------------------------------
Set_Display_Page Proc Near; (18_Jul_85)
;--------------------------------------
; 	1) Set_Display_Page is called to determine what page will be
;	   shown on the monitor.
;	2) Input Register Values:
;		AL -> Page number to be displayed.
;	3) Output Values:
;		Correct cursor position for this page
;		Current_Page
;		Displayed_Page
;		Crt_Start_Address
;
Public	Set_Display_Page

	PushReg	<ax,bx,dx,si,ds>
	call	DS_to_Rom_Data		;DS:= Rom Data



;*** START OF VIDEO PAGING CHANGE

	call	swap_pages		;Swap the page of memory and video variables

;*** END OF VIDEO PAGING CHANGE



	mov	bh,al			;Get page number in BH
	call	Set_Io_Page		;Set up I/O variables for page

	mov	Displayed_Page,al	;Keep track of page being displayed
	mov	Old_Displayed_Page,al	;Keep copy of old displayed page
	cbw				;Convert page number to a word
	mov	bx,ax			;Get page number in BX
	shl	bx,1			;Make BX into a word offset
	mov	si,Page_Base[bx]	;Get the current page bage
	cmp	Video_Mode,7		;Are we in monochrome mode?
	jne	SDP1			;No, have the correct address

	mov	si,Page_Base[8 * 2]	;Yes - treat monochrome as page 8
SDP1:	shr	si,1			;Make it into a hardware address
	mov	Crt_Start_Address,si	;Save display address
	mov	dx,Cursor_Position[bx]	;Get the current cursor position

	push	bx			;Save the page pointer
	mov	bh,Displayed_Page	;Get the new display page number
	call	Set_Cursor_Position	;Update the hardware cursor
	pop	bx			;Restore the page pointer

	mov	al,Video_On		;Turn on the video
	call	Out_Video_Mode		;Set up the CRT for the right mode
	mov	bx,si			;Get the CRT start address
	mov	al,Crt_Start_Port	;Get the CRT start addr port number
	call	Send_Video_Word		;Display the right page!

	PopReg	<ds,si,dx,bx,ax>
	ret

Set_Display_Page EndP

page
;----------------------------------------------------------------------
Set_IO_Page Proc Near; (17_Jul_85)
;---------------------------------
;	1) Set_IO_Page is called prior to reading, writing or scrolling
;	   information on a page, so that calling routines will properly
;	   wrap-around when information is displayed.
;	2) Entry Register Values:
;		BH -> Page to be displayed.
;	3) Exit Values:
;		Current_Page --- Number of the Current Page
;		Start_of_Page -- Pointer to First Byte of Current Page
;		End_of_Page ---- Pointer to Last Byte of Current Page
;
	push	bx
	cmp	Video_Mode,3		;If (alphanumeric mode ne True)
	jbe	SioP1

	mov	bh,0			;	New_Page_Number:= Forced to 0
SioP1:	mov	Current_Page,bh		;Current_Page:= New_Page_Number

	mov	bx,0			;BX:= 0 (PAGE_START)
	cmp	Video_Mode,7		;If (MonoChrome mode ne True)
	je	SioP2

	mov	bl,Current_Page
	mov	bh,0			;	BX:= New Page Number
	shl	bx,1			;	(Make page number word offset)
	mov	bx,Page_Start[bx]	;	BX:= Pointer to start of page

SioP2:	mov	Start_of_Page,bx	;Start_of_Page:= Pointer to Start Addr

	add	bx,Page_Size		;(BX:= Pointer to 1st byte next page)
	dec	bx			;(BX:= Pointer to last byte this page)
	mov	End_of_Page,bx		;End_of_Page:= Pointer to Last Addr
	pop	bx
	ret				;Return

Set_IO_Page	EndP

page
;----------------------------------------------------------------------
Set_Cursor_Position Proc Near; (9_Jul_85)
;----------------------------------------
;	1) Set_Cursor_Position is called to set up the ROM variables
;	   for a screen coordinate. Also updates the hardware cursor if
;	   the page specified is the currently active page.
;	2) Entry Register Values:
;		BH -> Page number
;		DL -> Column number
;		DH -> Row number
;	3) Exit Register Values:
;		Sets global variables CURRENT_PAGE and CURSOR_POSITION.
;
Public	Set_Cursor_Position

	PushReg	<ax,bx,dx,ds>
	call	DS_to_Rom_Data		;DS:= Rom Data
	mov	Current_Page,bh		;Set the page for future I/O
	mov	bl,bh			;Get the page as a word in BX
	mov	bh,0
	shl	bx,1			;Make it a word offset
	mov	Cursor_Position[bx],dx	;Update the cursor position variable
	mov	al,Current_Page		;Get the new page number
	cmp	al,Displayed_Page	;Are we displaying this page?
	jne	SCP3			;No, finished with update

	mov	al,dh			;Get the row number
	mul	Byte Ptr Number_of_Columns ;Generate offset of start of row
	add	al,dl			;Add column to get right offset
	adc	ah,0
	add	ax,Crt_Start_Address	;Add in start of current page
	cmp	ax,Screen_Bytes / 2	;Wrapped around?
	jbe	SCP1			;No, all is OK

	sub	ax,Screen_Bytes		;Yes - wrap-around to rest of screen

SCP1:	cmp	Video_Mode,4		;If (We're in graphics mode)
	jb	SCP2
	cmp	Video_Mode,6
	ja	SCP2

;**	call	Set_Graphic_Cursor	;	Update the software cursor
	jmp	SCP3			;	Return

SCP2:	mov	bx,ax			;Store CRT address in BX
	mov	dx,Video_Base		;Point DX to the CRT chip
	mov	al,Crt_Cursor_Start	;Point to the CRT cursor address reg
	out	dx,al			;Select M.S. byte of cursor addr
	inc	dx			;Now, point to the parameter port
	mov	al,bh			;Get the M.S. cursor addr byte
	out	dx,al			;Output it...
	dec	dx			;Point back to the control port
	mov	al,Crt_Cursor_Start+1
	out	dx,al			;Select L.S. byte of cursor address
	inc	dx			;Point to parameter port again
	mov	al,bl			;Output L.S. byte of address
	out	dx,al

SCP3:	PopReg	<ds,dx,bx,ax>
	ret

Set_Cursor_Position EndP

page
;----------------------------------------------------------------------
Get_Cursor_Position Proc Near; (9_Jul_85)
;----------------------------------------
; 	1) Get_Cursor_Position is called to read the cursor's current
;	   position from the global variables.  In addition, the starting
;	   and ending line defining the hardware cursor are returned.
;	2) Input Register Values:
;		BH -> Current page number
;	3) Output Register Values:
;		DL -> Column number
;		DH -> Row number
;		CH -> Cursor's starting scan line
;		CL -> Cursor's ending scan line
;
Public	Get_Cursor_Position

	PushReg	<bx,ds>
	call	DS_to_Rom_Data		;DS:= Rom Data

	mov	bl,bh			;Get the page number
	mov	bh,0			;Extend it to a word
	shl	bx,1			;Make it a word offset

	mov	dx,Cursor_Position[bx]	;Get the cursor's position
	mov	cx,Cursor_Style		;Get the current cursor style

	PopReg	<ds,bx>
	ret

Get_Cursor_Position EndP

page
;----------------------------------------------------------------------
Set_Cursor_Style Proc Near; (9_Jul_85)
;-------------------------------------
;	1) Set_Cursor_Style sets up the CRT controller chip for a particular
;	   cursor style.
;	2) Input Register Values:
;		CH -> Starting scan line
;		CL -> Ending scan line
;
Public	Set_Cursor_Style

	PushReg	<ax,bx,ds>		;Save registers
	call	DS_to_Rom_Data		;DS:= Rom Data

	mov	Cursor_Style,cx		;Place the cursor style in memory
	mov	al,Crt_Cursor_Style	;Get the cursor style port	

	mov	bx,cx			;Place start & end scan lines in BX
	call	Send_Video_Word		;Tell CRT controller

	PopReg	<ds,bx,ax>		;Restore registers
	ret				;Return

Set_Cursor_Style EndP

page
;**********************************************************************
; SET_COLOR: (PALLETTE, COLOR)
;
;	Set_Color is called to set the screen colors used.  It can
; select background colors, as well as the foreground colors in
; medium-resolution graphics mode.
;
; Input:
;	BH: Color register selection.  If BH = 0, then the color value
;	    in BL is used for the background color (for alphanumeric
;	    modes) - legal values are from 0-15 (bits 0-3), with bit 4
;	    accepted as the high-intensity flag.  This may also be used 
;	    for medium-resolution graphics mode, except that the high-
;	    intensity flag is ignored.
;	    If BH = 1, then the value in BL will contain the 'pallette'
;	    used for medium-resolution graphics (320 by 200).
;	BL: Value to use for color register.  In medium-resolution
;	    graphics mode, two different pallettes are used:
;	    If BL=0, then:
;			Color 0 is the background color
;			Color 1 is green
;			Color 2 is red
;			Color 3 is yellow
;	    If BL=1, then:
;			Color 0 is the background color
;			Color 1 is cyan
;			Color 2 is magenta
;			Color 3 is white
;**********************************************************************
SET_COLOR PROC NEAR

	PUSHREG	<AX,BX,CX,DX>
	MOV	AL,CURRENT_COLORS	;Get the current color values in AL
	TEST	BH,BH			;Setting background colors?
	JNZ	SCLR1			;No, continue

	AND	AL,NOT BACKGROUND_COLOR_MASK ;Yes - mask out old background
	AND	BL,BACKGROUND_COLOR_MASK;Mask out unused bits in new value
	JMP	SHORT SCLR2

SCLR1:	AND	AL,NOT FOREGROUND_COLOR_MASK ;Mask out old foreground color
	MOV	CL,5			;Shift count to put bit in right posn
	SHL	BL,CL			;Place pallette select in right bit
	AND	BL,FOREGROUND_COLOR_MASK;Mask out unused bits of new value
SCLR2:	OR	AL,BL			;Insert new color value
	MOV	DX,VIDEO_BASE		;Point to the color selection port
	ADD	DX,CRT_COLOR_SELECT
	OUT	DX,AL			;Output new colors
	MOV	CURRENT_COLORS,AL	;Save new value
	POPREG	<DX,CX,BX,AX>
	RET

SET_COLOR ENDP

page
;**********************************************************************
; GET_LPEN_STATUS:    (12_30_85)
;
;	Get_LPen_Status returns AH = 0 since no light pen is usable
;	in the Pivot II.
;**********************************************************************
GET_LPEN_STATUS PROC NEAR
	mov	ah,0		; return no light pen flag
	RET

GET_LPEN_STATUS ENDP

page
;**********************************************************************
; GET_VIDEO_PARMS:
;
;	Get_Video_Parms returns a pointer to the video parameter 
; block associated with the current video mode.
;
; Output:
;    ES:SI: Pointing to the currently active video parameters
;**********************************************************************
GET_VIDEO_PARMS PROC NEAR

	PUSHREG	<AX,CX>
	PUSH	DS			;Save the monitor data segment
	MOV	AL,VIDEO_PARMS_INTR	;Get intr # of the video parameters
	CALL	GET_INTR_PTR		;Point to the video parameter tables

	PUSH	DS			;Save pointer to video parms segment
	POP	ES			;Point ES to the video parms segment
	POP	DS			;Restore pointer to data segment
	MOV	AL,VIDEO_MODE		;Get the video mode again
	CBW				;Convert mode into a word
	SHR	AX,1			;Turn mode into a page type
	CMP	VIDEO_MODE,6		;Entering high-res graphics mode?
	JNE	GVP1			;No, pointing to right parameters

	DEC	AX			;Yes - use common graphics parameters
GVP1:	MOV	CL,SIZE VID_PARMS	;Get the size of the video parameters
	MUL	CL			;Generate offset of correct table
	ADD	SI,AX			;Now point to the right table
	POPREG	<CX,AX>
	RET
GET_VIDEO_PARMS ENDP


page
;**********************************************************************
; SET_SCREEN_ADDRESS:
;
;	Set_Screen_Address is called to update the global
; cursor pointer with the address of the current character.
; This routine will return a pointer to a CHARACTER, not
; a pixel.
;
; Input:
;	    Global variables CURRENT_PAGE, CURSOR_POSITION
;
; Output:
;    ES:DI: Pointing to current screen character
;**********************************************************************
SET_SCREEN_ADDRESS PROC NEAR

	PUSHREG	<BX,CX>
	MOV	BL,CURRENT_PAGE		;Get the current page in BX
	MOV	BH,0
	SHL	BX,1			;Make it a word offset
	MOV	CX,CURSOR_POSITION[BX]	;Get the current cursor position
	CALL	SET_SCREEN_PTR		;Set ES:DI to the right address
	POPREG	<CX,BX>
	RET

SET_SCREEN_ADDRESS ENDP

page
;**********************************************************************
; SET_SCREEN_PTR: (POSITION)
;
;	Set_Screen_Ptr is similar to Set_Screen_Address,
; except that it permits a variable screen position to be
; used.
;
; Input:
;	CX: Screen position (CH: row, and CL: column)
;
; Output:
;    ES:DI: Pointing to the correct screen location
;**********************************************************************
SET_SCREEN_PTR PROC NEAR

	PUSHREG	<AX,CX,DX>
	MOV	DL,CH			;Place row number in DX
	MOV	DH,0
	MOV	CH,DH			;Extend row and column to words
	MOV	AH,TRUE			;Indicate character mode
	CALL	VIDEO_PTR		;Point to the right location
	POPREG	<DX,CX,AX>
	RET

SET_SCREEN_PTR ENDP

page
;**********************************************************************
; VIDEO_PTR: (Alpha_Flag, Row, Column)
;
;	Video_Ptr is called to transfer a screen coordinate 
; into an absolute screen address.
;
; Input:
;	AH: If set (non-zero) then an alphanumeric mode
;	    address will be returned.  Otherwise, a
;	    graphics address will be returned.
;	CX: Column number (or horizontal position for graphics)
;	DX: Row number (or vertical position)
;
; Output:
;    ES:DI: Pointing to the actual screen address for this
;	    coordinate.
;	AX: If a graphics address was requested, then AH
;	    will contain the pixel number (0-3, or 0-7 for
;	    medium and high resolution graphics modes,
;	    respectively), and AL will contain the number
;	    of bits/pixel.
;**********************************************************************
VIDEO_PTR PROC NEAR

	PUSHREG	<BX,CX,DX,BP>

	; Generate offset of start of line
	TEST	AH,AH			;Are we in alphanumeric mode?
	JNE	VDP1			;Yes - continue

	SHR	DX,1			;No, divide vertical position by 2
	PUSHF				;Save Even/Odd scan line status on stack
VDP1:	MOV	AL,0			;Place only alpha mode flag in AX
	MOV	BP,AX			;Save alpha mode flag in BP
	MOV	AX,DX			;Get the row number
	TEST	BP,BP			;In alpha mode?
	JE	VDP2			;No, generate graphics address

	MUL	ROW_INCREMENT		;Yes - mult by bytes between rows
	JMP	VDP3
VDP2:	PUSH	CX			;No, save horizontal position
	MOV	CL,SCAN_LINE_INCREMENT	;Get the offset between scan lines
	MUL	CL			;Multiply by bytes between scan lines
	POP	CX

VDP3:	MOV	DX,AX			;Save start-of-line offset in DX
	TEST	BP,BP			;Are we in graphics mode?
	JNE	VDP5			;No, generate alphanumeric address

	MOV	AH,CL			;Yes - get the horizontal position
;
; Generate pixel number for graphics mode & adjust horizontal position
;
	SHR	CX,1			;Divide horiz position by pixels/byte
	SHR	CX,1			;Assume medium resolution...
	CMP	VIDEO_MODE,6		;Are we in high-resolution mode?
	JNE	VDP4			;No, use 2 bits/pixel

	AND	AH,8 - 1		;Yes - have 1 bit/pixel (8 / byte)
	MOV	AL,1			;Set return parameter to 1 bit/pixel
	SHR	CX,1			;Adjust horiz position for high-res
	JMP	SHORT VDP6

VDP4:	AND	AH,4 - 1		;Get medium resolution pixel number
	MOV	AL,2			;Set return parm as 2 bits/pixel
	JMP	SHORT VDP6
;
; Add line offset, and create screen pointer
;
VDP5:	CMP	COLUMN_INCREMENT,1	;Is the column increment > 1?
	JLE	VDP6			;No, already have line offset

	SHL	CX,1			;Yes - get offset in the current row
VDP6:	ADD	DX,CX			;Now have screen offset in DX
	MOV	BL,CURRENT_PAGE		;Get the currently active page
	MOV	BH,0			;Extend it to a word
	SHL	BX,1			;Make it a word offset
	MOV	BX,PAGE_BASE[BX]	;Point to start of the current page
	CMP	VIDEO_MODE,7		;In monochrome text mode?
	JNE	VDP65			;No, have correct page base

	MOV	BX,PAGE_BASE[8 * 2]	;Yes - treat monochrome as page 8
VDP65:	ADD	DX,BX			;Add in offset - now have pointer
	CMP	VIDEO_MODE,3		;Is the hardware in graphics mode?
	JBE	VDP7			;No, continue

	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	VDP7			;Yes - cannot use graphics!

	TEST	BP,BP			;Generating graphics address?
	JE	VDP7			;Yes - use full screen addressing

	CMP	DX,OFFSET END_OF_PART	;Alpha/graphics - past end of part 1?
	JBE	VDP8			;No, continue

	SUB	DX,SCREEN_BYTES SHR 1	;Yes - wrap to start of partition
	JMP	SHORT VDP8

VDP7:	CMP	DX,END_OF_PAGE		;Beyond end of screen?
	JBE	VDP8			;No, continue

	SUB	DX,PAGE_SIZE		;Yes - wrap-around
VDP8:	TEST	BP,BP			;In alphanumeric mode?
	JNE	VDP9			;Yes - return

	POPF				;Restore Even/Odd scan line flag
	JNC	VDP9			;On even scan line in graphics mode

	XOR	DX,SCREEN_PARTITION_OFS	;On odd scan line - point to it
VDP9:	MOV	ES,VIDEO_SEGMENT	;Prepare return address...
	MOV	DI,DX			;...in ES:DI
	POPREG	<BP,DX,CX,BX>
	RET

VIDEO_PTR ENDP

page
;**********************************************************************
; EXPAND_FONT: (FONT_BYTE)
;
;	Expand_Font is called in the medium-resolution
; graphics mode to expand a font character (byte) into
; a word suitable for display.  This is accomplished by
; doubling each bit in the byte.
;
; Input:
;	AL: Font byte
;
; Output:
;	AX: Expanded font word
;**********************************************************************
EXPAND_FONT PROC NEAR

	PUSHREG	<BX,CX>
	MOV	CX,8			;Repeat for each bit of the font byte
EF1:	SHL	BX,1			;Shift dest left for next bit
	SHL	BX,1
	SHL	AL,1			;Get a bit of the font into carry
	JNC	EF2			;Bit not set - continue

	OR	BX,11B			;Bit set - set correct bits in dest
EF2:	LOOP	EF1			;Continue until whole word expanded
	MOV	AX,BX			;Place result in AX
	POPREG	<CX,BX>
	RET

EXPAND_FONT ENDP

page
;**********************************************************************
; COMPRESS_FONT: (FONT_WORD)
;
;	Compress_Font takes a medium-resolution graphics word and
; compacts it into a single byte.  Of the 8 pixels in the graphics
; word, any pixel with a color other than the background color is
; considered 'ON'.
;
; Input:
;	AX: Graphics word
;
; Output:
;	AL: Packed font byte
;	AH: will not be informative
;**********************************************************************
COMPRESS_FONT PROC NEAR

	PUSHREG	<BX,CX,DX>
	MOV	BX,1100000000000000B	;Mask for a single pixel
	MOV	CX,8			;Number of pixels to work on
CF1:	SHL	DL,1			;Align dest for next bit
	PUSH	AX			;Save font value
	AND	AX,BX			;Mask out a single pixel
	JZ	CF2			;Pixel not set - continue

	OR	DL,1			;Pixel on - set dest bit
CF2:	POP	AX			;Restore word value
	SHR	BX,1			;Shift BX for new pixel
	SHR	BX,1
	LOOP	CF1			;Loop till all 8 bits gotten

	MOV	AL,DL			;Get return parameter
	POPREG	<DX,CX,BX>
	RET

COMPRESS_FONT ENDP

page
;**********************************************************************
; INCREMENT_POSITION: (DIRECTION, INCREMENT, POINTER1, POINTER2)
;
;	Increment_Position is used to advance a pair of
; screen pointers either forward or backward by a fixed
; amount.  This routine will detect and correct overflow
; problems.
;
; Input:
;	AH: Direction flag.  If AH is set (non-zero), then
;	    the increment will be subtracted from the
;	    pointers.  Otherwise, the increment will be
;	    added to them.
;	DX: Increment to add to pointers
;	SI: Screen pointer
;	DI: Screen pointer
;
; Output:
;	SI: Updated screen pointer
;	DI: Updated screen pointer
;**********************************************************************
INCREMENT_POSITION PROC NEAR

	PUSHREG	<BX,CX,BP>
	MOV	CX,START_OF_PAGE	;Point to the start of this page
	MOV	BX,PAGE_SIZE		;Get the page size, too
	MOV	BP,END_OF_PAGE		;...as well as the end of the page
	CMP	VIDEO_MODE,3		;In alphanumeric mode?
	JBE	IP2			;Yes - parameters set correctly

	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	IP2			;Yes - parameters are OK

	MOV	BX,SCREEN_BYTES	SHR 1	;Get the graphics partition size
	CMP	SI,OFFSET END_OF_PART	;Are we in the first partition?
	JA	IP1			;No, set second partition parms

	MOV	BP,OFFSET END_OF_PART	;Partition 1 - set end of partition
	JMP	SHORT IP2		;Continue

IP1:	MOV	CX,OFFSET END_OF_PART+1	;Partition 2 - set start address
IP2:	TEST	AH,AH			;Are we adding the increment?
	JNE	IP4			;No, subtract it

	ADD	SI,DX			;Yes - advance pointer1
	CMP	SI,BP			;Beyond end of page on pointer1?
	JBE	IP3			;No, continue

	SUB	SI,BX			;Yes - wrap-around
IP3:	ADD	DI,DX			;Advance pointer2
	CMP	DI,BP			;Pointer2 beyond end of page?
	JBE	IP6			;No, return

	SUB	DI,BX			;Yes - wrap-around pointer1
	JMP	SHORT IP6

IP4:	SUB	SI,DX			;Subtract increment from pointer1
	CMP	SI,CX			;Below start of page?
	JGE	IP5			;No wrap-around required

	ADD	SI,BX			;Wrap-around pointer1
IP5:	SUB	DI,DX			;Subtract increment from pointer2
	CMP	DI,CX			;Below start of page?
	JGE	IP6			;No wrap required - return

	ADD	DI,BX			;Wrap pointer2 around
IP6:	POPREG	<BP,CX,BX>
	RET

INCREMENT_POSITION ENDP

page
;**********************************************************************
; CLEAR_VIDEO_RAM:
;
;	Clear_Video_RAM is called when a mode change has occurred - 
; it clears all of the pages.
;**********************************************************************
CLEAR_VIDEO_RAM PROC NEAR
PUBLIC	CLEAR_VIDEO_RAM

	PUSHREG	<AX,CX,DI,ES>
	MOV	ES,VIDEO_SEGMENT	;Point ES to the screen segment
	LEA	DI,SCREEN		;Point to the base screen address
	MOV	AX,(DEFAULT_ATTRIBUTE SHL 8) OR ' '	;Assume alpha mode
	CMP	VIDEO_SEGMENT,MONO_SEGMENT ;Is this the monochrome card?
	JE	CVR1			;Yes - have right data to fill with

	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JBE	CVR1			;No, fill screen with blanks

	XOR	AX,AX			;Color card graphics mode - fill w/00H
CVR1:	MOV	CX,SCREEN_WORDS		;Get the screen size
	REP	STOS WORD PTR ES: [DI]	;Clear screen
	POPREG	<ES,DI,CX,AX>
	RET

CLEAR_VIDEO_RAM ENDP

page
;**********************************************************************
; CLEAR_LINE: (LINE_TYPE, LINE_START_ADDRESS, ATTRIBUTE, LENGTH)
;
;	Clear_Line is used to erase a line of information
; from the screen.  Either individual scan lines, or char-
; acter lines may be erased.  The 'line' is actually a
; string of sequential characters of arbitrary length.
; Also, the attribute for the new line may be set to any
; desired value.
;
; Input:
;	AL: Line-Type flag.  If set (non-zero), only a single
;	    scan line is cleared.  Otherwise (if AL is 0),
;	    an entire character line is cleared.
;    ES:DI: Start address of line
;	BH: Attribute byte for the line
;	CX: Length of the line in characters
;**********************************************************************
CLEAR_LINE PROC NEAR

	PUSHREG	<AX,BX,CX,DX,SI,DI,BP>
	MOV	BP,1			;Assume 1 scan line / char line
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	CL1			;Yes - clear text lines

	CMP	VIDEO_MODE,3		;Is graphics mode enabled?
	JBE	CL1			;No, continue

	TEST	AL,AL			;Clearing a single scan line?
	JNE	CL1			;Yes - count already set to 1

	MOV	BP,FONT_HEIGHT		;No, get the number of scan lines
CL1:	PUSH	DI			;Save the line's start address
	PUSH	CX			;Save the line length, too
CL2:	CMP	COLUMN_INCREMENT,1	;Are the characters bytes or words?
	JE	CL3			;Bytes - have the right count already

	SHL	CX,1			;Words - include attrib byte in count
CL3:	MOV	AH,FALSE		;Clear the wrap-around flag
	MOV	SI,DI			;Point to area to be cleared
	ADD	SI,CX			;Add in length of line
	CMP	VIDEO_MODE,7		;Are we in monochrome mode?
	JE	CL4			;Yes - don't bother with graphics

	CMP	VIDEO_MODE,3		;Are we in graphics mode?
	JBE	CL4			;No, continue

	CMP	DI,OFFSET END_OF_PART	;Are we in the second partition?
	JBE	CL4			;No, check first partition

	CMP	SI,OFFSET END_OF_SCREEN	;Yes - will the clear wrap around?
	JBE	CL6			;No, continue

	MOV	DX,OFFSET END_OF_SCREEN+1 ;Wrap will occur - get part. length
	JMP	SHORT CL5		;Calculate clear parameters

CL4:	CMP	SI,END_OF_PAGE		;Will the clear wrap-around screen?
	JBE	CL6			;No, continue

	MOV	DX,END_OF_PAGE		;Yes - get the last screen offset
	INC	DX			;Make it the addr of the next page
CL5:	SUB	DX,DI			;Find how many bytes can be cleared
	XCHG	DX,CX			;Place new count in CX, save old
	SUB	DX,CX			;Subtract from orig. count # bytes
	MOV	AH,TRUE			;Set the wrapped-around flag
CL6:	PUSH	AX			;Save the line-type flag
	MOV	AH,BH			;Get attribute (if used)
	MOV	AL,' ' 			;Assume space for fill character
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	CL7			;Yes - have correct fill character

	CMP	VIDEO_MODE,3		;In graphics mode?
	JBE	CL7			;No, continue

	MOV	AX,0			;Yes - set zero as fill character
CL7:	SHR	CX,1			;Make byte count a word count
	JNC	CL8			;Moving even number of words

	STOS	BYTE PTR ES: [DI]	;Moving odd count - move the odd byte
CL8:	REP	STOS WORD PTR ES: [DI]	;Clear a scan line of words
	POP	AX			;Restore the line-type flag
	TEST	AH,AH			;Did we wrap-around?
	JZ	CL9			;No, go on to next scan line

	MOV	CX,DX			;Yes - get remaining byte count
	MOV	DI,START_OF_PAGE	;Point to first byte of page
	CMP	VIDEO_MODE,7		;In monochrome mode?
	JE	CL3			;Yes - have correct screen start

	CMP	VIDEO_MODE,3		;In graphics mode?
	JBE	CL3			;No, continue

	CMP	SI,OFFSET END_OF_SCREEN	;Was the wrap past partition 2?
	JBE	CL3			;No, finish clearing scan line

	MOV	DI,OFFSET END_OF_PART+1	;Yes - point to start of partition 2
	JMP	SHORT CL3		;Clear rest of this scan line

CL9:	POP	CX			;Restore line length
	POP	DI			;Restore start-of-line pointer
	DEC	BP			;Cleared all of character line yet?
	JZ	CL11			;Yes - return

	XOR	DI,SCREEN_PARTITION_OFS	;No, pnt to the (even/odd) scan line
	TEST	BP,1			;Did we just clear an odd scan line?
	JNE	CL10			;No, pointing to an odd scan line now

	ADD	DI,SCAN_LINE_INCREMENT	;Yes - point to the even scan line
	CMP	DI,OFFSET END_OF_PART	;Did we go past the grafix partition?
	JBE	CL10			;No, go on to erase next scan line

	SUB	DI,SCREEN_BYTES SHR 1	;Yes - wrap-around to next line
CL10:	JMP	CL1			;Go on to the next scan line

CL11:	POPREG	<BP,DI,SI,DX,CX,BX,AX>
	RET

CLEAR_LINE ENDP

page
;**********************************************************************
; CLEAR_SCREEN:
;
;	Clear_Screen erases the contents of the currently displayed
; page, and homes the cursor.  The attribute used in text modes is the
; usual normal-intensity white characters on black background.
;**********************************************************************
CLEAR_SCREEN PROC NEAR

	PUSHREG	<AX,BX,CX,DX,DI,ES>	;Save registers
	MOV	CX,(0 SHL 8) OR 0	;Set position of upper-left hand corner
	MOV	DH,LAST_ROW		;Set position of lower right-hand
	MOV	DL,BYTE PTR NUMBER_OF_COLUMNS	;...corner of screen in DX
	DEC	DL
	MOV	BH,DEFAULT_ATTRIBUTE	;Set BH to the default attribute
	CALL	SCROLL_UP		;Erase the entire page
	MOV	DX,CX			;Now, set DX to the home position
	MOV	BH,DISPLAYED_PAGE	;Set the current display page
	CALL	SET_CURSOR_POSITION	;Home the logical and physical cursors
	POPREG	<ES,DI,DX,CX,BX,AX>	;Restore registers
	RET

CLEAR_SCREEN ENDP

;page
;----------------------------------------------------------------------
;Wait_Vertical_Sync Proc Near  NOT NEEDED FOR PIVOT II
;---------------------------
;	1) Wait_Vertical_Sync waits for a vertical sync pulse
;	2) Note that the monochrome card does not have a VSYNC indicator
;	   so this routine will immediately return if the monochrome card
;	   is being referenced.
;	3) Note that the routine contains a timeout to prevent looping forever!
;
;
;	PushReg	<ax,cx,dx>
;	mov	dx,Video_Base		;DX:= Pointer to the CRT status port
;	cmp	dx,Mono_Card		;If (this is NOT a monochrome card)
;	je	WVS3
;
;	add	dx,Crt_Status_Port	;	DX:= pointer video status port
;	mov	cx,10000		;	CX:= Maximum wait for VSYNC
;WVS1:	in	al,dx			;	Loop	wait for NOT VSYNC
;	and	AL,Vertical_Sync	;		(stops mult. scrolling
;	LoopNz	WVS1			;		during 1 sync time)
;
;	jcxz	WVS3			;	If (Sync Pulse Found)
;
;WVS2:	in	al,dx			;		Repeat Check VSync
;	and	al,Vertical_Sync
;	LoopZ	WVS2			;		Until (VSync received)
;
;WVS3:	PopReg	<dx,cx,ax>
;	ret				;Return
;
;Wait_Vertical_Sync EndP

page
;----------------------------------------------------------------------
In_Morrow_6845 Proc Near; (9_Jul_85)
;-----------------------------------
;	1) This routine will output a byte to the Morrow simulated 6845 chip.
;	2) Input Register Values:
;		AL -> 6845 register
;
Public	In_Morrow_6845

	push	dx
	mov	dx,Color_Card		;Address the NMI status register

	pushf				;Don't change the interrupt flag
	cli
	out	dx,al
	inc	dx
	in	al,dx			;Get the status (index to a routine)
	popf

	pop	dx
	ret

In_Morrow_6845	EndP

;----------------------------------------------------------------------
Out_Morrow_6845 Proc Near; (9_Jul_85)
;------------------------------------
;	1) This routine will output a byte to the Morrow simulated 6845 chip.
;	2) Entry Register Values
;		AL -> 6845 register
;		AH -> Value to output	
;
;
Public	Out_Morrow_6845

	push	dx
	mov	dx,Color_Card

	pushf
	cli
	out	dx,al
	inc	dx
	xchg	al,ah
	out	dx,al
	popf

	xchg	al,ah
	pop	dx
	ret

Out_Morrow_6845	EndP

page
;----------------------------------------------------------------------
Enable_Cursor_Ram Proc Near; (27_Jun_85)
;---------------------------------------
;
Public	Enable_Cursor_Ram

	PushReg	<ax,cx>
	or	Dcr_Save,Cursor_Ram_Enable
	mov	ah,Dcr_Save

	mov	al,Display_Control_Register
	call	Out_Morrow_6845

	mov	cx,Delay_One_ms/10	;Get delay of ~100 microsec
Ecr2:	loop	Ecr2

	PopReg	<cx,ax>
	ret

Enable_Cursor_Ram EndP

;----------------------------------------------------------------------
Disable_Cursor_Ram Proc Near; (14_May_85)
;----------------------------------------
;
Public	Disable_Cursor_Ram

	PushReg	<ax,cx>
	and	Dcr_Save,Not Cursor_Ram_Enable
	mov	ah,Dcr_Save

	mov	al,Display_Control_Register
	call	Out_Morrow_6845

	mov	cx,Delay_One_ms/10	;Get delay of ~100 microsec
Dcr2:	loop	Dcr2

	PopReg	<cx,ax>
	ret

Disable_Cursor_Ram EndP

page
;**********************************************************************
; OUT_VIDEO_MODE: (MODE)
;
;	Out_Video_Mode outputs a mode value to the video hardware.
; It takes the mode set by the user, and adds to the mode the
; current hardware alternate font selection.
;
; Input:
;	AL: New mode value to be output.
;**********************************************************************
OUT_VIDEO_MODE PROC NEAR
PUBLIC	OUT_VIDEO_MODE

	PUSHREG	<AX,CX,DX>
	MOV	CH,AL			;Save the value to be output
	MOV	DX,VIDEO_BASE		;Point to the correct I/O board
	CMP	DX,MONO_CARD		;Is this the monochrome card?
	JE	OVM1			;Yes - don't have control port!!!

	ADD	DX,CRT_STATUS_PORT	;No, point to the CRT status port
	IN	AL,DX			;Read the CRT status
	MOV	CL,3			;Shift font select to right position
	SHR	AL,CL
	AND	AL,CRT_FONT_SELECT	;Mask out all but font select bits
	OR	AL,CH			;OR in font select with new mode
	OUT	DX,AL			;Select right mode, font
	MOV	CRT_CONTROL,AL		;Save the value written to the control
OVM1:	POPREG	<DX,CX,AX>
	RET

OUT_VIDEO_MODE ENDP

page
;**********************************************************************
; SEND_VIDEO_WORD: (START_PORT, VALUE)
;
;	Send_Video_Word sends a word value to the 6845 CRT controller
; chip.  The high byte is sent first, then the low byte is sent to the
; next register.
;
; Input:
;	AL: First register number in the 6845
;	BX: Value to be output
;**********************************************************************
SEND_VIDEO_WORD PROC NEAR
PUBLIC	SEND_VIDEO_WORD

	PUSHREG	<AX,DX>
	MOV	DX,VIDEO_BASE		;Point to the 6845 registers
	OUT	DX,AL			;Select correct register

	PUSH	AX			;Save register number
	MOV	AL,BH			;Get the high byte to be sent
	INC	DX			;Point to the parameter port
	OUT	DX,AL			;Output high byte
	POP	AX			;Restore register number

	INC	AL			;Point to the next register
	DEC	DX			;Point DX back to the control reg
	OUT	DX,AL			;Select the second register
	INC	DX			;Point back to parameter register
	MOV	AL,BL			;Get the low byte to send
	OUT	DX,AL			;Send the low byte
	POPREG	<DX,AX>			;Restore registers
	RET

SEND_VIDEO_WORD ENDP

page
;**********************************************************************
; RECV_VIDEO_WORD: (START_PORT)
;
;	Recv_Video_Word reads a word value from a sequential register
; pair in the 6845 CRT controller chip.
;
; Input:
;	AL: Starting port number.
;
; Output:
;	AX: Value read
;**********************************************************************
RECV_VIDEO_WORD PROC NEAR
PUBLIC	RECV_VIDEO_WORD

	PUSHREG	<CX,DX>
	MOV	DX,VIDEO_BASE		;Point DX to the controller chip
	OUT	DX,AL			;Select the first register
	MOV	CL,AL			;Copy register number into CL
	INC	DX			;Point to the parameter port
	IN	AL,DX			;Read high byte
	MOV	AH,AL			;Place in high byte of AX
	MOV	AL,CL			;Get the starting register
	INC	AL			;Make it the second register number
	DEC	DX			;Point back to the control port
	OUT	DX,AL			;Select the second register
	INC	DX			;Point to the parameter port
	IN	AL,DX			;Read low byte from controller
	POPREG	<DX,CX>
	RET
RECV_VIDEO_WORD ENDP

page
;----------------------------------------------------------------------
Set_Default_Video_Mode Proc Near
;-------------------------------
;	1) Set_Default_Video_Mode polls the system configuration DIP
;	   switches to determine what mode should be used following power-up
;	   or reset, and then sets the system video to that selected.  This
;	   routine also forces the scroll mode to 0 (compatible mode).
;
Public	Set_Default_Video_Mode

	push	ax
;	mov	Scroll_Mode,0		;Force software scrolling mode
	call	Read_Dip_Switches	;Read the system DIP switches

SdvM0:	and	ah,Video_Mode_Mask	;Mask out all but video mode
	mov	al,3			;Assume 80 by 25 color...
	cmp	ah,Video_Mode_Mono	;Is this monochrome mode?
	jne	SdvM1			;No, try next mode

	mov	al,7			;Yes - set correct video mode
SdvM1:	cmp	ah,Video_Mode_40	;Is 40 column output requested?
	jne	SdvM2			;No, the selected mode

	mov	al,1			;Yes - set 40 by 25 color mode
SdvM2:	mov	ah,0			;Set AH to set-video-mode
	mov	Old_Video_Mode,al	;Ignore old video mode completely
	mov	Video_Mode,al		;Force skip of old-video-mode check
	int	Video_IO_Intr		;Set the default video mode

	mov	Super_Compatible,False	;Start up in FAST mode
	pop	ax
	ret				;Return

Set_Default_Video_Mode EndP

page
;----------------------------------------------------------------------
;**Set_Graphic_Cursor Proc Near; (31_Oct_85)
;---------------------------------------
;	1) This routine updates the Cursor_Ptr (Cursor Pointer) location
;	   while in graphics mode.
;	2) Entry Register Values
;		DX -> New cursor position (DH:row, and DL:column)
;
;**	cmp	Video_Mode,4		;If (We're in graphics mode)
;**	jb	Ugc2
;**	cmp	Video_Mode,6
;**	ja	Ugc2

;**	PushReg	<ax,bx,dx>
;**	mov	bl,dh			;	Get the current row number
;**	mov	bh,0			;	Extend it to a word offset
;**	shl	bx,1
;**	mov	bx,cs:Graphics_Line_Offset[bx] ;Pick up the offset to this line
;**	add	bx,Page_Base[0]		;	Add in address of start of page
;**	cmp	Old_Video_Mode,6	;	(2 bytes/char in 320x200 mode)
;**	jz	Ugc1

;**	shl	dl,1
;**Ugc1:	add	bl,dl			;	Now point to new cursor posn
;**	adc	bh,0
;**	and	bx,Graphics_Screen_Mask	;	Ensure that the ptr is valid
;**	mov	Cursor_Ptr,bx
;**	PopReg	<dx,bx,ax>
;**
;**Ugc2:	ret				;Return

;**Set_Graphic_Cursor EndP

page
;**********************************************************************
;	 C R T   M O D E   V A R I A B L E   T E M P L A T E S
;**********************************************************************

PUBLIC	  ROM_VARS
ROM_VARS VID_VARS <8,2048,40,2,80,30H>	;Templates for 40 by 25
         VID_VARS <4,4096,80,2,160,30H>	;Templates for 80 by 25
         VID_VARS <1,8192,40,2,320,30H>	;Templates for Med-res
	 VID_VARS <1,8192,80,1,320,3FH>	;Templates for High-res
	 VID_VARS <1,4096,80,2,160,30H>	;Templates for monochrome card

page
;**********************************************************************
;	     C R T   H A R D W A R E   M O D E   T A B L E
;**********************************************************************

PUBLIC	CRT_MODE
CRT_MODE	DB	101100B		;40 by 25 B & W mode
		DB	101000B		;40 by 25 color mode
		DB	101101B		;80 by 25 B & W mode
		DB	101001B		;80 by 25 color mode
		DB	101010B		;320 by 200 color graphics mode
		DB	101010B		;320 by 200 B & W graphics mode
		DB	011110B		;640 by 200 B & W graphics mode
		DB	101001B		;80 by 25 mono card mode


;**********************************************************************
;	     L I G H T   P E N   A D J U S T   T A B L E
;**********************************************************************

PUBLIC	LPEN_ADJUST
LPEN_ADJUST	db	4,4,5,5,4,4,4,4	;By video mode..


;**********************************************************************
;		     P O I N T E R   T O   F O N T
;**********************************************************************

PUBLIC	FONT_PTR
FONT_PTR	DD	FONT		;Pointer to standard ROM font


;**********************************************************************
;		    G R A P H I C S   T A B L E S
;**********************************************************************

; Mask for high-resolution graphics pixels
PUBLIC	HIGH_RES_MASKS
HIGH_RES_MASKS		DB	10000000B		;Pixel 0
			DB	01000000B		;Pixel 1
			DB	00100000B		;Pixel 2
			DB	00010000B		;Pixel 3
			DB	00001000B		;Pixel 4
			DB	00000100B		;Pixel 5
			DB	00000010B		;Pixel 6
			DB	00000001B		;Pixel 7

; Mask for medium-resolution graphics pixels
PUBLIC	MED_RES_MASKS
MED_RES_MASKS		DB	11000000B		;Pixel 0
			DB	00110000B		;Pixel 1
			DB	00001100B		;Pixel 2
			DB	00000011B		;Pixel 3
			DB	11000000B		;Pixel 0
			DB	00110000B		;Pixel 1
			DB	00001100B		;Pixel 2
			DB	00000011B		;Pixel 3

; Table to spread a color value across a word
PUBLIC	MED_RES_COLOR
MED_RES_COLOR		DW	0000000000000000B	;Color 0
			DW	0101010101010101B	;Color 1
			DW	1010101010101010B	;Color 2
			DW	1111111111111111B	;Color 3

MONITOR_SEGMENT ENDS

	END
