using System;
using System.Collections.Generic;
using System.Text;

namespace PERQDisk
{
    /// <summary>
    /// Represents a FileInformationBlock (FIB)
    /// </summary>
    public class FileInformationBlock
    {
        public FileInformationBlock(Sector fibSector)
        {
            _fileSize = fibSector.ReadWord(0);
            _fileMisc = fibSector.ReadWord(2);
            _creationDate = fibSector.ReadDWord(4);
            _lastWriteDate = fibSector.ReadDWord(8);
            _lastAccessDate = fibSector.ReadDWord(12);
            _fileType = fibSector.ReadWord(16);
            _fileName = fibSector.ReadString(23, fibSector.ReadByte(22));

            //direct indices
            _directIndices = new uint[64];
            for (int i = 0; i < 64; i++)
            {
                _directIndices[i] = fibSector.ReadDWord(i * 4 + 104);
            }

            //indirect indices
            _indirectIndices = new uint[32];
            for (int i = 0; i < 32; i++)
            {
                _indirectIndices[i] = fibSector.ReadDWord(i * 4 + 360);
            }

            //double-indirect indices
            _doubleIndirectIndices = new uint[2];
            for (int i = 0; i < 2; i++)
            {
                _doubleIndirectIndices[i] = fibSector.ReadDWord(i * 4 + 488);
            }

            _segmentKind = fibSector.ReadWord(496);
            _blocksInUse = fibSector.ReadWord(498);
            _lastBlock = fibSector.ReadWord(500);
            _lastAddress = fibSector.ReadDWord(502);
            _lastNegativeBlock = fibSector.ReadWord(506);
            _lastNegativeAddress = fibSector.ReadDWord(508);

        }

        public ushort FileSize
        {
            get { return _fileSize; }
        }

        public ushort FileMisc
        {
            get { return _fileMisc; }
        }

        public uint CreationDate
        {
            get { return _creationDate; }
        }

        public uint LastWriteDate
        {
            get { return _lastWriteDate; }
        }

        public uint LastAccessDate
        {
            get { return _lastAccessDate; }
        }

        public ushort FileType
        {
            get { return _fileType; }
        }

        public string FileName
        {
            get { return _fileName; }
        }

        public uint[] DirectIndices
        {
            get { return _directIndices; }
        }

        public uint[] IndirectIndices
        {
            get { return _indirectIndices; }
        }

        public uint[] DoubleIndirectIndices
        {
            get { return _doubleIndirectIndices; }
        }

        public ushort SegmentKind
        {
            get { return _segmentKind; }
        }

        public ushort BlocksInUse
        {
            get { return _blocksInUse; }
        }

        public ushort LastBlock
        {
            get { return _lastBlock; }
        }

        public uint LastAddress
        {
            get { return _lastAddress; }
        }

        public ushort LastNegativeBlock
        {
            get { return _lastNegativeBlock; }
        }

        public uint LastNegativeAddress
        {
            get { return _lastNegativeAddress; }
        }

        ushort _fileSize;
        ushort _fileMisc;
        uint _creationDate;
        uint _lastWriteDate;
        uint _lastAccessDate;
        ushort _fileType;
        string _fileName;
        uint[] _directIndices;
        uint[] _indirectIndices;
        uint[] _doubleIndirectIndices;
        ushort _segmentKind;
        ushort _blocksInUse;
        ushort _lastBlock;
        uint _lastAddress;
        ushort _lastNegativeBlock;
        uint _lastNegativeAddress;
    }

    /// <summary>
    /// Represents the structure of a single directory entry in a directory file.
    /// </summary>
    public class DirectoryEntry
    {
        public DirectoryEntry(byte[] data, int offset)
        {            
            _inUse = (Helper.ReadWord(data, offset) & 0x1) != 0;
            _fibAddress = Helper.ReadDWord(data, offset + 2);
            _simpleName = Helper.ReadString(data, offset + 7, Helper.ReadByte(data, offset + 6));
        }

        /// <summary>
        /// Hacky override to allow an entry of sorts for partitions.
        /// Lame.
        /// </summary>
        /// <param name="name"></param>
        public DirectoryEntry(string name)
        {
            _simpleName = name;
        }

        public bool InUse
        {
            get { return _inUse; }
        }

        public uint FIBAddress
        {
            get { return _fibAddress; }
        }

        public string SimpleName
        {
            get { return _simpleName; }
        }

        bool _inUse;
        uint _fibAddress;
        string _simpleName;
    }

    /// <summary>
    /// Represents a File in a POS directory structure, which may be a directory (which contains other Files)
    /// or a File (which contains interesting data.)
    /// </summary>
    public class File
    {
        /// <summary>
        /// Creates a new directory given the address of its FIB.
        /// </summary>
        /// <param name="fib"></param>
        public File(PhysicalDisk disk, DirectoryEntry entry, FileInformationBlock fib)
        {
            if (disk == null)
            {
                throw new ArgumentNullException("disk");
            }

            if (fib == null)
            {
                throw new ArgumentNullException("fib");
            }

            _entry = entry;
            _disk = disk;
            _fib = fib;

            BuildDirectoryTree();
        }

        public string FullName
        {
            get { return _fib.FileName; }
        }

        public string SimpleName
        {
            get
            {
                if (_entry == null)
                {
                    return String.Empty;
                }
                else
                {
                    return _entry.SimpleName;
                }
            }
        }

        public List<File> Children
        {
            get { return _children; }
        }

        public bool IsDirectory
        {
            get { return _fib.FileType == 3; } //experimentation reveals directory files have type 3.  woo.
        }

        public bool IsSparse
        {
            get { return (_fib.FileMisc & 0x1000) != 0; }
        }

        public byte[] Data
        {
            get { return _data; }
        }

        public void PrintFileInfo()
        {
            Console.WriteLine(" Information for '{0}':", SimpleName);
            if (IsDirectory)
            {
                Console.WriteLine("  Directory '{0}' contains {1} files", _fib.FileName, _children.Count);
                Console.WriteLine("  File Type: {0}", _fib.FileType);
            }
            else
            {
                Console.WriteLine("  File '{0}' contains {1} bytes", _fib.FileName, _data.Length);
                Console.WriteLine("  Blocks: {0}\n  Bits: {1}\n  Last Block: {2}\n ", _fib.BlocksInUse, _fib.FileMisc & 0xfff, _fib.LastBlock);
                Console.WriteLine("  File Type: {0}", _fib.FileType);
            }

            
        }

        /// <summary>
        /// Dumps the data for this file to screen.  Skips character 0x07 since it beeps and is annoying.
        /// </summary>
        public void Print()
        {
            for (int i = 0; i < _data.Length; i++)
            {
                //For the sake of ears and sanity, we print everything except 0x07, which beeps.

                if (_data[i] != 0x07)
                {
                    Console.Write((char)_data[i]);
                }
            }
        }

        /// <summary>
        /// Recursively builds this file's structure.
        /// </summary>
        private void BuildDirectoryTree()
        {           
            //Read this file/directory's data blocks into memory.
            ReadFile();                      

            //If this is a directory, read the directory entries from it and read those files/directories...
            if (IsDirectory)
            {                
                ReadDirectory();
            }
        }

        /// <summary>
        /// Reads the child files of this directory into memory.
        /// </summary>
        private void ReadDirectory()
        {
            _children = new List<File>(16);

            for (int i = 0; i < _data.Length; i += 32)
            {
                DirectoryEntry entry = new DirectoryEntry(_data, i);

                if (entry.InUse)
                {                    
                    _children.Add(
                        new File(
                            _disk, 
                            entry,  
                            new FileInformationBlock(
                                _disk.GetSector(
                                    _disk.LDAToLogicalBlock(entry.FIBAddress)))));
                }
            }
        }

        /// <summary>
        /// Reads the disk blocks for this file into memory.
        /// TODO: this method and associated methods are _UGLY_.
        /// </summary>
        private void ReadFile()
        {
            //Trim any excess data not actually used in the last block of the file.
            int usedBits = _fib.FileMisc & 0xfff;

            // This is based on trial and error -- FileSize is _not_ used for Directories, but is accurate for Files.
            // LastBlock+2 _seems_ to be the correct size for directories, but I don't know if it's the Right Thing.
            // The POS documentation is _Very Vague_ about this stuff.
            int numBlocks = IsDirectory ? ((short)_fib.LastBlock) + 2: _fib.FileSize; // hack!

            // This is a stupid way to convey "if we have more than one block allocated to a file, then subtract off the
            // unused bytes of the last block; otherwise add the used bytes to the current allocation (0 blocks)."
            int dataSize = Math.Abs(numBlocks * 512 -(512 - (usedBits >> 3)));

            // Allocate some space for the data in this file
            _data = new byte[dataSize];            

            int blockCount = 0;

            // Read the direct indexed blocks
            blockCount += ReadBlocks(_fib.DirectIndices, 0, numBlocks - blockCount);

            if (blockCount >= numBlocks)
            {
                return;
            }

            // Read the indirect indexed blocks.
            blockCount += ReadIndexedBlocks(_fib.IndirectIndices, blockCount * 512, numBlocks - blockCount);

            if (blockCount >= numBlocks)
            {
                return;
            }

            // Read the double-indirect indexed blocks
            blockCount += ReadDoubleIndexedBlocks(_fib.DoubleIndirectIndices, blockCount * 512, numBlocks - blockCount);

            //We're done regardless at this point...                        
        }

        /// <summary>
        /// Really stupid way to get an array of uints (lda's) out of a sector.
        /// Meh.
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        private uint[] SectorDataToUintArray(Sector s)
        {
            uint[] uints = new uint[128];

            for (int i = 0; i < 128; i++)
            {
                uints[i] = s.ReadDWord(i * 4);
            }

            return uints;
        }

        private int ReadDoubleIndexedBlocks(uint[] indices, int offset, int maxBlocks)
        {
            int blockCount = 0;
            // Read the double-indirect blocks
            for (int i = 0; i < _fib.DoubleIndirectIndices.Length; i++)
            {
                Sector doubleIndirectSector = _disk.GetSector(_disk.LDAToLogicalBlock(_fib.DoubleIndirectIndices[i]));

                // The above points to a sector of indirect blocks.
                blockCount += ReadIndexedBlocks(SectorDataToUintArray(doubleIndirectSector), offset + blockCount * 512, maxBlocks - blockCount);

                if (blockCount >= maxBlocks)
                {
                    return blockCount;
                }
            }

            return blockCount;
        }

        private int ReadIndexedBlocks(uint[] indices, int offset, int maxBlocks)
        {
            int blockCount = 0;
            for (int i = 0; i < indices.Length; i++)
            {
                Sector indirectSector = _disk.GetSector(_disk.LDAToLogicalBlock(indices[i]));

                blockCount += ReadBlocks(SectorDataToUintArray(indirectSector), offset + blockCount * 512, maxBlocks - blockCount);

                if (blockCount >= maxBlocks)
                {
                    return blockCount;
                }
            }

            return blockCount;
        }

        private int ReadBlocks(uint[] indices, int offset, int maxBlocks)
        {
            // Read direct blocks
            for (int i = 0; i < Math.Min(indices.Length, maxBlocks); i++)
            {
                Sector fileSector = _disk.GetSector(_disk.LDAToLogicalBlock(indices[i]));

                if (offset + i * 512 + 512 <= _data.Length)
                {
                    fileSector.Data.CopyTo(_data, offset + i * 512);
                }
                else
                {
                    //Copy incomplete last sector by hand...
                    for (int j = offset + i * 512; j < _data.Length; j++)
                    {
                        _data[j] = fileSector.Data[j - (offset + i * 512)];
                    }

                    return i+1;
                }

            }

            return Math.Min(indices.Length, maxBlocks);
        }

        private DirectoryEntry _entry;
        private List<File> _children;
        private byte[] _data;
        private FileInformationBlock _fib;
        private PhysicalDisk _disk;
    }
}
