#include "stdio.h"
#include "ctype.h"

/* ------------------------------------------------------------------------ */
/*			     DEFINES					    */
/* ------------------------------------------------------------------------ */

#define CURR_DRIVE	0x19	/* DOS call to retrieve current drive */
#define FALSE		0
#define FCB_SIZE	37	/* size of the File Control Block */
#define	FIRST		1	/* Flag if want first entry of directory */
#define FIRST_ENTRY	0x11	/* DOS call to get first filename entry */
#define LEN_FILENAME	0x0B	/* Length of filename plus extension */
#define NEXT_ENTRY	0x12	/* DOS call to get next filename entry */
#define RENAME		0x17	/* DOS call to rename a file */
#define SET_DTA		0x1A	/* DOS call to set Disk Transfer Area */
#define SPACE		0x20	/* Value of a space */
#define TRUE		1

/* ------------------------------------------------------------------------ */
/*			     PROGRAM					    */
/* ------------------------------------------------------------------------ */

/* ---------------------------------------------------------------------------

NAME

	Rename - rename a file

SYNOPSIS

	ret = Rename(from, to);
	int ret;		return code
	char *from;		filename to rename
	char *to;		filename to rename to

DESCRIPTION

	Renames file 'from' to file 'to' using the DOS function call.


RETURNS

	ret = 0 if no match was found, an attempt ws made to rename
	      to a file that already existed, the filename was invalid,
	      or the drives specified are different.

CAUTIONS

	Wildcard characters, e.g. '?' and '*', in the filename will not
	be expanded.

--------------------------------------------------------------------------- */
Rename(from, to)
char *from;
char *to;
{
	int curr_drive;			/* Current default drive */
	char *drive1;			/* Drive number of from filename */
	char *drive2;			/* Drive number of to filename */
	char fcb[FCB_SIZE];		/* FCB used to rename file */

	drive1 = fcb;			/* point to the drive specifiers */
	drive2 = fcb + 16;

	/* Zap and setup the fcb */
	setmem(fcb, FCB_SIZE, SPACE);
	ParseFn(from, fcb);
	ParseFn(to, fcb + 16);

	/* make sure the drives specified are equal */
	curr_drive = bdos(CURR_DRIVE) + 1;
	if (!*drive1) *drive1 = curr_drive;
	if (!*drive2) *drive2 = curr_drive;
	if (*drive1 != *drive2) return(0);

	return(!bdos(RENAME, fcb));
}

/* ---------------------------------------------------------------------------

NAME

	ParseFn - parse filename

SYNOPSIS

	ParseFn(fn, fcb);
	char *fn;		filename to parse
	char *fcb;		FCB to parse fn into

DESCRIPTION

	Performs a simple parse of fn into the proper locations of fcb
	for DOS.

CAUTIONS

	This is a very primitive parser so don't expect much.  For instance,
	wildcard characters, e.g. '?' and '*', in the filename will not
	be expanded.

--------------------------------------------------------------------------- */
ParseFn (fn, fcb)
char *fn;
char *fcb;
{
	char *ext;		/* pointer to extension part of fn */

	ext = fcb + 9;		/* point to extension of fn */

	*fcb = 0;		/* initialize to the current drive */
	if (*(fn+1) == ':') {	/* drive was specified, set fcb accordingly */
		*fcb = tolower(*fn) - 'a' + 1;
		fn += 2;
	}

	fcb++;		/* store the rest of the filename w/ extension */
	while (*fn && *fn != '.') *fcb++ = *fn++;
	if (*fn) while (*++fn) *ext++ = *fn;
}

/* ---------------------------------------------------------------------------

NAME

	Dir - get directory entries

SYNOPSIS

	ret = Dir(flag, drive, fn);
	int ret			return code
	char flag;		flag for first or subsequent retrievals
	char drive;		drive letter of directory to read
	char fn[];		location to put filename entry

DESCRIPTION

	Retrieves a filename entry from the directory specified and places
	the filename in fn.  To get the first entry, call this function
	using FIRST as the value of 'flag'.  To get the subsequent entries
	call this function with some other value other than FIRST for 'flag'.

	fn is in the form:

		FFFFFFFFXXX
	where "F" is the filename and "X" is the extension.  The string
	is terminated with a null byte.

RETURNS

	ret = zero if there are no more entries.

CAUTIONS

	Be sure fn points to a location large enough for the filename.

--------------------------------------------------------------------------- */
Direct(flag, drive, fn)
char flag;
char drive;
char fn[];
{
	static char dta[FCB_SIZE]; /* Data Transfer Area to store filename */
	static char fcb[FCB_SIZE] = {0, '?', '?', '?', '?', '?', '?', '?',
					'?', '?', '?', '?'};
	int func;			/* DOS function to use to get entry */

	bdos(SET_DTA, dta);		/* set the DTA to receive filenames */

	/* 
	 * If they want the first entry, determine
	 * the drive and set it.  Set func for the
	 * first or subsequent entry to retrieve.
	 */
	if ( flag == FIRST ) {
		if ( drive == 0 ) *fcb = bdos(CURR_DRIVE) + 1;
		else *fcb = tolower(drive) - 'a' + 1;
		func = FIRST_ENTRY;
	}
	else func = NEXT_ENTRY;

	/* Get the entry, return if no more entries */
	if ( bdos(func, fcb) == 0xFF ) return(0);

	/*
	 * Copy the filename from the DTA to fn
	 * and end fn with a NULL byte
	 */
	movmem(dta + 1, fn, LEN_FILENAME);
	fn[LEN_FILENAME] = NULL;
	return(1);
}
