*    Parser for 6809 debugger
*    This parser is logically an LR(1) parser, with the following optimizations
*    1) The number of parse states and rules is less than 256.
*    2) Lookahead sets are computed dynamically from parser states as needed
*       (This is actually very cheap!)
*    3) If no exit of a state is applicable, and there is only a single
*       reduction, then that reduction is forced; no lookahead set is computed.
*    4) A state containing 0 exits and just 1 reduction occupies minimal space
*    5) A syntax error simply causes error signal; last token is thrown away
*    6) A DELete character causes last character entered to be erased,
*       BS, SPACE, BS to be output,
*       and the last unprocessed command input to be re-parsed
*
*    IT IS POSSIBLE TO PARSE WITHOUT THE USE OF LOOKAHEAD SETS.
*    TO SEE THIS, ONE MUST ONLY NOTICE THAT LOOKAHEAD SETS CONTAIN
*    ONLY TOKENS IN THE FOLLOW SET OF THE APPROPRIATE CONTEXT;
*    THE LR STATES CLEARLY CONTAIN THESE TOKENS AS EXIT TOKENS
*    THEREFORE THE LOOKAHEAD SETS CAN BE CONSTRUCTED DYNAMICALLY
*    FROM THE LR EXIT INFORMATION. TO MINIMIZE THE COST OF THIS COMPUTATION,
*    THE PARSER CAN TRY FOR EXITS FIRST; CLEARLY, IF A STATE EXIT EXISTS,
*    THE TOKEN CANNOT BE A MEMBER OF A LOOKAHEAD SET
*    AND SO NO REDUCTION CAN OCCUR. IF NO EXIT IS VALID, AND ONLY
*    ONE REDUCTION IS POSSIBLE, COMPUTATION OF THE LOOKAHEAD SET
*    CAN LIKEWISE BE AVOIDED.
*    FINALLY, IF SEVERAL (N) REDUCTIONS ARE POSSIBLE,
*    ONLY THE LOOKAHEAD SETS FOR THE FIRST N-1 (MAX) NEED BE COMPUTED.
*    The work invested to compute a lookahead set is asymptotically
*    proportional to LRstackdepth/(1-r) where r is the average number
*    of reductions per state, and LRstackdepth is the number of LR states
*    pushed onto the stack at the time the lookahead set needs computation.
*    Note that this work estimate is somewhat higher than actual practice.
*    For r<.8, this value is about 5 units of work with depth=100!!
*    SPACE NORMALLY USED FOR LOOKAHEAD SETS CAN INSTEAD BE DEDICATED
*    TO CODE TO DYNAMICALLY COMPUTE THEM; SINCE LOOKAHEAD SETS ARE UNNECESSARY,
*    AN SLR(0)? STATE MACHINE CAN BE USED WITH CONCOMITANT SPACE SAVINGS
*    A STATE CONTAINING A REDUCTION CAN REPRESENT THAT REDUCTION
*    WITH 1 BYTE: RULE NUMBER OF REDUCTION.
*    A TABLE INDEXED BY RULE NUMBER, CONTAINING REDUCE-TO TOKEN (LHS),
*    SIZE OF RIGHT HAND OF RULE, AND POINTER TO SEMANITIC ROUTINE
*    ANOTHER SAVINGS IS POSSIBLE IF THE TARGET STATE OF AN EXIT
*    CONTAINS ONLY A SINGLE REDUCTION: THE EXIT COULD CONTAIN THE EXIT TOKEN
*    AND A RULE TO REDUCE TO, THEREBY SAVING A STATE.

*    LR PARSING MECHANISM 9/8/81
*    DOES BOTTOM UP PARSE OF STREAM OF INPUT TOKENS FROM "GETTOKEN"
*    PARSING IS PERFORMED IN ACCORDANCE WITH LR(1) PARSE TABLES
*    (LRTABLES) THAT (X) SELECTS ON ENTRY TO THIS ROUTINE
*
*    DATA STRUCTURES:
*
*    TOKEN: TYPE INTEGER 0..MAXTOKEN (ASSERT: MAXTOKEN <= 255)
*    RULE: TYPE INTEGER 0..MAXRULE (ASSERT: MAXRULE <= 255)
*
*    ASSERT MAXTERMINALTOKEN<=MAXTOKEN
*
*    LRRULE: TYPE RECORD OF
*                 LRRULELHSTOKEN: TOKEN -- TOKEN TO REDUCE TO
*                 LRRULESIZE: INTEGER 0..255 -- SIZE OF RHS OF RULE
*                 LRRULESEMANTIC: POINTER TO FUNCTION -- SEMANTIC ROUTINE
*    LTOKEN<=MAXTOKEN
*
*    LRRULE: TYPE RECORD OF
*                 LRRULELHSTOKEN: TOKEN -- TOKEN TO REDUCE TO
*                 LRRULESIZE: INTEGER 0..255 -- SIZE OF RHS OF RULE
*                 LRRULESEMANTIC: POINTER TO FUNCTION -- SEMANTIC ROUTINE
*                 LRREDUCEDRULE: RULE
*                 END
*
*    LRSTATE: TYPE RECORD OF
*             LRSTATENEXITS: INTEGER 0..MAXTOKEN -- MAX 1 EXIT/TOKEN
*             LRSTATENREDUCTIONS: INTEGER 0..MAXRULE -- MAX 1 REDUCTION/RULE
*             LRSTATEEXI                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           AND RETURN TO CALLER WITH IT

LRPARSER ; *** ENTRY POINT TO LR PARSER; (X) POINTS TO "LRTABLES"
    LDD   LRMAXSTACKDEPTH,X      GET DEPTH OF LR STATE STACK, MAXIMUM
    STD   LRWMAXSTACKDEPTH       AND SAVE
    LDD   LRSTACK,X              GET POINTER TO STACK AREA
    STD   LRWSTACK               AND SAVE THAT
    LDD   LRFIRSTSTATE,X         GET POINTER TO FIRST STATE
    LDX   LRWSTACK               PUSH 1ST STATE
    STD   LRREDUCETOSTATE,X
    LEAX  LRSTACKOBJECTSIZE,X
    STX   LRWSTACK
    CLR   LRREJECTTOKEN          FLAG "DON'T REJECT TOKEN"
LRPARSEREXAMINESTATE ; INSPECT THIS STATE TO DECIDE WHAT TO DO
    LDX   LRWSTATESTACK          EXAMINE THE STATE WE ARE ENTERING
    LDX   LRCURRENTSTATE,X       GET ADDRESS OF STATE
    LDD   ,X++                   GET # EXITS AND REDUCTIONS IN THIS STATE
    STX   LRNEXTREDUCTION        SAVE POINTER TO 1ST EXIT (OR REDUCTION)
    STA   LRWSTATENREDUCTIONS    SAVE NUMBER OF REDUCTIONS
    TSTB                         SAVE NUMBER OF EXITS, ANY EXITS POSSIBLE ?
    BNE   LRPARSERCHECKEXITS     B/ SOME EXITS, SEE IF ONE IS APPLICABLE
    CMPB  #1                     IS THIS A FORCED REDUCTION ?
    BEQ   LRREDUCTIONFORCED      B/ YES, GO DO FORCED REDUCTION
    JSR   LRGETTOKEN             NO, READ LOOKAHEAD TOKEN
    BRA   LRPARSERCHECKREDUCTIONS GO SEARCH FOR REDUCTION

LRFOUNDEXIT ; PROCESS EXIT
    LDB   0,X                    GET INDEX OF EXIT TARGET STATE
    CLRA                         GET ADDRESS OF TARGET STATE FROM STATE INDEX
    ASLD
    LDX   #LRSTATEINDEX
    LDD   D,X                    FETCH ADDRESS OF TARGET STATE
    STD   LRNEXTREDUCTION        SAVE POINTER TO NEXT STATE
    LDX   LRWSTATESTACK          PUSH STACK OBJECT ONTO STATE STACK
    CPX   LRSTACKFULL            IS STATE STACK FULL?
    BEQ   ???                    ISSUE SUITABLE COMPLAINT, RECOVER LIKE SYNTAX
    LEAX  -LRSTACKOBJECTSIZE,X
    STD   LRCURRENTSTATE,X
    LDD   LRCURRENTTOKENVALUE    PUSH TOKEN VALUE ONTO STATE STACK
    STD   LRCURRENTVALUE,X
    STX   LRSTATESTACK           SAVE UPDATED STACK POINTER
    LDX   LRNEXTREDUCTION        GET POINTER TO TARGET STATE
    CMPX  #LRRULES               ONLY A FORCED REDUCTION IN TARGET STATE ?
    BCS   LRPARSEREXAMINESTATE   B/ NO, NOW GO INSPECT THE NEW STATE
    BRA   LRREDUCEFOUNDRULE         B/ YES, GO HANDLE!

LRPARSERCHECKEXITS ; CHECK TO SEE IF EXIT IS POSSIBLE WITH THIS TOKEN
    PSHB                         SAVE NUMBER OF EXITS
    JSR   LRGETTOKEN             GO GET TOKEN TO TRY ON EXITS
    LDX   LRNEXTREDUCTION        GET POINTER TO NEXT EXIT
    LDA   LRCURRENTTOKEN         GET CURRENT TOKEN
    PULB                         GET EXIT COUNT FOR FAST LOOP
LRPARSERCHECKEXIT ; CHECK TO SEE IF THIS EXIT IS CORRECT
    CMPA  ,X+                    DOES CURRENT TOKEN MATCH EXIT TOKEN ?
    BEQ   LRFOUNDEXIT            B/ YES, DO EXIT PROCESSING
    LEAX  1,X                    SKIP OVER INDEX OF TARGET STATE FOR EXIT
    DECB                         ANY MORE EXITS ?
    BNE   LRPARSERCHECKEXIT      B/ YES, CHECK IT
    TST   LRWSTATEREDUCTIONS     NO, ANY REDUCTIONS POSSIBLE IN THIS STATE ?
    BEQ   LRPARSERSYNTAXERROR    B/ NO, CURRENT TOKEN NOT LEGAL IN THIS STATE
LRPARSERCHECKREDUCTIONS ; CHECK TO SEE IF REDUCTION IS POSSIBLE
    LDA   LRCURRENTTOKEN         SET UP TO SCAN REDUCTION TABLE
LRPARSERCHECKREDUCTION ; CHECK TO SEE IF THIS REDUCTION APPLIES
    LDX   LRWNEXTREDUCTION       GET POINTER TO NEXT REDUCTION
    LDD   ,X++                   GET RULE NUMBER TO (A)
    STX   LRWNEXTREDUCTION       AND SAVE POINTER
    DEC   LRWSTATENREDUCTIONS    LAST REDUCTION IN THIS STATE ?
    BEQ   LRDOREDUCTION          B/ YES, DON'T BOTHER COMPUTING LOOKAHEAD SET!
    BSR   LRISMEMBERLOOKAHEADSET IS CURRENT TOKEN MEMBER OF LOOKAHEAD SET?
    BEQ   LRDOREDUCTION          B/ YES, DO THIS REDUCTION!
    DEC   LRWSTATENREDUCTIONS    ANY MORE REDUCTIONS TO TRY ?
    BRA   LRPARSERCHECKREDUCTION GO TRY NEXT REDUCTION

LRPARSERSYNTAXERROR ; CAN'T FIND LEGAL EXIT OR REDUCTION, ***ERROR***
    ISSUE ERROR SIGNAL HERE
    JMP   LRPARSEREXAMINESTATE   THROW TOKEN AWAY BY SIMPLY IGNORING IT

LRPARSEREXIT RTS                 SYNTAX ERROR WAS FATAL, LEAVE PARSER

LRDOREDUCTION ; REDUCTION FOUND, PROCESS IT
    INC   LRREJECTTOKEN          WE DON'T WANT THIS TOKEN THIS TIME
    LDX   LRWNEXTREDUCTION       GET POINTER TO REDUCTION AFTER ONE DESIRED
    LEAX  -2,X                   MAKE POINTER TO DESIRED REDUCTION
LRREDUCTIONFORCED ; HANDLE FORCED REDUCTION
    LDA   0,X                    GET RULE NUMBER TO (A)
    LDAB  #LRRULEREDUCTIONSIZE   # BYTES PER RULE/REDUCTION OBJECT
    MUL6809                      DISPLACEMENT INTO RULE TABLE
    LDX   LRRULESBASE            GET POINTER TO BASE OF RULES
    LEAX  D,X                    = ADDRESS OF RULE REDUCTION OBJECT
LRREDUCEFOUNDRULE ; FOUND RULE THAT APPLIES TO THIS REDUCTION
    LDA   LRRULELHSTOKEN,X       GET REDUCE-TO TOKEN
    PSHA                         SAVE REDUCE-TO TOKEN
    BSR   LRREDUCTION1           PUSH RETURN ADDRESS FOR CALL TO SEMANTIC
    LDX   LRSTATESTACK           SAVE VALUE OF SEMANTIC ROUTINE
    STD   LRSTACKEDSEMANTICVALUE,X
    LDX   LRSTACKEDSTATE,X       GET POINTER TO REDUCED-TO STATE
    LDD   ,X++                   GET # REDUCTIONS AND EXITS IN THIS STATE
*   WE DON'T NEED TO SAVE NUMBER OF REDUCTIONS BECAUSE WE'LL NEVER EXAMINE THEM
*   STA   LRWSTATENREDUCTIONS    SAVE NUMBER OF REDUCTIONS
    PULA                         GET REDUCED-TO TOKEN BACK; (B) = # EXITS
    BRA   LRPARSERCHECKEXIT      ASSERT: THERE IS AN APPLICABLE EXIT!!!!
*   NOTE: EVEN IF THERE IS AN ERROR IN THE INPUT, THE REDUCE-TO TOKEN...
*   WILL BE FOUND AMONG THE EXITS IN THE STATE SELECTED BY (X)
*   IF THE GRAMMER IS TRULY LR(1) (OR A SUBSET THEREOF!)

LRREDUCTION1 ; CONTINUE PROCESSING OF REDUCTION
    LDD   LRRULESEMANTIC,X       GET ADDRESS OF SEMANTIC ROUTINE TO CALL
    PSHD                         SAVE SEMANTIC ROUTINE ADDRESS FOR LATER CALL
    LDA   LRRULERHSSIZE,X        GET NUMBER OF TOKENS ON RIGHT HAND SIDE
    DECA                         POP 1 LESS THAN RHS LENGTH FROM STACK
    LDB   #LRSTACKOBJECTSIZE     = # BYTES PER STACK OBJECT
    MUL6809                      COMPUTE AMOUNT OF STATE STACK TO REMOVE
    ADDD  LRSTATESTACK           REMOVE STATE INFO FOR RHS ITEMS FROM STACK
    LDX   LRSTATESTACK           USE CURRENT VALUE OF STATE STACK FOR SEMANTIC
    STD   LRSTATESTACK           WHEN WE GET BACK, USE THIS VALUE
    RTS                          CALL SEMANTIC ROUTINE

*   LRISMEMBERLOOKAHEADSET -- IS CURRENT TOKEN MEMBER OF LOOKAHEAD SET?
*   Input Arguments:
*       LRCURRENTTOKEN   Token to match
*       (A)              Which Rule (number) to compute lookahead set on
*       (X)              Pointer to stacked state in which to apply Reduction
*   This routine examines the follow set of Rule to compute the LAS
*   If LRCURRENTTOKEN is not in the follow set, then LRISMEMBERLOOKAHEADSET
*   if recursively invoked on reductions in the target state.
*   Returns Z bit set if "YES", reset if "NO"
*   Note: this routine could return a count of number chained reductions;
*   that would obviate the need to compute the same lookahead set several
*   times successively when performing a chain of reductions.
*
LRISMEMBERLOOKAHEADSET
    LDAB  #LRRULEREDUCTIONSIZE   # BYTES PER RULE/REDUCTION OBJECT
    MUL6809                      DISPLACEMENT INTO RULE TABLE
    LDY   #LRRULESBASE           GET POINTER TO BASE OF RULES
    LEAY  D,Y                    = ADDRESS OF RULE REDUCTION OBJECT
LRISMEMBERLAS0 ; ENTRY POINT WITH (X)=STACK P, (Y)=RULE ADDRESS
    LDD   LRRULELHSTOKEN,Y       GET REDUCE-TO TOKEN, SIZE OF RHS OF RULE
    PSHA                         SAVE REDUCE-TO TOKEN
    LDA   #LRSTACKOBJECTSIZE     = # BYTES PER STACK OBJECT
    MUL6809                      COMPUTE AMOUNT OF STATE STACK TO REMOVE
    LEAX  D,X                    (COMPUTE STATE STACK ADDR AFTER REDUCTION)
    LDY   LRSTACKSTATE,X         = ADDRESS OF REDUCE-TO STATE
    LEAY  2,Y                    SKIP OVER # EXITS AND REDUCTIONS IN THIS STATE
    PULA                         FETCH REDUCE-TO TOKEN
LRISMEMBERLASFINDEXIT ; FIND EXIT FROM REDUCE-TO STATE TO TARGET STATE
    CMPA  ,Y+                    DOES REDUCE-TO TOKEN MATCH THIS EXIT ?
    BEQ   LRISMEMBERLAS1         B/ YES, 0,Y CONTAINS TARGET STATE NUMBER
    LEAY  1,Y                    FIND NEXT EXIT
    BRA   LRISMEMBERLASFINDEXIT  ASSERT: AN EXIT OF STATE MATCHES REDUCE-TO TKN

LRISMEMBERLAS1 ; FOUND EXIT FROM REDUCE-TO STATE
    LDB   0,Y                    FETCH STATE # WHOSE EXIT TKNS ARE LOOKAHEAD TKNS
    CLRA                         FETCH STATE ADDRESS
    ASLD                         MAKE INDEX INTO STATE POINTER VECTOR
    LDY   LRSTATEINDEX           FETCH POINTER FROM STATE POINTER VECTOR
    LDY   [D,Y]                  = ADDRESS OF "LOOKAHEAD" STATE
* NOTE: ADDRESS OF LOOKAHEAD STATE SHOULD BE PRESERVED, IT WAS HARD
* ENOUGH TO COMPUTE!
    CMPY  #LRRULES               DOES IT POINT TO A STATE ?
    BCS   LRISMEMBERLAS2         B/ YES, STATE'S EXITS ARE LOOKAHEAD TOKENS!
* NO -- THIS STATE IS AN OPTIMIZED, SINGLE-REDUCTION ONLY STATE
* HERE'S WHERE OUR RECURSIVE LOOKAHEAD STEPS IN!
    BSR   LRISMEMBERLAS0         NO, POINTS TO STATE, DO RECURSIVE CALL!
    BEQ   LRISMEMBERLASYES       B/ CURRENT TOKEN REALLY IS MEMBER OF LOOKAHEAD
LRISMEMBERLASNO ; CURRENT TOKEN IS NOT MEMBER OF LOOKAHEAD SET
    LDAA  #1                     SIGNAL "NOT MEMBER OF LOOKAHEAD SET"
    RTS

LRISMEMBERLAS2 ; STATE EXITS ARE LOOKAHEAD TOKENS, COMPARE TO CURRENTTOKEN
    LDD   ,Y++                   GET # EXITS AND REDUCTIONS IN THIS STATE
    STA   LRWSTATENREDUCTIONS    SAVE # REDUCTIONS IN THIS STATE
LRISMEMBERLASEXITL ; LOOP THRU EXIT ITEMS IN TARGET STATE
    LDD   ,Y++                   FETCH NEXT EXIT TOKEN/EXIT-TO STATE NUMBER PAIR
    CMPA  LRCURRENTTOKEN         THE EXIT TOKEN HERE IS MEMBER OF LOOKAHEAD SET
    BEQ   LRISMEMBERLASYES       B/ CURRENT TOKEN REALLY IS MEMBER OF LOOKAHEAD
    DECB                         MORE EXIT TOKENS TO ADD TO LOOKAHEAD SET?
    BNE   LRISMEMBERLASEXITL     B/ YES, GO CHECK THEM, TOO!
    LDB   LRWSTATENREDUCTIONS    NOT IN EXIT SET, SEE IF IN LAS OF REDUCTIONS
    BEQ   LRISMEMBERLASNO        B/ NO REDUCTIONS, IS NOT MEMBER OF LAS
LRISMEMBERLASREDUCTIONL ; LOOP THRU REDUCE-TO ITEMS IN TARGET STATE
    LDA   ,Y+                    GET RULE NUMBER OF REDUCTION
    PSHS  B,Y                    SAVE COUNT AND POINTER TO REST OF STATE
    BSR   LRISMEMBERLOOKAHEADSET RECURSIVELY FIND IF TOKEN IN LAS OF REDUCTION
    PULS  Y,B                    RESTORE COUNT AND POINTER
    BEQ   LRISMEMBERLAS2         B/ FOUND CURRENT TOKEN IN LOOKAHEAD SET!
    DECB                         NOT IN LAS FOR THAT REDUCE-TO ITEM
    BNE   LRISMEMBERLASREDUCTION B/ MORE REDUCE-TO ITEMS TO COMPUTE LAS ON
    BRA   LRISMEMBERLASNO        CURRENT TOKEN IS NOT IN LOOKAHEAD SET

LRISMEMBERLASYES ; SIGNAL CURRENT TOKEN IS MEMBER OF LOOKAHEAD SET
    RTS

    END   LRPARSER
