         page  Token Collector
*        Token Collector for IDB
*        Returns next token from InputStream
*        (A) = token number, (X) = token value
*        Tokens which have a single character ASCII representation...
*        have a token number equal to the ASCII value.
*
*        With the exception of <LF>,<TAB> and <RETURN>,
*        no control character is a token.
*
*        Several special token types warrant some additional explanation:
*
*        A NUMBER token is generated when a (hexadecimal) digit stream is seen.
*        A number can optionally begin with "$" to maintain consistency with
*        the assembler.  The value of the number is assumed to be hex;
*        a trailing "." can be used to signal that the value is decimal.
*        The value returned is the value of the number.
*
*        A LONGHEX token is up to 64 bytes of Hex digits, optionally prefixed
*        by "$". The token "value" returned points to a Count Byte,
*        followed by CountByte bytes containing the value of the token.
*
*        A SYMBOL token is up to 32 characters of ASM compatible symbol.
*        The token "value" returned points to a Count Byte,
*        followed by CountByte bytes containing the ASCII symbol name.
*
*        A TEXT token is up to 32 characters embedded in either single or
*        double quotes.  The token "value" returned points to a Count Byte,
*        followed by CountByte bytes containing the actual text bytes.
*
*        A FLOAT token is up to 64 bytes of value parsed by a user-specified
*        floating-point conversion routine.  The returned values are identical
*        to LONGHEX.
*
*        The PCR token is a reserved symbol name, used by (yuk) Motorola
*        compatible 6809 assemblers as an index register "name".
*
*        So called "illegal tokens" are collected by this token collector.
*        The LR syntax parser will reject them if present, so this routine
*        does not need to check for them.
*
         page
*        Token Name          Token number             Token Value
*        ----- ----          ------------             -----------
*        EndOfFile           $00                      End of File token
*        NUMBER              $02                      16 bit value of number
*        SYMBOL              $03                      pointer to symbol name
*        TEXT                $04                      pointer to text string
*        LONGHEX             $05                      pointer to value
*        FLOAT               $06                      Floating point value
*        REGISTERNAME        $07                      single letter register
*        <LF>                $0A                      none
*        <CR>                $0D                      none
*        OPINHERENT          $10                      inherent mode opcode
*        OPPUSHPULL          $11                      PSHS/U or PULS/U opcode
*        OPTFREXG            $12                      TFR/EXG opcode
*        OPRELATIVE          $13                      Relative branch opcode
*        OPMEMORY            $14                      Memory-reference opcode
*        OPIMPLIEDIMM        $15                      Opcode with implied imm
*        OPLONG              $16                      Long instruction sequence
*        !!                  $1B                      Exclusive-or operator
*        ->                  $1C                      "points to" operator
*        :=                  $1D                      "gets new value of" op
*        ##                  $1E                      Shift operator
*        =?                  $1F                      Display value operator
*        <TAB>               $20                      none
*        <SPACE>             $20                      none
*        !                   $21                      none
*        #                   $23                      none
*        %                   $25                      none
*        &                   $26                      none
*        (                   $28                      none
*        )                   $29                      none
*        *                   $2A                      none
*        +                   $2B                      none
*        ,                   $2C                      none
*        -                   $2D                      none
*        .                   $2E                      none
*        /                   $2F                      none
*        :                   $3A                      none
*        <                   $3C                      none
*        =                   $3D                      none
*        >                   $3E                      none
*        ?                   $3F                      none
*        @                   $40                      none
*        [                   $5B                      none
*        \                   $5C                      none
*        ]                   $5D                      none
*        ^                   $5E                      none
*        <underscore>        $5F                      illegal token
*        <accentgrave>       $60                      illegal token
*        {                   $7B                      illegal token
*        |                   $7C                      illegal token
*        }                   $7D                      illegal token
*        ~                   $7E                      illegal token
*        <rubout>            $7F                      illegal token
*        <singleletter>      See REGISTERNAME above   ascii code for letter
*        ;<letter>           Lower case Ascii code for letter

*
*        Special token code equates
*
Token:Number      equ        $02
Token:Symbol      equ        $03
Token:Text        equ        $04
Token:LongHex     equ        $05
Token:Float       equ        $06
Token:RegisterName equ       $07

Token:OpInherent  equ        $10
Token:OpPushPull  equ        $11
Token:OpTFREXG    equ        $12
Token:OpRelative  equ        $13
Token:OpMemory    equ        $14
Token:ImpliedImm  equ        $15
Token:OpLong      equ        $16

Token:Xor         equ        $1B
Token:PointsTo    equ        $1C
Token:Assign      equ        $1D
Token:Shift       equ        $1E
Token:DisplayValue equ       $1F
         Page
CollectTokenControl ; Collect control character token
         cmpa #Ascii:lf      Linefeed ?
         beq  CollectTokenDone b/ yes, all done
         cmpa #Ascii:tab     Tab ?
         beq  CollectTokenBlank b/ yes, convert to "blank" character
         cmpa #Ascii:cr      Carriage return ?
         beq  CollectTokenDone b/ yes, all done
CollectTokenIllegal ; Force illegal token to cause parser to barf!
         lda  #Ascii:Rubout  This token code is not legal.
CollectTokenDone ; Token is in (A), value in (X)
         rts

CollectTokenBlank ; Convert <TAB> to BLANK token
         lda #Ascii:blank
         rts

******************************************************************************
CollectToken ; Collect next token from input stream
******************************************************************************
         ldx  #TokenBuffer   in case of text, symbol or longhex object
         stx  TokenBufferFill where to start filling token buffer
         clr  TokenBuffercount = # bytes in token
         jsr  GetCharwithReject duh...
         cmpa #Ascii:Blank   a printing character ?
         blo  CollectTokenControl b/ no, a control character
*  Check for start of special tokens
         cmpa #'"            Text string start ?
         beq  CollectText    b/ yes, go process
         cmpa #'$            Hex number coming up ?
         lbeq CollectHex     b/ yes, go process
         cmpa #'%            Binary number coming up ?
         lbeq CollectBinary  b/ yes, go process
         cmpa #''            Another way to start a text string ?
         beq  CollectText    b/ yes, go process
         cmpa #';            ";<letter>" token ?
         beq  CollectSemicolon b/ yes, go process
         cmpa #'0            a digit or a letter ?
         blo  CollectTokenDone b/ no, this is a single character token
         cmpa #'9            a digit only ?
         bls  CollectDigit   b/ yes, go collect digit
         lbsr Isletter       a letter ?
         bcs  CollectTokendone b/ no, must be single character token
         page
CollectSymbolLoop ; save next symbol character
         lbsr StoreTokenCharacter save character in token buffer
         lbsr GetCharwithReject look at following character
         lbsr Isletter       is following character a letter ?
         bcc  CollectSymbolLoop b/ yes, save it
         cmpa #'0            is it a digit ?
         blo  CollectSymbolDone b/ no, done collecting symbol
         cmpa #'9            ...?
         bls  CollectSymbolLoop b/ yes, add to symbol
         cmpa #'$            Dollar sign is legal in symbol
         beq  CollectSymbolLoop
         cmpa #':            as are colons...
         beq  CollectSymbolLoop
CollectSymbolDone ; Symbol name is collected
         lda  TokenBufferCount how long is symbol name ?
         cmpa #1             single letter symbol name ?
         bne  CollectSymbolExit b/ no, return symbol token type
         lda  Tokenbuffer    yes, use symbol character as token name
         rts

CollectSymbolExit ; Multi-letter symbol name
         ldx  Tokenbuffer    Is it the special token, "PCR" ?
         cpx  #'P*256+'C     ...?
         bne  CollectSymbolSearch b/ no, go search opcode table
         lda  TokenBufferCount (is count=3 ?)
         ldb  TokenBuffer+2  (is final character "R" ?)
         cmpd #3*256+'R      ...?
         bne  CollectSymbolSearch b/ no, go search opcode table
         lda  #Token:RegisterName get proper token code
         ldx  #UserRegisterPC ??? is this right ???
         rts                 and exit with it!

CollectSymbolSearch ; Search opcode table for symbol
         lbsr SearchOpcodeTable see if symbol name is an opcode
         bcc  CollectOpcode  b/ its an opcode name !
         lda  #Token:Symbol  get code for symbol token
         ldx  #Tokenbuffer   get pointer to Token name
         rts

CollectOpcode ; Symbol name given is opcode value
         ripapartopcode to get instruction type --> token for opcode
         page
CollectSemicolon ; Collect ";<letter>"
         lbsr  GetCharwithReject get the next character
         lbsr Isletter       make sure its legal!
         bcs  CollectTokenIllegal b/ nope, feed parser the illegal token
         adda #'a-'A         generate token code (convert to lower case)
         rts

CollectText ; collect quoted text string as token
         sta  CollectQuote   save the quote character
CollectTextLoop ; collect next byte of quoted text token
         bsr  GetCharwithReject
         cmpa CollectQuote   match opening quote character ?
         bne  CollectTextByte b/ no, just save the byte
         lbsr GetCharwithReject see if doubled quote character...
         cmpa CollectQuote   ...?
         bne  CollectTextDone b/ no, all done!
CollectTextByte ; (A) is new byte for text string
         lbsr StoreTokenCharacter no, store character in token buffer
         bra  CollectTextLoop

CollectTextDone ; (A) is character past end of string
         inc  CharRejectedFlag so we see character again, next time through
         lda  #Token:Text    Get code for text token
         ldx  #Tokenbuffer   = pointer to count, contents
         rts
         page
*        CollectNumber -- collect number specified in decimal
         ...

         page
*        CollectHexNibble -- Collect next hex nibble from input
*        Returns C reset if hex nibble, (A) contains value of nibble
*        Returns C Set if input not hex, next character is rejected
*
CollectHexNibble ; Collect hex nibble
         bsr  GetCharwithReject get next hex digit
         cmpa #'A            upper case hex character ?
         blo  CollectHexDigit b/ no, must be digit !
         cmpa #'F
         bhi  CollectHexLower b/ if a digit, must be lower case hex digit
         suba #'A-10         convert to binary value of hex digit
         rts                 assert: Carry bit is reset here

CollectHexLower ; Must be lower case hex
         cmpa #'a            lower case hex character ?
         blo  CollectHexNotHex b/ not hex digit at all!
         cmpa #'f
         bhi  CollectHexNotHex b/ not hex digit at all!
         suba #'a-10         convert to binary value here
         rts                 assert: Carry bit is zero, here

CollectHexDigit ; Must be real digit to be hex number
         cmpa #'9            Is it a real live hex digit, daddy ?
         bhi  CollectHexNotHex b/ nope.
         suba #'0            a real digit ?
         blo  CollectHexNotHex b/ nope.
         rts                 binary value of hex digit is in (A)

CollectHexNotHex ; Next input stream character is NOT hex.
         inc  CharRejectedFlag So CollectToken sees it again!
         sec                 indicating no hex digit found
         rts
         page
CollectHex ; Collect Hex number
         bsr  CollectHexNibble get next hex nibble
         bcs  CollectHexEven b/ Even number of hex bytes collected
         asla                shift nibble into position
         asla
         asla
         asla
         psha                save nibble
         bsr  CollectHexNibble get 2nd nibble to complete byte
         bcs  CollectHexOdd  b/ odd number of hex bytes collect
         ora  ,s+            complete hex byte
         bsr  StoreTokenCharacter save byte in token buffer
         bra  CollectHex

CollectHexOdd ; Odd number of hex bytes collected, must shift right 1 nibble!
         ldx  #TokenBufferFill = address of last byte collected
         cpx  #TokenBufferEnd any room in buffer ?
         beq  SyntaxError    b/ no
         inc  TokenBufferCount
         clrb                = rightmost byte before right nibble shift
CollectHexRightShift ; right shift 1 nibble
         lda  ,x             get byte to extract nibble from
         lsrd                right shift one nibble
         lsrd
         lsrd
         lsrd
         stb  1,x
         ldb  ,x
         dex
         cpx  #TokenBuffer
         bne  CollectHexRightShift
         sta  ,x
CollectHexEven ; Even number of hex bytes collected
         lda  #Token:Longhex get token type
         ldx  #Tokenbuffer   where token is
         rts
         page
*        CollectBinary -- Collect a binary number
*
CollectBinary
         page
*
*        IsLetter -- Tests A register to see if upper/lower case letter
*        Returns C bit reset if yes, A register set to uppercase version
*        Otherwise, returns C bit set, A register unchanged
Isletter ; Is (A) a letter ?
         cmpa #'Z            an uppercase letter ?
         bls  IsletterU      b/ might be upper case letter
         cmpa #'z            ...?
         bls  IsletterL      b/ might be lower case letter
         sec                 Set carry to indicate "Not a letter"
IsletterRts ; Carry is set properly
         rts

IsLetterU ; might be uppercase letter
         cmpa #'A            too small to be a letter ?
         rts                 C set --> yes

IsLetterL ; might be lowercase letter
         cmpa #'a            lower case letter ?
         bcs  IsletterRts    b/ no, just exit
         suba #'a-'A         convert to upper case, reset carry
         rts
         page
*        GetCharwithReject -- Get character from input stream with Reject
*        If CharRejectedFlag is reset, reads next character from input stream
*        If Set, resets flag and returns last character read from input stream
*
GetCharwithReject ; Get character from input stream if not rejected
         lda  CharRejectedFlag was last character rejected ?
         bne  GetRejectedCharacter b/ yes, return it again
         ldx  InputStreamPointer fetch next byte from input stream
         cpx  InputStreamEnd Input Stream exhausted ?
         blo  GetChar1       b/ no
         cpx  InputStreamEOM yes, is end of Medium (available space) reached ?
         beq  GetCharCant    b/ yes, cannot get more from input stream
         jsr  GetConsoleChar no, fetch character from console
         ldx  InputStreamPointer and save it in the input buffer
         sta  ,x+
         stx  InputStreamEnd Update logical end of input
         dex
GetChar1 ; Fetch character from input buffer location (X)
         lda  ,x+            fetch the byte
         stx  InputStreamPointer save the updated pointer
         sta  RejectedCharacter In case it is rejected
GetRejectedCharacter ; need I say more ?
         lda  RejectedCharacter
         clr  CharRejectedFlag
         rts

GetCharCant ; no more input medium to read into, return a <RUBOUT>
         lda  #$7f
         rts
*
*        StoreTokenCharacter -- Stores (A) into next token buffer byte
*        Advances pointer, increments count
*        ?? what if buffer overflows?
*         presumably, yell at user and reject token!
*
StoreTokenCharacter ; Store (A) in next available token buffer byte
         ldx  TokenbufferFill get pointer
         cpx  #TokenbufferEnd is token buffer full ?
         beq  SyntaxError    Can't exceed token buffer capacity!
         sta  ,x+            save byte
         stx  TokenbufferFill
         inc  Tokenbuffercount = # bytes collected
         rts

SyntaxError ; Discovered syntax error in middle of token
         ???
         Page M6809 Instruction Mnemonics for Instruction Assembly/Disassembly
*   Opcode Table format:
*        <C><C><C><C><T><B><B>
*        where <C> is Ascii Character; blanks used to fill to 4 characters
*        and <T> is token number
*        and <B> is opcode byte(s); one byte of opcode if not = $10 or $11
*        otherwise <B> is two bytes of opcode
*        Memory Reference opcodes are stored with Direct page address mode
*        **** Note: opcodes are list in opcode order in table...
*        **** to allow "ListInstruction" to find text corresponding to opcode
*        **** Gaps in opcode sequence are filled with convenience instructions
*
OpcodeTable ; Table used to look up opcode names
*
TextNEG ; base of sub-table used to list opcodes $00-$3F
; !!! NOTE: FROM HERE TO "SWI" OPCODE: ORDER MUST NOT BE DISTURBED !!!
; (Instruction lister requires opcode names to be in opcode order)
 +'N,'E,'G,$20,Token:OpMemory,$00   Note: instruction assembler disallows NEG #..., etc.
 +'L,'D,'A,'A,Token:OpMemory,$96   Fills hole for illegal opcode $01
 +'S,'T,'A,'A,Token:OpMemory,$97   Fills hole for illegal opcode $02
 +'C,'O,'M,$20,Token:OpMemory,$03
 +'L,'S,'R,$20,Token:OpMemory,$04
 +'L,'D,'A,'B,Token:OpMemory,$D6   Fills hole for illegal opcode $05
 +'R,'O,'R,$20,Token:OpMemory,$06
 +'A,'S,'R,$20,Token:OpMemory,$07
 +'A,'S,'L,$20,Token:OpMemory,$08
 +'R,'O,'L,$20,Token:OpMemory,$09
 +'D,'E,'C,$20,Token:OpMemory,$0A
 +'L,'S,'L,$20,Token:OpMemory,$08   Fills hole for illegal opcode $0B
 +'I,'N,'C,$20,Token:OpMemory,$0C
 +'T,'S,'T,$20,Token:OpMemory,$0D
 +'J,'M,'P,$20,Token:OpMemory,$0E
 +'C,'L,'R,$20,Token:OpMemory,$0F

TextBSR ; Text for listing BSR opcode
 +'B,'S,'R,$20,Token:OpRelative,$8D  Fills hole for illegal opcode $10
 +'B,'L,'O,$20,Token:OpRelative,$25 Fills hole for illegal opcode $11
 +'N,'O,'P,$20,Token:OpInherent,$12
 +'S,'Y,'N,'C,Token:OpInherent,$13
 +'B,'H,'S,$20,Token:OpRelative,$24 Fills hole for illegal opcode $14
 +'O,'R,'A,'A,Token:OpMemory,$9A   Fills hole for illegal opcode $15
 +'L,'B,'R,'A,Token:OpRelative,$16 Note: LBRA and LBSR require special handling
 +'L,'B,'S,'R,Token:OpRelative,$17
 +'C,'P,'X,$20,Token:OpMemory,$9C   Fills hole for illegal opcode $18
 +'D,'A,'A,$20,Token:OpInherent,$19
 +'O,'R,'C,'C,Token:ImpliedImm,$1A
 +'O,'R,'A,'B,Token:OpMemory,$DA   Fills hole for illegal opcode $1B
TextANDC ; Text string for ANDCC opcode: note special handling due to 5 char op
 +'A,'N,'D,'C,Token:ImpliedImm,$1C
 +'S,'E,'X,$20,Token:OpInherent,$1D
 +'T,'F,'R,$20,Token:OpTFREXG,$1E  Note: instruction assembler checks postbyte validity
 +'E,'X,'G,$20,Token:OpTFREXG,$1F

 +'B,'R,'A,$20,Token:OpRelative,$20 Note: instruction assembler handles long/short
 +'B,'R,'N,$20,Token:OpRelative,$21
 +'B,'H,'I,$20,Token:OpRelative,$22
 +'B,'L,'S,$20,Token:OpRelative,$23
 +'B,'C,'C,$20,Token:OpRelative,$24
 +'B,'C,'S,$20,Token:OpRelative,$25
 +'B,'N,'E,$20,Token:OpRelative,$26
 +'B,'E,'Q,$20,Token:OpRelative,$27
 +'B,'V,'C,$20,Token:OpRelative,$28
 +'B,'V,'S,$20,Token:OpRelative,$29
 +'B,'P,'L,$20,Token:OpRelative,$2A
 +'B,'M,'I,$20,Token:OpRelative,$2B
 +'B,'G,'E,$20,Token:OpRelative,$2C
 +'B,'L,'T,$20,Token:OpRelative,$2D
 +'B,'G,'T,$20,Token:OpRelative,$2E
 +'B,'L,'E,$20,Token:OpRelative,$2F

*** s/b page mark before BSR opcode
         page
 +'L,'E,'A,'X,Token:OpMemory,$30    Note: instruction assembler checks for index postbyte
 +'L,'E,'A,'Y,Token:OpMemory,$31
 +'L,'E,'A,'S,Token:OpMemory,$32
 +'L,'E,'A,'U,Token:OpMemory,$33
 +'P,'S,'H,'S,Token:OpPushPull,$34
 +'P,'U,'L,'S,Token:OpPushPull,$35
 +'P,'S,'H,'U,Token:OpPushPull,$36
 +'P,'U,'L,'U,Token:OpPushPull,$37
 +'S,'T,'A,'B,Token:OpMemory,$D7    Fills hole for illegal opcode $38
 +'R,'T,'S,$20,Token:OpInherent,$39
 +'A,'B,'X,$20,Token:OpInherent,$3A
 +'R,'T,'I,$20,Token:OpInherent,$3B
 +'C,'W,'A,'I,Token:ImpliedImm,$3C
 +'M,'U,'L,$20,Token:OpInherent,$3D
 +'N,'E,'G,'A,Token:OpInherent,$40   Fills hole for illegal opcode $3E
 +'S,'W,'I,$20,Token:OpInherent,$3F
; !!! OPCODE SEQUENCE TO HERE MUST NOT BE DISTURBED !!!

         page
**** note: monadic to register consumes 6*16*2 == 200 bytes
**** probably should special case token buster to recognize xxxA or xxxB
**** if xxx is monadic opcode; would save space!!!
**** Note: these can be in any order
 +'N,'E,'G,'A,Token:OpInherent,$40
 +'C,'O,'M,'A,Token:OpInherent,$43
 +'L,'S,'R,'A,Token:OpInherent,$44
 +'R,'O,'R,'A,Token:OpInherent,$46
 +'A,'S,'R,'A,Token:OpInherent,$47
 +'A,'S,'L,'A,Token:OpInherent,$48
 +'L,'S,'L,'A,Token:OpInherent,$48
 +'R,'O,'L,'A,Token:OpInherent,$49
 +'D,'E,'C,'A,Token:OpInherent,$4A
 +'I,'N,'C,'A,Token:OpInherent,$4C
 +'T,'S,'T,'A,Token:OpInherent,$4D
 +'C,'L,'R,'A,Token:OpInherent,$4F
 +'N,'E,'G,'B,Token:OpInherent,$50
 +'C,'O,'M,'B,Token:OpInherent,$53
 +'L,'S,'R,'B,Token:OpInherent,$54
 +'R,'O,'R,'B,Token:OpInherent,$56
 +'A,'S,'R,'A,Token:OpInherent,$57
 +'A,'S,'L,'B,Token:OpInherent,$58
 +'L,'S,'L,'B,Token:OpInherent,$58
 +'R,'O,'L,'B,Token:OpInherent,$59
 +'D,'E,'C,'B,Token:OpInherent,$5A
 +'I,'N,'C,'B,Token:OpInherent,$5C
 +'T,'S,'T,'B,Token:OpInherent,$5D
 +'C,'L,'R,'B,Token:OpInherent,$5F
         page
TextSUBA ; base of sub-table used to display arithmetic class instructions
; !!! OPCODE SEQUENCE FROM HERE TO "STU" MUST NOT BE DISTURBED !!!
; (Instruction Lister requires these instructions in opcode order)
 +'S,'U,'B,'A,Token:OpMemory,$90
 +'C,'M,'P,'A,Token:OpMemory,$91
 +'S,'B,'C,'A,Token:OpMemory,$92
 +'S,'U,'B,'D,Token:OpMemory,$93 Note: instruction assembler handles 2 byte immediate
 +'A,'N,'D,'A,Token:OpMemory,$94
 +'B,'I,'T,'A,Token:OpMemory,$95
 +'L,'D,'A,$20,Token:OpMemory,$96
 +'S,'T,'A,$20,Token:OpMemory,$97  Note: instruction assembler disallows STq #...
 +'E,'O,'R,'A,Token:OpMemory,$98
 +'A,'D,'C,'A,Token:OpMemory,$99
 +'O,'R,'A,$20,Token:OpMemory,$9A
 +'A,'D,'D,'A,Token:OpMemory,$9B
 +'C,'M,'P,'X,Token:OpMemory,$9C
 +'J,'S,'R,$20,Token:OpMemory,$9D  Note: instruction assembler disallows JSR #...
 +'L,'D,'X,$20,Token:OpMemory,$9E
 +'S,'T,'X,$20,Token:OpMemory,$9F
 +'S,'U,'B,'B,Token:OpMemory,$D0
 +'C,'M,'P,'B,Token:OpMemory,$D1
 +'S,'B,'C,'B,Token:OpMemory,$D2
 +'A,'D,'D,'D,Token:OpMemory,$D3
 +'A,'N,'D,'B,Token:OpMemory,$D4
 +'B,'I,'T,'B,Token:OpMemory,$D5
 +'L,'D,'B,$20,Token:OpMemory,$D6
 +'S,'T,'B,$20,Token:OpMemory,$D7
 +'E,'O,'R,'B,Token:OpMemory,$D8
 +'A,'D,'C,'B,Token:OpMemory,$D9
 +'O,'R,'B,$20,Token:OpMemory,$DA
 +'A,'D,'D,'B,Token:OpMemory,$DB
 +'L,'D,'D,$20,Token:OpMemory,$DC
 +'S,'T,'D,$20,Token:OpMemory,$DD
 +'L,'D,'U,$20,Token:OpMemory,$DE
 +'S,'T,'U,$20,Token:OpMemory,$DF
; !!! OPCODE SEQUENCE TO HERE MUST NOT BE DISTURBED !!!
         page
TextLBRN ; base of sub-table used to list long branch instructions
; !!! OPCODE SEQUENCE FROM HERE THRU "LBLO" MUST NOT BE DISTURBED !!!
; (Instruction Lister requires these instructions in opcode order)
 +'L,'B,'R,'N,Token:OpRelative,$1021
 +'L,'B,'H,'I,Token:OpRelative,$1022
 +'L,'B,'L,'S,Token:OpRelative,$1023
 +'L,'B,'C,'C,Token:OpRelative,$1024
 +'L,'B,'C,'S,Token:OpRelative,$1025
 +'L,'B,'N,'E,Token:OpRelative,$1026
 +'L,'B,'E,'Q,Token:OpRelative,$1027
 +'L,'B,'V,'C,Token:OpRelative,$1028
 +'L,'B,'V,'S,Token:OpRelative,$1029
 +'L,'B,'P,'L,Token:OpRelative,$102A
 +'L,'B,'M,'I,Token:OpRelative,$102B
 +'L,'B,'G,'E,Token:OpRelative,$102C
 +'L,'B,'L,'T,Token:OpRelative,$102D
 +'L,'B,'G,'T,Token:OpRelative,$102E
 +'L,'B,'L,'E,Token:OpRelative,$102F
 +'L,'B,'H,'S,Token:OpRelative,$1024
 +'L,'B,'L,'O,Token:OpRelative,$1025
; !!! OPCODE SEQUENCE TO HERE MUST NOT BE DISTURBED !!!
         page
TextCMPD ; Text for listing CMPD opcode
 +'C,'M,'P,'D,Token:OpMemory,$1093
TextCMPY ; Text for listing CMPY opcode
 +'C,'M,'P,'Y,Token:OpMemory,$109C
TextLDY ; Text for listing LDY opcode
 +'L,'D,'Y,$20,Token:OpMemory,$109E
TextSTY ; Text for listing STY opcode
 +'S,'T,'Y,$20,Token:OpMemory,$109F
TextLDS ; Text for listing LDS opcode
 +'L,'D,'S,$20,Token:OpMemory,$10DE
TextSTS ; Text for listing STS opcode
 +'S,'T,'S,$20,Token:OpMemory,$10DF
TextCMPU ; Text for listing CMPU opcode
 +'C,'M,'P,'U,Token:OpMemory,$1193
TextCMPS ; Text for list CMPS opcode
 +'C,'M,'P,'S,Token:OpMemory,$119C

TextSWI2 ; Text for listing SWI2 opcode
 +'S,'W,'I,'2,Token:OpInherent,$103F
TextSWI3 ; Text for listing SWI3 opcode
 +'S,'W,'I,'3,Token:OpInherent,$113F
         page
*   6800 compatibility opcodes
*
 +'I,'N,'X,$20,Token:OpLong,2,$3001
 +'D,'E,'X,$20,Token:OpLong,2,$301F
 +'I,'N,'S,$20,Token:OpLong,2,$3061
 +'D,'E,'S,$20,Token:OpLong,2,$307F
 +'P,'S,'H,'A,Token:OpLong,2,$3402
 +'P,'S,'H,'B,Token:OpLong,2,$3404
 +'P,'S,'H,'D,Token:OpLong,2,$3406
 +'P,'S,'H,'X,Token:OpLong,2,$3410
 +'P,'U,'L,'A,Token:OpLong,2,$3502
 +'P,'U,'L,'B,Token:OpLong,2,$3504
 +'P,'U,'L,'D,Token:OpLong,2,$3506
 +'P,'U,'L,'X,Token:OpLong,2,$3510
 +'T,'A,'B,$20,Token:OpLong,3,$1F89,$5D
 +'T,'B,'A,$20,Token:OpLong,3,$1F98,$4D
 +'T,'X,'D,$20,Token:OpLong,2,$1F10
 +'T,'D,'X,$20,Token:OpLong,2,$1F01
 +'T,'X,'S,$20,Token:OpLong,2,$1F14
 +'T,'S,'X,$20,Token:OpLong,2,$1F41
 +'A,'B,'A,$20,Token:OpLong,4,$3404,$ABE0
 +'S,'B,'A,$20,Token:OpLong,4,$3404,$A0E0
 +'C,'B,'A,$20,Token:OpLong,4,$3404,$A1E0
 +'C,'O,'M,'D,Token:OpLong,2,$4353
 +'N,'E,'G,'D,Token:OpLong,4,$4350,$8200
 +'R,'O,'L,'D,Token:OpLong,2,$5949
 +'R,'O,'R,'D,Token:OpLong,2,$4656
 +'A,'S,'L,'D,Token:OpLong,2,$5849
 +'L,'S,'L,'D,Token:OpLong,2,$5849
 +'A,'S,'R,'D,Token:OpLong,2,$4756
 +'L,'S,'R,'D,Token:OpLong,2,$4456
 +'T,'S,'T,'D,Token:OpLong,3,$83,#$0000
 +'C,'L,'C,$20,Token:OpLong,2,$1CFE
 +'C,'L,'I,$20,Token:OpLong,2,$1CEF
 +'C,'L,'V,$20,Token:OpLong,2,$1CFD
 +'S,'E,'C,$20,Token:OpLong,2,$1A01
 +'S,'E,'I,$20,Token:OpLong,2,$1A10
 +'S,'E,'V,$20,Token:OpLong,2,$1A02
 +'T,'A,'P,$20,Token:OpLong,2,$1F8A
 +'T,'P,'A,$20,Token:OpLong,2,$1FA8
OpcodeTableEnd ; No opcode entries beyond here
         Page
