
/*
 * V Kernel - Copyright (c) 1981 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 * Copyright (c) 1982 Stanford University.
 *
 * Interrupt service routine that calls a machine-independent
 * C routine to handle the interrupt.  The routine does four things:
 *
 *	1)  Save r0-r5.  This is because C functions destroy those registers.
 *	2)  Call the interrupt handler.
 *	3)  Restore the registers.
 *	4)  Cause a process switch if necessary.
 *
 * The parameter to the macro should be the name of the function to
 *  handle the interrupt.  If the C function name is "func", the
 *  interrupt vector needs to be plugged with the address of the assembly
 *  language label "_Asm_func" and Call_inthandler needs to be called
 *  with the parameter func.
 * You have to say "extern Asm_func();" yourself because lint gaks when there
 *  asm's and a function declaration in the same line and there is no way to
 *  get cpp to insert a newline.
 *
 * The location KernelInterrupted is set to a nonzero value if the
 *  interrupt occurred during a kernel operation (indicated by being
 *  in kernel mode and active not being equal to the Idle process).
 *  If the kernel was not interrupted, we set ProcessInterrupted
 *  to allow the exception handler code to reliably detect exceptions
 *  in interrupt routines.
 */

#ifndef INTERRUPT_H
#define INTERRUPT_H

#ifndef PROCESS_H
# include "process.h"
#endif
#ifndef ASMDEFS_H
# include "asmdefs.h"
#endif
#ifndef ASMDEFINES_H
# include "asmdefines.h"
#endif

extern int		(*SysControlBlock[])();

#define setexvec(vecname,vecnum)  /* Initialize a vector */	\
SysControlBlock[vecnum] = (int (*)())(vecname)

/* Used by the interrupt handler */
extern short KernelInterrupted, ProcessInterrupted;

#define Call_inthandler(handler) \
	intserv(_/**/handler,_Asm_/**/handler)

#define Call_timer_inthandler(handler) \
	timer_intserv(_/**/handler,_Asm_/**/handler)
/*
 * Assembly language front-end to device interrupt routines.
 * The ".align 2" is required since the low-order two bits of
 * the interrupt vector are used by the hardware to determine
 * which stack to use and whether to call user control store.
 * Here the two bits are always 0 so the kernel stack is used
 * instead of the interrupt stack.
 */
#define	intserv(_handler,_Asm_handler)				\
asm("	.text");							\
asm("	.align	2");							\
asm("	.globl	_Asm_handler");						\
asm("_Asm_handler:");							\
PrimaryDisable;				/* Disable interrupts */	\
KLock(); 								\
asm("	movq	r4, -(sp)");		/* Save regs that C destroys */	\
asm("	movq	r2, -(sp)");						\
asm("	movq	r0, -(sp)");						\
									\
AsmGetProcessorRecord(r0);						\
asm("   bitl    $_PSL_CURMOD,28(sp)");   /* Were we in kernel mode ? */  \
asm("	bneq	3f");			/* No, we were in a process */	\
asm("   bitl    $_PSL_IPL,28(sp)");   /* Is saved IPL = 0? */         \
asm("	bneq	1f");			/* no, we were in the kernel */	\
									\
asm("	moval 	_PR_IDLE_PROCESS(r0),r1");				\
asm("	cmpl	_PR_ACTIVE(r0),r1");	/* active = Idle_process? */	\
asm("	beql	3f");			/* yes, try process switching */\
asm("1:");				/* Kernel interrupted */	\
asm("	incw	_KernelInterrupted");	/* Kernel interrupted */	\
asm("	calls	$0, _handler");						\
asm("	decw	_KernelInterrupted");					\
asm("2:");								\
asm("	movq	(sp)+, r0");		/* Restore registers */		\
asm("	movq	(sp)+, r2");						\
asm("	movq	(sp)+, r4");						\
KUnlock(); 								\
									\
asm("	rei");				/* Return */			\
asm("3:");				/* Kernel not interrupted */	\
asm("	incw	_ProcessInterrupted");	/* Process interrupted */	\
asm("	calls	$0, _handler");						\
asm("	decw	_ProcessInterrupted");					\
									\
AsmGetProcessorRecord(r2);						\
asm("	tstl	_PR_READYQ(r2)");	/* Check ready queue empty */	\
asm("	beql	2b");			/* If yes, we are done. */	\
/* Check if active is still higher/equal priority than ready queue head */\
asm("	movl	_PR_ACTIVE(r2), r0");					\
asm("	movl	_PR_READYQ(r2), r1");					\
asm("	cmpw	_PRIORITY(r0), _PRIORITY(r1)");				\
asm("	blequ	2b");			/* If so, continue */		\
/*									\
 * Now test for the case where the active process has been destroyed,	\
 * in which case we reschedule immediately. No need to restore 		\
 * registers and the the kernel stack pointer here. 			\
 */									\
asm("	cmpl	_LOCALPID(r0),$_INVALID_PROCESS");			\
asm("	bneq	0f");							\
asm("	moval 	_PR_IDLE_PROCESS(r2),r1");				\
asm("	cmpl	r0,r1");						\
asm("	beql	0f");							\
asm("	jmp	ActivateReadyqHead");					\
asm("0:");								\
asm("	pushl	r0" );		/* Arg to Addready */		\
asm("	calls	$1, _Addready");	/* Add active to readyq	*/	\
asm("	movq	(sp)+, r0");						\
asm("	movq	(sp)+, r2");						\
asm("	movq	(sp)+, r4");						\
asm("	jmp	Switch")

#ifdef FIREFLY
#define	timer_intserv(_handler,_Asm_handler)				\
asm("	.text");							\
asm("	.align	2");							\
asm("	.globl	_Asm_handler");						\
asm("_Asm_handler:");							\
PrimaryDisable;				/* Disable interrupts */	\
asm("	movq	r4, -(sp)");		/* Save regs that C destroys */	\
asm("	movq	r2, -(sp)");						\
asm("	movq	r0, -(sp)");						\
									\
AsmGetProcessorRecord(r0);						\
asm("   bitl    $_PSL_CURMOD,28(sp)");   /* Were we in kernel mode ? */  \
asm("	bneq	3f");			/* No, we were in a process */	\
asm("   bitl    $_PSL_IPL,28(sp)");   /* Is saved IPL = 0? */         \
asm("	bneq	1f");			/* no, we were in the kernel */	\
									\
asm("	moval 	_PR_IDLE_PROCESS(r0),r1");				\
asm("	cmpl	_PR_ACTIVE(r0),r1");	/* active = Idle_process? */	\
asm("	beql	3f");			/* yes, try process switching */\
asm("1:");				/* Kernel interrupted */	\
asm("	incw	_KernelInterrupted");	/* Kernel interrupted */	\
asm("	calls	$0, _handler");						\
asm("	decw	_KernelInterrupted");					\
asm("2:");								\
asm("	movq	(sp)+, r0");		/* Restore registers */		\
asm("	movq	(sp)+, r2");						\
asm("	movq	(sp)+, r4");						\
asm("	rei");				/* Return */			\
asm("3:");				/* Kernel not interrupted */	\
asm("	incw	_ProcessInterrupted");	/* Process interrupted */	\
asm("	calls	$0, _handler");						\
asm("	decw	_ProcessInterrupted");					\
									\
asm("	tstl	_HoldKLock");		/* If hold KernelLock */	\
asm("	beql	4f");	                				\
asm("	clrl	_HoldKLock");						\
asm("	bbcci	$0,_KernelLock,4f");	/* Release KernelLock */	\
asm("	clrl	_KLockHolder");						\
asm("4:");								\
AsmGetProcessorRecord(r2);						\
asm("	tstl	_PR_READYQ(r2)");	/* Check ready queue empty */	\
asm("	beql	2b");			/* If yes, we are done. */	\
/* Check if active is still higher/equal priority than ready queue head */\
asm("	movl	_PR_ACTIVE(r2), r0");					\
asm("	movl	_PR_READYQ(r2), r1");					\
asm("	cmpw	_PRIORITY(r0), _PRIORITY(r1)");				\
asm("	blequ	2b");			/* If so, continue */		\
PrimaryDisable;								\
asm(" 	bbssi	$0,_KernelLock,2b");	/* Try for kernel lock */       \
asm("	movl	r2,_KLockHolder");					\
/*									\
 * Now test for the case where the active process has been destroyed,	\
/*									\
 * Now test for the case where the active process has been destroyed,	\
 * in which case we reschedule immediately. No need to restore 		\
 * registers and the the kernel stack pointer here. 			\
 */									\
asm("	cmpl	_LOCALPID(r0),$_INVALID_PROCESS");			\
asm("	bneq	0f");							\
asm("	moval 	_PR_IDLE_PROCESS(r2),r1");				\
asm("	cmpl	r0,r1");						\
asm("	beql	0f");							\
asm("	jmp	ActivateReadyqHead");					\
asm("0:");								\
asm("	pushl	r0" );		/* Arg to Addready */		\
asm("	calls	$1, _Addready");	/* Add active to readyq	*/	\
asm("	movq	(sp)+, r0");						\
asm("	movq	(sp)+, r2");						\
asm("	movq	(sp)+, r4");						\
asm("	jmp	Switch")
#else
  
#define	timer_intserv(_handler,_Asm_handler)				\
asm("	.text");							\
asm("	.align	2");							\
asm("	.globl	_Asm_handler");						\
asm("_Asm_handler:");							\
disable;				/* Disable interrupts */	\
asm("	movq	r4, -(sp)");		/* Save regs that C destroys */	\
asm("	movq	r2, -(sp)");						\
asm("	movq	r0, -(sp)");						\
									\
asm("	mfpr	$esp,r0");						\
asm("   bitl    $_PSL_CURMOD,28(sp)");   /* Were we in kernel mode ? */  \
asm("	bneq	3f");			/* No, we were in a process */	\
asm("   bitl    $_PSL_IPL,28(sp)");   /* Is saved IPL = 0? */         \
asm("	bneq	1f");			/* no, we were in the kernel */	\
									\
asm("	moval 	_PR_IDLE_PROCESS(r0),r1");				\
asm("	cmpl	_PR_ACTIVE(r0),r1");	/* active = Idle_process? */	\
asm("	beql	3f");			/* yes, try process switching */\
asm("1:");				/* Kernel interrupted */	\
asm("	incw	_KernelInterrupted");	/* Kernel interrupted */	\
asm("	pushl	_PR_ACTIVE(r0)");					\
asm("	calls	$1, _handler");						\
asm("	decw	_KernelInterrupted");					\
asm("2:");								\
asm("	movq	(sp)+, r0");		/* Restore registers */		\
asm("	movq	(sp)+, r2");						\
asm("	movq	(sp)+, r4");						\
asm("	rei");				/* Return */			\
asm("3:");				/* Kernel not interrupted */	\
asm("	incw	_ProcessInterrupted");	/* Process interrupted */	\
asm("	pushl	_PR_ACTIVE(r0)");					\
asm("	calls	$1, _handler");						\
asm("	decw	_ProcessInterrupted");					\
									\
asm("	mfpr	$esp,r2");						\
asm("	tstl	_PR_READYQ(r2)");	/* Check ready queue empty */	\
asm("	beql	2b");			/* If yes, we are done. */	\
/* Check if active is still higher/equal priority than ready queue head */\
asm("	movl	_PR_ACTIVE(r2), r0");					\
asm("	movl	_PR_READYQ(r2), r1");					\
asm("	cmpw	_PRIORITY(r0), _PRIORITY(r1)");				\
asm("	blequ	2b");			/* If so, continue */		\
/*									\
 * Now test for the case where the active process has been destroyed,	\
 * in which case we reschedule immediately. No need to restore 		\
 * registers and the the kernel stack pointer here. 			\
 */									\
asm("	cmpl	_LOCALPID(r0),$_INVALID_PROCESS");			\
asm("	bneq	0f");							\
asm("	moval 	_PR_IDLE_PROCESS(r2),r1");				\
asm("	cmpl	r0,r1");						\
asm("	beql	0f");							\
asm("	jmp	ActivateReadyqHead");					\
asm("0:");								\
asm("	pushl	r0" );		/* Arg to Addready */		\
asm("	calls	$1, _Addready");	/* Add active to readyq	*/	\
asm("	movq	(sp)+, r0");						\
asm("	movq	(sp)+, r2");						\
asm("	movq	(sp)+, r4");						\
asm("	jmp	Switch")
#endif FIREFLY

#endif INTERRUPT_H
