/*
 * Keyboard and mouse input routines for the DEC-SRC MDC (Monochrome Display
 *   Controller?).  There isn't a particularly good reason for keeping these
 *   in a separate file from the routines which control the rest of the MDC;
 *   it just seems like a fairly neat partition, and I like small files.
 *
 * $Revision: 1.6.1.3 $
 * $Locker:  $
 * $State: Exp $
 */

#include "process.h"
#include "cbuff.h"
#include "console.h"
#include "devman.h"
#include "mdc.h"
#include "timerfuncs.h"
#include "mouse.h"

/* Forward */
void MdcPollInput _TAKES(());
void MdcDoFunctionKey _TAKES(( int, int));

/*
 * Auto-repeat policy (arbitrary; if you don't like it, change it) :
 *   - auto-repeat applies to anything which actually generates characters,
 *     even RETURN, TAB, ESCAPE, RUBOUT and the function keys
 *   - auto-repeat is enabled whenever we generate a new character
 *   - auto-repeat is disabled when the key being repeated is released or
 *     some other character-generating key is depressed (in which case we'll
 *     then enable auto-repeat for the newly pressed key).  [So we don't
 *     support cyclic auto-repetition of a group of keys that are depressed].
 *   - a change in the state of SHIFT, CAPS LOCK or CTRL does not disable
 *     auto-repeat, but does alter appropriately the characters generated by
 *     auto-repetition
 *   - there two delay values: one for the first delay after enabling, and one
 *     for all subsequent repetitions for that key.
 */
static int	    autoRepeat_mask = 0;
static int	    autoRepeat_index;
static int	    autoRepeat_keynum;	/* Mask and index could be inferred  */
					/* from this, but it's faster not to */
static int	    autoRepeat_delay;

	int	    autoRepeat_init_delay = 50;	/* Clicks (i.e. lots of */
	int	    autoRepeat_next_delay = 5;	/*   ten milliseconds)  */

#define Index(keyPos)	((keyPos) >> 4)
#define Mask(keyPos)	(1 << ((keyPos) & 15))
#define	InvIndex(index)	((index) << 4)	/* map index -> first keyPos */
#define KEY_CAPS_LOCK	71
#define KEY_SHIFT_LEFT	41
#define	KEY_SHIFT_RIGHT	47
#define	KEY_CTRL_LEFT	33
#define KEY_CTRL_RIGHT	23
#define	MOUSE_BUTTON_LEFT	5
#define MOUSE_BUTTON_MIDDLE	6
#define	MOUSE_BUTTON_RIGHT	7
#define MOUSE_BUTTON_INDEX	0
#define MOUSE_BUTTON_MASK  0x00E0	/* Bits 5..7 */
	
/* Bits in key_table.flags: */
#define	CAPS_LOCK	1	/* If CAPS LOCK toggle is on, behave as if */
				/*   SHIFT was pressed			   */
#define	FUNCTION_KEY	2	/* Invoke the function-key magic	   */
#define	IGNORE_CTRL	4	/* Ignore the state of the CTRL key	   */
				/*   (currently unused?)		   */
#define	UNUSED		8

static struct key_table {
    unsigned char unshifted, shifted;
    unsigned short flags;
  }
keyTable[16*MDC_KEY_STATE_WORDS] = {
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{ ' ', ' ', 0},
{0,0,0}, /*   5: mouse - left */
{0,0,0}, /*   6: mouse - middle */
{0,0,0}, /*   7: mouse - right */
	{'\t','\t', 0},
	{ 'z', 'Z', CAPS_LOCK },
	{ 'c', 'C', CAPS_LOCK },
	{ 'b', 'B', CAPS_LOCK },
	{ '8', '*', 0 },
	{ '0', ')', 0 },
	{ '=', '+', 0 },
	{ '\\', '|', 0 },
	{   4,  10, FUNCTION_KEY }, /* F5 */
	{ 'q', 'Q', CAPS_LOCK },
	{ 'e', 'E', CAPS_LOCK },
	{ 't', 'T', CAPS_LOCK },
	{ 'u', 'U', CAPS_LOCK },
	{ 'o', 'O', CAPS_LOCK },
	{ '[', '{', 0 },
{0,0,0}, /*  23: CTRL - RIGHT */
	{   2,   8, FUNCTION_KEY }, /* F3 */
{0,0,0}, /*  25: OPTION */
	{ '3', '#', 0 },
	{ 'v', 'V', CAPS_LOCK },
	{ 'h', 'H', CAPS_LOCK },
	{ '9', '(', 0 },
	{ '-', '_', 0 },
	{ '`', '~', 0 },
	{   0,   6, FUNCTION_KEY }, /* F1 */
{0,0,0}, /*  33: CTRL */
	{ 'w', 'W', CAPS_LOCK },
	{ 'r', 'R', CAPS_LOCK },
	{ 'y', 'Y', CAPS_LOCK },
	{ 'i', 'I', CAPS_LOCK },
	{ 'p', 'P', CAPS_LOCK },
	{ ']', '}', 0 },
	{   1,   7, FUNCTION_KEY }, /* F2 */
{0,0,0}, /*  41: SHIFT - LEFT */
	{ 's', 'S', CAPS_LOCK },
	{ 'f', 'F', CAPS_LOCK },
	{ '7', '&', 0 },
	{ 'k', 'K', CAPS_LOCK },
	{ ';', ':', 0 },
{0,0,0}, /*  47: SHIFT - RIGHT */
	{   3,   9, FUNCTION_KEY }, /* F4 */
	{ '1', '!', 0 },
	{ 'x', 'X', CAPS_LOCK },
	{ '5', '%', 0 },
	{ 'n', 'N', CAPS_LOCK },
	{ ',', '<', 0 },
	{ '/', '?', 0 },
	{'\r','\r', 0 },
	{   5,  11, FUNCTION_KEY }, /* F6 */
	{ 'a', 'A', CAPS_LOCK },
	{ 'd', 'D', CAPS_LOCK },
	{ 'g', 'G', CAPS_LOCK },
	{ 'j', 'J', CAPS_LOCK },
	{ 'l', 'L', CAPS_LOCK },
	{ '\'', '\"', 0 },
	{ 127, 127, IGNORE_CTRL}, /* RUBOUT */
	{  27,  27, 0},		  /* ESC    */
	{ '2', '@', 0 },
	{ '4', '$', 0 },
	{ '6', '^', 0 },
	{ 'm', 'M', CAPS_LOCK },
	{ '.', '>', 0 },
{0,0,0}, /*  70: OPTION - RIGHT */
{0,0,0}, /*  71: CAPS LOCK */
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED },
	{   0,   0, UNUSED }
  };

static int keyboard_ready = 0;	/*    0 -> uninitialized		   */
				/* 1..3 -> just initialized; read it a few */
				/*	   times before starting to record */
				/*	   keystrokes			   */
				/*    4 -> normal operation		   */

static int	    caps_lock_on = 0;
static MDC_KeyState watch_down;	/* Keys whose down-transitions interest us */
static MDC_KeyState last_keyState;
static MDC_Point    last_mouseLoc;

static NextTimerFunc ntf;

void
MdcInputInit()
  {
    register struct key_table *p = keyTable;
    register int i,j;
    register unsigned bit, mask;

    /*
     * We could compile in a value for watch_down instead of calculating it
     *   now, but who cares about the slight startup overhead?
     */
    for (i = 0; i < MDC_KEY_STATE_WORDS; i++)
      {
	for (mask = 0, bit = 1, j = 0; j < 16; bit <<= 1, j++, p++)
	    if (p->shifted != 0
 /*------------ should be some magic function on *p */)
		mask |= bit;
	watch_down.word[i] = mask;
      }
    keyboard_ready = 1;
    Add_timer_func((int_func)MdcPollInput, &ntf);	/* Tell timer routines to call */
					/*   MdcPollInput every click  */
    MousePowerup();	/* Init machine-independent part of mouse handler */
  }

/*
 * Polls keyboard and mouse
 */
void
MdcPollInput()
  {
    MDC_KeyState now_keyState;
    MDC_Point    now_mouseLoc;
    register int i, knum;
    register unsigned new_bits;
    int new_keynum;

    do_next_timer_func(ntf);	/* See mi/timerfuncs.h */

    if (keyboard_ready < 4)
      {
	if (keyboard_ready > 0)
	  {
	    last_mouseLoc = ((MDC_HeadRecord *)commandMemory_virt_addr)->mouseLoc;
	    last_keyState = ((MDC_HeadRecord *)commandMemory_virt_addr)->keyState;
	    keyboard_ready++;
	  }
	return;
      }

    /*
     * The MicroVAX/Firefly version of the kernel defines the disable-
     *   interrupt macro ("disable") so that it does NOT disable interrupts
     *   from the MicroVAX interval timer.  Much of the kernel assumes that,
     *   if it invokes "disable", it won't run into trouble with interrupt
     *   routines playing with data structures (in particular, the ready-
     *   queue).
     * In order to report mouse or keyboard activity we have to monkey
     *   with the ready-queue and with data in the mouse and console files.
     * This routine is executed whenever we get an interrupt from the
     *   MicroVAX interval timer.
     * So... if the interrupt occurred when we were in any part of the kernel,
     *   we return and wait for a more favourable occasion.
     * If we're spending a lot of time in the kernel we may lose keystrokes
     *   or mouse squeaks.  In the extreme, this may mean that we can't kill
     *   a runaway process that's making us spend almost all our time in the
     *   kernel.  Hmmm.
     */
    if (KernelMutex())
	return;

    now_mouseLoc = ((MDC_HeadRecord *)commandMemory_virt_addr)->mouseLoc;
    now_keyState = ((MDC_HeadRecord *)commandMemory_virt_addr)->keyState;

    if (now_mouseLoc.x != last_mouseLoc.x ||
	now_mouseLoc.y != last_mouseLoc.y ||
	(( now_keyState.word[MOUSE_BUTTON_INDEX] ^
	  last_keyState.word[MOUSE_BUTTON_INDEX]) & MOUSE_BUTTON_MASK) != 0 )
      {
	MouseX += (/*signed*/ short)(now_mouseLoc.x - last_mouseLoc.x);
	MouseY += (/*signed*/ short)(now_mouseLoc.y - last_mouseLoc.y);
	MouseButtons =
	 ~now_keyState.word[MOUSE_BUTTON_INDEX]	>> MOUSE_BUTTON_LEFT  -2 & 4 |
	 ~now_keyState.word[MOUSE_BUTTON_INDEX] >> MOUSE_BUTTON_MIDDLE-1 & 2 |
	 ~now_keyState.word[MOUSE_BUTTON_INDEX] >> MOUSE_BUTTON_RIGHT    & 1;
		       
/*  ( ~now_keyState.word[MOUSE_BUTTON_INDEX] &
			 MOUSE_BUTTON_MASK ) >> MOUSE_BUTTON_FIRSTBIT;
*/
	    /* Note the "~", because the code in mi/mouse.c does another */
	    /*   inversion, presumably because some Sun mouse needed it  */
	MouseEvent = 1;
	CheckMouseReader();
      }
	
    /*
     * Do this bit of the auto-repeat logic up here to ensure that it gets
     *   executed every time.  Rest is done further down after we've had a
     *   proper look at the keyboard.
     */
    if (autoRepeat_mask)
	if ( (now_keyState.word[autoRepeat_index] & autoRepeat_mask) == 0 )
	    autoRepeat_mask = 0;
	else
	    autoRepeat_delay--;
    /*
     * Hack for CAPS LOCK.  If there were other keys that did interesting
     *   things, it'd be nice to be a bit more general
     */
    if ( now_keyState.word[Index(KEY_CAPS_LOCK)] &
	 ~last_keyState.word[Index(KEY_CAPS_LOCK)] & Mask(KEY_CAPS_LOCK) )
	caps_lock_on = ~caps_lock_on;

    new_keynum = -1;
    for (i = 0; i < MDC_KEY_STATE_WORDS; i++)
      {
	if ( (new_bits = now_keyState.word[i] &
			 ~last_keyState.word[i] & watch_down.word[i]) != 0 )
	    for (knum = InvIndex(i); new_bits != 0;  new_bits >>= 1, knum++)
		if (new_bits & 1)
		    if (new_keynum != -1)
			/* Two or more keys have gone down since the last */
			/*   time we looked; ignore them all (is there a  */
			/*   more reasonable decision one could make?)    */
			new_keynum = -2;
		    else
			new_keynum = knum;
      }
    if (new_keynum >= 0)
      {
	autoRepeat_delay = autoRepeat_init_delay;
	autoRepeat_index = Index(new_keynum);
	autoRepeat_mask  = Mask(new_keynum);
	autoRepeat_keynum= new_keynum;
      }
    else if (new_keynum == -2)
	autoRepeat_mask = 0;
    else /* new_keynum == -1, i.e. no new keys */
	 if (autoRepeat_mask && autoRepeat_delay <= 0)
      {
	autoRepeat_delay = autoRepeat_next_delay;
	new_keynum = autoRepeat_keynum;
      }
    if (new_keynum >= 0)
      {
	register struct key_table *p = &keyTable[new_keynum];
	int shift_on, ctrl_on;
	unsigned char c;

	shift_on =
	  now_keyState.word[Index(KEY_SHIFT_LEFT )] & Mask(KEY_SHIFT_LEFT ) ||
	  now_keyState.word[Index(KEY_SHIFT_RIGHT)] & Mask(KEY_SHIFT_RIGHT);
	ctrl_on  =
	  now_keyState.word[Index(KEY_CTRL_LEFT  )] & Mask(KEY_CTRL_LEFT  ) ||
	  now_keyState.word[Index(KEY_CTRL_RIGHT )] & Mask(KEY_CTRL_RIGHT );

	if (shift_on || caps_lock_on && p->flags & CAPS_LOCK)
	    c = p->shifted;
	else
	    c = p->unshifted;
	if (p->flags & FUNCTION_KEY)
	    MdcDoFunctionKey(c, ctrl_on);
	else
	  {
	    if (ctrl_on && ~(p->flags & IGNORE_CTRL))
		c &= 0x1f;
	    ConsoleGotChar(c);
	  }
      }
    last_mouseLoc = now_mouseLoc;
    last_keyState = now_keyState;
  }

char *MdcFunctionKeyTable[] = {
	"\33[11~",			/* f1 */
	"\33[12~",			/* f2 */
	"\33[13~",			/* f3 */
	"\33[14~",			/* f4 */
	"\33[15~",			/* f5 */
	"\33[17~",			/* f6 */
	"\33[18~",			/* shift-f1 */
	"\33[19~",			/* shift-f2 */
	"\33[20~",			/* shift-f3 */
	"\33[21~",			/* shift-f4 */
	"\33[22~",			/* shift-f5 */
	"\33[23~"			/* shift-f6 */
  };

/*
 * Called with:  n = 6*shift + (function_key_number-1)    (so 0 <= n <= 11)
 *		 ctrl_on = ("CRTL key is depressed")
 *
 * Since we can't think of anything useful to do with the function keys at
 * present, we'll just generate escape sequences that are sort-of vaguely 
 * compatible with what you'd get on a MicroVAX II with QVSS and Ultrix.
 *
 * The CTRL key could be used as an extra shift if you wanted yet more function
 * keys, but we'll ignore it.
 *
 * Could get fancy and have programmable function keys, but what for?
 */
void
MdcDoFunctionKey(n, ctrl_on)
  int n;
  int ctrl_on;
  {
    register char *s = MdcFunctionKeyTable[n];

    while (*s)
	ConsoleGotChar(*s++);
  }

/* StartMouse() {;} */

/* StopMouse() {;} */

/*
 * ------ need to implement the stuff in console.h if we're to support the
 *	  X server's thirst for raw keystroke information.  Keyclick can go
 *	  take a running jump.
 */
