/*
 * V Kernel - Copyright (c) 1985 by Stanford University
 *
 * Kernel Framebuffer support routines
 *
 * The "qvss" provides access to the framebuffer and to the framebuffer's
 * machine registers.  Opening this device causes the framebuffer to be
 * mapped into your address space, and reading from or writing to this
 * device causes a read/write to the machine register that controls it.
 *
 * $Revision: 1.14.1.5 $
 * $Locker:  $
 * $State: Exp $
 */

#include <Venviron.h>
#include <Viocommon.h>
#include <Vnamingprotocol.h>
#include <Vquerykernel.h>
#include "devman.h"
#include "externals.h"
#include "uvaxmemory.h"
#include "memory.h"
#include "interrupt.h"
#include "qvss.h"

/* Forward */
void QvssEnableVideo _TAKES(());
ResponseCode QvssCreate _TAKES(( Process *, DeviceInstance *));
ResponseCode QvssRead _TAKES(( Process *, DeviceInstance *));
ResponseCode QvssWrite _TAKES(( Process *, DeviceInstance *));
ResponseCode QvssQuery _TAKES(( Process *, DeviceInstance *, unsigned short));
ResponseCode QvssRelease _TAKES(( Process *, DeviceInstance *));

/* type of script with which to initialize the entire qvss */
typedef	struct {
    int		regnum;	/* register number to store value into */
    short	value;	/* value to be stored */
} QvssScript;

QvssScript  QvssInitScript[] = {
	    /* ignore interrupts and turn video on */
		{QV_CSR, QV_ENABLEVID},
	    /* initialize the CRT controller */
		{QV_CRTCA, 0}, {QV_CRTCD, 0x27},
		{QV_CRTCA, 1}, {QV_CRTCD, 0x20},
		{QV_CRTCA, 2}, {QV_CRTCD, 0x21},
		{QV_CRTCA, 3}, {QV_CRTCD, 0xb4},
		{QV_CRTCA, 4}, {QV_CRTCD, 0x38},
		{QV_CRTCA, 5}, {QV_CRTCD, 0x5},
		{QV_CRTCA, 6}, {QV_CRTCD, 0x36},
		{QV_CRTCA, 7}, {QV_CRTCD, 0x36},
		{QV_CRTCA, 8}, {QV_CRTCD, 0x4},
		{QV_CRTCA, 9}, {QV_CRTCD, 0xf},
		{QV_CRTCA, 10}, {QV_CRTCD, 0x20},
		{QV_CRTCA, 11}, {QV_CRTCD, 0},
		{QV_CRTCA, 12}, {QV_CRTCD, 0},
		{QV_CRTCA, 13}, {QV_CRTCD, 0},
		{QV_CRTCA, 14}, {QV_CRTCD, 0},
		{QV_CRTCA, 15}, {QV_CRTCD, 0},
	    /* Initialize the two UARTs */
		{QV_UCMD+UARTAOFFS, 0x15},
		{QV_UMODE+UARTAOFFS, 0x17}, /* doc uses 0x13, Ultrix 0x17 */
		{QV_UMODE+UARTAOFFS, 0x07},
		{QV_USCSR+UARTAOFFS, 0x99}, /* supposed to be 4800 rx, tx */
		{QV_UCMD+UARTBOFFS, 0x15},
		{QV_UMODE+UARTBOFFS, 0x17},
		{QV_UMODE+UARTBOFFS, 0x07},
		{QV_USCSR+UARTBOFFS, 0x99},
		{QV_UISM, 0x2},		/* rxints only */
	    /* Initialize the Interrupt controller */
	    /*
	     * The index into the SCB that the QVSS interrupts at is
	     * equal to SCBOffset = (0x80 | (ValueStored>>2)).
	     */
		{QV_ICSR, 0},		/* reset the controller */
		{QV_ICSR, 0x40},	/* Clear the IRR */
		{QV_ICSR, 0x80},	/* Set Interrupt, MultiVec, RotPrior */
		{QV_ICSR, 0xc0},	/* Select the ACR */
		{QV_ICDR, 0xff},	/* All are auto clear */
		{QV_ICSR, 0xe0},	/* Select UART int Vec */
		{QV_ICDR, 0xff&(VecQvssUart<<2)},
		{QV_ICSR, 0x28},	/* enable ints for that device */
		{QV_ICSR, 0xe1},	/* Select Vsync int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x29},	/* enable ints for that device */
		{QV_ICSR, 0xe4},	/* Select MouseA int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x2c},	/* enable ints for that device */
		{QV_ICSR, 0xe5},	/* Select MouseB int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x2d},	/* enable ints for that device */
		{QV_ICSR, 0xe6},	/* Select MouseC int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x2e},	/* enable ints for that device */
		
		{QV_ICSR, 0xb0},	/* Select IMR */
		{QV_ICDR, 0x0c},	/* mask irrelevant ints */
		{QV_ICSR, 0xa1},	/* Enable ints and ISR */
		{QV_CSR, QV_INTENABLE|QV_ENABLEVID}
	};

char *qvss_mem_v;

/*
 * FindQvss(): Determines whether there is a QVSS at the standard location.
 *	       Uses physical, not virtual, addressing.
 */
unsigned char 
FindQvss(enable_video_func)
    int (**enable_video_func)();
  {
    if (Probe((short *)QVSS_REGS_P, PROBE_READ))
      {
	printx("Found a QVSS framebuffer, mouse and keyboard\n");
	AddPeripheral(PRF_MOUSE_QVSS);
	AddPeripheral(PRF_FRAMEBUFFER_QVSS);
	AddPeripheral(PRF_SERIAL_QVSS);		/* ??? */
	*enable_video_func = (int (*) ())QvssEnableVideo;
	return PRF_FRAMEBUFFER_QVSS;
      }
    else
	return PRF_NONE;
  }

/*
 * BuildSysMap_Qvss() must be called at boot time when the system memory map is
 *		      being constructed.  It asks the memory manager to map the
 *		      256K of QVSS memory into the system page table and tell
 *		      us the starting virtual address we should use to refer
 *		      to it.
 */
void
BuildSysMap_Qvss()
  {
    qvss_mem_v = BuildSysMap_MapPhysPages(QVSS_MEM_P, QVSS_MEM_SIZE);
  }

/*
 * Do boot-time initialization of the QVSS and add it to the list of kernel
 *   device drivers.
 */
void
QvssPowerup()
  /*
   * This powers up the entire physical Qvss device -- which contains
   * four logical devices: the mouse, framebuffer, and two serial ports,
   * of which the first is the keyboard and the second is currently unused.
   *
   * The reason it is all done here is that interrupts need to be
   * coordinated among the four logical devices.  So even though there
   * is a qvssmouse.c and a qvssserial.c, their respective initializations
   * appear here.
   */
  {
    register int i;
    register qvss_reg_t *qvss = (qvss_reg_t *)QVSS_REGS_V;

    /* Map it so that user processes can read and write the framebuffer */
    /*   (but not the QVSS registers)					*/
    ChangeSysPagesProtection((unsigned)qvss_mem_v, QVSS_MEM_SIZE, VM_RWRW);

     /* plug the interrupt vectors */
    setexvec(Asm_IntQvssUart, VecQvssUart);
    setexvec(Asm_IntQvssMousePoll, VecQvssMousePoll);

    for (i = 0; i < sizeof(QvssInitScript)/sizeof(QvssScript); i++)
	qvss->regs[QvssInitScript[i].regnum] = QvssInitScript[i].value;
    
    QvssKeyboardInit();
    QvssMousePowerup();

    AddDevice("qvss", 0, QvssCreate, 0, 0);
  }



/* Device Routines */

ResponseCode
QvssCreate( pd, inst )  
    Process *pd;
    DeviceInstance *inst;
  {
  /*
   * Create an instance for the Qvss.
   */
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for id and first instance's owner.
     */


    inst->readfunc = QvssRead;
    inst->writefunc = QvssWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = QvssQuery;
    inst->releasefunc = QvssRelease;

    inst->id |= (UIO_READABLE+UIO_WRITEABLE|UIO_RANDOM_ACCESS);
    inst->blocksize = sizeof(short);
    inst->lastbytes = inst->blocksize;
    inst->lastblock = NUMUSERQVSSREGS - 1;

    return OK;

  }

ResponseCode
QvssRead( pd, inst )
    Process		*pd;
    DeviceInstance	*inst;

   /* Handle a read instance request for the console
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register unsigned short *buf = (unsigned short *)&(reply->shortbuffer[0]);

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber >= NUMUSERQVSSREGS ) return( END_OF_FILE );
    
    *buf = ((qvss_reg_t *)QVSS_REGS_V)->regs[req->blocknumber];
    
    return (OK);
    
  }

ResponseCode 
QvssWrite( pd, inst )
    register Process *pd;
    register DeviceInstance	*inst;

   /* Handle a write instance request to a serial line
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register unsigned short *buf = (unsigned short *)&(reply->shortbuffer[0]);

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber >= NUMUSERQVSSREGS ) return ( END_OF_FILE );

    ((qvss_reg_t *)QVSS_REGS_V)->regs[req->blocknumber] = *buf;
   
    return (OK);
    
  } /* QvssWrite */

ResponseCode
QvssQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryDeviceReply *reply = (QueryDeviceReply *) &(pd->msg);

    reply->addr = (unsigned)qvss_mem_v;
    reply->length = QVSS_MEM_SIZE;

    return( OK );
  }

ResponseCode 
QvssRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release VAX screen (console) instance
   */
  {
    inst->owner = 0;
    
    return( OK );
  }

/*
 * QvssEnableVideo: turn on "video enable" bit in the Qvss' CSR.  Called from
 *   Kabort() to ensure that error messages are seen even if software outside
 *   the kernel asked for video to be disabled (screen saver).  Just in case
 *   we get a Kabort while running without memory mapping, we check and do the
 *   right thing.  If memory mapping is enabled, we half-heartedly check
 *   whether we can access the QVSS.
 */
void
QvssEnableVideo()
  {
    register int r11;

    ;asm("	mfpr	$mapen,	r11");
    if ( (r11 & 1) == 0 )
	/* Physical addressing */
	((qvss_reg_t *)QVSS_REGS_P)->regs[QV_CSR] |= QV_ENABLEVID;
    else if (KernelCanRead((unsigned) &((qvss_reg_t *)QVSS_REGS_V)->regs[QV_CSR],
			    sizeof(short) ))
	/* Should really test for write access, but... */
	((qvss_reg_t *)QVSS_REGS_V)->regs[QV_CSR] |= QV_ENABLEVID;
    /* else do nothing */
  }
