/*
 * interp.c
 *
 * macrocode interpreter
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "funcall.h"
#include "utils.h"
#include "interp.h"

#include <stdio.h>

struct context context;

/* FIXME: May want to be u64? */
u32 ppss_mask(int ppss)
{
    return (1 << (ppss & 077)) - 1;
}

int ppss_position(int ppss)
{
    return (ppss >> 6) & 077;
}

u16 get_opcode(void)
{
    u32 tmp;

    tmp = memread(context.function + (context.location_counter >> 1));

    if (context.location_counter & 1) {
	return tmp >> 16;
    } else {
	return tmp;
    }
}

void do_branch(u16 opcode)
{
    if (opcode & 0x100) {
	context.location_counter += (0xffffff00 + (opcode & 0xff));
    } else {
	context.location_counter += (opcode & 0xff);
    }
}

int math_plus(lisp_q src1, lisp_q src2)
{
    lisp_q result;
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to + is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }
    
    /* get fixnum values and sign extend */
    src1 = ADDRESS(src1);
    src2 = ADDRESS(src2);
    if (src1 & 0x01000000) src1 |= 0xfe000000;
    if (src2 & 0x01000000) src2 |= 0xfe000000;
    
    result = src1 + src2;
    
    if ((result > 0x00ffffff) && (result < 0xff000000)) {
	printf("Result from + overflows FIXNUM.\n");
	return 0;
    }
    
    result = DTP_FIX | ADDRESS(result);
    push_cdrnext(result);
    
    return 1;
}

int math_minus(lisp_q src1, lisp_q src2)
{
    lisp_q result;
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to - is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }
    
    /* get fixnum values and sign extend */
    src1 = ADDRESS(src1);
    src2 = ADDRESS(src2);
    if (src1 & 0x01000000) src1 |= 0xfe000000;
    if (src2 & 0x01000000) src2 |= 0xfe000000;
    
    result = src1 - src2;
    
    if ((result > 0x00ffffff) && (result < 0xff000000)) {
	printf("Result from - overflows FIXNUM.\n");
	return 0;
    }
    
    result = DTP_FIX | ADDRESS(result);
    push_cdrnext(result);

    return 1;
}

int math_times(lisp_q src1, lisp_q src2)
{
    lisp_q result;

    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to * is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }

    /* get fixnum values and sign extend */
    src1 = ADDRESS(src1);
    src2 = ADDRESS(src2);
    if (src1 & 0x01000000) src1 |= 0xfe000000;
    if (src2 & 0x01000000) src2 |= 0xfe000000;

    /* FIXME: Better overflow checking? */
    result = src1 * src2;
	
    if ((result > 0x00ffffff) && (result < 0xff000000)) {
	printf("Result from * overflows FIXNUM.\n");
	return 0;
    }

    result = DTP_FIX | ADDRESS(result);
    push_cdrnext(result);

    return 1;
}

lisp_q load_target(u16 opcode)
{
    int offset;

    offset = opcode & 0x3f;

    switch((opcode >> 6) & 7) {
    case 0: /* FEF|xx */
	return memread_inviz(context.function + offset);
	
    case 1: /* FEF|64+xx */
	return memread_inviz(context.function + offset + 64);
	
    case 2: /* FEF|128+xx */
	return memread_inviz(context.function + offset + 128);
	
    case 3: /* higher lexical context */
	printf("unimplemented source: higher lexical context.\n");
	exit(-1);
	
    case 4: /* SELF mapping table */
	printf("unimplemented source: SELF mapping table.\n");
	exit(-1);
	
    case 5: /* local|xx */
	return memread(context.local_pointer + offset);
	
    case 6: /* arg|xx */
	return memread(context.arg_pointer + offset);
	
    case 7: /* PDL-POP */
	return pop();
    }

    exit(-1); /* I wish GCC would just plain shut up */
}

void store_target(u16 opcode, lisp_q value)
{
    int offset;

    offset = opcode & 0x3f;
    
    switch((opcode >> 6) & 7) {
    case 0: /* FEF|xx */
	memwrite_inviz(context.function + offset, value);
	break;
	
    case 1: /* FEF|64+xx */
	memwrite_inviz(context.function + offset + 64, value);
	break;
	
    case 2: /* FEF|128+xx */
	memwrite_inviz(context.function + offset + 128, value);
	break;
	
    case 3: /* higher lexical context */
	printf("unimplemented destination: higher lexical context.\n");
	exit(-1);
	
    case 4: /* SELF mapping table */
	printf("unimplemented destination: SELF mapping table.\n");
	exit(-1);
	
    case 5: /* local|xx */
	memwrite(context.local_pointer + offset, value);
	break;
	
    case 6: /* arg|xx */
	memwrite(context.arg_pointer + offset, value);
	break;
	
    case 7: /* PDL-PUSH */
	push_cdrnext(value);
    }
}

lisp_q locative_target(u16 opcode)
{
    int offset;
    lisp_q retval;

    offset = opcode & 0x3f;
    retval = 0;

    switch((opcode >> 6) & 7) {
    case 0: /* FEF|xx */
	retval = context.function + offset;
	break;
	
    case 1: /* FEF|64+xx */
	retval = context.function + offset + 64;
	break;
	
    case 2: /* FEF|128+xx */
	retval = context.function + offset + 128;
	break;
	
    case 3: /* higher lexical context */
	printf("unimplemented target: higher lexical context.\n");
	exit(-1);
	
    case 4: /* SELF mapping table */
	printf("unimplemented target: SELF mapping table.\n");
	exit(-1);
	
    case 5: /* local|xx */
	retval = context.local_pointer + offset;
	break;
	
    case 6: /* arg|xx */
	retval = context.arg_pointer + offset;
	break;
	
    case 7: /* PDL-POP */
	printf("meaningless target: PDL.\n");
	exit(-1);
    }

    retval = DTP_LOCATIVE | ADDRESS(retval);

    return retval;
}

void load_sg(lisp_q sg)
{
    context.current_sg = sg;
    
    context.pdl = memread(sg - SG_REGULAR_PDL);
    context.pdl_pointer = context.pdl + 3; /* FIXME: Wrong */

    context.spdl = memread(sg - SG_SPECIAL_PDL);
    context.spdl_pointer = memread(sg - SG_SPECIAL_PDL_POINTER);
}

int process_data(u8 *data, int length)
{
    lisp_q spia; /* Scratch Pad Init Area */

    memory_init((lisp_q *)data);

    spia = get_area_address(2);

    printf("scratch pad init area:\n");
    dump_q(memread(spia), spia);
    dump_q(memread(spia+1), spia+1);
    dump_q(memread(spia+2), spia+2);
    dump_q(memread(spia+3), spia+3);
    dump_q(memread(spia+4), spia+4);

#if 0
    printf("initial stack group:\n");
/*     dump_raw_array(spia[3]); */
    dump_q(memread(spia[3]-3), spia[3]-3);
    dump_q(memread(memread(spia[3]-3)), memread(spia[3]-3));
    dump_q(memread(memread(spia[3]-3)+1), memread(spia[3]-3)+1);
    dump_q(memread(memread(spia[3]-3)+2), memread(spia[3]-3)+2);
    dump_q(memread(memread(spia[3]-3)+3), memread(spia[3]-3)+3);
    dump_q(memread(memread(spia[3]-3)+4), memread(spia[3]-3)+4);
#endif

    /*
     * Load the static part of the initial stack group, set the stack
     * pointers to 0, and fake a function call to the initial function.
     * The call-info word for the function call ends up as the first
     * thing on the stack.
     *
     * The call-info word will be 0x4a001000. (no, we need to set D-MICRO)
     * Just as a quick hack, we will push 4 NILs after the call-info
     * word and fake up whatever is nessecary for the initial function.
     *
     * The call process seems to be a matter of:
     *
     *     find the call target
     *     fixup for outgoing function
     *     fixup arglist
     *     save outgoing state
     *     load incoming state
     *     fixup local variables
     */

    load_sg(memread(spia + SPIA_INITIAL_SG));

    funcall(memread(memread(spia + SPIA_INITIAL_FUNCTION)), DTP_FIX | 0x001000); /* FIXME: wrong dest */

    while(step());
    
#if 0
    printf("initial function:\n");
    dump_q(memread(spia[0]), spia[0]);
    dump_raw_fef(memread(spia[0]));
#endif

/*     dump_q(memread(01000 + 22), 0); */
    return 0;
}

/* EOF */
