/*
 * memory.c
 *
 * memory load and access functions
 */

#include "memory.h"

#include "exploiter.h"
#include "context.h"
#include "funcall.h"
#include "utils.h"

#include <stdio.h>

lisp_q *image;
lisp_q *sca;
lisp_q *area_origin_area;
lisp_q *dpmt;

lisp_q *region_origin_area;
lisp_q *region_length_area;
lisp_q *region_free_pointer_area;
lisp_q region_gc_pointer_area_address;
lisp_q *region_bits_area;
lisp_q region_list_thread_area_address;

lisp_q *area_name_area;
lisp_q *area_region_list_area;
lisp_q *area_region_bits_area;
lisp_q *area_region_size_area;
lisp_q *area_maximum_size_area;

lisp_q m_self;
lisp_q m_self_mapping_table;

lisp_q a_default_cons_area;
lisp_q a_number_cons_area;

int num_wired_words;

#define CLUSTER_NUMBER(q) (((q) >> 13) & 0xfff)
#define PAGE_NUMBER(q) (((q) >> 9) & 0xf)

#if 0 /* VMM emulation not complete yet */

/* VMM status has 128K 2-bit entries. That's 256 bits. That's 32k bytes. */

u8 vmm_status[0x8000];

#define VMS_VALID 2
#define VMS_BANK  1

#define VMSB_LEFT  0
#define VMSB_RIGHT 1

int get_vmm_status(int index)
{
    u8 vmm_status_byte;

    vmm_status_byte = vmm_status[index >> 2];
    return (vmm_status_byte >> ((index & 3) << 1)) & 3;
}

void set_vmm_status(int index, int value)
{
    u8 vmm_status_byte;

    vmm_status_byte = vmm_status[index >> 2];

    vmm_status_byte &= ~(3 << ((index & 3) << 1));
    vmm_status_byte |= value << ((index & 3) << 1);

    vmm_status[index >> 2] = vmm_status_byte;
}

/* the VMM is two 16k entry 32bit maps */
u32 vmm_left_bank[0x4000];
u32 vmm_right_bank[0x4000];
u32 const *vmm_banks[2] = { vmm_left_bank, vmm_right_bank };

#define VMME_ACCESS 0x80000000

u32 mapped_read(lisp_q address)
{
    int vmm_status;
    u32 vmm_entry;
    u32 physaddr;
    
    address = ADDRESS(address);

    vmm_status = get_vmm_status((address >> 8) & 0x1ffff);

    if (!(vmm_status & VMS_VALID)) {
	/* FIXME: map miss page exception */
    }

    vmm_entry = vmm_banks[vmm_status & VMS_BANK][(address >> 8) & 0x3fff];

    if (!(vmm_entry & VMME_ACCESS)) {
	/* FIXME: page exception */
    }

    physaddr = vmm_entry << 10;
    physaddr |= (address & 0xff) << 2;

    return phys_read_u32(physaddr);
}
#endif /* VMM emulation */

u32 physical_memory_map[8] = {
    /*
     * We pretend that the first memory board found was in slot 4 and that
     * it has been set up with 1 Quantum (2 megs) of memory. All other boards
     * in the map are empty.
     */
    0xf4000001, -1, -1, -1, -1, -1, -1, -1
};

lisp_q read_amem(int address)
{
    if (address == M_SELF) {
	return m_self;
    } else if (address == M_SELF_MAPPING_TABLE) {
	return m_self_mapping_table;
    } else if (address == A_DEFAULT_CONS_AREA) {
	return a_default_cons_area;
    } else if (address == A_NUMBER_CONS_AREA) {
	return a_number_cons_area;
    } else if (address == 0x52) { /* %COUNTER-BLOCK-A-MEM-ADDRESS */
	printf("AMEM: read %%COUNTER-BLOCK-A-MEM-ADDRESS.\n");
	return DTP_FIX | 0x100; /* FIXME: where is it really? */
    } else if (address == 0x56) { /* inhibit scheduling flag */
	printf("AMEM: read inhibit scheduling flag.\n");
	return 0;
    } else if (address == 0x65) { /* alphabetic case affects string comparison */
	printf("AMEM: read alphabetic case affects string comparison.\n");
	return 0;
    } else if (address == 0x8a) { /* microcode type */
	printf("AMEM: read microcode type.\n");
	return 0;
    } else if (address == 0x8b) { /* processor type */
	printf("AMEM: read processor type.\n");
	return 0;
    } else if ((address >= 0x198) && (address < 0x200)) {
	printf("AMEM: read PMM[%d].\n", address & 7);
	return physical_memory_map[address & 7];
    } else {
	printf("AMEM: 0x%03x.\n", address);
    	return 0;
    }
}

void write_amem(int address, lisp_q data)
{
    if (address == M_SELF) {
	m_self = data;
    } else if (address == M_SELF_MAPPING_TABLE) {
	m_self_mapping_table = data;
    } else if (address == A_DEFAULT_CONS_AREA) {
	a_default_cons_area = data;
	printf("AMEM: default CONS area set: 0x%08lx.\n", data);
    } else if (address == A_NUMBER_CONS_AREA) {
	a_number_cons_area = data;
	printf("AMEM: number CONS area set: 0x%08lx.\n", data);
    } else {
	printf("AMEM: write 0x%03x = 0x%08lx.\n", address, data);
    }
}

#define DPMT_STATUS_A(foo) (((foo)[0] >> 29) & 7)
#define DPMT_STATUS_B(foo) (((foo)[0] >> 21) & 7)

#define DPMT_DEVICE_A(foo) (((foo)[0] >> 24) & 31)
#define DPMT_DEVICE_B(foo) (((foo)[0] >> 16) & 31)

#define DPMT_OFFSET_A(foo) (((foo)[1] >> 16) & 0xffff)
#define DPMT_OFFSET_B(foo) ( (foo)[1]        & 0xffff)

lisp_q *find_memory_location(lisp_q address)
{
    lisp_q *dpmt_entry;
    int is_device_b;
    int device_number;
    int device_status;
    int device_offset;
    
    dpmt_entry = &dpmt[CLUSTER_NUMBER(address) << 1];
/*     printf("DPMT: %08lx %08lx.\n", dpmt_entry[0], dpmt_entry[1]); */

    /* FIXME: make sure memory is mapped */
    is_device_b = dpmt_entry[0] & (1 << PAGE_NUMBER(address));

    if (is_device_b) {
	device_status = DPMT_STATUS_B(dpmt_entry);
	device_number = DPMT_DEVICE_B(dpmt_entry);
	device_offset = DPMT_OFFSET_B(dpmt_entry);
    } else {
	device_status = DPMT_STATUS_A(dpmt_entry);
	device_number = DPMT_DEVICE_A(dpmt_entry);
	device_offset = DPMT_OFFSET_A(dpmt_entry);
    }

    /* FIXME: the code below is semantically unfaithful to the real system */
    
    if (device_status != 1) {
	printf("find_memory_location: DPMT device status %d, expected 1.\n", device_status);
	exit(-1);
    }

    if (device_number != 0) {
	printf("find_memory_location: DPMT device number %d, expected 0.\n", device_number);
	exit(-1);
    }
    
    return &(image[0x2000 * device_offset + (address & 0x1fff)]);
}

lisp_q memread_unboxed(lisp_q address)
{
    address &= Q_BITS_ADDRESS;

    if (address < num_wired_words) {
	return image[address];
    } else if (address >= VM_M_BASE) {
	return read_amem(address & 0x3ff);
    } else if (address >= 0x01fdfc00) {
	printf("IO: 0x%05lx.\n", (address + 0x400) & 0x1ffff);
	return 0;
    }

    return *find_memory_location(address);
}

lisp_q memread(lisp_q address)
{
    /* FIXME: Read barrier goes here */
    return memread_unboxed(address);
}

void memwrite_unboxed(lisp_q address, lisp_q data)
{
    address &= Q_BITS_ADDRESS;

    if (address < num_wired_words) {
	image[address] = data;
	return;
    } else if (address >= VM_M_BASE) {
	write_amem(address & 0x3ff, data);
	return;
    } else if (address >= 0x01fdfc00) {
	printf("IO: 0x%05lx = 0x%08lx.\n", (address + 0x400) & 0x1ffff, data);
	return;
    }

    *find_memory_location(address) = data;
}

void memwrite(lisp_q address, lisp_q data)
{
    /* FIXME: Write barrier goes here */
    memwrite_unboxed(address, data);
}

lisp_q inviz(lisp_q address, lisp_q *final_address)
{
    /*
     * Oh, for multiple-value-bind...
     */
    lisp_q foo;

    foo = memread(address);
    
    if (DTP(foo) == DTP_EVCP) {
	printf("INVIZ: EVCP 0x%08lx.\n", foo);
    } else if (DTP(foo) == DTP_ONE_Q_FORWARD) {
	printf("INVIZ: 1QF 0x%08lx.\n", foo);
    } else {
	if (final_address) *final_address = address;
	return foo;
    }

    return inviz(foo, final_address);
}

lisp_q memread_inviz(lisp_q address)
{
    return inviz(address, NULL);
}

void memwrite_inviz(lisp_q address, lisp_q data)
{
    lisp_q tmp;
    lisp_q tmpaddr;

    tmp = inviz(address, &tmpaddr);
    memwrite(tmpaddr, CDRCODE(tmp) | NOT_CDRCODE(data));
}

lisp_q return_barrier(lisp_q data)
{
    /* FIXME: Implement */
    return data;
}

void push(lisp_q foo)
{
    memwrite(context.pdl_pointer++, foo);
}

void push_cdrnext(lisp_q foo)
{
    push(foo | CDR_NEXT);
}

lisp_q pop(void)
{
    return memread(--context.pdl_pointer);
}

lisp_q pop_loc(void)
{
    return --context.pdl_pointer;
}

lisp_q car(lisp_q cons)
{
    lisp_q foo;
    lisp_q fooaddr;

    foo = inviz(cons, &fooaddr);
    
    if (NOT_CDRCODE(fooaddr) == C_NIL) return C_NIL;
    
    return foo;
}

lisp_q cdr(lisp_q cons)
{
    lisp_q foo;

    /*
     * Okay, we're passed a DTP_LIST, a DTP_STACK_LIST, or a DTP_SYMBOL.
     * If we're a symbol, it had best be NIL, for which we are NIL. Next,
     * we have to find the CDR of the list, which means checking a CDR
     * code. Is this the CDR code on the forwarding pointer? It is, isn't
     * it? We then have to clamp out any forwarding pointers on it.
     */

    if (NOT_CDRCODE(cons) == C_NIL) return C_NIL;

    foo = memread(cons);
    
    switch (CDRCODE(foo)) {
    case CDR_NIL:
	return C_NIL;

    case CDR_NEXT:
	return cons + 1;

    case CDR_NORMAL:
	return memread_inviz(cons + 1);

    case CDR_ERROR:
	printf("CDR of CDR-ERROR taken.\n");
	exit(-1);

    default:
	exit(-1); /* GCC can't see that this can never be reached. :-( */
    }
}

void spdl_push(lisp_q data)
{
    memwrite(context.spdl + context.spdl_pointer++, data);
}

lisp_q spdl_pop(void)
{
    return memread(context.spdl + --context.spdl_pointer);
}

void bind(lisp_q where, lisp_q data)
{
    if (DTP(where) != DTP_LOCATIVE) {
	printf("bind(): where not DTP_LOCATIVE.\n");
	exit(-1);
    }

    where = NOT_CDRCODE(where);
    if (!(context.call_info & CI_BINDINGBLOCK)) {
	where |= SPECPDL_BLOCK_START_FLAG;
	context.call_info |= CI_BINDINGBLOCK;
    }
    
    spdl_push(memread(where));
    spdl_push(where);

    memwrite(where, data);
}

void unbind_1(void)
{
    lisp_q where;
    lisp_q data;

    /* FIXME: What about interactions with catch and unwind-protect? */
    
    where = spdl_pop();
    data = spdl_pop();

    if (where & SPECPDL_BLOCK_START_FLAG) {
	context.call_info &= ~CI_BINDINGBLOCK;
    }

    memwrite(where, data);
}

void unbind_block(void)
{
    while (context.call_info & CI_BINDINGBLOCK) {
	unbind_1();
    }
}

void unbind_n(int n)
{
    while(n--) {
	unbind_1();
    }
}

lisp_q get_area_address(int area)
{
    return area_origin_area[area];
}

void dump_region_bits(lisp_q bits)
{
    printf("%08lx: Status: %02ld Rep: %ld O: %ld GEN: %ld U: %ld Type: %02ld S: %ld V: %ld VO: %ld C: %ld R: %ld Swap: %ld.\n", bits,
           (bits >> 20) & 0x1f, (bits >> 18) & 3,
           (bits >> 17) & 1, (bits >> 14) & 7,
           (bits >> 13) & 1, (bits >> 9) & 15,
           (bits >> 8) & 1, (bits >> 7) & 1,
           (bits >> 5) & 3, (bits >> 4) & 1,
           (bits >> 3) & 1, bits & 7);
}

void dump_region_info(int region_num)
{
    dump_q(region_origin_area[region_num], 0);
    dump_q(region_length_area[region_num], 0);
    dump_q(region_free_pointer_area[region_num], 0);
    dump_q(region_bits_area[region_num], 0);
    dump_q(memread(region_gc_pointer_area_address + region_num), 0);
    dump_q(memread(region_list_thread_area_address + region_num), 0);

    dump_region_bits(region_bits_area[region_num]);
}

#define REGIONBITS_REP(x) (((x) >> 18) & 3)
#define RB_O         0x00020000
#define RB_TYPE      0x00001e00
#define RT_NEWSPACE  0x00000400
#define RT_COPYSPACE 0x00001800

int find_newspace_region(int area_num, int nqs, int representation)
{
    /*
     * returns the region number of the first newspace region in
     * area area_num with representation type representation and
     * nqs Qs of free storage available or 0 if no such region is
     * available (allocating in the RESIDENT-SYMBOL-AREA will
     * cause this to choke).
     */

    lisp_q current_region;
    lisp_q next_region;
    lisp_q region_bits;

    next_region = ADDRESS(area_region_list_area[area_num]);

    /* Loop while we have a valid region (negative region is end of chain) */
    while (!(next_region & 0x10000000)) {
	current_region = next_region;
	
	region_bits = region_bits_area[current_region];
	next_region = memread(region_list_thread_area_address + current_region);
    
	dump_region_info(current_region);

	if (REGIONBITS_REP(region_bits) != representation) continue;
	if (!(region_bits & RB_O)) continue; /* No oldspace */
	/* FIXME: Generation check */

	/* FIXME: Check to see if this is right */
	if (((region_bits & RB_TYPE) != RT_NEWSPACE) &&
	    ((region_bits & RB_TYPE) != RT_COPYSPACE)) continue;

	/* FIXME: Do we need to check volatility? */

	if ((ADDRESS(region_length_area[current_region]) -
	     ADDRESS(region_free_pointer_area[current_region])) < nqs) continue;

	return current_region;
    }

    return 0;
}

lisp_q allocate_structure_qs(lisp_q area, int nqs)
{
    int area_num;
    int region_num;
    lisp_q retval;
    
    printf("allocate_structure_qs: unimplemented (%d qs).\n", nqs);

    printf("area: ");
    if (DTP(area) == DTP_SYMBOL) {
	dump_string(memread(area + SYM_PRINTNAME));
    } else if (DTP(area) == DTP_FIX) {
	printf("%ld\n", ADDRESS(area));
    } else {
	printf("unrecognized DTP.\n");
	dump_q(area, 0);
    }

    if (NOT_CDRCODE(area) == C_NIL) {
	printf("allocate_structure_qs: AREA is NIL, assuming DEFAULT-CONS-AREA.\n");
	area = a_default_cons_area;
    }

    if ((DTP(area) == DTP_FIX) && (ADDRESS(area) < 0x100)) {
	printf("allocate_structure_qs: using area %ld.\n", ADDRESS(area));
	area_num = ADDRESS(area);
    } else {
	printf("allocate_structure_qs: Unknown area code.\n");
	exit(-1);
    }

    dump_q(area_name_area[area_num], area_num);
    dump_string(memread(area_name_area[area_num] + SYM_PRINTNAME));

    dump_q(area_region_list_area[area_num], area_num);
    
    region_num = find_newspace_region(area_num, nqs, 1);

    if (!region_num) {
	printf("allocate_structure_qs: Suitable region not found.\n");
	exit(-1);
    }

    printf("allocate_structure_qs: allocating in region %d.\n", region_num);
    
    retval = region_origin_area[region_num] +
	region_free_pointer_area[region_num];
    
    region_free_pointer_area[region_num] += nqs;

    printf("allocate_structure_qs: returning 0x%08lx.\n", retval);
    
    return retval;
}

void memory_init(lisp_q *data)
{
    image = data;
    sca = image + 01000;
    area_origin_area = image + ADDRESS(sca[0]);
    dpmt = image + ADDRESS(area_origin_area[9]);
    num_wired_words = ADDRESS(sca[13]);

    region_origin_area       = image + ADDRESS(area_origin_area[4]);
    region_length_area       = image + ADDRESS(area_origin_area[5]);
    region_free_pointer_area = image + ADDRESS(area_origin_area[7]);
    region_gc_pointer_area_address   = ADDRESS(area_origin_area[11]);
    region_bits_area         = image + ADDRESS(area_origin_area[6]);
    region_list_thread_area_address  = ADDRESS(area_origin_area[12]);

    /*
     * NOTE: some of the region tables are in wired memory, so we can
     * reference them directly. The area tables are all half a page in
     * length, and almost certainly aligned to the half-page, so we can
     * likewise reference them directly. The -other- region tables,
     * however, needs must be accessed the hard way.
     *
     * There should probably be some checks here to make sure that none
     * of the area tables cross a page boundary. Or at least a cluster
     * boundary. We can check this with a low-bits test on the address
     * (see if the botton N bits are less than or equal to 2^(N-1)).
     */

    area_name_area         = find_memory_location(area_origin_area[14]);
    area_region_list_area  = find_memory_location(area_origin_area[15]);
    area_region_bits_area  = find_memory_location(area_origin_area[16]);
    area_region_size_area  = find_memory_location(area_origin_area[17]);
    area_maximum_size_area = find_memory_location(area_origin_area[18]);

    sca[12] = DTP_FIX | 0x80000; /* pretend we have 2 megs of memory */
    sca[28] = DTP_FIX | 0xf1; /* pretend processor is in NuBus slot 1 */
    sca[32] = DTP_FIX | 0x1fffd98; /* pretend the PMM is at AMEM:0x298 */
}

/* EOF */
