/*
 * amem.c
 *
 * A-Memory and M-Memory interface functions and variables
 */

#include "amem.h"
#include <stdio.h>
#include <stdlib.h>

lisp_q m_self;
lisp_q m_self_mapping_table;

lisp_q a_default_cons_area;
lisp_q a_number_cons_area;
lisp_q a_background_cons_area;

lisp_q a_disk_run_light;
lisp_q a_disk_switches;
lisp_q a_inhibit_scheduling;
u32 a_slots_i_own;
u32 a_physical_resource_bitmap;
u32 a_io_space_virtual_address;

lisp_q a_pht_search_depth;
lisp_q a_pht_addr;
lisp_q a_pht_index_size;
lisp_q a_pht_index_limit;
lisp_q a_ppd_addr;
lisp_q a_ppd_end;

u32 a_boot_command_block;
u32 a_boot_lod_device;
u32 a_boot_memory;
u32 a_boot_monitor;
u32 a_boot_keyboard;
u32 a_boot_device;
u32 a_boot_mcr_name;
u32 a_boot_lod_name;

u32 physical_memory_map[8] = {
    /*
     * We pretend that the first memory board found was in slot 4 and that
     * it has been set up with 1 Quantum (2 megs) of memory. All other boards
     * in the map are empty.
     */
    0xf4000001, -1, -1, -1, -1, -1, -1, -1
};

lisp_q read_amem(int address)
{
    if (address == M_SELF) {
	return m_self;
    } else if (address == M_SELF_MAPPING_TABLE) {
	return m_self_mapping_table;
    } else if (address == A_DEFAULT_CONS_AREA) {
	return a_default_cons_area;
    } else if (address == A_NUMBER_CONS_AREA) {
	return a_number_cons_area;
    } else if (address == A_COUNTER_BLOCK_A_MEM_ADDRESS) {
	printf("AMEM: read %%COUNTER-BLOCK-A-MEM-ADDRESS.\n");
	return DTP_FIX | A_COUNTER_BLOCK_BASE;
    } else if (address == A_INHIBIT_SCHEDULING_FLAG) {
	printf("AMEM: read inhibit scheduling flag.\n");
	return a_inhibit_scheduling;
    } else if (address == A_ALPHABETIC_CASE_AFFECTS_STRING_COMPARISON) {
	printf("AMEM: read alphabetic case affects string comparison.\n");
	return 0;
    } else if (address == A_DISK_SWITCHES) {
	printf("AMEM: read disk switches.\n");
	return a_disk_switches;
    } else if (address == A_INHIBIT_READ_ONLY) {
	printf("AMEM: read inhibit read only.\n");
	return 0;
    } else if (address == A_DISK_RUN_LIGHT) {
	return a_disk_run_light;
    } else if (address == A_BACKGROUND_CONS_AREA) {
	printf("AMEM: read A-BACKGROUND-CONS-AREA.\n");
	return a_background_cons_area;
    } else if (address == A_MICROCODE_TYPE) {
	printf("AMEM: read microcode type.\n");
	return 0;
    } else if (address == A_PROCESSOR_TYPE) {
	printf("AMEM: read processor type.\n");
	return 1; /* FIXME: Should this be a #defined constant? */
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHT_SEARCH_DEPTH)) {
	printf("AMEM: read PHT search depth.\n");
	return a_pht_search_depth;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PAGE_HASH_TABLE_ADDRESS)) {
	printf("AMEM: read PHT address.\n");
	return a_pht_addr;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHT_INDEX_SIZE)) {
	printf("AMEM: read PHT index size.\n");
	return a_pht_index_size;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHT_INDEX_LIMIT)) {
	printf("AMEM: read PHT index limit.\n");
	return a_pht_index_limit;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHYSICAL_PAGE_DATA_ADDRESS)) {
	printf("AMEM: read PPD address.\n");
	return a_ppd_addr;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHYSICAL_PAGE_DATA_END)) {
	printf("AMEM: read PPD end.\n");
	return a_ppd_end;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_SLOTS_I_OWN)) {
	printf("AMEM: read SLOTS-I-OWN.\n");
	return a_slots_i_own;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_IO_SPACE_VIRTUAL_ADDRESS)) {
	printf("AMEM: read IO-SPACE-VIRTUAL-ADDRESS.\n");
	return a_io_space_virtual_address;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_CNFG_PARTITION_SLOT_UNIT)) {
	printf("AMEM: read CNFG-PARTITION-SLOT-UNIT.\n");
	return 0;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_CNFG_PARTITION_NAME)) {
	printf("AMEM: read CNFG-PARTITION-NAME.\n");
	return 0x31474643; /* 'CFG1' */
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHYSICAL_RESOURCE_BITMAP)) {
	printf("AMEM: read PHYSICAL-RESOURCE-BITMAP.\n");
	return a_physical_resource_bitmap;
    } else if ((address >= A_PMM_BASE) && (address < (A_PMM_BASE + A_PMM_SIZE))) {
	printf("AMEM: read PMM[%d].\n", address & 7);
	return physical_memory_map[address & 7];
    } else if (address == A_BOOT_COMMAND_BLOCK) {
	printf("AMEM: read A-BOOT-COMMAND-BLOCK.\n");
	return a_boot_command_block;
    } else if (address == A_BOOT_KEYBOARD) {
	printf("AMEM: read A-BOOT-KEYBOARD.\n");
	return a_boot_keyboard;
    } else if (address == A_BOOT_DEVICE) {
	printf("AMEM: read A-BOOT-DEVICE.\n");
	return a_boot_device;
    } else {
	printf("AMEM: 0x%03x.\n", address);
	exit(-1);
    	return 0;
    }
}

void write_amem(int address, lisp_q data)
{
    if (address == M_SELF) {
	m_self = data;
    } else if (address == M_SELF_MAPPING_TABLE) {
	m_self_mapping_table = data;
    } else if (address == A_DEFAULT_CONS_AREA) {
	a_default_cons_area = data;
	printf("AMEM: default CONS area set: 0x%08lx.\n", data);
    } else if (address == A_NUMBER_CONS_AREA) {
	a_number_cons_area = data;
	printf("AMEM: number CONS area set: 0x%08lx.\n", data);
    } else if (address == A_INHIBIT_SCHEDULING_FLAG) {
	a_inhibit_scheduling = data;
	printf("AMEM: inhibit scheduling flag set: 0x%08lx.\n", data);
    } else if (address == A_ALPHABETIC_CASE_AFFECTS_STRING_COMPARISON) {
	printf("AMEM: string comparison case sensitivity set ignored: 0x%08lx.\n", data);
    } else if (address == A_DISK_SWITCHES) {
	a_disk_switches = data;
	printf("AMEM: disk switch set: 0x%08lx.\n", data);
    } else if (address == A_INHIBIT_READ_ONLY) {
	printf("AMEM: inhibit read only set ignored: 0x%08lx.\n", data);
    } else if (address == A_DISK_RUN_LIGHT) {
	a_disk_run_light = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHT_SEARCH_DEPTH)) {
	printf("AMEM: write PHT search depth = 0x%08lx.\n", data);
	a_pht_search_depth = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PAGE_HASH_TABLE_ADDRESS)) {
	printf("AMEM: write PHT address = 0x%08lx.\n", data);
	a_pht_addr = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHT_INDEX_SIZE)) {
	printf("AMEM: write PHT index size = 0x%08lx.\n", data);
	a_pht_index_size = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHT_INDEX_LIMIT)) {
	printf("AMEM: write PHT index limit = 0x%08lx.\n", data);
	a_pht_index_limit = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHYSICAL_PAGE_DATA_ADDRESS)) {
	printf("AMEM: write PPD address = 0x%08lx.\n", data);
	a_ppd_addr = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_PHYSICAL_PAGE_DATA_END)) {
	printf("AMEM: write PPD end = 0x%08lx.\n", data);
	a_ppd_end = data;
    } else if (address == (A_COUNTER_BLOCK_BASE + ACB_IO_SPACE_VIRTUAL_ADDRESS)) {
	printf("AMEM: write IO-SPACE-VIRTUAL-ADDRESS = 0x%08lx.\n", data);
	a_io_space_virtual_address = data;
    } else if ((address >= A_PMM_BASE) && (address < (A_PMM_BASE + A_PMM_SIZE))) {
	printf("AMEM: write PMM[%d] = 0x%08lx.\n", address & 7, data);
	physical_memory_map[address & 7] = data;
    } else {
	printf("AMEM: write 0x%03x = 0x%08lx.\n", address, data);
	exit(-1);
    }
}

/* EOF */
