;;; -*- cold-load:t; Mode:Common-Lisp; Package:SI; Base:8. -*-

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (c)(1)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1985-1989 Texas Instruments Incorporated.  All rights reserved.

;;; This file contains SIB and low-level window system data structure initializations. 

;;;
;;; Edit History
;;;
;;;                   Patch
;;;   Date    Author  Number   Description
;;;---------------------------------------------------------------------------------
;;; 02-06-86    ab     --      Derived this file from WINDOW;COLD #204.
;;;                            Eliminated #-explorer code.
;;;                            Converted to Common Lisp.
;;;                            Removed cold to support old-style processor ROMS
;;;                              from Setup-Slot-Variables
;;; 02-13-86    ab             Changed Startup-Keyboard and Startup-TV-Interrupt
;;;                              to test for Cold-Booting in the Cold case
;;;                              (instead of just their variables being unbound)
;;;                              so that *TV-Frame-Interrupt-Descriptor* and
;;;                              *Keyboard* don't have to be unbound in the saved band.
;;;                              This change helps the Lisp version of Disk-Save.
;;; 03-24-86    sdk	       Turned all the :element-type (:unsigned-byte 16.) specifiers to
;;;			       make-array into :type 'art-16b because subtypes don't work this
;;;			       early in the cold band.
;;; 11-11-86    ab             Moved a couple of vars here from LISP-REINITIALIZE.
;;;  2/18/87    GRH            Change kbd-initialize-warm and RESET-SIB to use new 
;;;			       WAIT-REAL-MILLISECONDS function for Exp2 compatibility.
;;;  6-25-87    AB for GRH     Add support for CSIB. - change some event vector addresses and
;;;			       don't do a lot of initialization for the CSIB which does it for us.
;;; 12/20/87    RJF for M.Y.   Fixed a problem with enable-sib-interrupts not working properly
;;;                            with Meta-crtl-meta-ctrl-rubout.
;;; 11/03/88    clm            Made two separate versions of WAIT-REAL-MILLISECONDS, one for
;;;                            the Explorers and one for the MX.  The Explorer version accesses
;;;                            the SIB, and the Mac version uses the microsecond clock.
;;; 01/27/89    KJF   Sys 4-85 [may] Change to setup-sib-slots to set tv:*csib-slot*.
;;; 02/27/89    JLM            Added MP support.
;;; 3/27/89     JLM		Added test for MP-Initialize-OK 


;;
;; General Purpose
;;

(Defmacro Byte-Mask (field)
  `(DPB -1 ,field 0))

(DefMacro aref-32b (array index)
  `(dpb (aref ,array (1+ (* 2 ,index)))
	#o2020
	(aref ,array (* 2 ,index))))

(DefMacro Aset-32b (value array index)
  `(progn 'compile
	  (setf (aref ,array (1+ (* 2 ,index))) (ldb #o2020 ,value))
	  (setf (aref ,array (* 2 ,index)) (ldb #o0020 ,value))))




;; Moved here from Lisp-Reinitialize
(DEFVAR Processor-Slot-Number :UNBOUND)


(DEFVAR TV:TV-SLOT-NUMBER nil "Slot number for tv board")
(DEFVAR TV:SIB-SLOT-NUMBER :UNBOUND "Slot number for SIB board")
(DEFVAR Nupi-Slot-Number :UNBOUND)

;;; This is used to run simple hardware initializations
;;; Initializations on this list get done for a cold or warm boot before any
;;; other initializations get run
;;; Anything on this list must be able to be evaled in a very primitive cold
;;; load environment
(Defparameter Cold-Hardware-Initializations
	  '((Setup-Slot-Variables)
	    (Clear-SIB-Event-Registers)
	    (Reset-SIB)                
	    (Startup-TV-Interrupt)
	    (Startup-Keyboard)
	    (Initialize-SIB-Event-Posting)
	    (Enable-SIB-Interrupts)
	    ))


;; Some new variables to support the CSIB since many registers are at 
;; different locations on the CSIB.
(DEFVAR tv:sib-is-csib nil
  "Non-nil if using CSIB instead of SIB which has some different control registers.")	
(DEFVAR tv:configuration-register-offset :UNBOUND
  "Offset into SIB or CSIB of configuration register.")



;; new 7/88 grh - support for multiple monitors
(DEFVAR TV:*CSIB-SLOTS* nil
  "List of all slots containing CSIBs.  This may include the keyboard slot.")
(DEFVAR TV:*SIB-SLOTS*  nil
  "List of all slots containing SIBs.  This may include the keyboard slot.")
;; new 7/88 grh
(DEFVAR tv:*aux-csib-fslots* nil
  "A list containing CSIB fslots except the keyboard slot.")

(DEFUN setup-sib-slots ()
  ;; find all SIBs/CSIBs and put in TV:*SIB-SLOTS* and TV:*CSIB-SLOTS*.
  ;; includes primary sib, tv:sib-slot-number in list.
  (SETQ tv:*csib-slots* nil)
  (SETQ tv:*sib-slots* nil)
  (SETQ tv:*aux-csib-fslots* nil) ;; may 01/27/89 
  (DO ((slots-i-own (get-paging-parameter %Slots-I-Own))
       (bit-index 0 (1+ bit-index)))
      ((= bit-index 16.))
    ;; Make sure we own the board
    (WHEN (LDB-TEST (BYTE 1 bit-index) slots-i-own)
      (LET ((slot (DPB bit-index (BYTE 4. 0.) #xf0)))
	(COND ((STRING= "SIB" (board-type slot))
	       (SETQ tv:*sib-slots* (CONS bit-index tv:*sib-slots*)))
	      ((STRING= "CSI" (board-type slot))
	       (SETQ tv:*csib-slots* (CONS bit-index tv:*csib-slots*)))))))
  ;; Set *csib-slot* to the primary CSIB.  07/27/88 KJF.
  ;; It used to get set by tv:init-csib-registers and :before :init of tv:control-register-flavor.
  (SETQ tv:*csib-slot* (AND tv:sib-is-csib (- tv:sib-slot-number tv:*slot-space*)))
  (dolist (slot tv:*csib-slots* tv:*aux-csib-fslots*)
    (when (neq (ldb #o4 slot) (ldb #o4 tv:sib-slot-number))
      (setq tv:*aux-csib-fslots* (cons (dpb slot #o4 #x+F0) tv:*aux-csib-fslots*)))))

;;; Setup the slot variables from the boot parameters left in high a-memory.
;;; Boot parameters of interest are:
;;;   a-boot-monitor  (x3FB) 
;;;   a-boot-keyboard (x3FC)
;;;   a-boot-device   (x3FD)
;;; For now we assume that the SIB, monitor & keyboard are all on the same board
;;; Eventually should have si:keyboard-slot-number, si:network-slot-number.

(Defun Setup-Slot-Variables ()
  ;; Modified to get value from constant a-memory register for Explorer, -ab
  (setq TV:TV-SLOT-NUMBER 
 	(dpb #x+F (byte 4. 4.)
 	     (%p-ldb %%NuBus-Slot-Bits
 		     (+ sys:a-memory-virtual-address %A-Boot-KBD-Device-Address))))
  ;;;The following code makes TV:SIB-SLOT-NUMBER an "alias" of TV:TV-SLOT-NUMBER.
  (forward-value-cell 'TV:SIB-SLOT-NUMBER 'TV:TV-SLOT-NUMBER)
  ;; Get Processor slot number from SCA where microcode stored it during boot. -ab
  (setq PROCESSOR-SLOT-NUMBER (system-communication-area %SYS-COM-PROCESSOR-SLOT))
  (setq NUPI-SLOT-NUMBER
	(dpb #x+F (byte 4. 4.)
	     (%p-ldb %%NuBus-Slot-Bits
		     (+ sys:a-memory-virtual-address %A-Boot-MCR-Device-Address))))

  (setq tv:sib-is-csib
	(= #\C (int-char (ldb (byte 8. 0)
			      (%nubus-read tv:tv-slot-number CROMO-Board-Type-Offset-Name))))) ; CSIB
  (setq tv:configuration-register-offset
	(if tv:sib-is-csib
	    %CSIB-CONFIGURATION-REGISTER-OFFSET
	    %SIB-Reset))
  ;; find all SIBs/CSIBs and put in TV:*SIB-SLOTS* and TV:*CSIB-SLOTS*.
  (setup-sib-slots)
  )



;; Disable SIB event posting before zeroing out these locations
;; so that warm boots will work right.  Problem will otherwise
;; occur because the event posting address for the Graphics
;; controller command acknowledge (at #x+FxF00014) cannot be zero
;; for any length of time during warm boot because the SIB bus
;; master enable IS active on warm boot (but not at this point
;; during cold boot).  So during warm boot an event can be
;; generated during the time the posting address is zero; this
;; first interrupt never gets cleared, causing all sorts of
;; of timing problems.
;; System patch 2-54, -ab
;; Use symbolic names for offsets.
(Defun Clear-SIB-Event-Registers ()
  (when (mp-initialize-ok)	;jlm 3/28/89
    (cond ((not tv:sib-is-csib)
	   ;; Disable Master SIB event posting first (important for warm boot)
	   (%nubus-write tv:Sib-Slot-Number tv:configuration-register-offset 0.)
	   ;; zero out all even vector addresses.
	   (Do ((offset
		  (symbol-value (car SIB-Event-Address-Registers))
		  (+ offset 4.)))
	       ((> offset (symbol-value (car (last SIB-Event-Address-Registers)))))
	     (%nubus-write tv::Sib-Slot-Number offset 0)))
	  (t
	   ;; Clear master enable bit on CSIB
	   (%NuBus-Write TV:TV-Slot-Number tv:configuration-register-offset    
			 (dpb 0 #o0101
			      (%NuBus-Read TV:TV-SLOT-NUMBER tv:configuration-register-offset))))
	  )))



(defun millisecond-time ()
  "Return the current millisecond time from the real time clock."
  (do* ((slot tv:sib-slot-number)
	(milli-address #xf80004)
	(status-address #xf80050)
	status
        milli-value
	(brake 0 (1+ brake)))
       (())
    (setq milli-value (%nubus-read-8b slot milli-address))
    ; If we are running too fast put on the brakes till we get a good result.
    (do ((b brake (1- b))) ((< b 0)))
    (setq status (ldb (byte 1 0) (%nubus-read-8b slot status-address)))
    (do ((b brake (1- b))) ((< b 0)))
    (when (or (zerop status)	; read was valid
	      (= brake 100.))   ; give up
      (return (+ (* 100. (ldb (byte 4 4) milli-value))
		 (* 10. (ldb (byte 4 0) milli-value)))))))
	  
;;;clm 11/03/88 - made separate version of this function for the MX,
;;;as it has no SIB.
(defun wait-real-milliseconds (n)
  "Wait N milliseconds.  The microsecond clock is used instead of the 
 real time clock which only exists on the Explorers."
 (if (mx-p)
  (without-interrupts 
    (let* ((this-time)
	   (time-waited 0)
	   (n (* 1000 n))
	   (last-time (si:%microsecond-time)))
      (do () (())
	(setq this-time (si:%microsecond-time)
	      time-waited (+ time-waited
			     (time:microsecond-time-difference
			       this-time last-time))
	      last-time this-time)
	(when (>= time-waited n)
	  (return (values (floor time-waited 1000)))
	  ))))
   (without-interrupts 
    (let* ((last-time (millisecond-time))
	   (this-time)
	   (time-waited 0))
      (do () (())
	(setq this-time (millisecond-time)
	      time-waited (+ time-waited
			     (if (< this-time last-time)
				 (+ (- 1000. last-time) this-time) ; we wrapped
				 (- this-time last-time)))
	      last-time this-time)
	(when (>= time-waited n)
	  (return time-waited)))))))




;; This performs the required SIB cold initialization sequence.
;; Use the real time functions above, for Exp 2 compatibility. GRH 2/18/87
(Defun Reset-SIB ()
  (DECLARE (SPECIAL SIB-CRT-Init-Sequence-List))	;defined in LROY-QDEV
  ;; Only do full reset on Cold Boot.
  (when (and Cold-Booting
	     (mp-initialize-ok))	;jlm 3/28/89
    (let ((slot tv:SIB-Slot-Number))

      ;; Initialize RTC
      ;;   first disable RTC interrupts
      (%nubus-write-8b slot %SIB-RTC-Interrupt-Control 0)	;; #x+F80044
      ;;  Wait 100 units. (units = ?) before reading the interrupt status.
      (do ((x 100. (1- x))) ((< x 0)))
      ;;   Reset any pending interrupts on RTC (by reading it)
      (%nubus-read-8b  slot %SIB-RTC-Interrupt-Status)		;; #x+F80040
      
      (unless
	tv:sib-is-csib

	;; Reset CRT controller by writing byte of zeros to reset register.
	(%nubus-write-8b slot %SIB-CRT-Controller-Reset-Register 0)	;; #x+E00058
	
	;; Reset SIB
	(%nubus-write-8b slot tv:configuration-register-offset
			 (byte-mask %%SIB-Reset)) 	;; #x+F00040
	
	;; Send CRT initialization sequence
	(do* ((adr %SIB-CRT-Controller-Register-Base (+ adr 4))
	      (val-lst SIB-CRT-Init-Sequence-List (cdr val-lst))
	      (value (car val-lst) (car val-lst)))
	     ((>= adr %SIB-CRT-Controller-Reset-Register))
	  (%nubus-write-8b slot adr value))
	
	;; After initializing CRT controller (by loading 23 registers)
	;; we must delay (at least 600 microseconds) to ensure that the data
	;; link is up
	
	(wait-real-milliseconds 10.)		; mimimum wait available
	
	))))


;;; TV frame interrupt initialization

(DefVar *TV-Frame-Interrupt-Descriptor*)

;; Displace array into device-descriptor-area instead of permanent-storage-area.
(Defun Cold-Initialize-TV-Interrupt ()
  (Setq *TV-Frame-Interrupt-Descriptor*
	(Make-Array (* 2. 2.)       ;(length %IO-Descriptor-Words)
		    :type art-16b
		    :Displaced-To (allocate-device-descriptor-space 2) 
                    ))
  (Aset-32b 0
	    *TV-Frame-Interrupt-Descriptor*
	    SI:%IO-Device-Block-Link)            ;Init Link Word   
  (Aset-32b SI:%TV-Frame-Event-Type            
	    *TV-Frame-Interrupt-Descriptor*	;Init info word
	    SI:%IO-Device-Descriptor-Word)     
  (%Add-Interrupt *TV-Frame-Interrupt-Descriptor* SI:%TV-Frame-Event-Level))

(Defun Warm-Initialize-TV-Interrupt ()
  (when (mp-initialize-ok)	;jlm 3/28/89
    (Let* ((Processor-Slot-Address 
	     (DPB PROCESSOR-SLOT-NUMBER %%NuBus-F-And-Slot-Bits 0))
	   (Event-Address (+ Processor-Slot-Address
			     SI:%SLOT-POWER-FAIL-EVENT      
			     (* SI:%TV-Frame-Event-Level 4.))))
      ;; Setup the event address
      (%NuBus-Write TV:TV-SLOT-NUMBER
		    (if tv:sib-is-csib
			%CSIB-TV-Event-Address
			%SIB-TV-Event-Address)
		    Event-Address)
      ;; enable vertical retrace interrupt
      (unless tv:sib-is-csib			; done later for CSIB
	;; enable vertical retrace interrupt
	(%NuBus-Write TV:TV-SLOT-NUMBER
		      %SIB-TV-Interrupt-Enable
		      (Byte-Mask %%SIB-TV-Frame-Interrupt-Enable)))
      )))

(Defun Startup-TV-Interrupt ()
  (if (or Cold-Booting (Not (Boundp '*TV-Frame-Interrupt-Descriptor*)))
      (Cold-Initialize-TV-Interrupt))
  (Warm-Initialize-TV-Interrupt))

(Defun Shutdown-TV-Interrupt ()
  (if tv:sib-is-csib
      (%NuBus-Write TV:TV-Slot-Number %CSIB-TV-INTERRUPT-ENABLE    
		    (dpb 0 #o0201                     ;;use read modify write instead of writing a zero
			 (%NuBus-Read TV:TV-SLOT-NUMBER %CSIB-TV-INTERRUPT-ENABLE)))
      (%NuBus-Write TV:TV-Slot-Number %SIB-TV-Interrupt-Enable 0))
  (makunbound '*TV-Frame-Interrupt-Descriptor*))



(DefVar *Keyboard*)
(Defprop *Keyboard* "Keyboard device descriptor" :Documentation)

(DefParameter *KBD-Buffer-Size* 128.)

(DefVar *Keyboard-Char-Buffer*)
(Defprop *Keyboard-Char-Buffer* "Keyboard character buffer" :Documentation)

(Defun Initialize-KBD-Char-Buffer ()
  (Let ((KBD-Start-Address (%Pointer (Locf (Lisp:aref *Keyboard-Char-Buffer* 0) ))))
    (Aset-32b KBD-Start-Address *Keyboard* %KBD-Buffer-Start)
    (Aset-32b (+ KBD-Start-Address *KBD-Buffer-Size*) *Keyboard* %KBD-Buffer-End)
    (Aset-32b KBD-Start-Address *Keyboard* %KBD-Buffer-Front)
    (Aset-32b KBD-Start-Address *Keyboard* %KBD-Buffer-Back))
  )   

(Defun Initialize-KBD-Cold ()
  (Setq *Keyboard*
	(Make-Array
	  (* (Length KBD-Descriptor-Block) 2)
	  :type art-16b
	  :displaced-to (allocate-device-descriptor-space      
			   (length KBD-Descriptor-Block))
	  ))
  (Setq *Keyboard-Char-Buffer*
	(Make-Array
	  *KBD-Buffer-Size*
	  :type art-q
	  :displaced-to 
	  (allocate-device-descriptor-space *KBD-Buffer-Size*)
	  ))
  ;; Initialize keyboard descriptor.
  ;; Clear link word.
  (Aset-32b 0 *Keyboard* %KBD-Link)
  ;; Information word contains device type.
  (Aset-32b %KBD-Event-Type *Keyboard* %KBD-Information)
  (Aset-32b
    (dpb tv:sib-slot-number
	 %%NuBus-F-And-Slot-Bits
	 (if tv:sib-is-csib
	     %CSIB-KBD-UART-Receive-Data  ; not used anymore by microcode anyway - CSIB 4/10/87
	     %SIB-KBD-UART-Mode))
    *Keyboard*
    %KBD-Control-Space-Address)  
  (Aset-32b 0 *Keyboard* %KBD-Undefined)
  (Initialize-KBD-Char-Buffer)
  ;; Add keyboard descriptor to interrupt decode structure.
  (%Add-Interrupt *Keyboard* %KBD-Event-Level)
  )

;; Use the real time functions above, for Exp 2 compatibility. GRH 2/18/87
(Defun Initialize-KBD-Warm (&key force) ;; (&aux (count 0))
  (when (or force ;; marky 12-20-88
	    (mp-initialize-ok))	;jlm 3/28/89
    (Let ((slot TV:SIB-SLOT-NUMBER))
      ;; Reset the character buffer.
      (Initialize-KBD-Char-Buffer)
      
      ;; Set up event posting address for Keyboard Events.
      (Let* ((Processor-Slot-Address
	       (DPB PROCESSOR-SLOT-NUMBER %%NuBus-F-And-Slot-Bits 0))
	     (Event-Address (+ Processor-Slot-Address SI:%SLOT-POWER-FAIL-EVENT
			       (* SI:%KBD-Event-Level 4.))))
	(%NuBus-Write slot
		      (if tv:sib-is-csib
			  %CSIB-KBD-Event-Address 
			  %SIB-KBD-Event-Address)
		      Event-Address))
      
      (unless tv:sib-is-csib
	;; Now, enable SIB interrups briefly.  When we're done with the
	;; USART/keyboard initializations, the keyboard sends a 2-character
	;; sequence to acknowledge.  We want the Ucode interrupt routine to
	;; handle these, hence interrupts must be on here.
	;;  It's ok to turn on interrupts here because the vertical retrace
	;; event address has already been setup by Startup-TV-Interrupt.
	(%nubus-write tv:Sib-Slot-Number tv:configuration-register-offset
		      (byte-mask %%SIB-Interrupt-Enable))
	
	;; This is worst-case power-up initialization for the USART.
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode #x+0)
	;; need to wait 3.2 MICROseconds when writing to the USART
	;; this really waits 10 MILLIseconds.
	(wait-real-milliseconds 1.)	 			
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode #x+0)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode #x+0)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode %KBD-Usart-Reset)	;#x40
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode %KBD-Usart-Mode)	;#x7F
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode %KBD-Usart-Command)	;#x14
	(wait-real-milliseconds 1.)	
	;; 
	;;  After transmitting .. special escape sequence [warmboot c-m-c-m-whatever],
	;;  the keyboard microcomputer [on the keyboard!!!] ignores keyboard entries 
	;;  except for special chords. ... The keyboard microcomputer resumes normal
	;;  acceptance of keyboard entries after it receives a reset command as part
	;;  of self-test procedures or a system reset.  Since a full reset is only
	;;  done on cold boot, the following keyboard soft reset is required for warm
	;;  boot (and won't hurt on cold boots).
	;;
	;; Wait for TXRDY (transmit ready)
	
	(wait-real-milliseconds 100.)
	
	;; Enable Transmit interrupts for response (note that this keeps receive enable on).
	(%nubus-write-8b slot %SIB-KBD-UART-Mode #x+5)
	
	;; Send Reset Command to keyboard microcomputer.  This will cause the keyboard
	;; to send a 2-char response (00, x+70) which will be intercepted by the Ucode
	;; interrupt handler and placed in *Keyboard-Char-Buffer*.
	(%nubus-write-8b slot %SIB-KBD-UART-Data #x+7)
	
	;; Now we must wait at LEAST 10 milliseconds for the keyboard reset to take.
	;; Then re-initialize the USART so it is in a state where it can send the 2-char
	;; response back.  The loop below should be 10 times that long.
	
	(wait-real-milliseconds 100.)
	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode #x+0)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode #x+0)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode #x+0)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode %KBD-Usart-Reset)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode %KBD-Usart-Mode)
	(wait-real-milliseconds 1.)	
	(%NuBus-Write-8b slot %SIB-KBD-UART-Mode %KBD-Usart-Command)
	(wait-real-milliseconds 1.)	
	
	;; Get the response characters out of the keyboard buffer as soon as the Ucode
	;; gets them so that they will not be fed into someone else's IO stream later.
	
	(wait-real-milliseconds 100.)
	(kbd-get-hardware-char)
	
	(wait-real-milliseconds 100.)
	(kbd-get-hardware-char)
	
	;; Disable SIB events again.  They will be turned on for good later.
	(%nubus-write slot tv:configuration-register-offset 0)
	
	;; Finally, reset the keyboard buffer again so that it looks empty.
	(Initialize-KBD-Char-Buffer)
	))))

(Defun Startup-Keyboard ()
  (If (or Cold-Booting (Not (Boundp '*KEYBOARD*)))
      (Initialize-KBD-COLD))
  (Initialize-KBD-Warm))

;;; this actually shuts down all of the SIB interrupts
(Defun shutdown-KBD ()
  (if tv:sib-is-csib
      ;; Clear master enable bit on CSIB
      (%NuBus-Write TV:TV-Slot-Number tv:configuration-register-offset    
		    (dpb 0 #o0101
			 (%NuBus-Read TV:TV-SLOT-NUMBER tv:configuration-register-offset)))
      (%NuBus-Write TV:SIB-SLOT-NUMBER tv:configuration-register-offset 0))
  (makunbound '*keyboard*))


(Defun Initialize-SIB-Event-Posting ()

  (when (mp-initialize-ok)	;jlm 3/28/89
    ;; Overtemp Event.
    ;; Set posting location to zero.
    (setf (system-communication-area %SYS-COM-OVERTEMP-EVENT) 0)
    ;; Place NuBus address of posting location in SIB event register.
    (%nubus-write TV:SIB-SLOT-NUMBER
		  (if tv:sib-is-csib
		      %CSIB-Overtemp-Warn-Event-Address	; CSIB
		      %SIB-Overtemp-Warn-Event-Address)
		  (%physical-address 
		    (locf (system-communication-area %SYS-COM-OVERTEMP-EVENT))))
    
    ;; Fiber Optic Disconnect Event.
    ;; Set fiber optic warning event location to zero.
    (setf (system-communication-area %SYS-COM-FIBER-OPTIC-WARNING-EVENT) 0)
    ;; Modify Video Attribute register to enable Fiber Optic interrupt.
    (unless
      tv:sib-is-csib				; How do we enable this on the CSIB; is is not in the video attribute register ??
      (%nubus-write-8b TV:SIB-SLOT-NUMBER
		       %SIB-TV-Video-Attribute 			
		       (dpb 1 %%SIB-Fiber-Optic-Warning-Enable
			    (%nubus-read-8b TV:SIB-SLOT-NUMBER %SIB-TV-Video-Attribute))))
    ;; Place NuBus address of posting location in SIB event register.
    (%nubus-write TV:SIB-SLOT-NUMBER
		  (if tv:sib-is-csib
		      %CSIB-Fiber-Optic-Link-Warning-Address	; CSIB
		      %SIB-Fiber-Optic-Link-Warning-Address)
		  (%physical-address 
		    (locf (system-communication-area %SYS-COM-FIBER-OPTIC-WARNING-EVENT))))
    ))


(Defun Enable-SIB-Interrupts (&key force)
  (when (or force ;; marky 12-20-88
	    (mp-initialize-ok))	;jlm 3/28/89
    (cond ((not tv:sib-is-csib)
	   ;; Set master enable bit on SIB
	   (%NuBus-Write TV:SIB-SLOT-NUMBER
			 tv:configuration-register-offset
			 (Byte-Mask %%SIB-Interrupt-Enable))
	   ;; Enable warm boot event
	   (%Nubus-Write TV:SIB-SLOT-NUMBER %SIB-Warmboot-Event-Address 
			 (dpb Processor-Slot-Number
			      %%NuBus-F-And-Slot-Bits 
			      %Slot-Event-Receivers-Base))
	   ;; Enable powerfail event
	   (%Nubus-Write TV:SIB-SLOT-NUMBER %SIB-Powerfail-Event-Address-1
			 (dpb Processor-Slot-Number
			      %%NuBus-F-And-Slot-Bits 
			      %Slot-Power-Fail-Event))
	   )
	  (t
	   ;; Enable warm boot event
	   (%Nubus-Write TV:SIB-SLOT-NUMBER %CSIB-Warmboot-Event-Address 
			 (dpb Processor-Slot-Number
			      %%NuBus-F-And-Slot-Bits 
			      %Slot-Event-Receivers-Base))
	   ;; Enable powerfail event
	   (%Nubus-Write TV:SIB-SLOT-NUMBER %CSIB-Powerfail-Event-Address-1 
			 (dpb Processor-Slot-Number
			      %%NuBus-F-And-Slot-Bits 
			      %Slot-Power-Fail-Event))
	   ;; Clear the keyboard fifo of any leftover characters
	   (dotimes (x 64. nil)
	     (%NuBus-Write-8b tv::sib-slot-number %CSIB-KBD-UART-Receive-Status 0))
	   ;; Clear tv-frame retrace interrupt in case set
	   (%nubus-read tv:tv-slot-number %CSIB-TV-STATUS)
	   ;; Clear CSIB event status register
	   (%nubus-write tv:tv-slot-number %CSIB-EVENT-STATUS-REGISTER-OFFSET 0)
	   ;; Set master enable bit on CSIB
	   (%NuBus-Write TV:TV-Slot-Number tv:configuration-register-offset    
			 (dpb -1 #o0101
			      (%NuBus-Read TV:TV-SLOT-NUMBER tv:configuration-register-offset)))
	   
	   ;; Clear any keyboard chars that occurred between the fifo clearing above and the NuBus
	   ;; master enabling above. This will allow CSIB to continue posting events which were
	   ;; just now enabled via the NuBus master enable bit in the config register. This fixes
	   ;; the problem which occurs when a ctrl-meta-ctrl-meta-rubout chord is pressed AND HELD DOWN
	   ;; which causes a char to be sent AFTER the fifo clearing but BEFORE the nubus master
	   ;; enabling, which meant the keyboard did not work because CSIB was waiting for LISP to take
	   ;; that char, and LISP was waiting for an event!
	   (%NuBus-Write-8b tv::sib-slot-number %CSIB-KBD-UART-Receive-Status 0)
	   
	   ;; enable vertical retrace interrupt
	   (%NuBus-Write
	     TV:TV-SLOT-NUMBER
	     %CSIB-TV-INTERRUPT-ENABLE
	     (dpb -1 #o0201
		  (%NuBus-Read TV:TV-SLOT-NUMBER %CSIB-TV-INTERRUPT-ENABLE)))
	   ))))

;; marky 12-20-88 new
(defun reset-keyboard ()
  "Restore operation to locked up keyboard"
  ;; stop keyboard interrupts - taken from sys:Shutdown-KBD
  (if tv:sib-is-csib
      ;; Clear master enable bit on CSIB
      (%NuBus-Write TV:TV-Slot-Number tv:configuration-register-offset    
		    (dpb 0 #o0101
			 (%NuBus-Read TV:TV-SLOT-NUMBER tv:configuration-register-offset)))
      (%NuBus-Write TV:SIB-SLOT-NUMBER tv:configuration-register-offset 0))
  (Initialize-KBD-Warm :force t) 	;; reset uart
  (Enable-SIB-Interrupts :force t) 	;; this will also clear out extra keyboard characters
  )


;;;
;;; Misc slot initialization
;;;

(DEFVAR *addin-board-slot* #xF6)
(DEFVAR *addin-memory-slot* #xFC)

(DEFUN setup-processor-slot ()
  (SETQ *addin-board-slot* 
	(DPB #xF (BYTE 4. 4.)
	     (get-processor-slot)))
  (SETQ *addin-memory-slot* 
	(LDB %%Nubus-F-And-Slot-Bits (%physical-address 0)))
  (SETQ processor-slot-number
	(DPB #xF (BYTE 4. 4.)
	     (get-processor-slot)))
  )


;;;
;;; Keyboard Support
;;;


(defparameter *keyboard-valid-address*  (+ %Driver-Data-Start %DD-Kbd-Valid) "short") 
(defparameter *keyboard-buffer-address* (+ %Driver-Data-Start %DD-Kbd-Buffer) "long")

(PROCLAIM '(inline mx-char-avail))
(DEFUN mx-char-avail ()
  (WHEN (PLUSP (%nubus-read-16b *addin-memory-slot*
				*keyboard-valid-address*))
    (%nubus-read *addin-memory-slot*
		 *keyboard-buffer-address*))
  )

(PROCLAIM '(inline mx-get-key))
(DEFUN mx-get-key ()
  (WHEN (PLUSP (%nubus-read-16b *addin-memory-slot*
				*keyboard-valid-address*))
    (WITHOUT-INTERRUPTS
      (PROG1  
	(%nubus-read *addin-memory-slot*
		     *keyboard-buffer-address*)
	(%nubus-write-16b *addin-memory-slot*
			  *keyboard-valid-address*
			  0)))))


(DEFVAR tv:*mac-keyboard-p* nil
  "If nil use the LISPM keyboard, otherwise use the Mac keyboard.")


(Defun kbd-hardware-char-available ()
  "Returns T if a character is available. Uses the Explorer microcode interrupt buffer or
the Mac command block depending on the selected window."
  (IF tv:*mac-keyboard-p*
      (mx-char-avail)
      (%IO %KBD-IO-Char-Available *Keyboard*)))


(Defun kbd-get-hardware-char ()
  "Returns the next available character or nil if there is none. Uses the Explorer microcode
interrupt buffer or the Mac command block depending on the selected window."
  (LET ((kbd-char
	  (IF tv:*mac-keyboard-p*
	      (mx-get-key)
	      (%IO %KBD-Get-Char *Keyboard*))))
    kbd-char))

