#include <iostream.h>
#include <fstream.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "macroop.h"
#include "macroop.h"
#include "memobj.h"


// Miscellaneous macroinsn bit-mangling.

e3MacroInstruction::e3MacroInstruction(e3u16 my_bits)
{
  setBits(my_bits);
}

e3u16 e3MacroInstruction::getBits(void) const
{
#ifdef E3_LITTLE_ENDIAN_PLATFORM
  return(myBits);
#else
  // Shift low bits to high, high bits to low.
  e3u16 big_endian_bits = ((myBits & 0xff) << 8) | ((myBits & 0xff00) >> 8);
  return(big_endian_bits);
#endif // E3_LITTLE_ENDIAN_PLATFORM
}

void e3MacroInstruction::setBits(e3u16 my_bits)
{
#ifdef E3_LITTLE_ENDIAN_PLATFORM
  myBits = my_bits;
#else
  myBits = ((my_bits & 0xff) << 8) | ((my_bits & 0xff00) >> 8);
#endif // E3_LITTLE_ENDIAN_PLATFORM
}

e3u16 e3MacroInstruction::selectBitsMask(e3u16 mask) const
{
  e3u16 my_bits = getBits();
  e3u16 result = my_bits & mask;
  return(result);
}

e3u16 e3MacroInstruction::selectBitsPPSS(e3u16 position, e3u16 size) const
{
#ifdef PARANOID
  if (size >= 16)
    {
      cerr << "e3MacroInstruction::selectBitsPPSS -- size too big for word size " << size << endl;
      abort();
      return(0);
    }
  else if (position >= 16)
    {
      cerr << "e3MacroInstruction::selectBitsPPSS -- size too big for word size " << size << endl;
      abort();
      return(0);
    }
  else
#endif // PARANOID
    {
      // Construct mask of correct width.
      e3u16 mask = 0;
      for (e3u16 i=0; i<size; i++)
        { mask = (mask << 1) | 1; }

      // Shift mask to position.
      mask = mask << position;

      // Extract the bits.
      e3u16 bits = selectBitsMask(mask);

      // Shift back down to low bits.
      bits = bits >> position;
      return(bits);
    }
}


// Instruction dumping.

void e3MacroInstruction::dump(ostream &os) const
{
  e3DefOp full_opcode = QMIFullOpcode();

// A spoonful of preprocessor makes the medicine go down...
#define PRINT_OP(op) \
    case DEFOP_ ## op: \
      cout << "   " #op; \
      dumpMainOperands(os); \
      break;
#define PRINT_BRANCHOP(branchop) \
    case DEFBRANCHOP_ ## branchop: \
      cout << " b " #branchop; \
      dumpShortBranchOperands(os); \
      break;
#define PRINT_IMMEDOP_FIX(immedop) \
    case DEFOP_ ## immedop: \
      cout << " i " #immedop; \
      dumpImmedOperands(os, IMM_FIXNUM); \
      break;
#define PRINT_IMMEDOP_PPSS(immedop) \
    case DEFOP_ ## immedop: \
      cout << " i " #immedop; \
      dumpImmedOperands(os, IMM_PPSS); \
      break;
#define PRINT_CALLOP(callop) \
    case DEFCALLOP_ ## callop: \
      cout << " c " #callop; \
      dumpMainOperands(os); \
      break;

// Auxops and miscops have no operands (they are taken from stack).
#define PRINT_AUXOP(auxop) \
        case AUXOP_ ## auxop: \
          cout << #auxop; \
          break;
#define PRINT_MISCOP(miscop) \
        case MISCOP_ ## miscop: \
          cout << #miscop; \
          break;
#define PRINT_AREFIOP(arefiop) \
        case AREFIOP_ ## arefiop: \
          cout << #arefiop; \
          break;

  switch(full_opcode)
    {
    case DEFOP_AUX_GROUP:
      cout << " a ";
      switch(QMIAuxOpcode())
        {
          PRINT_AUXOP(RESERVED_ILLEGAL);
          PRINT_AUXOP(BREAKPOINT);
          PRINT_AUXOP(HALT);
          PRINT_AUXOP(CRASH);

          PRINT_AUXOP(EXCHANGE);
          PRINT_AUXOP(SPREAD);
          PRINT_AUXOP(ASSURE_PDL_ROOM);
          PRINT_AUXOP(POP_M_FROM_UNDER_N);
          PRINT_AUXOP(POPJ);
          PRINT_AUXOP(UNWIND_PROTECT_CLEANUP);
          PRINT_AUXOP(USING_BINDING_INSTANCES);
          PRINT_AUXOP(UNBIND_TO_INDEX);

          PRINT_AUXOP(SET_SELF_MAPPING_TABLE);

          PRINT_AUXOP(STACK_GROUP_RETURN);
          PRINT_AUXOP(STACK_GROUP_RESUME);
          PRINT_AUXOP(STORE_IN_HIGHER_CONTEXT);
          PRINT_AUXOP(LEXICAL_UNSHARE_ALL);
          PRINT_AUXOP(STORE_KEY_WORD_ARGS);

          PRINT_AUXOP(COMPLEX_CALL_TO_INDS);
          PRINT_AUXOP(COMPLEX_CALL_TO_PUSH);
          PRINT_AUXOP(COMPLEX_CALL_TO_RETURN);
          PRINT_AUXOP(COMPLEX_CALL_TO_TAIL_REC);
          PRINT_AUXOP(APPLY_TO_INDS);
          PRINT_AUXOP(APPLY_TO_PUSH);
          PRINT_AUXOP(APPLY_TO_RETURN);
          PRINT_AUXOP(APPLY_TO_TAIL_REC);
          PRINT_AUXOP(APPLY_METHOD_TO_INDS);
          PRINT_AUXOP(APPLY_METHOD_TO_PUSH);
          PRINT_AUXOP(APPLY_METHOD_TO_RETURN);
          PRINT_AUXOP(APPLY_METHOD_TO_TAIL_REC);
          PRINT_AUXOP(LEXPR_FUNCALL_WITH_MAPPING_TABLE_TO_INDS);
          PRINT_AUXOP(LEXPR_FUNCALL_WITH_MAPPING_TABLE_TO_PUSH);
          PRINT_AUXOP(LEXPR_FUNCALL_WITH_MAPPING_TABLE_TO_RETURN);
          PRINT_AUXOP(LEXPR_FUNCALL_WITH_MAPPING_TABLE_TO_TAIL_REC);
          PRINT_AUXOP(RETURN_N);
          PRINT_AUXOP(RETURN_LIST);
          PRINT_AUXOP(RETURN_NIL);
          PRINT_AUXOP(RETURN_T);
          PRINT_AUXOP(OPEN_CATCH);
          PRINT_AUXOP(OPEN_CATCH_MULTIPLE_VALUE);
          PRINT_AUXOP(OPEN_CATCH_TAIL_RECURSIVE);
          PRINT_AUXOP(OPEN_CATCH_MV_LIST);
          PRINT_AUXOP(THROW);
          PRINT_AUXOP(THROW_N);
          PRINT_AUXOP(UNWIND_STACK);
          PRINT_AUXOP(CLOSE_CATCH);
          //PRINT_AUXOP(CLOSE_CATCH_RETURN);
          PRINT_AUXOP(CLOSE_CATCH_UNWIND_PROTECT);
          PRINT_AUXOP(RETURN_PRED);
          PRINT_AUXOP(RETURN_NOT_INDS);
          //PRINT_AUXOP(CHANGE_PAGE_STATUS);
          PRINT_AUXOP(ENABLE_NUPI_LOCKING);
          PRINT_AUXOP(DISABLE_NUPI_LOCKING);
          PRINT_AUXOP(PAGE_IN);
          PRINT_AUXOP(ADD_PAGE_DEVICE);
          //PRINT_AUXOP(SCRUB);
          PRINT_AUXOP(CREATE_PHYSICAL_PAGE);
          PRINT_AUXOP(DELETE_PHYSICAL_PAGE);
          PRINT_AUXOP(GC_FREE_REGION);
          PRINT_AUXOP(GC_FLIP);
          PRINT_AUXOP(GC_SCAVENGE);
          PRINT_AUXOP(GC_CONS_WORK);
          PRINT_AUXOP(FLUSH_EXTRA_PDL);
          PRINT_AUXOP(IMPORT_OBJECT);
          PRINT_AUXOP(DISK_RESTORE);
          PRINT_AUXOP(DISK_SAVE);
          PRINT_AUXOP(LONG_BR_NULL_ELSE_POP);
          PRINT_AUXOP(LONG_BR_NOT_NULL_ELSE_POP);
          PRINT_AUXOP(LONG_BR_NULL);
          PRINT_AUXOP(LONG_BR_NOT_NULL);
          PRINT_AUXOP(LONG_BR_ATOM);
          PRINT_AUXOP(LONG_BR_NOT_ATOM);
          PRINT_AUXOP(LONG_BR_ZEROP);
          PRINT_AUXOP(LONG_BR_NOT_ZEROP);
          PRINT_AUXOP(LONG_BR_SYMBOLP);
          PRINT_AUXOP(LONG_BR_NOT_SYMBOLP);

          PRINT_AUXOP(LONG_BR_NULL_LIKELY);
          PRINT_AUXOP(LONG_BR_NOT_NULL_LIKELY);
          PRINT_AUXOP(LONG_BR);
          PRINT_AUXOP(LONG_PUSHJ);
          
          PRINT_AUXOP(UNBIND_1);
          PRINT_AUXOP(UNBIND_2);
          PRINT_AUXOP(UNBIND_3);
          PRINT_AUXOP(UNBIND_4);
          PRINT_AUXOP(UNBIND_5);
          PRINT_AUXOP(UNBIND_6);
          PRINT_AUXOP(UNBIND_7);
          PRINT_AUXOP(UNBIND_8);
          PRINT_AUXOP(UNBIND_9);
          PRINT_AUXOP(UNBIND_10);
          PRINT_AUXOP(UNBIND_11);
          PRINT_AUXOP(UNBIND_12);
          PRINT_AUXOP(UNBIND_13);
          PRINT_AUXOP(UNBIND_14);
          PRINT_AUXOP(UNBIND_15);
          PRINT_AUXOP(UNBIND_16);
          PRINT_AUXOP(UNBIND_17);
          PRINT_AUXOP(UNBIND_18);
          PRINT_AUXOP(UNBIND_19);
          PRINT_AUXOP(UNBIND_20);
          PRINT_AUXOP(UNBIND_21);
          PRINT_AUXOP(UNBIND_22);
          PRINT_AUXOP(UNBIND_23);
          PRINT_AUXOP(UNBIND_24);
          PRINT_AUXOP(UNBIND_25);
          PRINT_AUXOP(UNBIND_26);
          PRINT_AUXOP(UNBIND_27);
          PRINT_AUXOP(UNBIND_28);
          PRINT_AUXOP(UNBIND_29);
          PRINT_AUXOP(UNBIND_30);
          PRINT_AUXOP(UNBIND_31);
          PRINT_AUXOP(UNBIND_32);
          PRINT_AUXOP(UNBIND_33);
          PRINT_AUXOP(UNBIND_34);
          PRINT_AUXOP(UNBIND_35);
          PRINT_AUXOP(UNBIND_36);
          PRINT_AUXOP(UNBIND_37);
          PRINT_AUXOP(UNBIND_38);
          PRINT_AUXOP(UNBIND_39);
          PRINT_AUXOP(UNBIND_40);
          PRINT_AUXOP(UNBIND_41);
          PRINT_AUXOP(UNBIND_42);
          PRINT_AUXOP(UNBIND_43);
          PRINT_AUXOP(UNBIND_44);
          PRINT_AUXOP(UNBIND_45);
          PRINT_AUXOP(UNBIND_46);
          PRINT_AUXOP(UNBIND_47);
          PRINT_AUXOP(UNBIND_48);
          PRINT_AUXOP(UNBIND_49);
          PRINT_AUXOP(UNBIND_50);
          PRINT_AUXOP(UNBIND_51);
          PRINT_AUXOP(UNBIND_52);
          PRINT_AUXOP(UNBIND_53);
          PRINT_AUXOP(UNBIND_54);
          PRINT_AUXOP(UNBIND_55);
          PRINT_AUXOP(UNBIND_56);
          PRINT_AUXOP(UNBIND_57);
          PRINT_AUXOP(UNBIND_58);
          PRINT_AUXOP(UNBIND_59);
          PRINT_AUXOP(UNBIND_60);
          PRINT_AUXOP(UNBIND_61);
          PRINT_AUXOP(UNBIND_62);
          PRINT_AUXOP(UNBIND_63);
          PRINT_AUXOP(UNBIND_64);
          PRINT_AUXOP(POP_PDL_1);
          PRINT_AUXOP(POP_PDL_2);
          PRINT_AUXOP(POP_PDL_3);
          PRINT_AUXOP(POP_PDL_4);
          PRINT_AUXOP(POP_PDL_5);
          PRINT_AUXOP(POP_PDL_6);
          PRINT_AUXOP(POP_PDL_7);
          PRINT_AUXOP(POP_PDL_8);
          PRINT_AUXOP(POP_PDL_9);
          PRINT_AUXOP(POP_PDL_10);
          PRINT_AUXOP(POP_PDL_11);
          PRINT_AUXOP(POP_PDL_12);
          PRINT_AUXOP(POP_PDL_13);
          PRINT_AUXOP(POP_PDL_14);
          PRINT_AUXOP(POP_PDL_15);
          PRINT_AUXOP(POP_PDL_16);
          PRINT_AUXOP(POP_PDL_17);
          PRINT_AUXOP(POP_PDL_18);
          PRINT_AUXOP(POP_PDL_19);
          PRINT_AUXOP(POP_PDL_20);
          PRINT_AUXOP(POP_PDL_21);
          PRINT_AUXOP(POP_PDL_22);
          PRINT_AUXOP(POP_PDL_23);
          PRINT_AUXOP(POP_PDL_24);
          PRINT_AUXOP(POP_PDL_25);
          PRINT_AUXOP(POP_PDL_26);
          PRINT_AUXOP(POP_PDL_27);
          PRINT_AUXOP(POP_PDL_28);
          PRINT_AUXOP(POP_PDL_29);
          PRINT_AUXOP(POP_PDL_30);
          PRINT_AUXOP(POP_PDL_31);
          PRINT_AUXOP(POP_PDL_32);
          PRINT_AUXOP(POP_PDL_33);
          PRINT_AUXOP(POP_PDL_34);
          PRINT_AUXOP(POP_PDL_35);
          PRINT_AUXOP(POP_PDL_36);
          PRINT_AUXOP(POP_PDL_37);
          PRINT_AUXOP(POP_PDL_38);
          PRINT_AUXOP(POP_PDL_39);
          PRINT_AUXOP(POP_PDL_40);
          PRINT_AUXOP(POP_PDL_41);
          PRINT_AUXOP(POP_PDL_42);
          PRINT_AUXOP(POP_PDL_43);
          PRINT_AUXOP(POP_PDL_44);
          PRINT_AUXOP(POP_PDL_45);
          PRINT_AUXOP(POP_PDL_46);
          PRINT_AUXOP(POP_PDL_47);
          PRINT_AUXOP(POP_PDL_48);
          PRINT_AUXOP(POP_PDL_49);
          PRINT_AUXOP(POP_PDL_50);
          PRINT_AUXOP(POP_PDL_51);
          PRINT_AUXOP(POP_PDL_52);
          PRINT_AUXOP(POP_PDL_53);
          PRINT_AUXOP(POP_PDL_54);
          PRINT_AUXOP(POP_PDL_55);
          PRINT_AUXOP(POP_PDL_56);
          PRINT_AUXOP(POP_PDL_57);
          PRINT_AUXOP(POP_PDL_58);
          PRINT_AUXOP(POP_PDL_59);
          PRINT_AUXOP(POP_PDL_60);
          PRINT_AUXOP(POP_PDL_61);
          PRINT_AUXOP(POP_PDL_62);
          PRINT_AUXOP(POP_PDL_63);
          PRINT_AUXOP(POP_PDL_64);
          PRINT_AUXOP(RETURN_0);
          PRINT_AUXOP(RETURN_1);
          PRINT_AUXOP(RETURN_2);
          PRINT_AUXOP(RETURN_3);
          PRINT_AUXOP(RETURN_4);
          PRINT_AUXOP(RETURN_5);
          PRINT_AUXOP(RETURN_6);
          PRINT_AUXOP(RETURN_7);
          PRINT_AUXOP(RETURN_8);
          PRINT_AUXOP(RETURN_9);
          PRINT_AUXOP(RETURN_10);
          PRINT_AUXOP(RETURN_11);
          PRINT_AUXOP(RETURN_12);
          PRINT_AUXOP(RETURN_13);
          PRINT_AUXOP(RETURN_14);
          PRINT_AUXOP(RETURN_15);
          PRINT_AUXOP(RETURN_16);
          PRINT_AUXOP(RETURN_17);
          PRINT_AUXOP(RETURN_18);
          PRINT_AUXOP(RETURN_19);
          PRINT_AUXOP(RETURN_20);
          PRINT_AUXOP(RETURN_21);
          PRINT_AUXOP(RETURN_22);
          PRINT_AUXOP(RETURN_23);
          PRINT_AUXOP(RETURN_24);
          PRINT_AUXOP(RETURN_25);
          PRINT_AUXOP(RETURN_26);
          PRINT_AUXOP(RETURN_27);
          PRINT_AUXOP(RETURN_28);
          PRINT_AUXOP(RETURN_29);
          PRINT_AUXOP(RETURN_30);
          PRINT_AUXOP(RETURN_31);
          PRINT_AUXOP(RETURN_32);
          PRINT_AUXOP(RETURN_33);
          PRINT_AUXOP(RETURN_34);
          PRINT_AUXOP(RETURN_35);
          PRINT_AUXOP(RETURN_36);
          PRINT_AUXOP(RETURN_37);
          PRINT_AUXOP(RETURN_38);
          PRINT_AUXOP(RETURN_39);
          PRINT_AUXOP(RETURN_40);
          PRINT_AUXOP(RETURN_41);
          PRINT_AUXOP(RETURN_42);
          PRINT_AUXOP(RETURN_43);
          PRINT_AUXOP(RETURN_44);
          PRINT_AUXOP(RETURN_45);
          PRINT_AUXOP(RETURN_46);
          PRINT_AUXOP(RETURN_47);
          PRINT_AUXOP(RETURN_48);
          PRINT_AUXOP(RETURN_49);
          PRINT_AUXOP(RETURN_50);
          PRINT_AUXOP(RETURN_51);
          PRINT_AUXOP(RETURN_52);
          PRINT_AUXOP(RETURN_53);
          PRINT_AUXOP(RETURN_54);
          PRINT_AUXOP(RETURN_55);
          PRINT_AUXOP(RETURN_56);
          PRINT_AUXOP(RETURN_57);
          PRINT_AUXOP(RETURN_58);
          PRINT_AUXOP(RETURN_59);
          PRINT_AUXOP(RETURN_60);
          PRINT_AUXOP(RETURN_61);
          PRINT_AUXOP(RETURN_62);
          PRINT_AUXOP(RETURN_63);
        }
      break;

    case DEFOP_TEST_MISC_GROUP:
      cout << " m ";
      switch(QMIMiscOpcode())
        {
        default:
          printf("miscop=#o%o", QMIMiscOpcode());
          break;
        }
      break;

    case DEFOP_TEST_MODULE_GROUP:
      cout << " d ";
      switch(QMIModuleNumber())
        {
        default:
          printf("modnum=#o%o modop=#o%o", QMIModuleNumber(), QMIModuleOpcode());
        }
      break;

    case DEFOP_TEST_AREFI:
      cout << " A ";
      switch(QMIArefiRefKind())
        {
          PRINT_AREFIOP(AREF);
          PRINT_AREFIOP(ARRAY_LEADER);
          PRINT_AREFIOP(INSTANCE_REF);
          PRINT_AREFIOP(CL_AREF);
          PRINT_AREFIOP(SETF_AREF);
          PRINT_AREFIOP(SETF_ARRAY_LEADER);
          PRINT_AREFIOP(SETF_INSTANCE_REF);
          PRINT_AREFIOP(UNUSED);
        default:
          printf("refkind=#o%o", QMIArefiRefKind());
          break;
        }
      dumpArefiOperands(os);
      break;

    case DEFOP_PUSH_MISC_GROUP:
      cout << " m.";
      switch(QMIMiscOpcode())
        {
        default:
          printf("miscop=#o%o", QMIMiscOpcode());
          break;
        }
      break;

    case DEFOP_PUSH_MODULE_GROUP:
      cout << " d.";
      switch(QMIModuleNumber())
        {
        default:
          printf("modnum=#o%o modop=#o%o", QMIModuleNumber(), QMIModuleOpcode());
        }
      break;

    case DEFOP_PUSH_AREFI:
      cout << " A.";
      switch(QMIArefiRefKind())
        {
          PRINT_AREFIOP(AREF);
          PRINT_AREFIOP(ARRAY_LEADER);
          PRINT_AREFIOP(INSTANCE_REF);
          PRINT_AREFIOP(CL_AREF);
          PRINT_AREFIOP(SETF_AREF);
          PRINT_AREFIOP(SETF_ARRAY_LEADER);
          PRINT_AREFIOP(SETF_INSTANCE_REF);
          PRINT_AREFIOP(UNUSED);
        default:
          printf("refkind=#o%o", QMIArefiRefKind());
          break;
        }
      dumpArefiOperands(os);
      break;

      PRINT_IMMEDOP_FIX(EQ_IMMED);
      PRINT_IMMEDOP_FIX(EQUALS_IMMED);
      PRINT_IMMEDOP_FIX(GREATERP_IMMED);
      PRINT_IMMEDOP_FIX(LESSP_IMMED);
      //PRINT_OP(TEST_AREFI);

      PRINT_OP(TEST);
      PRINT_OP(TEST_CAR);
      PRINT_OP(TEST_CDR);
      PRINT_OP(TEST_CADR);
      PRINT_OP(TEST_CDDR);
      PRINT_OP(TEST_CAAR);
      PRINT_OP(TEST_CDAR);
      PRINT_OP(RETURN);

      PRINT_OP(EQUALS);
      PRINT_OP(GREATERP);
      PRINT_OP(LESSP);
      PRINT_OP(EQ);
      PRINT_OP(EQL);
      PRINT_OP(EQUAL);
      PRINT_OP(EQUALP);
      // Skip OP027, default case.

      PRINT_OP(NUMBERP);
      PRINT_OP(ARRAYP);
      PRINT_OP(CLI_LISTP);
      PRINT_OP(STRINGP);
      PRINT_OP(FIXNUMP);
      PRINT_OP(INTEGERP);
      PRINT_OP(PLUSP);
      PRINT_OP(MINUSP);

      // Skip OP040, default case.
      //PRINT_OP(PUSH_MISC_GROUP);
      //PRINT_OP(PUSH_MODULE_GROUP);
      PRINT_IMMEDOP_FIX(ADD_IMMED);
      PRINT_IMMEDOP_PPSS(LDB_IMMED);
      PRINT_OP(PUSH_NUMBER);
      PRINT_OP(PUSH_NEG_NUMBER);
      //PRINT_OP(PUSH_AREFI);

      PRINT_OP(PUSH);
      PRINT_OP(PUSH_CAR);
      PRINT_OP(PUSH_CDR);
      PRINT_OP(PUSH_CADR);
      PRINT_OP(PUSH_CDDR);
      PRINT_OP(PUSH_CADDR);
      PRINT_OP(PUSH_CONS);
      PRINT_OP(PUSH_GET);

      PRINT_OP(PLUS);
      PRINT_OP(DIF);
      PRINT_OP(TIMES);
      PRINT_OP(LOGAND);
      PRINT_OP(LOGXOR);
      PRINT_OP(ONE_PLUS);
      PRINT_OP(ONE_MINUS);
      PRINT_OP(PUSH_AR_1);

      PRINT_OP(PUSH_LONG_FEF);
      PRINT_OP(SELECT);
      PRINT_OP(DISPATCH);
      // Skip OP073, default case.
      PRINT_OP(MAKE_STACK_CLOSURE);
      PRINT_OP(STACK_CLOSURE_DISCONNECT);
      PRINT_OP(STACK_CLOSURE_UNSHARE);
      PRINT_OP(STACK_CLOSURE_DISCONNECT_FIRST);

      PRINT_CALLOP(CALL_0_DEST_INDS);
      PRINT_CALLOP(CALL_0_DEST_PUSH);
      PRINT_CALLOP(CALL_0_DEST_RETURN);
      PRINT_CALLOP(CALL_0_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_1_DEST_INDS);
      PRINT_CALLOP(CALL_1_DEST_PUSH);
      PRINT_CALLOP(CALL_1_DEST_RETURN);
      PRINT_CALLOP(CALL_1_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_2_DEST_INDS);
      PRINT_CALLOP(CALL_2_DEST_PUSH);
      PRINT_CALLOP(CALL_2_DEST_RETURN);
      PRINT_CALLOP(CALL_2_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_3_DEST_INDS);
      PRINT_CALLOP(CALL_3_DEST_PUSH);
      PRINT_CALLOP(CALL_3_DEST_RETURN);
      PRINT_CALLOP(CALL_3_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_4_DEST_INDS);
      PRINT_CALLOP(CALL_4_DEST_PUSH);
      PRINT_CALLOP(CALL_4_DEST_RETURN);
      PRINT_CALLOP(CALL_4_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_5_DEST_INDS);
      PRINT_CALLOP(CALL_5_DEST_PUSH);
      PRINT_CALLOP(CALL_5_DEST_RETURN);
      PRINT_CALLOP(CALL_5_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_6_DEST_INDS);
      PRINT_CALLOP(CALL_6_DEST_PUSH);
      PRINT_CALLOP(CALL_6_DEST_RETURN);
      PRINT_CALLOP(CALL_6_DEST_TAIL_REC);
      PRINT_CALLOP(CALL_N_DEST_INDS);
      PRINT_CALLOP(CALL_N_DEST_PUSH);
      PRINT_CALLOP(CALL_N_DEST_RETURN);
      PRINT_CALLOP(CALL_N_DEST_TAIL_REC);

      PRINT_OP(POP);
      PRINT_OP(MOVEM);
      PRINT_OP(SETE_CDR);
      PRINT_OP(SETE_CDDR);
      PRINT_OP(SETE_1_PLUS);
      PRINT_OP(SETE_1_MINUS);
      // Skip OP0146, default case.
      PRINT_OP(PUSH_CDR_STORE_CAR_IF_CONS);

      PRINT_OP(PUSH_LOC);
      PRINT_OP(BIND_NIL);
      PRINT_OP(BIND_T);
      PRINT_OP(BIND_POP);
      PRINT_OP(BIND_CURRENT);
      PRINT_OP(SET_NIL);
      PRINT_OP(SET_T);
      PRINT_OP(SET_ZERO);

      PRINT_BRANCHOP(BR_NIL_ELSE_POP);
      PRINT_BRANCHOP(BR_NOT_NIL_ELSE_POP);
      PRINT_BRANCHOP(BR_NIL);
      PRINT_BRANCHOP(BR_NOT_NIL);
      PRINT_BRANCHOP(BR_ATOM);
      PRINT_BRANCHOP(BR_NOT_ATOM);
      PRINT_BRANCHOP(BR_ZEROP);
      PRINT_BRANCHOP(BR_NOT_ZEROP);

      PRINT_BRANCHOP(BR_SYMBOLP);
      PRINT_BRANCHOP(BR_NOT_SYMBOLP);
      // Skip OP0172, default case.
      // Skip OP0173, default case.
      PRINT_BRANCHOP(BR_NIL_LIKELY);
      PRINT_BRANCHOP(BR_NOT_NIL_LIKELY);
      PRINT_BRANCHOP(BR_ALWAYS_NIL);
      // Skip OP0177, default case.

    default:
      printf(" full-opcode=#o%o contents=#o%o opcode=#o%o",
             full_opcode, getBits(), QMIFullOpcode());
      break;
    }
}

void e3MacroInstruction::dumpMainOperands(ostream &os) const
{
  switch(QMIRegister())
    {
    case REG_FEF:
      printf(" FEF|#o%o", QMIOffset());
      break;
    case REG_FEF_PLUS_64:
      printf(" FEF+64|#o%o", QMIOffset());
      break;
    case REG_FEF_PLUS_128:
      printf(" FEF+128|#o%o", QMIOffset());
      break;
    case REG_HIGHER_LEXICAL_CONTEXT:
      printf(" LEX|#o%o", QMIOffset());
      break;
    case REG_SELF_MAPPING_TABLE:
      printf(" IVAR|#o%o", QMIOffset());
      break;
    case REG_LOCAL_VARIABLES:
      printf(" LOCAL|#o%o", QMIOffset());
      break;
    case REG_ARGUMENTS:
      printf(" ARG|#o%o", QMIOffset());
      break;
    case REG_PDL_POP:
      printf(" PDL|#o%o", QMIOffset());
      break;
    default:
      printf(" ????reg=#o%o, off=#o%o", QMIRegister(), QMIOffset());
      break;
    }
}

void e3MacroInstruction::dumpShortBranchOperands(ostream &os) const
{
  printf(" disp=#o%o", QMIDisplacement());
}

void e3MacroInstruction::dumpImmedOperands(ostream &os, e3MacroInstructionImmedType type) const
{
  switch(type)
    {
    case IMM_PPSS:
      // FIXME: Should decode into two separate half-bytes.
      printf(" immppss=#o%o", QMIImmedValue());
      break;
    // Default case is to treat value field as a fixnum.
    case IMM_FIXNUM:
    default:
      printf(" immfix=#o%o", QMIImmedValue());
      break;
    }
}

void e3MacroInstruction::dumpArefiOperands(ostream &os) const
{
  printf(" idx=#o%o", QMIArefiIndex());
}


// Methods for dissecting opcodes.

// %%QMI-FULL-OPCODE
e3DefOp e3MacroInstruction::QMIFullOpcode(void) const
{
  e3u16 opcode = selectBitsPPSS(011, 7);
  return((e3DefOp)opcode);
}

// %%QMI-DEST-OPCODE
e3u8 e3MacroInstruction::QMIDestOpcode(void) const
{
  e3u8 opcode = selectBitsPPSS(012, 6);
  return(opcode);
}


// %%QMI-REGISTER
e3MacroInstructionRegister e3MacroInstruction::QMIRegister(void) const
{
  e3u8 qmi_register = selectBitsPPSS(6, 3);
  return((e3MacroInstructionRegister)qmi_register);
}

// %%QMI-OFFSET
e3u8 e3MacroInstruction::QMIOffset(void) const
{
  e3u8 offset = selectBitsPPSS(0, 6);
  return(offset);
}

// %%QMI-CALL-DEST
e3u8 e3MacroInstruction::QMICallDest(void) const
{
  e3u8 dest = selectBitsPPSS(011, 2);
  return(dest);
}

e3u8 e3MacroInstruction::QMICallNumargs(void) const
{
  e3u8 numargs = selectBitsPPSS(013, 3);
  return(numargs);
}

// %%QMI-BR-OFFSET
e3u16 e3MacroInstruction::QMIDisplacement(void) const
{
  e3u16 displacement = selectBitsPPSS(0, 011);
  return(displacement);
}


// FIXME: Which %%QMI symbol?  %%QMI-INST-ADR?
e3u16 e3MacroInstruction::QMIImmedValue(void) const
{
  e3u16 value = selectBitsPPSS(0, 011);
  return(value);
}


// %%QMI-AUX-OP
e3u8 e3MacroInstruction::QMIAuxOpcode(void) const
{
  e3u8 opcode = selectBitsPPSS(0, 011);
  return(opcode);
}


e3u8 e3MacroInstruction::QMIAuxCallType(void) const
{
  e3u8 type = selectBitsPPSS(2, 2);
  return(type);
}

e3u8 e3MacroInstruction::QMIAuxCallDest(void) const
{
  e3u8 dest = selectBitsPPSS(0, 2);
  return(dest);
}


e3u8 e3MacroInstruction::QMIAuxLBTest(void) const
{
  e3u8 test = selectBitsPPSS(1, 3);
  return(test);
}

e3u8 e3MacroInstruction::QMIAuxLBSense(void) const
{
  e3u8 sense = selectBitsPPSS(0, 1);
  return(sense);
}


e3u8 e3MacroInstruction::QMIAuxCountOp(void) const
{
  e3u8 operation = selectBitsPPSS(6, 2);
  return(operation);
}

e3u8 e3MacroInstruction::QMIAuxCountCount(void) const
{
  e3u8 count = selectBitsPPSS(0, 6);
  return(count);
}


// %%QMI-MISC-OP
e3u16 e3MacroInstruction::QMIMiscOpcode(void) const
{
  e3u16 opcode = selectBitsPPSS(0, 011);
  return(opcode);
}


e3u8 e3MacroInstruction::QMIArefiRefKind(void) const
{
  e3u8 refkind = selectBitsPPSS(6, 3);
  return(refkind);
}

e3u8 e3MacroInstruction::QMIArefiIndex(void) const
{
  e3u8 index = selectBitsPPSS(0, 6);
  return(index);
}


// %%QMI-EXTERNAL-MODULE-NUMBER
e3u8 e3MacroInstruction::QMIModuleNumber(void) const
{
  e3u8 modnum = selectBitsPPSS(3, 6);
  return(modnum);
}

// %%QMI-MODULE-OP
e3u8 e3MacroInstruction::QMIModuleOpcode(void) const
{
  e3u8 opcode = selectBitsPPSS(0, 2);
  return(opcode);
}

// %%QMI-MOD-DEST
e3u8 e3MacroInstruction::QMIModuleDest(void) const
{
  e3u8 dest = selectBitsPPSS(016, 1);
  return(dest);
}


// Macroinsn execution engine.

e3MacroOpCodeEngine::e3MacroOpCodeEngine() : e3MemMap()
{
  setDebugging(e3True);

  myDTPFunction.clear();
  myErrorHandlerStackGroup.clear();
  myCurrentStackGroup.clear();
  myInitialStackGroup.clear();
  myLastArrayElementAccessed.clear();

  myFEFPC = 0;
}

e3MacroOpCodeEngine::~e3MacroOpCodeEngine()
{ ; }

e3Boolean e3MacroOpCodeEngine::setDebugging(e3Boolean new_state)
{
  e3Boolean retval = myDebuggingOn;

  myDebuggingOn = new_state;

  if (myDebuggingOn)
    {
      cout << "e3MacroOpCodeEngine::setDebugging -- debugging is ON\n";
    }

  return(retval);
}

e3Word e3MacroOpCodeEngine::fetchOperand(e3MacroInstruction i) const
{
  if (myDebuggingOn)
    {
      i.dumpMainOperands(cout);
    }
  
  e3Word     retval;
  e3WordAddr location;
  e3WordAddr offset;

  switch(i.QMIRegister())
    {
    case REG_FEF:
      location = myDTPFunction.getPointer();
      offset = i.QMIOffset();
      break;
    case REG_FEF_PLUS_64:
    case REG_FEF_PLUS_128:
    case REG_HIGHER_LEXICAL_CONTEXT:
    case REG_SELF_MAPPING_TABLE:
    case REG_LOCAL_VARIABLES:
    case REG_ARGUMENTS:
    case REG_PDL_POP:
    default:
      i.dumpMainOperands(cout);
      abort();
      break;
    }

  retval = readWordFollowingInvisiblePointers(location + offset);

  if (myDebuggingOn)
    {
      cout << "returning address " << (location+offset) << " retval " << retval << endl;
    }

  return(retval);
}

e3MacroInstruction e3MacroOpCodeEngine::currentInstruction(void) const
{
  memFEFHeader fef(myDTPFunction.getPointer());

#ifdef PARANOID
  if (myFEFPC >= fef.numberOfMacroInstructions())
    {
      abort();
      return(e3False);
    }
  else
#endif // PARANOID
    {
      e3MacroInstruction instr = fef.readMacroInstruction(fef.locationCounterOffset(), myFEFPC);
      return(instr);
    }
}

e3Boolean e3MacroOpCodeEngine::runMacroOpCode(void)
{
  cout << "myDTPFunction = " << hex << myDTPFunction << endl;

  e3MacroInstruction instr = currentInstruction();

  e3DefOp full_opcode = instr.QMIFullOpcode();

  cout << "full_opcode = #o" << oct << full_opcode << endl;

  e3MacroOpBaseClass::theExecuteDispatchTable[full_opcode](this);
  return(e3False);		// $$$$$$$$$$$
}


// Protected methods.

void e3MacroOpCodeEngine::setInitialStateFromScratchPadInitArea(void)
{
  /*
   * Get initial state out of the SCRATCH-PAD-INIT-AREA
   * These offsets are defined in qdefs.lisp
   * Thery are the SCRATCH-PAD-POINTERS
   *
   * $$$$$ Should not use hard-coded 0x400.
   * (Get out of region slot, I should think.)
   */

  setInitialTopLevelFunction(readWord(0x400 + 0));
  setErrorHandlerStackGroup(readWord(0x400 + 1));
  setCurrentStackGroup(readWord(0x400 + 2));
  setInitialStackGroup(readWord(0x400 + 3));
  setLastArrayElementAccessed(readWord(0x400 + 4));
}
  

void e3MacroOpCodeEngine::setInitialTopLevelFunction(e3Word itlf_locative)
{
  cout << "itlf_locative = " << hex << itlf_locative << endl;

  e3WordAddr initial_dtp_function_addr = itlf_locative.getPointer();
  e3Word     initial_dtp_function_q = readWord(initial_dtp_function_addr);

#ifndef NOTDEF
  cout << "initial_dtp_function_addr = " << hex << initial_dtp_function_addr << endl;
  cout << "initial_dtp_function_q = " << hex << initial_dtp_function_q << endl;
  disassembleMemory(initial_dtp_function_addr, 20);
#endif // NOTDEF

  myDTPFunction = initial_dtp_function_q; 
}

void e3MacroOpCodeEngine::setErrorHandlerStackGroup(e3Word ehsh)
{
  cout << "e3MacroOpCodeEngine::setErrorHandlerStackGroup(" << ehsh << ")\n";
  myErrorHandlerStackGroup = ehsh;
}
void e3MacroOpCodeEngine::setCurrentStackGroup(e3Word csg)
{
  cout << "e3MacroOpCodeEngine::setCurrentStackGroup(" << csg << ")\n";
  myCurrentStackGroup = csg;

#ifdef NOTDEF
  e3WordAddr csg_addr = csg.getPointer();
  disassembleMemory(csg_addr, 2);
#endif // NOTDEF
}

void e3MacroOpCodeEngine::setInitialStackGroup(e3Word isg)
{
  cout << "e3MacroOpCodeEngine::setInitialStackGroup(" << isg << ")\n";
  myInitialStackGroup = isg;

#ifdef NOTDEF
  e3WordAddr isg_addr = isg.getPointer();
  disassembleMemory(isg_addr, 2);
#endif // NOTDEF
}

void e3MacroOpCodeEngine::setLastArrayElementAccessed(e3Word laea)
{
  cout << "e3MacroOpCodeEngine::setLastArrayElementAccessed(" << laea << ")\n";
  myLastArrayElementAccessed = laea;

#ifdef NOTDEF
  /* $$$$$$$$$$ This is broken. */
  e3i32 laea_integer = laea.getFixed();
  cout << "laea_integer = " << laea_integer << endl;
#endif // NOTDEF
}


// Random debugging stuff.

void e3MacroOpCodeEngine::findNIL(void)
{
  cout << "e3LispM::findNIL -- myNumberOfWordsLoaded = " << hex << myNumberOfWordsLoaded << endl;

  for (e3WordAddr a=0; a<myNumberOfWordsLoaded; a++)
    {
      e3Word tmp = readWord(a);

#ifdef NOTDEF
      cout << a << " " << tmp << endl;
#endif
      if (tmp.getBits() == 0x804c494e)
	{
	  e3Word hdr_maybe = readWord(a-1);
	  
	  if (hdr_maybe.getBits() == 0x30489003)
	    {
	      dumpMemory(a-1, 2);
	    }
	}
    }

  cout << "Done!\n";
}

void e3MacroOpCodeEngine::dumpInterestingAreas(void)
{
  e3WordAddr symbol_for_nil = 0;
  cout << "Should be NIL\n";
  disassembleMemory(symbol_for_nil, 5);

  e3WordAddr nil_print_name_cell = readWord(symbol_for_nil).getPointer();
  disassembleMemory(nil_print_name_cell, 1);

  e3WordAddr symbol_for_t = 5;
  cout << "Should be T\n";
  disassembleMemory(symbol_for_t, 5);

  e3WordAddr t_print_name_cell = readWord(symbol_for_t).getPointer();
  disassembleMemory(t_print_name_cell, 1);

#ifdef NOTDEF
  e3Word nil_pname = readWord(0x6981a9);
  cout << "Nil name =?= " << nil_pname << endl;

  disassembleMemory(0x6981a9, 32);	// NIL is in the middle here.
  disassembleMemory(0xa0dac9, 32);	// Initial FEF?

  cout << "++++Region Origin Area?\n";
  dumpMemory(0x800, 2048);

  cout << "++++Region Length Area?\n";
  dumpMemory(0x1000, 2048);

  cout << "++++Region Bits Area?\n";
  dumpMemory(0x1800, 2048);

  cout << "++++Region Free Pointer Area?\n";
  dumpMemory(0x2000, 2048);

  cout << "++++Region GC Pointer Area?\n";
  dumpMemory(0x2800, 2048);

  cout << "++++Region List Thread Area?\n";
  dumpMemory(0x6800, 2048);

  e3u16 words_of_allocated_mem = 0;

  for (e3u16 r=0; r<2048; r++)
    {
      e3Word w = readWord(0x2000+r);
      e3u16 len = w.getPointer();
      words_of_allocated_mem += len;
    }

  cout << "words_of_allocated_mem=" << words_of_allocated_mem << endl;
#endif // NOTDEF

#ifdef NOTDEF
  dump(cout);
#endif // NOTDEF
}

void e3MacroOpCodeEngine::dumpMemory(e3WordAddr start, e3u32 len)
{
  cout << "e3LispM::dumpMemory -- dumping 0x" << len << " words starting at 0x" ;
  cout.fill('0');
  cout.width(8);
  cout << start << endl;

  for (e3WordAddr i=start; i<(start+len); i++)
    {
      e3Word tmp = readWord(i);

      cout << hex;
      cout.fill('0');
      cout.width(8);

      cout << i << ": " << tmp << endl;
    }
}

void e3MacroOpCodeEngine::disassembleMemory(e3WordAddr start, e3u32 len)
{
  /* Now, dump some words in memory. */

  cout << "e3LispM::disassembleMemory -- disassembling 0x" << len << " words starting at 0x" ;
  cout.fill('0');
  cout.width(8);
  cout << start << endl;

  for (e3WordAddr i=start; i<(start+len);)
    {
      cout << hex;
      cout.fill('0');
      cout.width(8);

      cout << i << ": ";

      memObj *o = memObj::makeRightObjType(i);

      o->dump(cout);
      i += o->sizeOfInWords();
      delete(o);
    }
}



e3Boolean e3MacroOpBaseClass::theTrace = e3True;
disassemble_signature e3MacroOpBaseClass::theDisassembleDispatchTable[theLengthOfDispatchTables];
execute_signature     e3MacroOpBaseClass::theExecuteDispatchTable[theLengthOfDispatchTables];


void e3MacroOpBaseClass::execute(e3MacroOpCodeEngine *e)
{
  cout << "e3MacroOpBaseClass::execute -- unimplemented\n";
  abort();
}

int e3MacroOpBaseClass::disassemble(e3MacroOpCodeEngine *e)
{
  cout << "e3MacroOpBaseClass::disassemble -- unimplemented\n";
  abort();
  return(0);
}


// Here lies more debugging cruft.

#ifdef NOTDEF
DEFOP(jmp, 88);

void jmp::execute(e3MacroOpCodeEngine *e)
{
  cout << "jmp::execute -- gotcha\n";
  return(2);
}

int jmp::disassemble(e3MacroOpCodeEngine *e)
{
  cout << "jmp::disassemble -- gotcha\n";
  return(2);
}

DEFOP(call, 99);

void call::execute(e3MacroOpCodeEngine *e)
{
  cout << "call::execute -- gotcha\n";
  return(2);
}

int call::disassemble(e3MacroOpCodeEngine *e)
{
  cout << "jmp::disassemble -- gotcha\n";
  return(2);
}
#endif // NOTDEF

#include "lispm.h" 

void e3MacroOpBaseClass::test(void)
{
#ifdef NOTDEF
  e3MacroOpCodeEngine *e = new e3LispM();

  for (int i=0; i<0xffff; i++)
    {
      if (e3MacroOpBaseClass::theExecuteDispatchTable[i])
	{
	  cout
	    << "theExecuteDispatchTable entry "
	    << i
	    << " = "
	    << e3MacroOpBaseClass::theExecuteDispatchTable[i]
	    << endl;
	  e3MacroOpBaseClass::theExecuteDispatchTable[i](e);
	}

      if (e3MacroOpBaseClass::theDisassembleDispatchTable[i])
	{
	  cout
	    << "theDisassembleDispatchTable entry "
	    << i
	    << " = " << e3MacroOpBaseClass::theDisassembleDispatchTable[i]
	    << endl;
	  e3MacroOpBaseClass::theDisassembleDispatchTable[i](e);
	}
    }
    
  cout << "sizeof jmp_instance = " << sizeof(jmp_instance) << endl;
  cout << "sizeof call_instance = " << sizeof(call_instance) << endl;
#endif // NOTDEF
  exit(0);
}


//
// Local Variables:
// compile-command: "make cycle"
// fill-column: 78
// End:
