/*
 * interp.c
 *
 * macrocode interpreter
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "nupi.h"
#include "cpu.h"
#include "funcall.h"
#include "utils.h"
#include "interp.h"
#include "sib.h"

#include <stdio.h>
#include <stdlib.h>

struct context context;

/* FIXME: May want to be u64? */
u32 ppss_mask(int ppss)
{
    return (1 << (ppss & 077)) - 1;
}

int ppss_position(int ppss)
{
    return (ppss >> 6) & 077;
}

int ppss_numbits(int ppss)
{
    return ppss & 077;
}

u16 get_opcode(void)
{
    u32 tmp;

    tmp = memread(context.function + (context.location_counter >> 1));

    if (context.location_counter & 1) {
	return tmp >> 16;
    } else {
	return tmp;
    }
}

void do_branch(u16 opcode)
{
    if (opcode & 0x100) {
	context.location_counter += (0xffffff00 + (opcode & 0xff));
    } else {
	context.location_counter += (opcode & 0xff);
    }
}

int fixnum_value(lisp_q fixnum)
{
    return (fixnum & 0x01000000)? fixnum | 0xfe000000: fixnum & ~0xfe000000;
}

u32 positive_integer_value(lisp_q x)
{
    lisp_q header;
    u32 val;
    
    if (DTP(x) == DTP_FIX) return fixnum_value(x);
    if (DTP(x) != DTP_EXTENDED_NUMBER) {
	printf("positive_integer_value: x not integer.\n");
	exit(-1);
    }

    header = memread(x);
    val = memread(x + 1);
    if ((header & BH_BITS_LENGTH) > 1) {
	val |= memread(x + 2) << 31;
    }

    return val;
}

int math_plus_bigfix_internal(lisp_q big1, lisp_q bigheader, int fixval)
{
    lisp_q newbig;
    u32 tmp;
    lisp_q result;
    int i;

    newbig = allocate_number_qs((bigheader & BH_BITS_LENGTH) + 1);

    memwrite(newbig, bigheader);

    for (i = 1; i <= (bigheader & BH_BITS_LENGTH); i++) {
	tmp = memread(big1 + i) + fixval;
	memwrite(newbig + i, tmp & 0x7fffffff);
	fixval = !!(tmp & 0x80000000);
	printf("%d: %08lx.\n", i, tmp);
    }

    result = DTP_EXTENDED_NUMBER | ADDRESS(newbig);
    push_cdrnext(result);
    
    return 1;
}

int math_minus_bigfix_internal(lisp_q big1, lisp_q bigheader, int fixval)
{
    lisp_q newbig;
    u32 tmp;
    lisp_q result;
    int i;

    newbig = allocate_number_qs((bigheader & BH_BITS_LENGTH) + 1);

    memwrite(newbig, bigheader);

    for (i = 1; i <= (bigheader & BH_BITS_LENGTH); i++) {
	tmp = memread(big1 + i) - fixval;
	memwrite(newbig + i, tmp & 0x7fffffff);
	fixval = !!(tmp & 0x80000000);
	printf("%d: %08lx.\n", i, tmp);
    }

    result = DTP_EXTENDED_NUMBER | ADDRESS(newbig);
    push_cdrnext(result);
    
    return 1;
}

int math_plus_bigfix(lisp_q big1, lisp_q bigheader, lisp_q fix2)
{
    long fixval;

    fixval = fixnum_value(fix2);

    /*
     * Knuth's Classical Algorithms are predicated on positive integers.
     * Our next move therefore depends on the signs of the values we're
     * passed.
     *
     * positive bignum, positive fixnum:
     *    add fix to big.
     * positive bignum, negatve fixnum:
     *    subtract abs(fix) from big.
     * negative bignum, positive fixnum:
     *    subtract fix from abs(big), then negate.
     * negative bignum, negative fixnum:
     *    add abs(fix) to abs(big), then negate.
     *
     * Of course, we only handle the double-positive case for now.
     */

    if (bigheader & BH_BITS_SIGN) {
	printf("math_plus_bigfix: unimplemented.\n");
    
	return 0;
    }

    if (fixval >= 0) {
	return math_plus_bigfix_internal(big1, bigheader, fixval);
    } else {
	return math_minus_bigfix_internal(big1, bigheader, -fixval);
    }
}

int math_plus_fixfix(lisp_q fix1, lisp_q fix2)
{
    lisp_q src1;
    lisp_q src2;
    lisp_q result;
    
    /* get fixnum values and sign extend */
    src1 = fixnum_value(fix1);
    src2 = fixnum_value(fix2);
#if 0
    src1 = ADDRESS(fix1);
    src2 = ADDRESS(fix2);
    if (src1 & 0x01000000) src1 |= 0xfe000000;
    if (src2 & 0x01000000) src2 |= 0xfe000000;
#endif
    
    result = src1 + src2;
    
    if ((result > 0x00ffffff) && (result < 0xff000000)) {
	printf("Result from + overflows FIXNUM.\n");
	/* FIXME: Allocate a BIGNUM for the result. */
	return 0;
    }
    
    result = DTP_FIX | ADDRESS(result);
    push_cdrnext(result);
    
    return 1;
}

int math_plus(lisp_q src1, lisp_q src2)
{
    lisp_q header1;
    lisp_q header2;

    header1 = HT_ERROR;
    header2 = HT_ERROR;

    if (DTP(src1) == DTP_EXTENDED_NUMBER) header1 = memread(src1);
    if (DTP(src2) == DTP_EXTENDED_NUMBER) header2 = memread(src2);

    if ((DTP(src1) == DTP_FIX) && (DTP(src2) == DTP_FIX)) {
	return math_plus_fixfix(src1, src2);
    } else if ((HDR_TYPE(header1) == HT_BIGNUM) && (DTP(src2) == DTP_FIX)) {
	return math_plus_bigfix(src1, header1, src2);
    } else if ((HDR_TYPE(header2) == HT_BIGNUM) && (DTP(src1) == DTP_FIX)) {
	return math_plus_bigfix(src2, header2, src1);
    }
    
    printf("One or both args to + is of unsupported type, failing.\n");
    dump_q(src1, 1);
    dump_q(src2, 2);
    return 0;
}

int math_minus(lisp_q src1, lisp_q src2)
{
    lisp_q result;
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to - is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }
    
    /* get fixnum values and sign extend */
    src1 = ADDRESS(src1);
    src2 = ADDRESS(src2);
    if (src1 & 0x01000000) src1 |= 0xfe000000;
    if (src2 & 0x01000000) src2 |= 0xfe000000;
    
    result = src1 - src2;
    
    if ((result > 0x00ffffff) && (result < 0xff000000)) {
	printf("Result from - overflows FIXNUM.\n");
	return 0;
    }
    
    result = DTP_FIX | ADDRESS(result);
    push_cdrnext(result);

    return 1;
}

int math_times(lisp_q src1, lisp_q src2)
{
    lisp_q result;

    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to * is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }

    /* get fixnum values and sign extend */
    src1 = ADDRESS(src1);
    src2 = ADDRESS(src2);
    if (src1 & 0x01000000) src1 |= 0xfe000000;
    if (src2 & 0x01000000) src2 |= 0xfe000000;

    /* FIXME: Better overflow checking? */
    result = src1 * src2;
	
    if ((result > 0x00ffffff) && (result < 0xff000000)) {
	printf("Result from * overflows FIXNUM.\n");
	return 0;
    }

    result = DTP_FIX | ADDRESS(result);
    push_cdrnext(result);

    return 1;
}

int ldb_generic(lisp_q val, u32 mask, int pos, int bits, lisp_q *result)
{
    if (DTP(val) == DTP_EXTENDED_NUMBER) {
	lisp_q header;

	header = memread(val);

	if ((DTP(header) == DTP_HEADER) &&
	    ((header & HDR_BITS_TYPE) == HT_BIGNUM)) {
	    /* Okay, at this point we know we're dealing with a BIGNUM */
	    int start_word;
	    u32 tmp_src;
	    u32 tmp_dst;

	    printf("ldb_generic: Value is BIGNUM.\n");
	    printf("m: %08lx p: %d b: %d.\n", mask, pos, bits);

	    start_word = pos / 31;
	    pos %= 31;
	    
	    if (start_word > (header & BH_BITS_LENGTH)) {
		if (header & BH_BITS_SIGN) {
		    *result = DTP_FIX | Q_BITS_ADDRESS;
		} else {
		    *result = DTP_FIX;
		}
		return 1;
	    }
	    
	    tmp_src = memread(val + start_word + 1);
	    tmp_dst = tmp_src >> pos;

	    if ((pos + bits) > 31) {
		if ((start_word + 1) <= (header & BH_BITS_LENGTH)) {
		    tmp_src = memread(val + start_word + 2);
		    tmp_dst |= tmp_src << (31 - pos);
		}
	    }

	    if (header & BH_BITS_SIGN) {
		printf("ldb_generic: negative BIGNUM.\n");

		tmp_dst ^= -1;

		/*
		 * FIXME: If all the bits in the BIGNUM before pos
		 * are 0, add 1 to tmp_dst.
		 */
		
		return 0;
	    }

	    *result = DTP_FIX | (tmp_dst & mask);

	    printf("ldb_generic: result 0x%08lx.\n", *result);
	    
	    return 1;
	}
    }

    if ((DTP(val) != DTP_FIX) && (DTP(val) != DTP_CHARACTER)) {
	printf("ldb_generic: non-integer value.\n");
	return 0;
    }

    *result = DTP_FIX | ((fixnum_value(val) >> pos) & mask);

    return 1;
}

lisp_q load_target(u16 opcode)
{
    int offset;

    offset = opcode & 0x3f;

    switch((opcode >> 6) & 7) {
    case 0: /* FEF|xx */
	return memread_inviz(context.function + offset);
	
    case 1: /* FEF|64+xx */
	return memread_inviz(context.function + offset + 64);
	
    case 2: /* FEF|128+xx */
	return memread_inviz(context.function + offset + 128);
	
    case 3: /* higher lexical context */
	printf("unimplemented source: higher lexical context.\n");
	exit(-1);
	
    case 4: /* SELF mapping table */
	printf("unimplemented source: SELF mapping table.\n");
	exit(-1);
	
    case 5: /* local|xx */
	return memread(context.local_pointer + offset);
	
    case 6: /* arg|xx */
	return memread(context.arg_pointer + offset);
	
    case 7: /* PDL-POP */
	return pop();
    }

    exit(-1); /* I wish GCC would just plain shut up */
}

void store_target(u16 opcode, lisp_q value)
{
    int offset;

    offset = opcode & 0x3f;
    
    switch((opcode >> 6) & 7) {
    case 0: /* FEF|xx */
	memwrite_inviz(context.function + offset, value);
	break;
	
    case 1: /* FEF|64+xx */
	memwrite_inviz(context.function + offset + 64, value);
	break;
	
    case 2: /* FEF|128+xx */
	memwrite_inviz(context.function + offset + 128, value);
	break;
	
    case 3: /* higher lexical context */
	printf("unimplemented destination: higher lexical context.\n");
	exit(-1);
	
    case 4: /* SELF mapping table */
	printf("unimplemented destination: SELF mapping table.\n");
	exit(-1);
	
    case 5: /* local|xx */
	memwrite(context.local_pointer + offset, value);
	break;
	
    case 6: /* arg|xx */
	memwrite(context.arg_pointer + offset, value);
	break;
	
    case 7: /* PDL-PUSH */
	push_cdrnext(value);
    }

    context.indicators = value;
}

lisp_q locative_target(u16 opcode)
{
    int offset;
    lisp_q retval;

    offset = opcode & 0x3f;
    retval = 0;

    switch((opcode >> 6) & 7) {
    case 0: /* FEF|xx */
	retval = context.function + offset;
	inviz(retval, &retval);
	break;
	
    case 1: /* FEF|64+xx */
	retval = context.function + offset + 64;
	inviz(retval, &retval);
	break;
	
    case 2: /* FEF|128+xx */
	retval = context.function + offset + 128;
	inviz(retval, &retval);
	break;
	
    case 3: /* higher lexical context */
	printf("unimplemented target: higher lexical context.\n");
	exit(-1);
	
    case 4: /* SELF mapping table */
	printf("unimplemented target: SELF mapping table.\n");
	exit(-1);
	
    case 5: /* local|xx */
	retval = context.local_pointer + offset;
	break;
	
    case 6: /* arg|xx */
	retval = context.arg_pointer + offset;
	break;
	
    case 7: /* PDL-POP */
	printf("meaningless target: PDL.\n");
	exit(-1);
    }

    retval = DTP_LOCATIVE | ADDRESS(retval);

    return retval;
}

void load_sg(lisp_q sg)
{
    context.current_sg = sg;
    
    context.pdl = memread(sg - SG_REGULAR_PDL);
    context.pdl_pointer = context.pdl + 3; /* FIXME: Wrong */

    context.spdl = memread(sg - SG_SPECIAL_PDL);
    context.spdl_pointer = memread(sg - SG_SPECIAL_PDL_POINTER);
}

int process_data(u8 *data, int length)
{
    lisp_q spia; /* Scratch Pad Init Area */

    memory_init((lisp_q *)data);
    nupi_init();
    cpu_init();
    sib_create(0xf5);

    spia = get_area_address(2);

    printf("scratch pad init area:\n");
    dump_q(memread(spia), spia);
    dump_q(memread(spia+1), spia+1);
    dump_q(memread(spia+2), spia+2);
    dump_q(memread(spia+3), spia+3);
    dump_q(memread(spia+4), spia+4);

#if 0
    printf("initial stack group:\n");
/*     dump_raw_array(spia[3]); */
    dump_q(memread(spia[3]-3), spia[3]-3);
    dump_q(memread(memread(spia[3]-3)), memread(spia[3]-3));
    dump_q(memread(memread(spia[3]-3)+1), memread(spia[3]-3)+1);
    dump_q(memread(memread(spia[3]-3)+2), memread(spia[3]-3)+2);
    dump_q(memread(memread(spia[3]-3)+3), memread(spia[3]-3)+3);
    dump_q(memread(memread(spia[3]-3)+4), memread(spia[3]-3)+4);
#endif

    /*
     * Load the static part of the initial stack group, set the stack
     * pointers to 0, and fake a function call to the initial function.
     * The call-info word for the function call ends up as the first
     * thing on the stack.
     *
     * The call-info word will be 0x4a001000. (no, we need to set D-MICRO)
     * Just as a quick hack, we will push 4 NILs after the call-info
     * word and fake up whatever is nessecary for the initial function.
     *
     * The call process seems to be a matter of:
     *
     *     find the call target
     *     fixup for outgoing function
     *     fixup arglist
     *     save outgoing state
     *     load incoming state
     *     fixup local variables
     */

    load_sg(memread(spia + SPIA_INITIAL_SG));

    funcall(memread(memread(spia + SPIA_INITIAL_FUNCTION)), DTP_FIX | 0x001000); /* FIXME: wrong dest */

    while(step());
    
#if 0
    printf("initial function:\n");
    dump_q(memread(spia[0]), spia[0]);
    dump_raw_fef(memread(spia[0]));
#endif

/*     dump_q(memread(01000 + 22), 0); */
    return 0;
}

lisp_q equal(lisp_q foo, lisp_q bar)
{
    if ((DTP(foo) == DTP_FIX) || (DTP(foo) == DTP_SYMBOL)) {
	return (NOT_CDRCODE(foo) == NOT_CDRCODE(bar))? C_T: C_NIL;
    } else if (((DTP(foo) == DTP_LIST) || (DTP(foo) == DTP_STACK_LIST)) &&
	       ((DTP(bar) == DTP_LIST) || (DTP(bar) == DTP_STACK_LIST))) {
	if (equal(car(foo), car(bar)) == C_T) {
	    return equal(cdr(foo), cdr(bar));
	} else {
	    return C_NIL;
	}
    } else if ((DTP(foo) == DTP_ARRAY) && (DTP(bar) == DTP_ARRAY)) {
	lisp_q foo_hdr;
	lisp_q bar_hdr;

	lisp_q foo_length;
	lisp_q bar_length;

	int i;
	int offset;
	int status;
	lisp_q locative;
	lisp_q foo_data;
	lisp_q bar_data;

	foo_hdr = memread(foo);
	bar_hdr = memread(bar);

	dump_q(foo_hdr, 0);
	dump_q(bar_hdr, 0);

	if ((DTP(foo_hdr) != DTP_ARRAY_HEADER) ||
	    (DTP(bar_hdr) != DTP_ARRAY_HEADER)) {
	    printf("equal: Array header not DTP_ARRAY_HEADER.\n");
	    exit(-1);
	}

	if ((ARY_TYPE(foo_hdr) != ART_STRING) ||
	    (ARY_TYPE(bar_hdr) != ART_STRING)) {
	    printf("equal: Array not ART_STRING.\n");
	    exit(-1);
	}

	if (ARY_LEADER(foo_hdr) &&
	    (DTP(foo_length = memread(foo - 2)) == DTP_FIX)) {
	    foo_length = fixnum_value(foo_length);
	} else {
	    foo_length = ARY_INDEX(foo_hdr);
	}
	
	if (ARY_LEADER(bar_hdr) &&
	    (DTP(bar_length = memread(bar - 2)) == DTP_FIX)) {
	    bar_length = fixnum_value(bar_length);
	} else {
	    bar_length = ARY_INDEX(bar_hdr);
	}
	
	if (foo_length != bar_length) {
	    printf("equal: Array index lengths not equal.\n");
	    exit(-1);
	}

	for (i = 0; i < foo_length; i++) {
	    push(DTP_FIX | i);
	    status = resolve_aref(foo, 1, 0, context.pdl_pointer - 1, &foo_hdr, &locative, &offset);
	    if (!status) exit(-1);
	    status = read_aref(foo_hdr, locative, offset, &foo_data, 1);
	    if (!status) exit(-1);
	    status = resolve_aref(bar, 1, 0, context.pdl_pointer - 1, &bar_hdr, &locative, &offset);
	    if (!status) exit(-1);
	    status = read_aref(bar_hdr, locative, offset, &bar_data, 1);
	    if (!status) exit(-1);
	    context.pdl_pointer--;

	    if (foo_data != bar_data) {
		printf("difference.\n");
		return C_NIL;
	    }
	}
	
	return C_T;
    }

    printf("equal: Unhandled type.\n");
    dump_q(foo, 0);
    dump_q(bar, 1);
    exit(-1);
}

/* EOF */
