/*
 * memory.c
 *
 * memory load and access functions
 */

#include "memory.h"

#include "exploiter.h"
#include "context.h"
#include "funcall.h"
#include "utils.h"
#include "amem.h"
#include "nubus.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

lisp_q *image;
lisp_q *sca;
lisp_q *area_origin_area;
lisp_q *dpmt;

lisp_q *region_origin_area;
lisp_q *region_length_area;
lisp_q *region_free_pointer_area;
lisp_q region_gc_pointer_area_address;
lisp_q *region_bits_area;
lisp_q region_list_thread_area_address;

lisp_q *area_name_area;
lisp_q *area_region_list_area;
lisp_q *area_region_bits_area;
lisp_q *area_region_size_area;
lisp_q *area_maximum_size_area;

lisp_q *support_entry_vector_area;
lisp_q *micro_code_link_area;
lisp_q *micro_code_entry_area;
lisp_q *micro_code_entry_debug_info_area;

u16 ppd_lru_head; /* least recently used page */
u16 ppd_lru_tail; /* most recently used page */

int num_wired_words;

#define PHT_VALID 0x40
/*
 * NOTE: PHT hash key and rehash constant are -byte- indices into
 * the PHT, and that PHT entries are 8 bytes wide.
 */
#define PHT_REHASH_CONSTANT 0x3c38 /* lroy-qcom: (Lisp:* 4 #x+F0E) */
#define PHT_HASH_KEY(pht0) 0 /* FIXME: Real value? */

#define CLUSTER_NUMBER(q) (((q) >> 13) & 0xfff)
#define PAGE_NUMBER(q) (((q) >> 9) & 0xf)
#define PAGE_BITS(q) ((q) & 0x01fffe00)

#if 0 /* VMM emulation not complete yet */

/* VMM status has 128K 2-bit entries. That's 256 bits. That's 32k bytes. */

u8 vmm_status[0x8000];

#define VMS_VALID 2
#define VMS_BANK  1

#define VMSB_LEFT  0
#define VMSB_RIGHT 1

int get_vmm_status(int index)
{
    u8 vmm_status_byte;

    vmm_status_byte = vmm_status[index >> 2];
    return (vmm_status_byte >> ((index & 3) << 1)) & 3;
}

void set_vmm_status(int index, int value)
{
    u8 vmm_status_byte;

    vmm_status_byte = vmm_status[index >> 2];

    vmm_status_byte &= ~(3 << ((index & 3) << 1));
    vmm_status_byte |= value << ((index & 3) << 1);

    vmm_status[index >> 2] = vmm_status_byte;
}

/* the VMM is two 16k entry 32bit maps */
u32 vmm_left_bank[0x4000];
u32 vmm_right_bank[0x4000];
u32 const *vmm_banks[2] = { vmm_left_bank, vmm_right_bank };

#define VMME_ACCESS 0x80000000

u32 mapped_read(lisp_q address)
{
    int vmm_status;
    u32 vmm_entry;
    u32 physaddr;
    
    address = ADDRESS(address);

    vmm_status = get_vmm_status((address >> 8) & 0x1ffff);

    if (!(vmm_status & VMS_VALID)) {
	/* FIXME: map miss page exception */
    }

    vmm_entry = vmm_banks[vmm_status & VMS_BANK][(address >> 8) & 0x3fff];

    if (!(vmm_entry & VMME_ACCESS)) {
	/* FIXME: page exception */
    }

    physaddr = vmm_entry << 10;
    physaddr |= (address & 0xff) << 2;

    return phys_read_u32(physaddr);
}
#endif /* VMM emulation */

#define DPMT_STATUS_A(foo) (((foo)[0] >> 29) & 7)
#define DPMT_STATUS_B(foo) (((foo)[0] >> 21) & 7)

#define DPMT_DEVICE_A(foo) (((foo)[0] >> 24) & 31)
#define DPMT_DEVICE_B(foo) (((foo)[0] >> 16) & 31)

#define DPMT_OFFSET_A(foo) (((foo)[1] >> 16) & 0xffff)
#define DPMT_OFFSET_B(foo) ( (foo)[1]        & 0xffff)

lisp_q *find_memory_location(lisp_q address)
{
    lisp_q *dpmt_entry;
    int is_device_b;
    int device_number;
    int device_status;
    int device_offset;

    if (ADDRESS(address) < num_wired_words) return image + ADDRESS(address);
    
    dpmt_entry = &dpmt[CLUSTER_NUMBER(address) << 1];
/*     printf("DPMT: %08lx %08lx.\n", dpmt_entry[0], dpmt_entry[1]); */

    /* FIXME: make sure memory is mapped */
    is_device_b = dpmt_entry[0] & (1 << PAGE_NUMBER(address));

    if (is_device_b) {
	device_status = DPMT_STATUS_B(dpmt_entry);
	device_number = DPMT_DEVICE_B(dpmt_entry);
	device_offset = DPMT_OFFSET_B(dpmt_entry);
    } else {
	device_status = DPMT_STATUS_A(dpmt_entry);
	device_number = DPMT_DEVICE_A(dpmt_entry);
	device_offset = DPMT_OFFSET_A(dpmt_entry);
	printf("find_memory_location: Device A? (%08lx, %08lx, %08lx)\n",
	       dpmt_entry[0], dpmt_entry[1], address);
	if ((dpmt_entry[0] == 0x60200000) &&
	    (dpmt_entry[1] == 0x00000000)) {
	    printf("find_memory_location: address unmapped in band.\n");
	}
	exit(-1);
    }

    /* FIXME: the code below is semantically unfaithful to the real system */
    
    if (device_status != 1) {
	printf("find_memory_location: DPMT device status %d, expected 1.\n", device_status);
	exit(-1);
    }

    if (device_number != 0) {
	printf("find_memory_location: DPMT device number %d, expected 0.\n", device_number);
	exit(-1);
    }
    
    return &(image[0x2000 * device_offset + (address & 0x1fff)]);
}

lisp_q memread_unboxed(lisp_q address)
{
    address &= Q_BITS_ADDRESS;

    if (address < num_wired_words) {
	return image[address];
    } else if (address >= VM_M_BASE) {
	return read_amem(address & 0x3ff);
#if 1
    } else if (address >= 0x01fdfc00) {
	printf("IO: 0x%05lx.\n", (address + 0x400) & 0x1ffff);
	return 0;
#endif
    }

    return *find_memory_location(address);
}

lisp_q memread(lisp_q address)
{
    /* FIXME: Read barrier goes here */
    return memread_unboxed(address);
}

void memwrite_unboxed(lisp_q address, lisp_q data)
{
    address &= Q_BITS_ADDRESS;

/*     if (address == 0x137c7c) { */
/* 	printf("STOMP!\n"); */
/* 	exit(-1); */
/*     } */

    if (address < num_wired_words) {
	image[address] = data;
	return;
    } else if (address >= VM_M_BASE) {
	write_amem(address & 0x3ff, data);
	return;
#if 1
    } else if (address >= 0x01fdfc00) {
	printf("IO: 0x%05lx = 0x%08lx.\n", (address + 0x400) & 0x1ffff, data);
	return;
#endif
    }

    *find_memory_location(address) = data;
}

void memwrite(lisp_q address, lisp_q data)
{
    /* FIXME: Write barrier goes here */
    memwrite_unboxed(address, data);
}

lisp_q inviz(lisp_q address, lisp_q *final_address)
{
    /*
     * Oh, for multiple-value-bind...
     */
    lisp_q foo;

    foo = memread(address);
    
    if (DTP(foo) == DTP_EVCP) {
	printf("INVIZ: EVCP 0x%08lx.\n", foo);
    } else if (DTP(foo) == DTP_ONE_Q_FORWARD) {
	printf("INVIZ: 1QF 0x%08lx.\n", foo);
    } else {
	if (final_address) *final_address = address;
	return foo;
    }

    return inviz(foo, final_address);
}

lisp_q memread_inviz(lisp_q address)
{
    return inviz(address, NULL);
}

void memwrite_inviz(lisp_q address, lisp_q data)
{
    lisp_q tmp;
    lisp_q tmpaddr;

    tmp = inviz(address, &tmpaddr);
    memwrite(tmpaddr, CDRCODE(tmp) | NOT_CDRCODE(data));
}

lisp_q return_barrier(lisp_q data)
{
    /* FIXME: Implement */
    return data;
}

void push(lisp_q foo)
{
    memwrite(context.pdl_pointer++, foo);
}

void push_cdrnext(lisp_q foo)
{
    push(foo | CDR_NEXT);
}

lisp_q pop(void)
{
    return memread(--context.pdl_pointer);
}

lisp_q pop_loc(void)
{
    return --context.pdl_pointer;
}

lisp_q car(lisp_q cons)
{
    lisp_q foo;
    lisp_q fooaddr;

    foo = inviz(cons, &fooaddr);
    
    if (NOT_CDRCODE(fooaddr) == C_NIL) return C_NIL;
    if ((DTP(fooaddr) != DTP_LIST) &&
	(DTP(fooaddr) != DTP_STACK_LIST)) {
	printf("car: not list or NIL.\n");
    }
    
    return foo;
}

lisp_q cdr(lisp_q cons)
{
    lisp_q foo;

    /*
     * Okay, we're passed a DTP_LIST, a DTP_STACK_LIST, or a DTP_SYMBOL.
     * If we're a symbol, it had best be NIL, for which we are NIL. Next,
     * we have to find the CDR of the list, which means checking a CDR
     * code. Is this the CDR code on the forwarding pointer? It is, isn't
     * it? We then have to clamp out any forwarding pointers on it.
     */

    if (NOT_CDRCODE(cons) == C_NIL) return C_NIL;

    foo = memread(cons);
    
    switch (CDRCODE(foo)) {
    case CDR_NIL:
	return C_NIL;

    case CDR_NEXT:
	return cons + 1;

    case CDR_NORMAL:
	return memread_inviz(cons + 1);

    case CDR_ERROR:
	printf("CDR of CDR-ERROR taken.\n");
	exit(-1);

    default:
	exit(-1); /* GCC can't see that this can never be reached. :-( */
    }
}

void spdl_push(lisp_q data)
{
    memwrite(context.spdl + context.spdl_pointer++, data);
}

lisp_q spdl_pop(void)
{
    return memread(context.spdl + --context.spdl_pointer);
}

void bind(lisp_q where, lisp_q data)
{
    if (DTP(where) != DTP_LOCATIVE) {
	printf("bind(): where not DTP_LOCATIVE.\n");
	exit(-1);
    }

    where = NOT_CDRCODE(where);
    if (!(context.call_info & CI_BINDINGBLOCK)) {
	where |= SPECPDL_BLOCK_START_FLAG;
	context.call_info |= CI_BINDINGBLOCK;
    }
    
    spdl_push(memread(where));
    spdl_push(where);

    memwrite(where, data);
}

void unbind_1(void)
{
    lisp_q where;
    lisp_q data;

    /* FIXME: What about interactions with catch and unwind-protect? */
    
    where = spdl_pop();
    data = spdl_pop();

    if (where & SPECPDL_BLOCK_START_FLAG) {
	context.call_info &= ~CI_BINDINGBLOCK;
    }

    memwrite(where, data);
}

void unbind_block(void)
{
    while (context.call_info & CI_BINDINGBLOCK) {
	unbind_1();
    }
}

void unbind_n(int n)
{
    while(n--) {
	unbind_1();
    }
}

lisp_q get_area_address(int area)
{
    return area_origin_area[area];
}

void dump_region_bits(lisp_q bits)
{
    printf("%08lx: Status: %02ld Rep: %ld O: %ld GEN: %ld U: %ld Type: %02ld S: %ld V: %ld VO: %ld C: %ld R: %ld Swap: %ld.\n", bits,
           (bits >> 20) & 0x1f, (bits >> 18) & 3,
           (bits >> 17) & 1, (bits >> 14) & 7,
           (bits >> 13) & 1, (bits >> 9) & 15,
           (bits >> 8) & 1, (bits >> 7) & 1,
           (bits >> 5) & 3, (bits >> 4) & 1,
           (bits >> 3) & 1, bits & 7);
}

void dump_region_info(int region_num)
{
    dump_q(region_origin_area[region_num], 0);
    dump_q(region_length_area[region_num], 0);
    dump_q(region_free_pointer_area[region_num], 0);
    dump_q(region_bits_area[region_num], 0);
    dump_q(memread(region_gc_pointer_area_address + region_num), 0);
    dump_q(memread(region_list_thread_area_address + region_num), 0);

    dump_region_bits(region_bits_area[region_num]);
}

#define REGIONBITS_REP(x) (((x) >> 18) & 3)
#define RB_O         0x00020000
#define RB_TYPE      0x00001e00
#define RT_NEWSPACE  0x00000400
#define RT_STATIC    0x00001200
#define RT_COPYSPACE 0x00001800

int find_newspace_region(int area_num, int nqs, int representation)
{
    /*
     * returns the region number of the first newspace region in
     * area area_num with representation type representation and
     * nqs Qs of free storage available or 0 if no such region is
     * available (allocating in the RESIDENT-SYMBOL-AREA will
     * cause this to choke).
     */

    lisp_q current_region;
    lisp_q next_region;
    lisp_q region_bits;

    next_region = ADDRESS(area_region_list_area[area_num]);

    /* Loop while we have a valid region (negative region is end of chain) */
    while (!(next_region & 0x1000000)) {
	current_region = next_region;
	
	region_bits = region_bits_area[current_region];
	next_region = ADDRESS(memread(region_list_thread_area_address + current_region));
    
	dump_region_info(current_region);

	if (REGIONBITS_REP(region_bits) != representation) continue;
#if 0 /* no GC, no oldspace exceptions, we don't care about oldspace */
	if (!(region_bits & RB_O)) continue; /* No oldspace */
#endif
	/* FIXME: Generation check */

	/* FIXME: Check to see if this is right */
	if (((region_bits & RB_TYPE) != RT_NEWSPACE) &&
	    ((region_bits & RB_TYPE) != RT_STATIC) &&
	    ((region_bits & RB_TYPE) != RT_COPYSPACE)) continue;

	/* FIXME: Do we need to check volatility? */

	if ((ADDRESS(region_length_area[current_region]) -
	     ADDRESS(region_free_pointer_area[current_region])) < nqs) continue;

	return current_region;
    }

    return 0;
}

lisp_q allocate_area_qs(lisp_q area, int nqs, int representation)
{
    int area_num;
    int region_num;
    lisp_q retval;
    
    printf("allocate_structure_qs: unimplemented (%d qs).\n", nqs);

    printf("area: ");
    if (DTP(area) == DTP_SYMBOL) {
	dump_string(memread(area + SYM_PRINTNAME));
    } else if (DTP(area) == DTP_FIX) {
	printf("%ld\n", ADDRESS(area));
    } else {
	printf("unrecognized DTP.\n");
	dump_q(area, 0);
    }

    if (NOT_CDRCODE(area) == C_NIL) {
	printf("allocate_structure_qs: AREA is NIL, assuming DEFAULT-CONS-AREA.\n");
	area = a_default_cons_area;
    }

    if ((DTP(area) == DTP_FIX) && (ADDRESS(area) < 0x100)) {
	printf("allocate_structure_qs: using area %ld.\n", ADDRESS(area));
	area_num = ADDRESS(area);
    } else {
	printf("allocate_structure_qs: Unknown area code %ld.\n", ADDRESS(area));
	exit(-1);
    }

    dump_q(area_name_area[area_num], area_num);
    dump_string(memread(area_name_area[area_num] + SYM_PRINTNAME));

    dump_q(area_region_list_area[area_num], area_num);
    
    region_num = find_newspace_region(area_num, nqs, representation);

    if (!region_num) {
	printf("allocate_structure_qs: Suitable region not found.\n");
	exit(-1);
    }

    printf("allocate_structure_qs: allocating in region %d.\n", region_num);
    
    retval = region_origin_area[region_num] +
	region_free_pointer_area[region_num];
    
    region_free_pointer_area[region_num] += nqs;

    printf("allocate_structure_qs: returning 0x%08lx.\n", retval);
    
    return retval;
}

lisp_q allocate_list_qs(lisp_q area, int nqs)
{
    return allocate_area_qs(area, nqs, 0);
}

lisp_q allocate_structure_qs(lisp_q area, int nqs)
{
    return allocate_area_qs(area, nqs, 1);
}

lisp_q allocate_number_qs(int nqs)
{
    /* FIXME: Should make with the EXTRA-PDL-AREA */
    return allocate_structure_qs(C_NIL, nqs);
}

int find_region_number(lisp_q address)
{
    int i;

    address = ADDRESS(address);
    
    for (i = 0; i < 2048; i++) {
	if ((ADDRESS(region_origin_area[i]) <= address) &&
	    (ADDRESS(region_origin_area[i] +
		     region_length_area[i]) > address)) {
	    return i;
	}
    }

    printf("find_region_number: address #x%08lx not in region.\n", address);
    exit(-1);
}

int find_region_area(int region_number)
{
    int area;
    int region;

    for (area = 0; area < 256; area++) {
	for (region = ADDRESS(area_region_list_area[area]);
	     !(region & 0x01000000);
	     region = ADDRESS(memread(region_list_thread_area_address + region))) {
	    if (region == region_number) {
		return area;
	    }
	}
    }
    
    printf("find_region_area: region %d not found.\n", region_number);
    exit(-1);
}

lisp_q find_structure_header(lisp_q pointer)
{
    int region;
    lisp_q region_bits;
    lisp_q tmp;

    region = find_region_number(pointer);
    region_bits = region_bits_area[region];

#if 0
    dump_region_info(region);
#endif

    if (REGIONBITS_REP(region_bits) != 1) {
	printf("find_structure_header: Not structure space.\n");
	exit(-1);
    }

    do {
	tmp = memread(pointer);
	if (DTP(tmp) == DTP_SYMBOL_HEADER) break;
	if (DTP(tmp) == DTP_HEADER) break;
	if (DTP(tmp) == DTP_ARRAY_HEADER) break;
	if (DTP(tmp) == DTP_INSTANCE_HEADER) break;
	if (DTP(tmp) == DTP_FEF_HEADER) break;
    } while (pointer--);

    /* FIXME: Is this really supposed to be a locative? */
    return DTP_LOCATIVE | ADDRESS(pointer);
}

lisp_q find_structure_leader(lisp_q pointer)
{
    lisp_q tmp;

    pointer = find_structure_header(pointer);

    tmp = memread(pointer);

    if ((DTP(tmp) == DTP_ARRAY_HEADER) && ARY_LEADER(tmp)) {
	tmp = memread(pointer - 1);
	pointer = DTP_LOCATIVE | (ADDRESS(pointer) - ADDRESS(tmp) - 2);

	tmp = memread(pointer);
	if (DTP(tmp) != DTP_HEADER) {
	    printf("find_structure_header: missed array header.\n");
	    dump_q(tmp, 0);
	    exit(-1);
	}
    }

    return pointer;
}

lisp_q structure_boxed_size(lisp_q pointer)
{
    printf("structure_boxed_size: unimplemented.\n");
    exit(-1);
}

lisp_q structure_total_size(lisp_q pointer)
{
    int tmp;

    tmp = memread(pointer);

    dump_q(tmp, 0);

    if (DTP(tmp) == DTP_FEF_HEADER) {
	tmp = memread(pointer+1);
	dump_q(tmp, 1);
	return tmp;
    } else if (DTP(tmp) == DTP_SYMBOL_HEADER) {
	return DTP_FIX | 5;
    }
    
    printf("structure_total_size: unimplemented.\n");
    exit(-1);
}

void create_physical_page(int pfn)
{
    printf("create_physical_page(#x%x): ignoring.\n", pfn);
}

void init_virtual_memory_system_tables(void)
{
    /*
     * PHT is 4 bytes/word * 2 words/entry * 4 entries/page * number of pages
     * in 2MB of memory.
     *
     * PPD is 4 bytes/word * 1 word/entry * 1 entry/page * number of pages in
     * 2MB of memory.
     *
     * 2MB of memory / 2KB/page = 1024 pages.
     *
     * PHT is #x8000 bytes (16 pages) long.
     * PPD is #x1000 bytes (2 pages) long.
     *
     * PHT index size is length in bits of highest PHT entry number.
     */
    
    int i;

    printf("initializing PHT and PPD tables.\n");
    
    a_pht_addr = 0xf4000000;
    a_pht_search_depth = 0;
    a_pht_index_size = 13;
    a_pht_index_limit = 0x8000;

    a_ppd_addr = 0xf4008000;
    a_ppd_end = 0x1000;

    /*
     * we set the PHT up with no valid entries (they will be added as they
     * are set up by the band loader or paged in by the fault handler).
     */

    for (i = 0; i < 4096; i++) {
	nubus_write32(0xf4, i << 3, DTP_FIX);
    }

    /*
     * we set the PPD up to reserve the first 18 pages of memory for the
     * PHT and PPD and then leave the rest available with the link fields
     * set up reasonably.
     */

    for (i = 0; i < 18; i++) {
	nubus_write32(0xf4, 0x8000 + (i << 2), 0xffffffff);
    }

    for (i = 18; i < 1023; i++) {
	nubus_write32(0xf4, 0x8000 + (i << 2), 0xfffe | ((i + 1) << 16));
    }

    nubus_write32(0xf4, 0x8ffc, 0xfffdfffe);

    ppd_lru_head = 18;
    ppd_lru_tail = 1023;
}

u16 findcore(void)
{
    u16 retval;

    /* FIXME: This breaks once all the pages have been allocated */
    /* FIXME: This should probably re-link the new page at the LRU tail */

    retval = ppd_lru_head;
    if (NR_SUCCESS != nubus_read16(a_ppd_addr >> 24, (a_ppd_addr & 0xffffff) + (retval << 2) + 2, &ppd_lru_head)) {
	printf("NuBus fault during FINDCORE.\n");
	exit(-1);
    }

    return retval;
}

u32 pfn2ppn(int pfn)
{
    /* FIXME: Implement properly (iterate over the PMM) */
    return 0xf4000000 + pfn * 2048;
}

lisp_q get_pht_entry(lisp_q address)
{
    int hash_key;
    int search_depth;
    u32 pht0, page_num;
    lisp_q retval;

    /* NOTE: hash_key is a -byte- index into the PHT for an entry */

    hash_key = PHT_HASH_KEY(address);
    search_depth = 1;
    page_num = PAGE_BITS(address);
    retval = C_NIL;

    while (search_depth <= ADDRESS(a_pht_search_depth)) {
	hash_key &= a_pht_index_limit - 1;

	/* FIXME: Check return code */
	nubus_read32(a_pht_addr >> 24, (a_pht_addr & 0xffffff) + hash_key, &pht0);

	/* stop looking if entry isn't valid */
	if (!(pht0 & PHT_VALID)) break;

	if (PAGE_BITS(pht0) == page_num) {
	    retval = pht0;
	    break;
	}

	search_depth++;
	hash_key += PHT_REHASH_CONSTANT;
    }

    return retval;
}

void load_pht_entry(u32 pht_low, u32 pht_high)
{
    int hash_key;
    int search_depth;
    u32 pht0;

    /* NOTE: hash_key is a -byte- index into the PHT for an entry */

    hash_key = PHT_HASH_KEY(pht_low);
    search_depth = 1;

    while (1) {
	hash_key &= a_pht_index_limit - 1;
	
	/* FIXME: Check return code */
	nubus_read32(a_pht_addr >> 24, (a_pht_addr & 0xffffff) + hash_key, &pht0);

	/* stop looking if entry isn't valid */
	if (!(pht0 & PHT_VALID)) break;

	/* stop looking if entry is for this page */
	if (PAGE_BITS(pht0) == PAGE_BITS(pht_low)) break;

	search_depth++;
	hash_key += PHT_REHASH_CONSTANT;
    }

    /* FIXME: Check return codes */
    nubus_write32(a_pht_addr >> 24, (a_pht_addr & 0xffffff) + hash_key, pht_low);
    nubus_write32(a_pht_addr >> 24, (a_pht_addr & 0xffffff) + hash_key + 4, pht_high);
    
    if (search_depth > ADDRESS(a_pht_search_depth)) {
	a_pht_search_depth = DTP_FIX | search_depth;
    }
}

void init_load_band(void)
{
    int i;
    int num_wired_pages;
    int page;
    u32 ppn;
    void *page_data;
    lisp_q pht_low;
    u32 pht_high;

    printf("Initializing from load band.\n");
    
    num_wired_pages = num_wired_words >> 9;

    for (i = 0; i < num_wired_pages; i++) {
	page = findcore();
	
	/* mark page as being wired */
	nubus_write32(a_ppd_addr >> 24, (a_ppd_addr & 0xffffff) + (page << 2), 0xffffffff);

	/* copy page contents from load band */
	ppn = pfn2ppn(page);
	page_data = nubus_get_memory_base(ppn >> 24, ppn & 0xffffff);
	memcpy(page_data, image + (i * 2048), 2048);

	/* Set up PHT entry */
	/* FIXME: Are we really supposed to setup PHT for the low pages? */
	pht_low = DTP_FIX | 0x45 | (i << 9); /* valid, wired */
	pht_high = page; /* FIXME: map control data? */

	load_pht_entry(pht_low, pht_high);
    }
}

lisp_q get_support_entry_vector(int index)
{
    /* should check for area bounds */
    return support_entry_vector_area[index];
}

lisp_q get_micro_code_link(int index)
{
    /* should check for area bounds */
    return micro_code_link_area[index];
}

lisp_q get_micro_code_entry(int index)
{
    /* should check for area bounds */
    return micro_code_entry_area[index];
}

lisp_q get_micro_code_entry_debug_info(int index)
{
    /* should check for area bounds */
    return micro_code_entry_debug_info_area[index];
}

void memory_init(lisp_q *data)
{
    nubus_create_memory_board(0xf4, RSB_2MEG);
    init_virtual_memory_system_tables();

    image = data;

    sca = image + 01000;
    num_wired_words = ADDRESS(sca[13]);

    a_disk_switches = DTP_FIX | 0x0c8005;
    a_inhibit_scheduling = C_NIL;
    a_slots_i_own = 0x0074;
    a_physical_resource_bitmap = PRB_DISK;
    a_default_cons_area = DTP_FIX | 27; /* FIXME: Is this right? */
    a_background_cons_area = DTP_FIX | 27; /* FIXME: Is this right? */
    m_self = C_NIL;

    a_boot_command_block = 0; /* FIXME: Is this right? */
    a_boot_keyboard = 0xf5000000; /* SIB, slot 5 -- keyboard, unit 0 */
    a_boot_device = 0xf2000000; /* NPI, slot 2 -- bootdev, unit 0 */

    init_load_band();

    sca = find_memory_location(01000);
    area_origin_area = find_memory_location(sca[0]);
    dpmt = find_memory_location(area_origin_area[9]);
    
    printf("Setting up VM system pointers.\n");
    
    /*
     * NOTE: some of the region tables are in wired memory, so we can
     * reference them directly. The area tables are all half a page in
     * length, and almost certainly aligned to the half-page, so we can
     * likewise reference them directly. The -other- region tables,
     * however, needs must be accessed the hard way.
     *
     * There should probably be some checks here to make sure that none
     * of the area tables cross a page boundary. Or at least a cluster
     * boundary. We can check this with a low-bits test on the address
     * (see if the botton N bits are less than or equal to 2^(N-1)).
     */

    region_origin_area       = find_memory_location(area_origin_area[4]);
    region_length_area       = find_memory_location(area_origin_area[5]);
    region_free_pointer_area = find_memory_location(area_origin_area[7]);
    region_gc_pointer_area_address        = ADDRESS(area_origin_area[11]);
    region_bits_area         = find_memory_location(area_origin_area[6]);
    region_list_thread_area_address       = ADDRESS(area_origin_area[12]);

    area_name_area         = find_memory_location(area_origin_area[14]);
    area_region_list_area  = find_memory_location(area_origin_area[15]);
    area_region_bits_area  = find_memory_location(area_origin_area[16]);
    area_region_size_area  = find_memory_location(area_origin_area[17]);
    area_maximum_size_area = find_memory_location(area_origin_area[18]);

    support_entry_vector_area = find_memory_location(area_origin_area[20]);
    micro_code_link_area = find_memory_location(area_origin_area[3]);
    micro_code_entry_area = find_memory_location(area_origin_area[22]);
    micro_code_entry_debug_info_area = find_memory_location(area_origin_area[23]);
    
    sca[12] = DTP_FIX | 0x80000; /* pretend we have 2 megs of memory */
    sca[28] = DTP_FIX | 0xf6; /* pretend processor is in NuBus slot 6 */
    sca[32] = DTP_FIX | (VM_M_BASE + A_PMM_BASE); /* set pointer to PMM */

#if 0
/*     dump_raw_array(0xc8129a28); */
    printf("package name: ");
    dump_string(memread(0xc8129a28 + 1));

    dump_symbol_table(0x08160000);
    exit(-1);
#endif
}

/* EOF */
