;;; -*- Mode:Common-Lisp; Package:TV; Base:8 -*- 

(DEFUN DISPLAY-FONT (FONT &KEY (WINDOW SELECTED-WINDOW) COLUMNS (LABEL-BASE 8.) (LABEL-FONT FONTS:CPTFONT)
		     (SAMPLE-FONT FONTS:CPTFONT) (HEADER-FONT FONTS:HL12B)
		     MOUSE-SENSITIVE-ITEM-TYPE RESET-CURSOR-P)
  "    Displays a table showing FONT compared to SAMPLE-FONT on WINDOW
FONT - the font to be displayed
WINDOW - the window to put the display on
COLUMNS - number of columns to use for the display. Defaults to largest power of 2 that fits.
LABEL-BASE - Base to use for printing the label numbers
HEADER-FONT - The font to use for the heading
LABEL-FONT - The font to print the label number in.
SAMPLE-FONT - The font to print first as a comparison to FONT
MOUSE-SENSITIVE-ITEM-TYPE - When non-nil, makes each character mouse sensitive with this item type
RESET-CURSOR-P - IF non-nil, puts the cursor at 0 0 instead of at the end of the displayed table"

  (LET* ((FONT-SIZE (IF (NOT (ZEROP (FONT-FILL-POINTER FONT)))
                        (FONT-FILL-POINTER FONT)
                        ;;ELSE
                        (LENGTH (FONT-CHAR-WIDTH-TABLE FONT))))
	 (SAMPLE-FONT-MAX-RASTER-WIDTH (OR (LET ((CHAR-WIDTH-TABLE (FONT-CHAR-WIDTH-TABLE SAMPLE-FONT))
					  (MAX-CHAR-WIDTH 0))
				      (AND CHAR-WIDTH-TABLE
					   (DOTIMES (CHAR (LENGTH CHAR-WIDTH-TABLE) MAX-CHAR-WIDTH)
					     (SETQ MAX-CHAR-WIDTH (MAX MAX-CHAR-WIDTH
								       (AREF CHAR-WIDTH-TABLE CHAR))))))
				    (FONT-RASTER-WIDTH SAMPLE-FONT)))
	 (FONT-MAX-RASTER-WIDTH (OR (LET ((CHAR-WIDTH-TABLE (FONT-CHAR-WIDTH-TABLE FONT))
					  (MAX-CHAR-WIDTH 0))
				      (AND CHAR-WIDTH-TABLE
					   (DOTIMES (CHAR (LENGTH CHAR-WIDTH-TABLE) MAX-CHAR-WIDTH)
					     (SETQ MAX-CHAR-WIDTH (MAX MAX-CHAR-WIDTH
								       (AREF CHAR-WIDTH-TABLE CHAR))))))
				    (FONT-RASTER-WIDTH FONT)))
         (START-X-OFFSET  50.)
         (START-Y-OFFSET 100.)
         (X-OFFSET-LINE-FUDGE -6.)
         (Y-OFFSET-LINE-FUDGE -6.)
	 (SPACING-BETWEEN-FONTS 3.)
	 (COLUMN-INCREMENT (+ (* -2 X-OFFSET-LINE-FUDGE)
			      (MAX SAMPLE-FONT-MAX-RASTER-WIDTH FONT-MAX-RASTER-WIDTH)))
         (ROW-INCREMENT    (+ (FONT-RASTER-HEIGHT SAMPLE-FONT)
			      SPACING-BETWEEN-FONTS
			      (FONT-RASTER-HEIGHT FONT)
			      (* -2 Y-OFFSET-LINE-FUDGE)))
	 (COLUMNS (LET ((COLUMNS-THAT-FIT (FLOOR (- (SHEET-WIDTH WINDOW) START-X-OFFSET)
							   COLUMN-INCREMENT)))
			      (OR (WHEN COLUMNS
				    (IF (<= COLUMNS COLUMNS-THAT-FIT)
					(IF (PLUSP COLUMNS) COLUMNS
					    ;;else
					    (CHECK-ARG COLUMNS (PLUSP COLUMNS) "a positive number" :FIXNUM)
					    (CERROR T NIL NIL "There is only room for ~d columns - ~d specified~%"
						COLUMNS-THAT-FIT COLUMNS))))
				  (IF (PLUSP COLUMNS-THAT-FIT)
				      (EXPT 2 (FLOOR (LOG COLUMNS-THAT-FIT 2))) ;find power of 2 that fits
				      (PROGN
					(FORMAT WINDOW "~%This font is too large to fit on this window")
					(RETURN-FROM DISPLAY-FONT))))))
         (ROWS (CEILING FONT-SIZE COLUMNS))
	 (ROWS-THAT-FIT (MIN ROWS
			     (FLOOR (- (SHEET-HEIGHT WINDOW) START-Y-OFFSET Y-OFFSET-LINE-FUDGE 40.)
				    ROW-INCREMENT)))
	 (ROW-INDEX 0)
         (CHARACTER 0)
         CURSOR-X CURSOR-Y)
    ;;Loop thru the font, doing one screen's worth at a time
    (DO (ROWS-TO-DO)
	((= ROW-INDEX ROWS))
      (SETQ ROWS-TO-DO (MIN ROWS-THAT-FIT (- ROWS ROW-INDEX)))
      (SEND WINDOW :SET-CURSORPOS 0 0)
      (SEND WINDOW :CLEAR-EOF)
      
      (WHEN MOUSE-SENSITIVE-ITEM-TYPE
 	(SEND WINDOW :STRING-OUT-EXPLICIT 
	      "Mouse on any character to select it"
	      START-X-OFFSET
	      (MAX (- START-Y-OFFSET 75.) (TRUNCATE START-Y-OFFSET 4))
	      999. 999.
	      HEADER-FONT
	      ALU-IOR))
	      
      ;; Write out the name of this font centered
      (SEND WINDOW :STRING-OUT-CENTERED-EXPLICIT 
	    (FORMAT NIL "~A ~:[(Continued)~]" (FONT-NAME FONT) (EQ ROW-INDEX 0))
	    START-X-OFFSET
	    (MAX (- START-Y-OFFSET 50.)
		 (TRUNCATE START-Y-OFFSET 2))
	    (MIN (SHEET-INSIDE-RIGHT WINDOW)
		 (+ START-X-OFFSET (* COLUMNS COLUMN-INCREMENT)))
	    999.
	    HEADER-FONT)

      ;;Do the Vertical lines and put the number labels above them 
      (DOTIMES (COLUMN-INDEX COLUMNS)
	;; Draw the vertical lines
	(SEND WINDOW :DRAW-LINE
	      (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMN-INDEX COLUMN-INCREMENT))
	      (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
	      (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMN-INDEX COLUMN-INCREMENT))
	      (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE (* ROWS-TO-DO ROW-INCREMENT))
	      ALU-IOR)
	
	;; Draw the number labels at the top.
	(LET* ((LABEL-PIXEL-LENGTH (* SAMPLE-FONT-MAX-RASTER-WIDTH            ;width of the label
				      (IF (ZEROP COLUMN-INDEX)
					  1
					  (1+ (FLOOR (LOG COLUMN-INDEX 8.))))))
	       (X-LABEL-OFFSET	                                           ;x distance to move to write label
		 (- (FLOOR (MAX SAMPLE-FONT-MAX-RASTER-WIDTH               ;half the width of the character
				FONT-MAX-RASTER-WIDTH) 2)
				(FLOOR LABEL-PIXEL-LENGTH 2))))	           ;half the length of the label   
	  (SEND WINDOW :STRING-OUT-EXPLICIT
		(FORMAT NIL "~VR" LABEL-BASE COLUMN-INDEX)
		(+ START-X-OFFSET (* COLUMN-INDEX COLUMN-INCREMENT) X-LABEL-OFFSET)
		(+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE  (- (FONT-RASTER-HEIGHT LABEL-FONT)))
		9999. 9999. LABEL-FONT ALU-IOR)))

      ;; Draw the rightmost vertical line.
      (SEND WINDOW :DRAW-LINE
	    (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMNS COLUMN-INCREMENT))
	    (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
	    (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMNS COLUMN-INCREMENT))
	    (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE (* ROWS-TO-DO ROW-INCREMENT))
	    ALU-IOR)

      ;; Draw the topmost horizontal line.
      (SEND WINDOW :DRAW-LINE
	    (+ START-X-OFFSET X-OFFSET-LINE-FUDGE) (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
	    (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMNS COLUMN-INCREMENT))
	    (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
	    ALU-IOR)

      ;; Loop through all of the rows in the font that can fit on one screen.
      (BLOCK OUTER
	(DOTIMES (ROW ROWS-TO-DO)
	  (INCF ROW-INDEX)
	  (PROGN
	    (SETQ CURSOR-Y (+ START-Y-OFFSET (* ROW ROW-INCREMENT)))
	    ;; Draw a horizontal line at the bottom of this row.
	    (SEND WINDOW :DRAW-LINE
		  (+ START-X-OFFSET X-OFFSET-LINE-FUDGE) (+ CURSOR-Y ROW-INCREMENT Y-OFFSET-LINE-FUDGE)
		  (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMNS COLUMN-INCREMENT))
		  (+ CURSOR-Y ROW-INCREMENT Y-OFFSET-LINE-FUDGE)
		  ALU-IOR)
	    ;; Put out the row heading.
	    (SEND WINDOW :STRING-OUT-EXPLICIT
		  (FORMAT NIL "~V,4R" LABEL-BASE CHARACTER)
		  0 CURSOR-Y 9999. 9999. LABEL-FONT ALU-IOR)
	    
	    ;; Loop through all of the columns.
	    (LOOP FOR COLUMN-INDEX FROM 0 BELOW COLUMNS
		  DO
		  (PROGN
		    (SETQ CURSOR-X (+ START-X-OFFSET (* COLUMN-INDEX COLUMN-INCREMENT)))
		    ;; Here is where we actually draw the character.
		    ;; Draw a character in the sample font above the other character.
		    (PREPARE-SHEET (WINDOW)
		      (DRAW-CHAR SAMPLE-FONT
				 (IF (< CHARACTER (FONT-FILL-POINTER SAMPLE-FONT))
				     CHARACTER
				     40)	       ;draw a space character if no sample font chars there.
				 CURSOR-X
                                 CURSOR-Y
				 ALU-IOR WINDOW)
		      (DRAW-CHAR FONT
				 CHARACTER
				 CURSOR-X
				 (+ CURSOR-Y SPACING-BETWEEN-FONTS
				    (FONT-RASTER-HEIGHT SAMPLE-FONT))
				 ALU-IOR WINDOW))
		    ;;give this character a mouse sensitive box
		    (WHEN MOUSE-SENSITIVE-ITEM-TYPE
		      (SEND WINDOW :SEND-IF-HANDLES
			    :PRIMITIVE-ITEM
			    MOUSE-SENSITIVE-ITEM-TYPE
			    (INT-CHAR CHARACTER)			      ;mouse click will return char object
			    (- CURSOR-X 1)				      ;left
			    (- CURSOR-Y 1)                                    ;top
			    (+ CURSOR-X (MAX SAMPLE-FONT-MAX-RASTER-WIDTH     ;right
					     FONT-MAX-RASTER-WIDTH) 1)	      
			    (+ CURSOR-Y SPACING-BETWEEN-FONTS		      ;bottom
			       (FONT-RASTER-HEIGHT SAMPLE-FONT)
			       (FONT-RASTER-HEIGHT FONT)
			       1)))
		    (INCF CHARACTER)
		    ;; Get out when we don't have any more characters.
		    (WHEN (>= CHARACTER FONT-SIZE)
		      (RETURN-FROM OUTER NIL)))))))

      ;;see if we need to put out a MORE message
      (WHEN (< ROW-INDEX ROWS)
	(SEND WINDOW :SET-CURSORPOS (+ START-X-OFFSET X-OFFSET-LINE-FUDGE) ;put cursor at bottom of screen
	      (- (SHEET-HEIGHT WINDOW) (FONT-RASTER-HEIGHT (sheet-current-font window)) 5))
	(SETF (W:SHEET-MORE-FLAG WINDOW) 1) ;turn on more-processing flag manually
	(LET* ((SELF WINDOW)
	       (READ-ANY (SHEET-MORE-HANDLER :read-any)))  ;put out a more message and wait for user to do something
	  (WHEN (AND (CONSP READ-ANY) MOUSE-SENSITIVE-ITEM-TYPE)
	    (SEND WINDOW :UNREAD-ANY READ-ANY)
	    (RETURN-FROM DISPLAY-FONT))))) ; If this was a mouse click, put it back in the io buffer and get out.

    ;;Move the cursor back to 0 0 if requested.  Otherwise, put it after the table.
    (SEND WINDOW :SET-CURSORPOS 0 (IF RESET-CURSOR-P
				      0
				      (+ CURSOR-Y ROW-INCREMENT)))))


;; Compute the smallest raster width needed to store the specified char
;; as defined by the specified font.
;; low-level means we are looking at one sub-character in a wide font.

(DEFUN FONT-CHAR-MIN-RASTER-WIDTH (FONT CHAR-CODE &OPTIONAL LOW-LEVEL
				   &AUX BIT-POS WORD-POS TEM MIN-RASTER-WIDTH F-RASTER-WIDTH RASTER-HEIGHT)
  (COND
    ((AND (NOT LOW-LEVEL) (SETQ TEM (FONT-INDEXING-TABLE FONT)))
     ;; If it's a wide font, go by the number of vertical stripes,
     ;; but also see how wide the rightmost stripe really needs to be.
     (LET ((START-IDX (AREF TEM CHAR-CODE))
	   (END-IDX (AREF TEM (1+ CHAR-CODE))))
       (IF (= START-IDX END-IDX)
	 0
	 (MAX 0
	      (+ (* 32. (- END-IDX START-IDX)) -32.
		 (FONT-CHAR-MIN-RASTER-WIDTH FONT (1- END-IDX) T))))))
    (T
     (SETQ WORD-POS (* CHAR-CODE (FONT-WORDS-PER-CHAR FONT))
	   BIT-POS 0
	   MIN-RASTER-WIDTH 0
	   F-RASTER-WIDTH (FONT-RASTER-WIDTH FONT)
	   RASTER-HEIGHT (FONT-RASTER-HEIGHT FONT))
     (DOTIMES (VPOS RASTER-HEIGHT)
       (AND (> (+ BIT-POS F-RASTER-WIDTH) 32.) (SETQ BIT-POS 0
						    WORD-POS (1+ WORD-POS)))
       (DO ((HPOS 0 (1+ HPOS))
	    (INDEX (+ BIT-POS (LSH WORD-POS 5)) (1+ INDEX)))
	   ((= HPOS F-RASTER-WIDTH))
	 (OR (ZEROP (AREF FONT INDEX)) (SETQ MIN-RASTER-WIDTH (MAX (1+ HPOS) MIN-RASTER-WIDTH))))
       (SETQ BIT-POS (+ F-RASTER-WIDTH BIT-POS)))
     MIN-RASTER-WIDTH))) 
