;; -*- Mode: COMMON-LISP; Fonts: MEDFNT,HL12B,HL12BI; Package: USER; Base: 10 -*-

#| This program is used to translate a resource font file from a
Macintosh into an Explorer font file.  

|#

(DEFVAR DEBUG-ENABLED NIL)
(DEFVAR DEFAULT-MAC-FILE-TYPE "MACFONT")
(DEFVAR DEFAULT-MAC-DIRECTORY 
        (FS:MAKE-PATHNAME :HOST "CEREBUS" :DIRECTORY "NICHOLS.FONTS")
  "2Default directory for reading and writing bitmap files.*")


(DEFVAR FONT-TYPE          NIL)
(DEFVAR FIRST-CHAR         NIL)
(DEFVAR LAST-CHAR          NIL)
(DEFVAR WIDTH-MAX          NIL)
(DEFVAR KERN-MAX           NIL)
(DEFVAR NEGATIVE-DESCENT   NIL)
(DEFVAR FONT-RECTANGLE-MAX NIL)
(DEFVAR CHAR-HEIGHT        NIL)
(DEFVAR OW-TABLE-LOC       NIL)
(DEFVAR ASCENT             NIL)
(DEFVAR DESCENT            NIL)
(DEFVAR LEADING            NIL)
(DEFVAR ROW-WORDS          NIL)
(DEFVAR BIT-IMAGE          NIL)
(DEFVAR OW-TABLE           NIL)
(DEFVAR LOC-TABLE          NIL)

(DEFUN WORD-FLIP (WORD &AUX FLIPPED-WORD)
  "2Takes a WORD with bits ordered 0123...14 15 and changes their order to be 15 14...3210.*"
  ;; Note that this is quite slow, but it doesn't matter since it is
  ;; only used to initialize an array.
  (SETQ FLIPPED-WORD 0)
  (DOTIMES (LOW-TO-HIGH 16.)
    (INCF FLIPPED-WORD FLIPPED-WORD)
    (IF (LOGBITP LOW-TO-HIGH WORD)
        (INCF FLIPPED-WORD)))
  FLIPPED-WORD)


(DEFVAR WORDS-FLIPPED (MAKE-ARRAY (EXPT 2 16.) :TYPE 'ART-16B)
  "This is an array of WORDS with the WORDS flipped from low-to-high to high-to-low.")


(DEFUN MAKE-WORDS-FLIPPED ()
  "Initializes the WORDS-FLIPPED array."
  (DOTIMES (INDEX (EXPT 2 16.))
    (ASET (WORD-FLIP INDEX) WORDS-FLIPPED INDEX)))

(MAKE-WORDS-FLIPPED)

(DEFVAR CURRENT-ADDRESS 0)

(DEFUN MAC (&OPTIONAL (FILENAME "GENEVA9"))
  (LET* ((PATHNAME (FS:MERGE-PATHNAME-DEFAULTS FILENAME DEFAULT-MAC-DIRECTORY DEFAULT-MAC-FILE-TYPE))
         (FILE-NAME-COMPONENT (SEND PATHNAME :NAME))
         FONT-NAME
         (FONT-COUNTER 1)
         MAX-ADDRESS)
    (WITH-OPEN-FILE (MAC-STREAM PATHNAME :DIRECTION :INPUT :CHARACTERS NIL :BYTE-SIZE 8.)
      (SETQ CURRENT-ADDRESS 0)
      ;; Skip over the resource header stuff.
      (DOTIMES (SKIP-COUNT 2)
        (READ-WORD MAC-STREAM))
      (SETQ MAX-ADDRESS (+ (* (READ-WORD MAC-STREAM) (EXPT 2 16.)) (READ-WORD MAC-STREAM)))
      ;; Hack alert!  There are 2 fonts which have less in the front of them.
      (DOTIMES (SKIP-COUNT (+ #x7E (IF (OR (STRING-EQUAL FILE-NAME-COMPONENT "MONACO")
                                           (STRING-EQUAL FILE-NAME-COMPONENT "PRINCETON"))
                                           0 2)))
        (READ-WORD MAC-STREAM))
      (LOOP UNTIL (>= CURRENT-ADDRESS MAX-ADDRESS)
            DO
            (PROGN
              ;; Read in the header information for the font.
              (FORMAT T "Translating font ~A-~D, Current address=~16R, "
                      FILE-NAME-COMPONENT FONT-COUNTER CURRENT-ADDRESS)
              (SETQ FONT-TYPE (READ-WORD MAC-STREAM))
              ;; Validate the font type.
              (WHEN (NOT (= FONT-TYPE #x9000))
                (FORMAT T "Unknown font type ~16R.  Attempting to locate it..." FONT-TYPE)
                ;; Didn't have the correct font type.  Sometimes
                ;; it is nearby.  Try reading ahead a bit to locate it.
                (UNLESS (DOTIMES (I 5000.)
                          (SETQ FONT-TYPE (READ-WORD MAC-STREAM))
                          (WHEN (= FONT-TYPE #x9000)
                            (FORMAT T "~%Font type located after skipping ~D words, continuing.~%" I)
                            (RETURN T))
                          NIL)
                  (FERROR NIL "~%Unknown font type.  Unable to resynchronize." FONT-TYPE)))
              (SETQ FIRST-CHAR         (READ-WORD MAC-STREAM)      LAST-CHAR          (READ-WORD MAC-STREAM)
                    WIDTH-MAX          (READ-WORD MAC-STREAM)      KERN-MAX           (READ-WORD MAC-STREAM)
                    NEGATIVE-DESCENT   (READ-WORD MAC-STREAM)      FONT-RECTANGLE-MAX (READ-WORD MAC-STREAM)
                    CHAR-HEIGHT        (READ-WORD MAC-STREAM)      OW-TABLE-LOC       (READ-WORD MAC-STREAM)
                    ASCENT             (READ-WORD MAC-STREAM)      DESCENT            (READ-WORD MAC-STREAM)
                    LEADING            (READ-WORD MAC-STREAM)      ROW-WORDS          (READ-WORD MAC-STREAM))
              (FORMAT T "Height=~D~%" CHAR-HEIGHT)
              ;; The character bitmap follows.  Create one that is just large enough.
              (SETQ BIT-IMAGE (TV:MAKE-SHEET-BIT-ARRAY TV:DEFAULT-SCREEN (* ROW-WORDS 16.) CHAR-HEIGHT))
              (LET ((16B-ARRAY (MAKE-ARRAY (* ROW-WORDS CHAR-HEIGHT)
                                           :TYPE 'ART-16B :DISPLACED-TO BIT-IMAGE)))
                (DOTIMES (ROW-INDEX CHAR-HEIGHT)
                  (DOTIMES (COLUMN-INDEX ROW-WORDS)
                    (ASET (AREF WORDS-FLIPPED (READ-WORD MAC-STREAM))
                          16B-ARRAY (+ (* ROW-WORDS ROW-INDEX) COLUMN-INDEX)))))
              (SETQ FONT-NAME (FORM-CHARACTERS
                                MAC-STREAM
                                (STRING-APPEND FILE-NAME-COMPONENT
                                               (FORMAT NIL "-~D" FONT-COUNTER))))
              (RECALCULATE-WIDTHS MAC-STREAM FONT-NAME)
              (INCF FONT-COUNTER)
              ;; Skip over the header stuff for the next font
              (FORMAT T "Header address=~16R, data=" CURRENT-ADDRESS)
              (LET ((OLD-DEBUG-ENABLED DEBUG-ENABLED))
                (SETQ DEBUG-ENABLED T)
                (READ-WORD MAC-STREAM)     ; Skip flag indicator (x14 or xD)
                (LOOP UNTIL (NOT (ZEROP (READ-WORD MAC-STREAM))))
                (SETQ DEBUG-ENABLED OLD-DEBUG-ENABLED))
              (FORMAT T "~%"))))
    (1- FONT-COUNTER)))

(DEFUN READ-WORD (STREAM)
  "2Read in a single word, performing byte swaps as necessary.*"
  (LET ((BYTE-1 (SEND STREAM :TYI))
        (BYTE-2 (SEND STREAM :TYI))
        TEMP)
    (SETQ TEMP (DPB BYTE-1 (BYTE 8 8) BYTE-2))
    (INCF CURRENT-ADDRESS 2)
    (WHEN DEBUG-ENABLED
      ;;(FORMAT T "(~16,2,48R ~16,2,48R)=~16,4,48R, " BYTE-1 BYTE-2 TEMP))
      (FORMAT T "~16,4,48R " TEMP))
    TEMP))

(DEFUN MULTIPLE-32 (WIDTH)
  "2Make sure that WIDTH is an even multiple of 32.*"
2   *(1+ (DPB #o37 #o0005 WIDTH)))

(DEFVAR CHARACTER-MAGNIFICATION 1)
(DEFVAR CHARACTER-MAGNIFICATION-BIT-IMAGE NIL)
(DEFUN SET-CHARACTER-MAGNIFICATION (MAGNIFICATION)
  (SETQ CHARACTER-MAGNIFICATION MAGNIFICATION)
  ;; When the magnification is 1 then we simply copy the bits
  ;; directly.  Otherwise we create a square array (suitable for
  ;; BITBLT) which we will copy to the character descriptor when
  ;; we see a 1 in the bit-image.
  (WHEN (NOT (= MAGNIFICATION 1))
    (SETQ CHARACTER-MAGNIFICATION-BIT-IMAGE (TV:MAKE-SHEET-BIT-ARRAY
                                              TV:DEFAULT-SCREEN
                                              (MULTIPLE-32 MAGNIFICATION)
                                              MAGNIFICATION
                                              :INITIAL-ELEMENT 1))))

(DEFUN FORM-CHARACTERS (STREAM FONT-STRING)
  "2Form the individual charaters given the bitmap.*"
  (LET* ((FONT-SYMBOL (INTERN FONT-STRING "FONTS"))
         ;(FONT-NAME   (EVAL FONT-SYMBOL))
         FONT-DESC
         (LAST-LOCATION 0)
         THIS-LOCATION)
    ;; Since we are defining the font, we need to create a font descriptor from scratch.
    (SETQ FONT-DESC (FED:MAKE-FONT-DESCRIPTOR :MAKE-ARRAY (:TYPE 'ART-Q :LENGTH (1+ LAST-CHAR))))
    (SETQ LAST-LOCATION (READ-WORD STREAM))
    (LOOP FOR CHARACTER FROM FIRST-CHAR TO LAST-CHAR
          DO
          (PROGN
            (SETQ THIS-LOCATION (READ-WORD STREAM))
            ;; 1Hack alert: *The following code gets around a
            ;; problem that appears to be related to VAX or Macintosh
            ;; transmission.  What happens is that a byte #x8d
            ;; is translated to #x0d.  This code checks for the
            ;; error and corrects it.
            (WHEN (AND (MINUSP (- THIS-LOCATION LAST-LOCATION)) (= (LOGAND THIS-LOCATION #xFF) #x0d))
              (SETQ THIS-LOCATION (LOGIOR #x80 THIS-LOCATION)))
            ;; Stuff the bits into a character descriptor.
            (SAVE-CHARACTER CHARACTER FONT-DESC (- THIS-LOCATION LAST-LOCATION) CHAR-HEIGHT LAST-LOCATION)
            (SETQ LAST-LOCATION THIS-LOCATION)))
    ;; The we have finished specifying all of the characters.
    ;; Store these into a font descriptor and then translate
    ;; that into the internal font representation.
    (SETF (FED:FD-FILL-POINTER   FONT-DESC) (1+ LAST-CHAR)
          (FED:FD-NAME           FONT-DESC) FONT-SYMBOL
          (FED:FD-LINE-SPACING   FONT-DESC) (* CHAR-HEIGHT CHARACTER-MAGNIFICATION)
          (FED:FD-BASELINE       FONT-DESC) (* CHAR-HEIGHT CHARACTER-MAGNIFICATION)
          (FED:FD-BLINKER-HEIGHT FONT-DESC) (* CHAR-HEIGHT CHARACTER-MAGNIFICATION)
          (FED:FD-BLINKER-WIDTH  FONT-DESC) (FED:CD-CHAR-WIDTH (AREF FONT-DESC (CHAR-INT #\W)))
          (FED:FD-SPACE-WIDTH    FONT-DESC) (FED:CD-CHAR-WIDTH (AREF FONT-DESC (CHAR-INT #\X))))
    (FED:FONT-NAME-SET-FONT-AND-DESCRIPTOR FONT-SYMBOL FONT-DESC)
    (SETF (TV:FONT-CHAR-WIDTH (EVAL FONT-SYMBOL)) (FED:FD-SPACE-WIDTH FONT-DESC))
    ;; Skip over the last entry in the location table.  We will
    ;; then be pointing to the beginning of the offset/width table.
    (READ-WORD STREAM)
    (EVAL FONT-SYMBOL)))

(DEFUN SAVE-CHARACTER (CHARACTER FONT-DESC WIDTH HEIGHT BIT-IMAGE-X-ORIGIN)
  "2Save the character away in the font descriptor.*"
  (LET ((CHAR-DESC (FED:MAKE-CHAR-DESCRIPTOR
                   :MAKE-ARRAY (:TYPE 'ART-1B
                                :LENGTH (LIST (* HEIGHT CHARACTER-MAGNIFICATION)
                                              ;; Make the width an even multiple of 32.
                                              (MULTIPLE-32 (* WIDTH CHARACTER-MAGNIFICATION))))
                   CD-CHAR-WIDTH (* WIDTH CHARACTER-MAGNIFICATION)
                   CD-CHAR-LEFT-KERN 0)))
    (IF (= CHARACTER-MAGNIFICATION 1)
        (BITBLT TV:ALU-IOR WIDTH HEIGHT BIT-IMAGE BIT-IMAGE-X-ORIGIN 0 CHAR-DESC 0 0)
        ;;1ELSE*
        (DOTIMES (Y-INDEX HEIGHT)
          (DOTIMES (X-INDEX WIDTH)
            (UNLESS (ZEROP (AREF BIT-IMAGE Y-INDEX (+ BIT-IMAGE-X-ORIGIN X-INDEX)))
              (BITBLT TV:ALU-IOR CHARACTER-MAGNIFICATION CHARACTER-MAGNIFICATION
                      CHARACTER-MAGNIFICATION-BIT-IMAGE 0 0
                      CHAR-DESC (* CHARACTER-MAGNIFICATION X-INDEX) (* CHARACTER-MAGNIFICATION Y-INDEX))))))
    (FED:FD-STORE-CD FONT-DESC CHAR-DESC CHARACTER)))


(DEFUN RECALCULATE-WIDTHS (STREAM FONT-NAME)
  "2Recalculate the widths based upon what is in the offset/width table*"
  (LET (OFFSET-WIDTH-ENTRY
        (WIDTH-TABLE (TV:FONT-CHAR-WIDTH-TABLE FONT-NAME)))
    (LOOP FOR CHARACTER FROM FIRST-CHAR TO LAST-CHAR
        DO
        (PROGN
          (SETQ OFFSET-WIDTH-ENTRY (READ-WORD STREAM))
          (ASET (IF (= OFFSET-WIDTH-ENTRY #xFFFF)
                    0
                    ;;1ELSE*
                    (* (LOGAND #xFF OFFSET-WIDTH-ENTRY) CHARACTER-MAGNIFICATION))
                WIDTH-TABLE CHARACTER)))))

(DEFUN DISPLAY-FONT (FONT &OPTIONAL (WINDOW TV:SELECTED-WINDOW) (CHARACTERS-PER-COLUMN 32.))
  (UNLESS WINDOW
    (SETQ WINDOW TV:SELECTED-WINDOW))
  (SEND WINDOW :SELECT)
  (SEND WINDOW :SET-CURSORPOS 0 0)
  (SEND WINDOW :CLEAR-EOF)
  (LET* ((FONT-SIZE (IF (NOT (ZEROP (LENGTH FONT)))
                        (LENGTH FONT)
                        ;;1ELSE*
                        (LENGTH (TV:FONT-CHAR-WIDTH-TABLE FONT))))
         (LABEL-FONT  FONTS:CPTFONT)
         (SAMPLE-FONT FONTS:CPTFONT)
         (START-X-OFFSET  50.)
         (START-Y-OFFSET 100.)
         (X-OFFSET-LINE-FUDGE -10.)
         (Y-OFFSET-LINE-FUDGE -10.)
         (CHARACTERS-PER-ROW (CEILING FONT-SIZE CHARACTERS-PER-COLUMN))
         (COLUMN-INCREMENT (TRUNCATE (- (TV:SHEET-WIDTH  WINDOW) START-X-OFFSET)
                                     (* 1.05 CHARACTERS-PER-COLUMN)))
         (ROW-INCREMENT    (TRUNCATE (- (TV:SHEET-HEIGHT WINDOW) START-Y-OFFSET)
                                     (* 1.1 CHARACTERS-PER-ROW)))
         (CHARACTER 0)
         CURSOR-X CURSOR-Y)
    ;; Write out the name of this font
    (SEND WINDOW :STRING-OUT-CENTERED-EXPLICIT 
          (FORMAT NIL "~A" (TV:FONT-NAME FONT))
          (TV:SHEET-INSIDE-LEFT WINDOW) (MAX (- START-Y-OFFSET 50.) (TRUNCATE START-Y-OFFSET 2))
          (TV:SHEET-INSIDE-RIGHT WINDOW) 999. FONTS:MEDFNB)
    ;; Draw the label at the top.  Draw the vertical lines too.
    (DOTIMES (COLUMN-INDEX CHARACTERS-PER-COLUMN)
      (SEND WINDOW :DRAW-LINE
            (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMN-INDEX COLUMN-INCREMENT))
            (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
            (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* COLUMN-INDEX COLUMN-INCREMENT))
            (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE (* CHARACTERS-PER-ROW ROW-INCREMENT))
            TV:ALU-IOR)
      (SEND WINDOW :STRING-OUT-EXPLICIT
            (FORMAT NIL "~X" COLUMN-INDEX)
            (+ START-X-OFFSET (* COLUMN-INDEX COLUMN-INCREMENT))
            (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE  (- (TV:FONT-CHAR-HEIGHT LABEL-FONT)))
            9999. 9999. LABEL-FONT TV:ALU-IOR))
    ;; Draw the rightmost vertical line.
    (SEND WINDOW :DRAW-LINE
          (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* CHARACTERS-PER-COLUMN COLUMN-INCREMENT))
          (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
          (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* CHARACTERS-PER-COLUMN COLUMN-INCREMENT))
          (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE (* CHARACTERS-PER-ROW ROW-INCREMENT))
          TV:ALU-IOR)
    ;; Draw the topmost horizontal line.
    (SEND WINDOW :DRAW-LINE
          (+ START-X-OFFSET X-OFFSET-LINE-FUDGE) (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
          (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* CHARACTERS-PER-COLUMN COLUMN-INCREMENT))
          (+ START-Y-OFFSET Y-OFFSET-LINE-FUDGE)
          TV:ALU-IOR)
    ;; Loop through all of the rows in the font.
    (LOOP NAMED OUTER
          FOR ROW-INDEX FROM 0 BELOW CHARACTERS-PER-ROW
          DO
          (PROGN
            (SETQ CURSOR-Y (+ START-Y-OFFSET (* ROW-INDEX ROW-INCREMENT)))
            ;; Draw a horizontal line at the bottom of this row.
            (SEND WINDOW :DRAW-LINE
                  (+ START-X-OFFSET X-OFFSET-LINE-FUDGE) (+ CURSOR-Y ROW-INCREMENT Y-OFFSET-LINE-FUDGE)
                  (+ START-X-OFFSET X-OFFSET-LINE-FUDGE (* CHARACTERS-PER-COLUMN COLUMN-INCREMENT))
                  (+ CURSOR-Y ROW-INCREMENT Y-OFFSET-LINE-FUDGE)
                  TV:ALU-IOR)
            ;; Put out the row heading.
            (SEND WINDOW :STRING-OUT-EXPLICIT (FORMAT NIL "~16,4R " CHARACTER)
                  0 CURSOR-Y 9999. 9999. LABEL-FONT TV:ALU-IOR)
            ;; Loop through all of the columns.
            (LOOP FOR COLUMN-INDEX FROM 0 BELOW CHARACTERS-PER-COLUMN
                  DO
                  (PROGN
                    (SETQ CURSOR-X (+ START-X-OFFSET (* COLUMN-INDEX COLUMN-INCREMENT)))
                    ;; Here is where we actually draw the character.
                    (TV:PREPARE-SHEET (WINDOW)
                      ;; Draw a character in the sample font above the other character.
                      (TV:DRAW-CHAR SAMPLE-FONT (IF (>= CHARACTER (LENGTH SAMPLE-FONT))
                                                    (MOD CHARACTER (LENGTH SAMPLE-FONT))
                                                      ;;1ELSE*
                                                      CHARACTER)
                                    CURSOR-X CURSOR-Y
                                    TV:ALU-IOR WINDOW)
                      (TV:DRAW-CHAR FONT CHARACTER CURSOR-X (+ CURSOR-Y 3 (TV:FONT-CHAR-HEIGHT SAMPLE-FONT))
                                    TV:ALU-IOR WINDOW)))
                  (INCF CHARACTER)
                  ;; Get out when we don't have any more characters.
                  (WHEN (>= CHARACTER FONT-SIZE)
                    (RETURN-FROM OUTER NIL))))))
    (SEND WINDOW :SET-CURSORPOS 0 0))

(DEFUN MAC-AND-DISPLAY (FILENAME)
  (DOTIMES (FONT-INDEX (MAC FILENAME))
    (SEND TV:SELECTED-WINDOW :TYI)
    (DISPLAY-FONT (EVAL (READ-FROM-STRING (FORMAT NIL "FONTS:~A-~D"
                                                  (SEND (FS:PARSE-PATHNAME FILENAME) :NAME)
                                                  (1+ FONT-INDEX)))))))

(DEFUN MAC-AND-PRINT (FILENAME &OPTIONAL (ASK T))
  (LET ((OLD-IMAGE-PRINTER SI:*DEFAULT-SCREEN-IMAGE-PRINTER*)
        CH)
    (SETQ SI:*DEFAULT-SCREEN-IMAGE-PRINTER* "TALARIS")
    (DOTIMES (FONT-INDEX (MAC FILENAME))
      (WHEN ASK
        (SEND TV:SELECTED-WINDOW :STRING-OUT "Press any character to display the next font chart.")
        (SEND TV:SELECTED-WINDOW :TYI))

      (DISPLAY-FONT (EVAL (READ-FROM-STRING (FORMAT NIL "FONTS:~A-~D"
                                                    (SEND (FS:PARSE-PATHNAME FILENAME) :NAME)
                                                    (1+ FONT-INDEX)))))
      
      (WHEN (PROGN
              (WHEN ASK
                (SEND TV:SELECTED-WINDOW :SET-CURSORPOS 0 0)
                (SEND TV:SELECTED-WINDOW :STRING-OUT "Print this font chart?")
                (SETQ CH (CHAR-INT (TYI)))
                (SEND TV:SELECTED-WINDOW :SET-CURSORPOS 0 0)
                (SEND TV:SELECTED-WINDOW :CLEAR-EOL))
              (OR (NOT ASK)
                  (EQ CH (CHAR-INT #\Y))
                  (EQ CH (CHAR-INT #\SPACE))))
        (MULTIPLE-VALUE-BIND (LEFT TOP RIGHT BOTTOM)
            (SEND TV:SELECTED-WINDOW :INSIDE-EDGES) 
          (TV:PREPARE-SHEET (TV:SELECTED-WINDOW)
            (TV:MOUSE-WARP 0 0)
            (PRINT-BITMAP TV:SELECTED-WINDOW :START-X LEFT :START-Y TOP
                          :TO-FILE "V4:V4UD:[EKBERG.JUNK.TRASH]BITMAP"
                          :WIDTH (- RIGHT LEFT) :HEIGHT (- BOTTOM TOP))))))
    (SETQ SI:*DEFAULT-SCREEN-IMAGE-PRINTER* OLD-IMAGE-PRINTER)))

(DEFUN MAC-AND-PRINT-LIST (FILENAME-LIST)
  (DOLIST (FILENAME FILENAME-LIST)
    (MAC-AND-PRINT FILENAME NIL)))

#|

(MAC-AND-PRINT-LIST
  '(
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]ALEXIS.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]ASCII12.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]ATHENS.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BELLEVUE.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BOISE.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BOSTON_FONT.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BROADWAY.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BROADWAY24.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BUBBLES.MACFONT"
  ;;  "V4:EXP:[EXPSRC.PASCAL.MACFONTS]BUBBLES_2.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]CAIRO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]CAIRO18.MACFONT"
    "M5:EKBERG.MAC-FONTS;CARMEL.MACFONT"
    "M5:EKBERG.MAC-FONTS;CHICAGO_BY_NIGHT.MACFONT"
    "M5:EKBERG.MAC-FONTS;CHIC_MATH.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]CUPERTINO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]EAST_ORANGE.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]ELVISH12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]GENEVA.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]GEN_MATH.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]GREEK12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]HOLLYWOOD.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LASER_WRITER_FONTS.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LONDON.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LONDON_2.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LONG_ISLAND.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LOS_ANGELES.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LOS_ANGELES_2.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LOTHLORIEN.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]MOSCOW.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]MOSCOW12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]MOSEISLEY.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]NEWYORK.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]NEWYORK36.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]NEW_YORK36.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]OPHIR.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]PALO_ALTO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]PICA12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]RAVENNA.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]RIO72PT.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SANTIAGO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SANTIAGO12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SAN_FRANCISCO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SAN_FRANCISCO18.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SEATTLE.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SILICON_BEACH.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SILICON_BEACH_FONT.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SILICON_VALLEY12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]STENCIL.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]STUTTGART.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SYSTEM.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]TALIESIN18.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]TAMPICO72.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]TORONTO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]VECTORS.MACFONT"))

|#

(MAC-AND-PRINT-LIST
  '(
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]ELVISH12.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]LOTHLORIEN.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]MOSEISLEY.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]OPHIR.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SAN_FRANCISCO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]SAN_FRANCISCO18.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]TALIESIN18.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]TORONTO.MACFONT"
    "V4:EXP:[EXPSRC.PASCAL.MACFONTS]VECTORS.MACFONT"))
