;;; -*- Package: USER; Mode: LISP; Base: 10.; Fonts:(CPTFONT HL12B HL12BI) -*-

1;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;*		1               UTIL-MATH
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;

#|Documentation:

2     This file contains miscellaneous useful math functions.*

End documentation|# 

;;;Functions contained in this file:
;;;       cabs
;;;       c+
;;;       c*
;;;       sqr
;;;       amod
;;;       star
;;;       vector-avg
;;;       vector-diff
;;;       vector-length
;;;       vector-sum
;;;       apl-add
;;;       apl-average
;;;       apl-divide
;;;       apl-equalp
;;;       apl-greater-or-equalp
;;;       apl-greaterp
;;;       apl-inner-product
;;;       apl-lesser-or-equalp
;;;       apl-lessp
;;;       apl-max
;;;       apl-min
;;;       apl-multiply
;;;       apl-sort
;;;       apl-subtract
;;;       array-compress-columns
;;;       array-compress-rows
;;;       delete-vector
;;;       join-columns
;;;       join-rows
;;;       transpose*

(DEFMACRO sqr (x)
  `(* ,x ,x))

(DEFUN c+ (a b)
  (LET ((ar (REALPART a))
        (ai (IMAGPART a))
        (br (REALPART b))
        (bi (IMAGPART b)))
    (COMPLEX (+ ar br) (+ ai bi))))

(DEFUN c* (a b)
  (LET ((ar (REALPART a))
        (ai (IMAGPART a))
        (br (REALPART b))
        (bi (IMAGPART b)))
    (COMPLEX (- (* ar ai)
                (* br bi))
             (+ (* ar bi)
                (* ai br)))))

(DEFUN cabs (cnum)
  (LET ((x (REALPART cnum))
        (y (IMAGPART cnum)))
    (SQRT (+ (* x x) (* y y)))))

(DEFMACRO amod (x y)
  2"Performs the FORTRAN AMOD function."*
  `(- ,x (* ,y (// (FIX ,x) (FIX ,y)))))

(defmacro star (arg)
  `(circular-list ,arg))

(DEFUN VECTOR-AVG (VEC-LIST)
  "2VEC-LIST is a list of (x y) coordinate pairs. Returns the average.*"
  (LIST (FC-AVERAGE (STAR #'CAR) VEC-LIST)
        (FC-AVERAGE (STAR #'CADR) VEC-LIST))) 

(DEFUN VECTOR-DIFF (MINUEND SUBTRAHEND)
  "2Returns the (x y) coordinate vector difference.*"
  (LIST (DIFFERENCE (CAR MINUEND) (CAR SUBTRAHEND))
        (DIFFERENCE (CADR MINUEND) (CADR SUBTRAHEND)))) 

(DEFUN VECTOR-LENGTH (VEC)
  "2Length of (x y) coordinate vector VEC.*"
  (SQRT (PLUS (TIMES (FLOAT (CAR VEC)) (FLOAT (CAR VEC)))
                  (TIMES (FLOAT (CADR VEC)) (FLOAT (CADR VEC)))))) 

(DEFUN VECTOR-SUM (AD1 AD2)
  "2Sum of two (x y) coordinate vectors.*"
  (LIST (PLUS (CAR AD1) (CAR AD2)) (PLUS (CADR AD1) (CADR AD2)))) 

;**********************************************************************
;Available APL functions
;**********************************************************************

(defun get-operand-types (operator)
  (selectq operator
    (apl-min '((vector vector)))
    (apl-max '((vector vector)))
    (apl-average '((vector vector)))
    (apl-add '((scalar scalar vector array) (vector scalar vector) (array scalar array)))
    (apl-subtract '((scalar scalar vector array) (vector scalar vector) (array scalar array)))
    (apl-multiply '((scalar scalar vector array) (vector scalar vector) (array scalar array)))
    (apl-divide '((scalar scalar vector array) (vector scalar vector) (array scalar array)))
    (apl-inner-product '((vector vector)))
    (apl-sort '((vector) (array vector)))
    (delete-vector '((vector)))
    (transpose '((array)))
    (join-rows '((array array)))
    (join-columns '((array array)))
    (x-y-graph '((vector vector)))
    (apl-equalp '((scalar atom scalar vector array)
		  (vector atom scalar vector)
		  (array atom scalar array)
		  (atom atom scalar vector array)))
    (apl-lessp '((scalar atom scalar vector array)
		  (vector atom scalar vector)
		  (array atom scalar array)
		  (atom atom scalar vector array)))
    (apl-greaterp '((scalar atom scalar vector array)
		     (vector atom scalar vector)
		     (array atom scalar array)
		     (atom atom scalar vector array)))
    (apl-lesser-or-equalp '((scalar atom scalar vector array)
			     (vector atom scalar vector)
			     (array atom scalar array)
			     (atom atom scalar vector array)))
    (apl-greater-or-equalp '((scalar atom scalar vector array)
			      (vector atom scalar vector)
			      (array atom scalar array)
			      (atom atom scalar vector array)))
    (array-compress-rows '((array vector)))
    (array-compress-columns '((array vector)))
    (show-table '((scalar) (atom) (vector) (array)))
    (hide-table '((scalar) (atom) (vector) (array)))
    (label-table '((scalar) (atom) (vector) (array)))))

;GET-N-ARGS returns (Monadic), (dyadic) or (monadic dyadic) depending  on
;whether the OPERATOR  takes one  argument only,  two arguments  only, or
;either one argument or two arguments.

(defun get-n-args (operator)
  (selectq operator
    (apl-max '(monadic))
    (apl-min '(monadic))
    (apl-average '(monadic))
    (apl-add '(monadic dyadic))
    (apl-subtract '(dyadic))
    (apl-multiply '(monadic dyadic))
    (apl-divide '(dyadic))
    (apl-inner-product '(dyadic))
    (apl-sort '(monadic dyadic))
    (delete-vector '(monadic))
    (transpose '(monadic))
    (join-rows '(dyadic))
    (join-columns '(dyadic))
    (x-y-graph '(dyadic))
    (apl-equalp '(dyadic))
    (apl-lessp '(dyadic))
    (apl-greaterp '(dyadic))
    (apl-lesser-or-equalp '(dyadic))
    (apl-greater-or-equalp '(dyadic))
    (array-compress-rows '(dyadic))
    (array-compress-columns '(dyadic))
    (show-table '(monadic))
    (hide-table '(monadic))
    (label-table '(monadic))))

;APL-MAX finds the maximum element in a vector.

(defun apl-max ()
  (apply (function max) *first-arg-normal*))
  
;APL-MIN finds the minimum element in a vector.

(defun apl-min ()
  (apply (function min) *first-arg-normal*))

;APL-AVERAGE finds the average of the elements in a vector.

(defun apl-average ()
  (// (apply (function +) *first-arg-normal*) (float (length *first-arg-normal*))))


;APL-ADD supports addition and subtraction of scalars to either  scalars,
;vectors, or arrays; for vectors to  vectors and scalars; and for  arrays
;to either scalars or vectors.

(defun apl-add (&optional subtract)
  (if (not (null *second-arg*))
;dyadic functions:
      (selectq (first *first-arg*)  ;find the type of the first arg
	(scalar
	 (selectq (first *second-arg*) ;find the type of the second arg
	   (scalar (if subtract
		       (funcall (function -) *first-arg-normal* *second-arg-normal*)
		     (funcall (function +) *first-arg-normal* *second-arg-normal*)))
	   ((row column)
	    (apl-scalar-to-vector-add
	      *first-arg-normal* *second-arg-normal* nil subtract))
	   (array
	    (apl-scalar-to-array-add *first-arg-normal* *second-arg-normal* nil subtract))))
	((row column)
	 (selectq (first *second-arg*) ;find the type of the second arg
	   (scalar
	    (apl-scalar-to-vector-add *second-arg-normal* *first-arg-normal* subtract nil))
	   ((row column)
	    (apl-vector-to-vector-add *first-arg-normal* *second-arg-normal* subtract))
	   (array
	    (format *comment-window*
		    "You cannot add a vector to an array.~%"))))
	(array
	 (selectq (first *second-arg*) ;find the type of the second arg
	   (scalar
	    (apl-scalar-to-array-add *second-arg-normal* *first-arg-normal* subtract nil))
	   ((row column) (format *comment-window* "You cannot add an array to a vector.~%"))
	   (array (apl-array-to-array-add *first-arg-normal* *second-arg-normal* subtract))))
	(atom (format *comment-window* "Only numbers can be added together.~%") nil))
;monadic functions:
    (selectq (first *first-arg*) ;find the type of the first arg
      (atom (format *comment-window* "Only numbers can be positive.~%") nil)
      (scalar (if subtract (- *first-arg-normal*) *first-arg-normal*))
      ((row column) (apl-vector-sum *first-arg-normal* subtract))
      (array (apl-array-sum *first-arg-normal* subtract)))))


;APL-VECTOR-SUM computes the sum of the elements of a vector:

(defun apl-vector-sum (vector subtract)
  (if subtract (apply (function -) vector) (apply (function +) vector)))

;APL-VECTOR-PRODUCT computes the product of the elements of a vector:

(defun apl-vector-multiply (vector) (apply (function *) vector))

;APL-ARRAY-SUM computes the sum of the elements of an array:

(defun apl-array-sum (array subtract)
  (if subtract (apply (function -) (listarray array))
    (apply (function +) (listarray array))))

;APL-ARRAY-MULTIPLY computes the product of the elements of an array:

(defun apl-array-multiply (array) (apply (function *) (listarray array)))

;APL-SCALAR-TO-VECTOR-ADD adds a scalar to each element of a vector, returning the new vector:

(local-declare ((special scalar))

(defun apl-scalar-to-vector-add (scalar vector subtract-scalar subtract-vector)
  (cond (subtract-scalar
	   (mapcar (function (lambda (element) (- element scalar))) vector))
	(subtract-vector
	   (mapcar (function (lambda (element) (- scalar element))) vector))
	(t (mapcar (function (lambda (element) (+ element scalar))) vector))))
)

;APL-VECTOR-TO-VECTOR-ADD adds two vectors together:

(defun apl-vector-to-vector-add (vector1 vector2 subtract)
  (if subtract
      (mapcar (function (lambda (element1 element2) (- element1 element2)))
	      vector1 vector2)
    (mapcar (function (lambda (element1 element2) (+ element1 element2))) vector1 vector2)))

;APL-ARRAY-TO-ARRAY-ADD adds two arrays together, returning a new array:

(defun apl-array-to-array-add (array1 array2 subtract &aux dimensions)
  (if (equal (setq dimensions (array-dimensions array1)) (array-dimensions array2))
      (do ((answer (make-array dimensions))
	   (rows (first dimensions))
	   (columns (or (second dimensions) 1))
	   (row 0 (1+ row)))
	  (( row rows) answer)
	(do ((column 0 (1+ column)))
	    (( column columns))
	  (aset
	    (if subtract (- (aref array1 row column) (aref array2 row column))
	      (+ (aref array1 row column) (aref array2 row column)))
	    answer row column)))
    (format *comment-window*
	    "The two arrays have different dimensions and cannot be added.~%")))


;APL-SCALAR-TO-ARRAY-ADD adds a scalar to each element of an array, returning a new array:

(defun apl-scalar-to-array-add (scalar array subtract-scalar subtract-array
				&aux (dimensions (array-dimensions array)))
  (do ((answer (make-array dimensions))
       (rows (first dimensions))
       (columns (or (second dimensions) 1))
       (row 0 (1+ row)))
      (( row rows) answer)
    (do ((column 0 (1+ column)))
	(( column columns))
      (cond (subtract-scalar
	       (aset (- (aref array row column) scalar) answer row column))
	    (subtract-array
	       (aset (- scalar (aref array row column)) answer row column))
	    (t (aset (+ (aref array row column) scalar) answer row column))))))

;APL-SUBTRACT performs subtraction by adding the negative of the second argument

(defun apl-subtract () (apl-add t))

;APL-MULTIPLY supports multiplication of  scalars by vectors  and arrays;
;multiplication of vectors by scalars and vectors; and multiplication  of
;arrays by scalars and by other arrays (matrix multiplication).  Division
;is similarly supported, except division  of one array by  another is not
;allowed.

(defun apl-multiply (&optional divide)
  (if *second-arg*
      (selectq (first *first-arg*)		;get type of first arg
	(scalar
	 (selectq (first *second-arg*)		;get type of second arg
	   (scalar
	    (if divide
		(// *first-arg-normal* (float *second-arg-normal*))
	      (* *first-arg-normal* *second-arg-normal*)))
	   ((row column)
	    (apl-scalar-to-vector-multiply *first-arg-normal* *second-arg-normal* nil divide))
	   (array
	    (apl-scalar-to-array-multiply
	      *first-arg-normal* *second-arg-normal* nil divide))))
	((row column)
	 (selectq (first *second-arg*)
	   (scalar
	    (apl-scalar-to-vector-multiply *second-arg-normal* *first-arg-normal* divide nil))
	   ((row column)
	    (apl-vector-to-vector-multiply *first-arg-normal* *second-arg-normal* divide))
	   (array (format *comment-window* "You cannot multiply an array by a vector.~%")
		  nil)))
	(array
	 (selectq (first *second-arg*)
	   (scalar
	    (apl-scalar-to-array-multiply *second-arg-normal* *first-arg-normal* divide nil))
	   ((row column)
	    (format *comment-window* "You cannot multiply an array by a vector.~%")
	    nil)
	   (t (if divide
		  (progn (format *comment-window* "You cannot divide one array by another.~%")
			 nil)
		(apl-matrix-multiply))))))
    (selectq (first *first-arg*)
      (scalar *first-arg-normal*)
      ((row column) (apl-vector-multiply *first-arg-normal*))
      (array (apl-array-multiply *first-arg-normal*)))))

;APL-SCALAR-TO-VECTOR-MULTIPLY multiplies a scalar and a vector

(local-declare ((special scalar))

(defun apl-scalar-to-vector-multiply (scalar vector divide-scalar divide-vector)
  (cond (divide-scalar
	 (mapcar (function (lambda (element) (// element (float scalar)))) vector))
	(divide-vector
	 (mapcar (function (lambda (element) (// scalar (float element)))) vector))
	(t (mapcar (function (lambda (element) (* scalar element))) vector))))
)

;APL-SCALAR-TO-ARRAY-MULTIPLY multiplies a scalar  to each element  of an
;array, returning a new array:

(defun apl-scalar-to-array-multiply (scalar array divide-scalar divide-array
				     &aux (dimensions (array-dimensions array)))
  (do ((answer (make-array dimensions))
       (rows (first dimensions))
       (columns (or (second dimensions) 1))
       (row 0 (1+ row)))
      (( row rows) answer)
    (do ((column 0 (1+ column)))
	(( column columns))
      (cond (divide-scalar (aset (// (aref array row column) (float scalar))
				 answer row column))
	    (divide-array (aset (// (float scalar) (aref array row column))
				answer row column))
	    (t (aset (* (aref array row column) scalar) answer row column))))))

;APL-VECTOR-TO-VECTOR-MULTIPLY multiplies each  element of  a vector  by
;the corresponding element of another vector.

(defun apl-vector-to-vector-multiply (vector1 vector2 divide)
  (if divide
      (mapcar (function (lambda (element1 element2) (// element1 (float element2))))
	      vector1 vector2)
      (mapcar (function (lambda (element1 element2) (* element1 element2)))
	      vector1 vector2)))

;APL-DIVIDE divides by calling APL-MULTIPLY.

(defun apl-divide () (apl-multiply t))

;APL-MATRIX-MULTIPLY performs a matrix multiplication between two arrays.

(defun apl-matrix-multiply (&aux new-answer)
  (let ((answer
	  (funcall (function math:multiply-matrices) *first-arg-normal* *second-arg-normal*)))
;if the answer is a one dimensional array of one element reformat it as a 1 by 1 two
;dimensional array to be compatible with the rest of this system.
    (cond ((null (cdr (array-dimensions answer)))
	   (setq new-answer (make-array '(1 1)))
	   (aset (aref answer 0) new-answer 0 0)
	   new-answer)
	  (t answer))))

;APL-INNER-PRODUCT computes the inner product of two vectors.

(local-declare ((special sum))

(defun apl-inner-product (&aux (sum 0))
  (mapc (function (lambda (element1 element2) (setq sum (+ sum (* element1 element2)))))
	*first-arg-normal* *second-arg-normal*)
  sum)
)

;APL-SORT allows a  vector to  be sorted,  or an  array to  be sorted  by
;either a row  or a  column.  The  sorting function  used is  ALPHALESSP,
;which is the same as  < if the elements  being sorted are only  numbers.	
;However, ALPHALESSP allows symbols to be sorted also, unlike <.

(defun apl-sort ()
  (cond ((null *second-arg*)
	 (if (memq (first *first-arg*) '(row column))
	     (apl-vector-sort *first-arg-normal*)
	   (format
	     *comment-window*
	     "The first argument must be a vector if only one argument is given to Sort.~%")))
	((eq (first *second-arg*) 'row)
	 (apl-array-sort-by-row *first-arg-normal* (1- (second *second-arg*))))
	((eq (first *second-arg*) 'column)
	 (apl-array-sort-by-column *first-arg-normal* (1- (second *second-arg*))))))

;APL-VECTOR-SORT sorts a vector by ALPHALESSP.

(defun apl-vector-sort (elements) (sort elements (function alphalessp)))

;APL-ARRAY-SORT-BY-ROW allows an array to be supported by the elements of
;one of its rows.  Example:

;Sort an array by row ROW-INDEX 
;If the array was 
;1 2 3
;7 8 9
;6 2 7
;then sorting on the third row produces:
;2 1 3 
;8 7 9
;2 6 7

(defun apl-array-sort-by-row (array row-index
			      &aux old-row row rows columns indices new-array)
  (setq rows (first (array-dimensions array))
	columns (second (array-dimensions array)))
  (cond ((< row-index rows)
	 (dotimes (index columns)
	   (setq row (cons (list (aref array row-index index) (gensym)) row)))
	 (setq row (nreverse row))
;Row now looks like ((6 g0001) (2 g0002) (7 g0003) (6 g0004)...)  The gensyms
;allow unique identification of each element.
	 (setq old-row (copylist row))
;Now sort the row [note that by using alphalessp instead of just <, symbols
;can be sorted along with numbers.]
	 (setq row (sortcar row (function alphalessp)))
;Now find the column indices of each element in the row:
	 (setq indices
	       (do ((row-left row (cdr row-left))
		    (indices-list
		      nil
		      (cons
			(find-position-in-list-equal (car row-left) old-row) indices-list)))
		   ((null row-left) (nreverse indices-list))))
;Now create a new array with the elements copied according to the indices list
;The indices list looks like (2 1 3), indicating that the first column in the
;new sorted array comes from the second column in the unsorted array, etc.
	 (setq new-array (make-array (list rows columns)))
	 (do ((column 0 (1+ column))
	      (indices-list indices (cdr indices-list))
	      (column-to-copy))
	     (( column columns))
	   (setq column-to-copy (car indices-list))
	   (do ((row 0 (1+ row)))
	       (( row rows))
	     (aset (aref array row column-to-copy) new-array row column)))
	 new-array)
	(t
	 (format *comment-window*
		 "The row should be selected from the array to be sorted.~%")
	 nil)))

;APL-ARRAY-SORT-BY-COLUMN allows an array to be supported by the elements of
;one of its columns.  Example:

;Sort an array by column COLUMN-INDEX
;If the array was 
;1 2 3
;7 8 9
;6 2 7
;then sorting on the first column produces:
;1 2 3
;6 2 7
;7 8 9

(defun apl-array-sort-by-column (array column-index
				 &aux old-column column rows columns indices new-array)
  (setq rows (first (array-dimensions array))
	columns (second (array-dimensions array)))
  (cond ((< column-index columns)
	 (dotimes (index rows)
	   (setq column (cons (list (aref array index column-index) (gensym)) column)))
	 (setq column (nreverse column))
;Column now looks like ((6 g0001) (2 g0002) (7 g0003) (6 g0004)...)  The gensyms
;allow unique identification of each element.
	 (setq old-column (copylist column))
;Now sort the column [note that by using alphalessp instead of just <, symbols
;can be sorted along with numbers.]
	 (setq column (sortcar column (function alphalessp)))
;Now find the row indices of each element in the column:
	 (setq indices
	       (do ((column-left column (cdr column-left))
		    (indices-list
		      nil
		      (cons (find-position-in-list-equal (car column-left) old-column)
			    indices-list)))
		   ((null column-left) (nreverse indices-list))))
;Now create a new array with the elements copied according to the indices list
;The indices list looks like (2 1 3), indicating that the first row in the
;new sorted array comes from the second row in the unsorted array, etc.
	 (setq new-array (make-array (list rows columns)))
	 (do ((row 0 (1+ row))
	      (indices-list indices (cdr indices-list))
	      (row-to-copy))
	     (( row rows))
	   (setq row-to-copy (car indices-list))
	   (do ((column 0 (1+ column)))
	       (( column columns))
	     (aset (aref array row-to-copy column) new-array row column)))
	 new-array)
	(t
	 (format *comment-window*
		 "The column should be selected from the array to be sorted.~%"))))

;DELETE-VECTOR creates a new table  from the table containing  the vector
;selected.  The new table does not have the selected vector and thus  has
;either one fewer rows or columns.  If the new table would have either  0
;rows or columns no table is created and an explanation is given.	

(defun delete-vector ()
  (let ((table (third *first-arg*)))  ;*first-arg* has the form (row <row> <table>)
    (cond ((eq (first *first-arg*) 'row)
	   (copy-table-except-row table (second *first-arg*)))
	  (t (copy-table-except-column table (second *first-arg*))))))

;COPY-TABLE-EXCEPT-ROW creates a new table identical to the first  except
;row ROW is elided.

(defun copy-table-except-row (table row)
  (let* ((old-array (<- table ':array))
	 (old-dimensions (array-dimensions old-array))
	 (rows (1- (first old-dimensions)))
	 (columns (second old-dimensions)))
    (if (or (zerop rows) (zerop columns))
	(format *comment-window* "That will delete all the rows in that table.~%")
;copy the old array into the new array but skip row ROW.
      (do ((new-array (make-array (list rows columns)))
	   (i 0 (1+ i))
	   (row (1- row))  
;rows in descriptors are from 1 to ROWS, but in arrays they are from 0 to ROWS-1
	   (row-skipped nil))
	  ((> i rows) new-array)
;The variable ROWS is one less than the number of rows in the old array.
	(cond ((= i row) (setq row-skipped t))
	      (t
	       (do ((j 0 (1+ j)))
		   (( j columns))
		 (aset (aref old-array i j) new-array (if row-skipped (1- i) i) j))))))))


;COPY-TABLE-EXCEPT-COLUMN creates a new table identical to the first  except
;column COLUMN is elided.

(defun copy-table-except-column (table column)
  (let* ((old-array (<- table ':array))
	 (old-dimensions (array-dimensions old-array))
	 (rows (first old-dimensions))
	 (columns (1- (second old-dimensions))))
    (if (or (zerop rows) (zerop columns))
	(format *comment-window* "That will delete all the columns in that table.~%")
;copy the old array into the new array but skip column COLUMN.
      (do ((new-array (make-array (list rows columns)))
	   (j 0 (1+ j))
	   (column (1- column))  
;columns in descriptors are from 1 to COLUMNS, but in arrays they are from 0 to COLUMNS-1
	   (column-skipped nil))
	  ((> j columns) new-array)
;The variable COLUMNS is one less than the number of columns in the old array.
	(cond ((= j column) (setq column-skipped t))
	      (t
	       (do ((i 0 (1+ i)))
		   (( i rows))
		 (aset (aref old-array i j) new-array i (if column-skipped (1- j) j)))))))))


;TRANSPOSE transposes a table.

(defun transpose () (math:transpose-matrix *first-arg-normal*))

;JOIN-COLUMNS concatenates tables together by  adding the columns of  the
;second table to the first table to  produce a new table.  The number  of
;rows in each table must be the same.

(defun join-columns ()
  (let* ((dimensions-a (array-dimensions *first-arg-normal*))
	 (dimensions-b (array-dimensions *second-arg-normal*))
	 (rows-a (first dimensions-a))
	 (rows-b (first dimensions-b))
	 (columns-a (second dimensions-a))
	 (columns-b (second dimensions-b))
	 (new-columns (+ columns-a columns-b)))
    (cond ((= rows-a rows-b)
	   (do ((result (make-array (list rows-a new-columns)))
		(column 0 (1+ column)))
	       (( column new-columns) result)
	     (do ((row 0 (1+ row)))
		 (( row rows-a))
	       (aset (aref (if ( column columns-a) *second-arg-normal* *first-arg-normal*)
			   row
			   (if ( column columns-a) (- column columns-a) column))
		     result row column))))
	  (t (format *comment-window*
		     "The two tables must have the same number of rows.~%")
	     nil))))

;JOIN-ROWS concatenates tables together by  adding the rows of  the
;second table to the first table to  produce a new table.  The number  of
;columns in each table must be the same.

(defun join-rows ()
  (let* ((dimensions-a (array-dimensions *first-arg-normal*))
	 (dimensions-b (array-dimensions *second-arg-normal*))
	 (rows-a (first dimensions-a))
	 (rows-b (first dimensions-b))
	 (columns-a (second dimensions-a))
	 (columns-b (second dimensions-b))
	 (new-rows (+ rows-a rows-b)))
    (cond ((= columns-a columns-b)
	   (do ((result (make-array (list new-rows columns-a)))
		(row 0 (1+ row)))
	       (( row new-rows) result)
	     (do ((column 0 (1+ column)))
		 (( column columns-a))
	       (aset (aref (if ( row rows-a) *second-arg-normal* *first-arg-normal*)
			   (if ( row rows-a) (- row rows-a) row)
			   column)
		     result row column))))
	  (t (format *comment-window*
		     "The two tables must have the same number of rows.~%")
	     nil))))

;RELATIONAL OPERATORS allow predicates to  be applied to operands.   1 is
;returned if the predicate is true, and 0 if it is false.

(defun apl-lessp () (apl-compare (function alphalessp)))

(defun apl-greaterp ()
  (apl-compare (function (lambda (x y) (and (not (alphalessp x y))
					    (not (equal x y)))))))

(defun apl-equalp () (apl-compare (function (lambda (x y) (equal x y)))))

(defun apl-greater-or-equalp ()
  (apl-compare (function (lambda (x y) (not (alphalessp x y))))))

(defun apl-lesser-or-equalp ()
  (apl-compare (function (lambda (x y) (or (equal x y) (alphalessp x y))))))

;Compare the first argument  to the second  argument using the  predicate
;function.  Compose an answer  of the right  form (atom, list,  or array)
;having 1's where the relation holds and 0 where it does not.

(defun apl-compare (predicate-function)
      (selectq (first *first-arg*)		;get type of first arg
	((atom scalar)
	 (selectq (first *second-arg*)		;get type of second arg
	   ((atom scalar)
	    (if (funcall predicate-function *first-arg-normal* *second-arg-normal*) 1 0))
	   ((row column)
	    (mapcar
	      (function
		(lambda (second-arg)
		  (if (funcall predicate-function *first-arg-normal* second-arg) 1 0)))
	      *second-arg-normal*))
	   (array
	    (apl-scalar-to-array-compare
	      *first-arg-normal* *second-arg-normal* predicate-function))))
	((row column)
	 (selectq (first *second-arg*)
	   ((atom scalar)
	    (mapcar
	      (function
		(lambda (first-arg)
		  (if (funcall predicate-function first-arg *second-arg-normal*) 1 0)))
	      *first-arg-normal*))
	   ((row column)
	    (mapcar
	      (function
		(lambda (first-arg second-arg)
		  (if (funcall predicate-function first-arg second-arg) 1 0)))
		    *first-arg-normal* *second-arg-normal*))
	   (array (format *comment-window* "You cannot compare an array to a vector.~%")
		  nil)))
	(array
	 (selectq (first *second-arg*)
	   ((atom scalar)
	    (apl-scalar-to-array-compare
	      *second-arg-normal* *first-arg-normal* predicate-function 'array-first-arg))
	   ((row column)
	    (format *comment-window* "You cannot compare an array to a vector.~%")
	    nil)
	   (array
	    (apl-array-to-array-compare
	      *first-arg-normal* *second-arg-normal* predicate-function))))))

;APL-SCALAR-TO-ARRAY-COMPARE compares  SCALAR  to  each  element of ARRAY
;using PREDICATE-FUNCTION and returns a new array of the results.

;Note: SCALAR can also be a nonnumeric atom since ALPHALESSP and EQUAL are used for
;comparisons.

;Twiddle-args reverses the order of the arguments if the operation to  be
;performed is ARRAY OP SCALAR instead of SCALAR OP ARRAY.	


(defun apl-scalar-to-array-compare (scalar array predicate-function
				    &optional twiddle-args
				    &aux (dimensions (array-dimensions array)))
  (do ((answer (make-array dimensions))
       (rows (first dimensions))
       (columns (or (second dimensions) 1))
       (row 0 (1+ row)))
      (( row rows) answer)
    (do ((column 0 (1+ column)))
	(( column columns))
      (aset
	(if (if twiddle-args (funcall predicate-function (aref array row column) scalar)
	      (funcall predicate-function scalar (aref array row column)))
	    1 0)
	answer row column))))



;APL-ARRAY-TO-ARRAY-COMPARE compares two arrays together using the PREDICATE-FUNCTION,
;and returns the new array:

(defun apl-array-to-array-compare (array1 array2 predicate-function &aux dimensions)
  (if (equal (setq dimensions (array-dimensions array1)) (array-dimensions array2))
      (do ((answer (make-array dimensions))
	   (rows (first dimensions))
	   (columns (or (second dimensions) 1))
	   (row 0 (1+ row)))
	  (( row rows) answer)
	(do ((column 0 (1+ column)))
	    (( column columns))
	  (aset
	    (if (funcall predicate-function (aref array1 row column) (aref array2 row column))
		1 0)
	    answer row column)))
    (format *comment-window*
	    "The two arrays have different dimensions and cannot be compared.~%")))


;ARRAY COMPRESSION OPERATORS allow rows and columns to be deleted from an
;array using  a  mask  of  1's  and  0's,  such  as  might  be created by
;relational operators.

;ARRAY-COMPRESS-ROWS compresses the rows of the first argument,  which is
;an array, by the vector mask that is the second argument.  The length of
;the vector should match the number of rows of the array.  The  new array
;has a row copied wherever a one occurs in the mask, rows with a 0 in the
;mask are deleted.

(defun array-compress-rows (&aux (dimensions (array-dimensions *first-arg-normal*))
			    answer-rows)
  (let ((rows (first dimensions))
	(columns (second dimensions))
	(mask-length (length *second-arg-normal*)))
    (cond ((= rows mask-length)
	   (setq answer-rows
		 (do ((answer-length
			0
			(if (zerop (car vector-left)) answer-length (1+ answer-length)))
		      (vector-left *second-arg-normal* (cdr vector-left)))
		     ((null vector-left) answer-length)))
	   (cond ((= answer-rows 0)
		  (format *comment-window* "That results in an empty array.~%"))
		 (t
		  (do ((answer (make-array (list answer-rows columns)))
		       (next-row 0)
		       (row 0 (1+ row))
		       (vector-left *second-arg-normal* (cdr vector-left)))
		      (( row rows) answer)
		    (cond ((zerop (car vector-left)))
;;;don't copy rows where a zero occurs in the mask.
			  (t 
			   (do ((column 0 (1+ column)))
			       (( column columns))
			     (aset (aref *first-arg-normal* row column)
				   answer next-row column))
			   (setq next-row (1+ next-row))))))))
	  (t (format *comment-window*
		     "The length of the mask does not match the number of rows.~%")))))

;ARRAY-COMPRESS-COLUMNS compresses  the  columns  of  the first argument,
;which is an array, by the vector mask that is the second argument.   The
;length of the vector  should match the  number of columns  of the array.
;The new array has  a column copied  wherever a one  occurs in the  mask,
;columns with a 0 in the mask are deleted.

(defun array-compress-columns (&aux (dimensions (array-dimensions *first-arg-normal*))
			       answer-columns)
  (let ((rows (first dimensions))
	(columns (second dimensions))
	(mask-length (length *second-arg-normal*)))
    (cond ((= columns mask-length)
	   (setq answer-columns
		 (do ((answer-length
			0
			(if (zerop (car vector-left)) answer-length (1+ answer-length)))
		      (vector-left *second-arg-normal* (cdr vector-left)))
		     ((null vector-left) answer-length)))
	   (cond ((= answer-columns 0)
		  (format *comment-window* "That results in an empty array.~%"))
		 (t
		  (do ((answer (make-array (list rows answer-columns)))
		       (next-column 0)
		       (column 0 (1+ column))
		       (vector-left *second-arg-normal* (cdr vector-left)))
		      (( column columns) answer)
		    (cond ((zerop (car vector-left)))
;;;don't copy columns where a 0 occurs in the mask.
			  (t 
			   (do ((row 0 (1+ row)))
			       (( row rows))
			     (aset (aref *first-arg-normal* row column)
				   answer row next-column))
			   (setq next-column (1+ next-column))))))))
	  (t (format *comment-window*
		     "The length of the mask does not match the number of columns.~%")))))

