;;; This is a -*-Lisp-*- file.
;;;
;;; **********************************************************************
;;; This code was written as part of the Spice Lisp project at
;;; Carnegie-Mellon University, and has been placed in the public domain.
;;; Spice Lisp is currently incomplete and under active development.
;;; If you want to use this code or any part of Spice Lisp, please contact
;;; Scott Fahlman (FAHLMAN@CMUC). 
;;; **********************************************************************
;;;
;;; Functions to implement irrational functions for Spice Lisp 
;;; Written by David Adam.
;;;
;;; The irrational functions are part of the standard Spicelisp environment.
;;;
;;; **********************************************************************
;;;

;;; These are the constants needed for the irrational functions.

(defconstant pi #~F3.1415926535897932384)
(defconstant half-pi #~F1.5707963267948966192)
(defconstant e #~F2.71828187
  "Euler's number, short float version.")


;;; A set of support routines which I found useful in more than one set 
;;; of functions.

(defmacro sqr (x) `(* ,x ,x))

(defmacro exp2 (x)
  `(if (minusp ,x) (/ (ash 1 (- ,x))) (ash 1 ,x)))



;;; returns the value of the smallest power of two which is greater than the 
;;; number input, assumes positive number for input.  Will be faster for 
;;; numbers greater than one half, than numbers smaller than one half.
;;; This probably should be accomplished at the machine level, if there 
;;; are reasonable rotate instructions (for efficiency reasons.)

(defun high-bit (x)
  (if (> x #~F0.5)
      (do ((y 1 (ash y 1))
	   (count 0 (1+ count)))
	  ((< x y) count))
      (do ((y #~F0.5 (* y #~F0.5))
	   (count 0 (1- count)))
	  ((> x y) count))))


;;; Routine calculates the absolute value of its argument.

(defun abs (x)
  (if (minusp x) (- x) x))

(defun signum (x)
  (let ((res  (cond ((plusp x) 1)
		    ((zerop x) 0)	
		    (t -1))))
    (if (floatp x) (float res x) res)))

;;; This is a first attempt at the integer square root function.
;;; It performs the operation in log n time, fairly efficiently.
;;; At such time a better algorithm comes to my attention this will 
;;; be changed.

;;; This function checks to see that it has the right kind of value
;;; leaving in an error hook if it is not.  Then it uses a crude.
;;; approximation measure of taking an upper bound and a lower bound,
;;; and determining if their average is above or below the desired value.
;;; If above the middle value becomes the upper bound, otherwise the
;;; middle becomes the lower bound.  The process is repeated until 
;;; the upper bound and the lower bound are one apart.  The test
;;; is not an equal because of the case of 1, when the upper and lower
;;; bounds both start at 1. 

(defun jsqrt (x)
  (if (and (integerp x) (not (minusp x)))
      (if (zerop  x) 0
	  (do* ((p () (<= (* m m) x))
		(b 1 (if p m b))
		(h x (if p h m))
		(m (ash x -1) (ash (+ b h) -1)))
	       ((<= h (1+ b)) b)))
      (cerror t "~S argument out of range" x)))

(defun isqrt (x) 
  (do ((n (round (/ x (expt 2 (truncate (/ (integer-length x) 2)))))
	  (round (/ (+ n (/ x n)) 2)))
       (oldn -2 n)) 
      ((or (= n oldn) (= n (+ oldn 1))) oldn)
   ))

;;; The square root is calculated by first attaining a fairly accurate
;;; approximation.  This is done by limiting the value to between 
;;; 1/4 and 1, which is multiplied by a power of 2 which is even.
;;; This limited range value is approximated, and then newtons method is 
;;; used until the value has enough accuracy.  Probably only twice for
;;; short flonums.  This algorithm is taken from the 
;;; Science Library and Fortran Utility Programs.

(defun sqrt (x)
  (if (minusp x)
      (cerror t "~S Argument out of range." x)
;; return complex when added.
      (do* ((hold (high-bit x))
	    (b (if (minusp hold) (- (ash (- hold) -1))(ash hold -1)))
	    (f (/ x (exp2 (ash b 1))))
	    (sf (if (> #~F0.5 f) (+ (* #~F0.8125 f) #~F0.302734)
		    (+ (* #~F0.578125 f) #~F0.421875)))
	    (old2 -2 old)
	    (old -2 tot)
	    (tot (* (exp2 b) sf)
		 (* #~F0.5 (+ tot (/ x tot)))))
	   ((or (= old tot) (= old2 tot)) tot))))

;;; Function calculates the value of x raised to the nth power.
;;; It is an error if power is not an integer or is negative.
;;; This function calculates the successive squares of base,
;;; storing them in newbase, halving n at the same time.  If
;;; n is odd total is multiplied by base, at any given time (fix later)


(defun intexp (obase power)
  (if (= obase 2) (ash 1 power)
      (do* ((newbase obase (* newbase newbase))
	    (n power (ash n -1))
	    (total (if (oddp n) newbase 1)
		   (if (oddp n) (* newbase total) total)))
	   ((zerop n) total))))

;;; alternate macro version,  possibly preferable for efficiency.
(eval-when (compile)
  (defmacro mintexp (obase power)
	   `(do ((total 1 (if (oddp n) (* newbase total) total))
		 (newbase ,obase (* newbase newbase))
		 (n ,power (ash n -1)))
		((zerop n) total))))

;;; This function calculates x raised to the nth power.  It separates
;;; the  cases by the type of n, for efficiency reasons, as powers can
;;; be calculated more efficiently if n is a positive integer,  Therefore,
;;; All integer values of n are calculated as positive integers, and
;;; inverted if negative.

(defun expt (x n)
  (cond ((integerp n) (if (minusp n)
			  (/ (intexp x (- n)))
			  (intexp x n)))
	((minusp x)
	 (cerror t
		 "improper to have negative base ~S and non integer power ~S"
		 x n))
	(t (exp (* n (log x)))))) ; for now.
     
;;; This function calculates e raised to the x.  Again the cases are separated
;;; By whether x is an integer or not.   If x is an integer, then merely the
;;; Integer exponent of e to the x is needed.  If x is a real, then e to the x
;;; is converted to 2 to the (x * log base 2 of e).
;;; This value is then separated into an integer and fractional part, which
;;; are dealt with separately.

;;; This routine is currently accurate to 6 decimal places in maclisp.

(defun exp (x)
  (if (integerp x)
      (if (minusp x)
	  (/ (intexp e (- x)))
	  (intexp e x))
      (let* ((y (* x  #~F1.4426950408896340740))	;log2(e)
;; currently only set up for one type of floating point number,
;; to avoid errors.
	     (int (floor y))
	     (frac (mod y 1)))
	(* (if (minusp int) (/ (ash 1 (- int))) (ash 1 int))
;; The integer part of the exponent.
	   (cond
	    ((< #~F0.875 frac) (setq frac (- frac #~F0.875)) #~F1.83400811) ;(expt 2 0.875)
	    ((< #~F0.75 frac) (setq frac (- frac #~F0.75)) #~F1.68179283) ;(expt 2 0.75)
	    ((< #~F0.625 frac) (setq frac (- frac #~F0.625)) #~F1.54221082) ;(expt 2 0.625)
	    ((< #~F0.5 frac) (setq frac (- frac #~F0.5)) #~F1.41421354) ;(expt 2 0.5)
	    ((< #~F0.375 frac) (setq frac (- frac #~F0.375)) #~F1.29683954) ;(expt 2 0.375)
	    ((< #~F0.25 frac) (setq frac (- frac #~F0.25)) #~F1.1892071) ;(expt 2 0.25)
	    ((< #~F0.125 frac) (setq frac (- frac #~F0.125)) #~F1.09050773) ;(expt 2 0.125)
	    (t 1))					;(expt 2 0)
;; Leaves a number < 1/8 to evaluate.
	   (if (= frac #~F0.0) #~F1.0				;test for case when frac = 0
	       (1+ (/ #~F60.59319171733646311080		;(* 42 (log e 2))
		    (+ (/ #~F87.417497202235527474 frac)	;(* 42 (sqr (log e 2)))
		       #~F-30.296595858668231555		;(* 21 (log e 2))
		       (* #~F1.05 frac)
		       (/ #~F214.1728681454770423113	;(/ (* 1029 (sqr (log e 2))) 10)
			(+ frac (/ #~F87.417497202235527474 ;(* 42 (sqr (log e 2)))
				 frac)))))))))))




;;; hyperbolic trig functions.
;;; Each of the hyperbolic trig functions is calculated directly from 
;;; their definition.  Exp(x) is calculated only once for efficiency.

;;; The Hyperbolic sine.

(defun sinh (x)
  (let ((z (exp x)))
    (/ (- z (/ z)) 2)))

;;; The Hyperbolic cosine.

(defun cosh (x)
  (let ((z (exp x)))
    (/ (+ z (/ z)) 2)))

;;; The Hyperbolic tangent.

(defun tanh (x)
  (let* ((z (exp x))
	 (y (/ x)))
    (/ (- z y) (+ z y))))

(defun float-exponent (x) (high-bit x))
(defun float-fraction (x) (/ x (exp2 (high-bit x))))
;;; Natural LOG routine for type SHORT-FLOAT (20 significant bits)

(DEFCONSTANT SHORT-LOG-COEF-0 #~F0.7071067811)
(DEFCONSTANT SHORT-LOG-COEF-1 #~F0.693359375)
(DEFCONSTANT SHORT-LOG-COEF-2 #~F-0.0002121944400546905827679)
(DEFCONSTANT SHORT-LOG-COEF-5 #~F0.5)
(DEFCONSTANT SHORT-LOG-A0 #~F-0.5527074855)
(DEFCONSTANT SHORT-LOG-B0 #~F-6.632718214)
(defconstant short-pi #~F3.1415926535897932384)

(DEFUN SHORT-LOG (X) 
  (DECLARE (SHORT-FLOAT X))
  (COND ((PLUSP X)
	 (LET ((N (FLOAT-EXPONENT X)) (F (FLOAT-FRACTION X)))
	      (DECLARE (SHORT-FLOAT F) (FIXNUM N))
	      (LET ((Z (IF (> F SHORT-LOG-COEF-0)
			   (LET ((ZN (- F SHORT-LOG-COEF-5)))
				(SETQ N (- N 1))
				(/ ZN
				   (+ (* ZN SHORT-LOG-COEF-5)
				      SHORT-LOG-COEF-5)))
			   (/ (- (- F SHORT-LOG-COEF-5) SHORT-LOG-COEF-5)
			      (+ (* F SHORT-LOG-COEF-5) SHORT-LOG-COEF-5)))))
		   (DECLARE (SHORT-FLOAT Z))
		   (LET ((W (* Z Z)) (XN (FLOAT N Z)))
			(DECLARE (SHORT-FLOAT W XN))
			(+ (+ (* XN SHORT-LOG-COEF-2)
			      (+ Z
				 (* Z
				    (* W
				       (/ SHORT-LOG-A0 (+ W SHORT-LOG-B0))))))
			   (* XN SHORT-LOG-COEF-1))))))
	((ZEROP X) (ERROR "LOG of zero"))
	(T (THE (COMPLEX SHORT-FLOAT) (COMPLEX (SHORT-LOG (- X)) SHORT-PI)))))


;;; This routine calculates the log base e of n, using the algorithm found 
;;; in the fortran library.
;;; The log is calculated based on the log base e.
;;; if there is no base given then the log base e is simply returned.
;;; otherwise the log base e of the number is divided by the log base e
;;; of the base.

(defun log (number &optional pbase)
  (if pbase
      (/ (short-log number) (short-log pbase))
      (short-log number)))


;;; The inverse hyperbolic functions will be coded by definition like
;;; the hyperbolic functions.

;;; The inverse of the hyperbolic sine.

(defun asinh (x)
  (log (+ x (sqrt (+ (sqr x) 1)))))

;;; The inverse of the hyperbolic cosine.

(defun acosh (x)
  (if (plusp x)
      (log (+ x (sqrt (- (sqr x) 1))))
      (cerror t "~S argument out of range." x)))

;;; The inverse of the hyperbolic tangent.

(defun atanh (x)
  (if (< -1 x 1)
      (* #~F0.5 (log (/ (1+ x) (- 1 x))))
      (cerror t "~S argument out of range." x)))

(DEFCONSTANT SHORT-C2-SIN  #~F0.000967653589793)
(DEFCONSTANT SHORT-SIN-R1 #~F-0.1666665668)
(DEFCONSTANT SHORT-SIN-R2 #~F0.008333025139)
(DEFCONSTANT SHORT-SIN-R3 #~F-0.0001980741872)
(DEFCONSTANT SHORT-SIN-R4 #~F0.000002601903036)
(DEFCONSTANT SHORT-ZERO #~F0.0)

(DEFCONSTANT SHORT-C1-SIN #~F3.140625)

(DEFUN SHORT-SIN (X COS-FLAG SGN) 
  (MULTIPLE-VALUE-BIND (X1 X2)
		       (TRUNCATE X)
		       (LET* ((N (ROUND x SHORT-PI))
			      (XN (IF COS-FLAG
				      (- (FLOAT N SHORT-ZERO) #~F0.5)
				      (FLOAT N SHORT-ZERO)))
			      (F (- (+ (- X1 (* XN SHORT-C1-SIN)) X2)
				    (* XN SHORT-C2-SIN)))
			      (G (SQR F)))
			     (IF (ODDP N) (SETQ SGN (- SGN)))
			     (* (+ (* (* (+ (* (+ (* (+ (* SHORT-SIN-R4 G)
							SHORT-SIN-R3)
						     G)
						  SHORT-SIN-R2)
					       G)
					    SHORT-SIN-R1)
					 G)
				      F)
				   F)
				SGN))))
(defun sin (x)
  (short-sin (abs x) () (if (plusp x) 1 -1)))

(defun cos (x)
  (short-sin (abs x) t 1))

;;; The tangent works by taking the sine divided by the cosine
;;; of the input argument.

(defun tan (x)
  (/ (sin x) (cos x)))

;;; The arc tangent is determined using a power series described in 
;;; Computer Approximations by Hart & Cheney p122 (6.5.15)
;;; As the radius of convergence is 1 there is a very poor performance
;;; When the value of x is near 1.  This should be fixed at a later date.

(defun atan (y &optional x)
  (cond	((not x) (atan1 y))
	((= y x 0)
	 (cerror t "Error in double entry atan both 0." ))
	((= x 0) (* (signum y) half-pi))
	((= y 0) (if (plusp x) 0 pi))
	((and (plusp x)(plusp y)) (atan1 (/ y x)))
	((plusp y) (- pi (atan1 (/ (- y) x))))
	((plusp x) (- (atan1 (/ y (- x)))))
	(t (- (atan1 (/ y x)) pi))))

(defconstant sixth-pi #~f0.5236)
(defconstant sqrt3 #~f1.7305248)
(defconstant 2-sqrt3 #~f0.26794816)
(defconstant inv-2-sqrt3 #~f3.7320704)

(defun atan1 (x)
  (cond ((minusp x) (- (atan1 (- x))))
	((< x 2-sqrt3) (- half-pi (atan2 (/ x))))
	((< x 1)
	 (- (+ half-pi sixth-pi) (atan2 (/ (+ sqrt3 x) (1- (* x sqrt3))))))
	((< x inv-2-sqrt3)
	 (let* ((inv (/ x)))
	   (- (atan2 (/ (+ sqrt3 inv) (1- (* sqrt3 inv)))) sixth-pi)))
	(t (atan2 x))))

(defun atan2 (x)
  (do* ((sqr (- (/ (sqr x))))
	(int 1 (+ 2 int))
	(old 0 val)
	(pow (/ x) (* pow sqr))
	(val pow (+ val (/ pow int))))
       ((= old val)
	(- half-pi  val))))


;;; The arc sine is calculated from the arc tangent.

(defun asin (x)
  (if ( < -1 x 1)
      (atan (/ x (sqrt (- 1 (sqr x)))))
      (cerror t "argument ~S out of bounds." x)))

;;; The arc cosine is calculated from the arc tangent.

(defun acos (x)
  (if ( < -1 x 1)
      (- half-pi (atan (/ x (sqrt (- 1 (sqr x))))))
      (cerror t "argument ~S out of bounds." x)))
   