;;; -*- Mode:Common-Lisp; Package:USER; Base:10 -*-

(export 'grep 'global)

(defun grep (key paths &optional (verbose t))
  "A weak version of a Unix-like grep.  KEY may be a string or a list of
strings to search for.  PATHS are files to search for KEY in -- it may be
a pathname, a string, or a list of pathnames or strings (which may
contain wildcards).  Each line which contains a match for KEY is printed
on standard output unless VERBOSE is nil.  Returns a list of pathnames
which contain a match for KEY."
  
  
  (setq paths (if (consp paths) paths (list paths)))
  (let (files-with-match)
    (dolist (path paths)
      (setq path (parse-namestring path))
      (unless (send path :version)
	(setq path (send path :new-version :newest)))
      (cond ((send path :wild-p)
	     (dolist (match (send path :wildcard-map #'search-file nil nil key verbose))
	       (when match
		 (push match files-with-match))))
	    (t
	     (when (search-file path key verbose)
	       (push path files-with-match)))))
    files-with-match))

(defun search-file (path key verbose)
  "Search PATH for KEY printing each matching line if VERBOSE is non-nil.
Return PATH if any match found, otherwise return NIL."
  
  (with-open-file (stream path :error nil)
    (unless (errorp stream)
      (let (return-value)
	(loop
	  (multiple-value-bind (line eof) (send stream :line-in)
	    (when (> (length line) 0)
	      (cond ((consp key)
		     (dolist (k key)
		       (when (string-search k line)
			 (setq return-value path)
			 (when verbose
			   (format t "~&~A:  ~A" path line)))))
		    (t
		     (when (string-search key line)
		       (setq return-value path)
		       (when verbose
			 (format t "~&~A:  ~A" path line))))))
	    (when eof
	      (return return-value))))))))

