;;; -*- Mode: LISP ; Base: 10. ; Package: user ; Fonts:CPTFONT,HL12B,HL12BI -*-


(DEFUN FILE-AUTHOR (FILENAME-OR-STREAM)
  "Return file's author's name (a string).  Specify pathname, namestring or file stream."
  (IF (OR (STRINGP FILENAME-OR-STREAM)
	  (SYMBOLP FILENAME-OR-STREAM)
	  (TYPEP FILENAME-OR-STREAM 'PATHNAME))
      (WITH-OPEN-FILE (STREAM FILENAME-OR-STREAM ':DIRECTION NIL)
	(SEND STREAM ':GET ':AUTHOR))
    (SEND FILENAME-OR-STREAM ':GET ':AUTHOR)))

(DEFUN WRITE-AUTHOR (FILENAME name)
  "2Insert file's author's name (a string).  Specify pathname, namestring or file stream."*
  (fs:change-file-properties filename t ':author name))

(DEFUN change-author (dir name)
  (LET ((files (LOOP for f in (fs:directory (fs:parse-pathname dir))
                     collect (STRING f))))
    (LOOP for file in files
          DO (write-author file name))))

(DEFUN locate-file (path &optional (type 'file))
  "2DETERMINE PRESENCE OF FILE/DIRECTORY*"
  (LET* ((parsed        (fs:parse-pathname path))
         (direct        (SEND parsed ':directory))
         (host-name     (fs:host-namestring parsed))
         (direct-string host-name)
         (direct-list  (fs:directory-list (STRING-APPEND host-name "~;*.*")))
         (valid         nil))
    (COND ((TYPEP direct ':string) (SETQ direct (LIST direct))))
    (*CATCH 'end 'end
            (LOOP for  node in direct
                  for  end from (length direct) downto 1
                  for  bad = t
                  with dot = ""
                  DO
                  (LOOP for dir in (CDR direct-list)
                        WHEN (AND (GET dir ':directory)
                                  (equalp (SEND (CAR dir) ':name) node))
                        DO (PROGN
                             (SETQ direct-string
                                   (STRING-APPEND direct-string dot node)
                                   dot "."
                                   bad nil)
                             (COND (( end 1)
                                    (SETQ direct-list
                                          (fs:directory-list
                                            (STRING-APPEND direct-string ";*.*"))))))
                        AND DO (RETURN))
                  WHEN bad DO (RETURN)
                  finally (SETQ valid t))
            (COND ((EQ type 'dir) (*THROW 'end 'dir)))
            (COND ((NULL valid)   (*THROW 'end 'file)))
            (SETQ valid nil)
            (SETQ direct-list (fs:directory-list
                               (STRING-APPEND direct-string ";*.*")))
            (LOOP for file in (REVERSE (CDR direct-list))
                  WHEN (AND (NOT (GET file ':directory))
                            (EQUALP (SEND parsed ':name) (SEND (CAR file) ':name))
                            (EQUALP (SEND parsed ':type) (SEND (CAR file) ':type)))
                  DO (SETQ valid t)
                  AND DO (RETURN)))
    valid))

    
(DEFUN delete-top-directory (parsed)
  "2DELETE TOP-DIRECTORY AND FILES BELOW*"
  (LET ((direct-string nil)
        (direct        (SEND parsed ':directory)))
    (SETQ direct-string (STRING-APPEND (fs:host-namestring parsed) direct ";"))
    (fs:delete-file (STRING-APPEND direct-string "*.*"))
    (fs:expunge-directory direct-string)
    (SETQ direct-string (STRING-APPEND (fs:host-namestring parsed) "~;"))
    (fs:delete-file (STRING-APPEND direct-string direct ".DIRECTORY"))
    (fs:expunge-directory direct-string)))


(DEFUN delete-sub-directory (parsed)
  "2DELETE SUB-DIRECTORY AND FILES BELOW*"
  (LET ((direct-string nil)
        (direct        (SEND parsed ':directory)))
    (SETQ direct-string
          (LOOP for name in direct
                with str = "" AND dot = ""
                DO (SETQ str (STRING-APPEND str dot name)
                         dot ".")
                finally (RETURN
                          (STRING-APPEND (fs:host-namestring parsed) str ";"))))
    (fs:delete-file (STRING-APPEND direct-string "*.*"))
    (fs:expunge-directory direct-string)
    (SETQ direct-string
          (LOOP for name in (BUTLAST direct)
                with str = "" AND dot = ""
                DO (SETQ str (STRING-APPEND str dot name)
                         dot ".")
                finally (RETURN
                          (STRING-APPEND (fs:host-namestring parsed) str ";"))))
    (fs:delete-file (STRING-APPEND direct-string (CAR (LAST direct)) ".DIRECTORY"))
    (fs:expunge-directory direct-string)))


(DEFUN delete-directory (path)
  "2DELETE DIRECTORY AND FILES BELOW*"
  (LET* ((parsed (fs:parse-pathname path))
         (type   (TYPEP (SEND parsed ':directory))))
    (COND ((EQ type ':string) (delete-top-directory parsed))
          ((EQ type ':cons  ) (delete-sub-directory parsed))
          (t                  (FERROR nil
                                      "DIRECTORY TYPE NOT RECOGNIZED - ~A" TYPE)))))


(DEFUN create-directory (direct)
  "2CREATE DIRECTORY - direct ( pathname string )*"
  (fs:create-directory direct))


(DEFUN file-delete (path)
  "2DELETE FILE AND EXPUNGE DIRECTORY*"
  (DELETE-FILE path)
  (fs:expunge-directory path))


(DEFUN keep-last-two-files (file-list)
  "2DETERMINE WHCIH FILES ARE NOT THE LATEST TWO*"
  (LOOP for dir in  (REVERSE file-list)
        for  new-file = (SEND (CAR dir) ':name)
        with old-file = "" AND file-count = 0
        WHEN (NOT (STRING-EQUAL old-file new-file))
        DO (SETQ old-file new-file
                 file-count 0)
        WHEN (STRING-EQUAL old-file new-file)
        DO (SETQ file-count (ADD1 file-count))
        WHEN (> file-count 2)
        collect dir))


(DEFUN keep-last-file-only (file-list)
  "2DETERMINE WHCIH FILES ARE NOT THE LAST ONE*"
  (LOOP for dir in  (REVERSE file-list)
        for  new-file = (SEND (CAR dir) ':name)
        with old-file = "" AND file-count = 0
        WHEN (NOT (STRING-EQUAL old-file new-file))
        DO (SETQ old-file new-file
                 file-count 0)
        WHEN (STRING-EQUAL old-file new-file)
        DO (SETQ file-count (ADD1 file-count))
        WHEN ( file-count 2)
        collect dir))


(DEFUN selective-file-delete (dir-name file-type option)
  "2DELETE FILE-TYPE VERSIONS AS INDICATED BY OPTION*"
  (LET* ((parsed        (fs:parse-pathname dir-name))
         (host-name     (fs:host-namestring parsed))
         (direct        (SEND parsed ':directory))
         (direct-string nil)
         (direct-list   nil)
         (file-list     nil)
         (delete-list   nil))
    (COND ((TYPEP direct ':string) (SETQ direct (LIST direct))))
    (SETQ direct-string
          (STRING-APPEND
            host-name
            (LOOP for name in direct
                  with str = "" AND dot =""
                  DO (SETQ str (STRING-APPEND str dot name)
                           dot ".")
                  finally (RETURN str))
            ";"))
    (SETQ direct-list (fs:directory-list (STRING-APPEND direct-string "*.*")))
    (SETQ file-list (LOOP for dir in (CDR direct-list)
                          WHEN (AND (NOT (GET dir ':directory))
                                    (EQ (READ-FROM-STRING (SEND (CAR dir) ':type))
                                        file-type))
                          collect dir))
    (SETQ delete-list
          (SELECTQ option
            ('keep-last-two  (keep-last-two-files file-list))
            ('keep-last-one (keep-last-file-only file-list))))
    (LOOP for dir in (REVERSE delete-list)
          for file = (CAR dir)
          DO (fs:delete-file file)
          DO (FORMAT t "~% File Deleted: ~A" (send file ':string-for-printing))
          finally (fs:expunge-directory direct-string))))
    
3(DEFUN walk (&optional (walk-string "Hello There..") (window terminal-io) (y 30))*
  "2Cute little routine that scrolls a little piece of text horizontally across the screen in
a background process.*"
  (LET* ((font fonts:43vxms))
    (LOOP for x from 100 to 350 by 5
          DO
          (UNWIND-PROTECT
            (PROGN (SEND window :string-out-explicit walk-string x y 1000 1000 font tv:alu-xor)
                   (PROCESS-SLEEP 15 "Zzzz..."))
            (SEND window :string-out-explicit walk-string x y 1000 1000 font tv:alu-xor)))))

(DEFCOM COM-REAP-FILE "Delete multiple versions of the specified file." ()
  (LET ((PATHNAME (READ-DEFAULTED-WILD-PATHNAME "Reap file" (DEFAULT-PATHNAME))))
    (PROMPT-LINE "")
    (REAP-FILE PATHNAME *MODE-LINE-WINDOW*))
  (FUNCALL STANDARD-OUTPUT ':MAKE-COMPLETE)
  DIS-NONE)

(DEFUN REAP-FILE (&OPTIONAL (PATHNAME "")
			    (PROMPT-STREAM STANDARD-OUTPUT))
  "Delete all but the last few versions of PATHNAME.
It tells you which versions there are and which it will delete,
then asks for confirmation."
  (SETQ PATHNAME (FS:MERGE-PATHNAME-DEFAULTS PATHNAME
					     (FUNCALL FS:LAST-FILE-OPENED ':NEW-PATHNAME
						      ':TYPE ':WILD ':VERSION ':WILD)
					     ':WILD ':WILD))
  (FORMAT PROMPT-STREAM "~&Reaping ~A" PATHNAME)
  (REAP-DIRECTORY PATHNAME STANDARD-OUTPUT))


(DEFCOM COM-CLEAN-DIRECTORY "Delete multiple versions in the specified directory." ()
  (LET ((PATHNAME (READ-DIRECTORY-NAME "Clean directory" (DEFAULT-PATHNAME))))
    (PROMPT-LINE "")
    (CLEAN-DIRECTORY PATHNAME STANDARD-OUTPUT))
  (FUNCALL STANDARD-OUTPUT ':MAKE-COMPLETE)
  DIS-NONE)

(DEFUN CLEAN-DIRECTORY (&OPTIONAL (PATHNAME FS:LAST-FILE-OPENED)
				  (PROMPT-STREAM STANDARD-OUTPUT))
  "Delete all but the last N-TO-KEEP versions of each file in PATHNAME.
PATHNAME may (and does by default) contain wildcards
so you can process all the files in a directory.
It tells you which versions there are and which it will delete,
then asks for confirmation, for each filename individually."
  (SETQ PATHNAME (FS:MERGE-PATHNAME-DEFAULTS PATHNAME
					     (FUNCALL FS:LAST-FILE-OPENED ':NEW-PATHNAME
						      ':NAME ':WILD ':TYPE ':WILD
						      ':VERSION ':WILD)
					     ':WILD ':WILD))
  (IF *numeric-arg-p*
      (FORMAT PROMPT-STREAM "~&Cleaning ~A, saving greatest ~D versions.~%"
	      PATHNAME *numeric-arg*)
    (FORMAT PROMPT-STREAM 
	    "~&Cleaning ~A deleting as follows:~%~%~{~{~5TFor ~A save latest ~A.~32T~
            ~:[Deleting from latest unbroken sequence of versions.~;~
               ALL older versions not specifically saved will be deleted.~]~%~}~}"
	    PATHNAME (mapcar #'(lambda (entry)
				 (append (firstn 2 entry)
					 (list (eq (caddr entry)
						   ':non-contig-doesnt-matter )) ))
			     *type-hog-alist*))
    (FORMAT PROMPT-STREAM  "~&~2TFor other file types save latest ~D versions.~%~%"
	     *FILE-VERSIONS-KEPT*) )
  (REAP-DIRECTORY PATHNAME STANDARD-OUTPUT))

(DEFPROP DIRED-PATHNAME-LESSP COMPILER:OBSOLETE COMPILER:STYLE-CHECKER)
(DEFPROP DIRED-PATHNAME-LESSP "is an obsolete function; use FS:PATHNAME-LESSP"
	 COMPILER:OBSOLETE)
(DEFF DIRED-PATHNAME-LESSP 'FS:PATHNAME-LESSP)

(DEFUN REAP-DIRECTORY (PATHNAME STREAM &AUX N-TO-KEEP DIRECTORY-LIST SOMETHING-DELETED)
  "Loops through directory 'PATHNAME' marking extra files for deletion, asks user about expunge."
  (SETQ DIRECTORY-LIST (FS:DIRECTORY-LIST PATHNAME ':SORTED))
  (SETQ DIRECTORY-LIST (DELQ (ASSQ NIL DIRECTORY-LIST) DIRECTORY-LIST))
  (DO* ((HEAD DIRECTORY-LIST)
	(LIST DIRECTORY-LIST (CDR LIST))
	(PATHNAME (CAAR LIST) (CAAR LIST))
	(PREV-NAME (AND PATHNAME (FUNCALL PATHNAME ':NAME))
		   NAME)
	(NAME PREV-NAME 
	      (AND PATHNAME (FUNCALL PATHNAME ':NAME)))
	(PREV-TYPE (AND PATHNAME (FUNCALL PATHNAME ':TYPE))
		   TYPE)
	(TYPE PREV-TYPE 
	      (AND PATHNAME (FUNCALL PATHNAME ':TYPE))) )
       ((NOT HEAD))
    (COND ((OR (NULL LIST)
	       (NOT (EQUAL PREV-NAME NAME))
	       (AND (NOT (EQUAL PREV-TYPE TYPE))
		    (NEQ (FUNCALL PATHNAME ':VERSION) ':UNSPECIFIC)))
	   (AND (SETQ N-TO-KEEP (COND (*NUMERIC-ARG-P* *NUMERIC-ARG*)
				      ((cadr (assoc TYPE *type-hog-alist*)))
				      (T *FILE-VERSIONS-KEPT*)))
		(REAP-ONE-FILE HEAD LIST N-TO-KEEP STREAM)
		(SETQ SOMETHING-DELETED T))
	   (SETQ HEAD LIST) )))
  (AND SOMETHING-DELETED
       (SEND PATHNAME ':UNDELETABLE-P)
       (LET ((QUERY-IO TERMINAL-IO))
	 (FQUERY NIL "Expunge ~A? " (SEND PATHNAME ':STRING-FOR-DIRECTORY)))
       (FORMAT T "~&~D blocks reclaimed." (FS:EXPUNGE-DIRECTORY PATHNAME)) ))

(DEFUN REAP-ONE-FILE (HEAD TAIL N-TO-KEEP STREAM
		      &AUX FIRST-DELETION-VERSION (N-VERSIONS 0) DELETE-LIST KEEP-LIST
		           contig-matters)
 "Returns T if something was deleted.
  Handles one file name and type, all versions.
  (Head - tail) is the set of versions of the current file, sorted by increasing version #."
  (setq contig-matters
	(neq (caddr (assoc (FUNCALL (CAAR head) ':type) *type-hog-alist*))
	     ':non-contig-doesnt-matter ))
  (DO* ((LIST HEAD (CDR LIST))
	(THIS-VERSION (AND LIST (FUNCALL (CAAR LIST) ':VERSION))
		      (AND LIST (FUNCALL (CAAR LIST) ':VERSION)))
	(LAST-VERSION nil) )
       ((EQ LIST TAIL))
    (WHEN (NUMBERP THIS-VERSION)
      (IF (AND LAST-VERSION
	       contig-matters
	       ( (1+ LAST-VERSION) THIS-VERSION))
	  (SETQ LAST-VERSION NIL
		N-VERSIONS 0))
      (UNLESS LAST-VERSION (SETQ FIRST-DELETION-VERSION THIS-VERSION))
      (SETQ N-VERSIONS (1+ N-VERSIONS)
	    LAST-VERSION THIS-VERSION)))
  ;; At this point:
  ;; FIRST-DELETION-VERSION is lowest version number to delete.
  ;; That is the bottom of the sequence of consecutive versions
  ;; that ends with the most recent version.
  ;; N-VERSIONS is number of versions that exist, starting with that version.
  (DO ((LIST HEAD (CDR LIST))
       (N-TO-DELETE -1)
       (FILE) (PATHNAME) (VERSION))
      ((EQ LIST TAIL)
       (SETQ DELETE-LIST (NREVERSE DELETE-LIST)
	     KEEP-LIST (NREVERSE KEEP-LIST)))
    (SETQ FILE (CAR LIST)
	  PATHNAME (CAR FILE)
	  VERSION (FUNCALL PATHNAME ':VERSION))
    (IF (EQ VERSION FIRST-DELETION-VERSION)
	(SETQ N-TO-DELETE (- N-VERSIONS N-TO-KEEP)))
    (IF (AND (OR (AND (NUMBERP VERSION) (PLUSP N-TO-DELETE))
		 (MEMBER (FUNCALL PATHNAME ':TYPE) *TEMP-FILE-TYPE-LIST*))
	     (NOT (GET FILE ':DONT-REAP)))
	(PUSH FILE DELETE-LIST)
      (PUSH FILE KEEP-LIST))
    (AND (NUMBERP VERSION)
	 (SETQ N-TO-DELETE (1- N-TO-DELETE))))
  ;; List versions to keep and delete, and check with the user.
  (COND (DELETE-LIST
	 (COND (KEEP-LIST
		(FORMAT STREAM "~&Keeping the following file~P:  (in ~A)~%"
			(LENGTH KEEP-LIST) (FUNCALL (CAAR KEEP-LIST) ':STRING-FOR-DIRECTORY))
		(DOLIST (FILE KEEP-LIST)
		  (FUNCALL *DIRECTORY-SINGLE-FILE-LISTER* FILE))))
	 (FORMAT STREAM "~&Deleting the following file~P:~:[ (in ~A)~]~%"
		 (LENGTH DELETE-LIST) KEEP-LIST
		 (FUNCALL (CAAR DELETE-LIST) ':STRING-FOR-DIRECTORY))
	 (DOLIST (FILE DELETE-LIST)
	   (FUNCALL *DIRECTORY-SINGLE-FILE-LISTER* FILE))
	 (AND (Y-OR-N-P "Ok? " STREAM)
	      (DOLIST (L DELETE-LIST T)
		(LET ((PATHNAME (CAR L)))
		  (CONDITION-CASE (ERROR)
		      (SEND PATHNAME ':DELETE nil)
		    (FS:FILE-ERROR
		      (FORMAT STREAM "~&Cannot delete ~A because ~A.~%" PATHNAME ERROR)))))))))



(DEFCOM COM-CHANGE-FILE-PROPERTIES "Change properties on a file" ()
  (LET ((PATHNAME (READ-DEFAULTED-PATHNAME "Change properties for" (PATHNAME-DEFAULTS)
					   NIL NIL ':DELETED)))
    (CHANGE-FILE-PROPERTIES PATHNAME))
  DIS-NONE)

;;; Really nice printing for ZWEI's Change File Properties
(DEFPROP :DONT-DELETE "Don't Delete" PRETTY-NAME)
(DEFPROP :DONT-REAP "Don't Reap" PRETTY-NAME)

(DEFUN CHANGE-FILE-PROPERTIES (PATHNAME &AUX DIRECTORY INDICATORS VALUES CHOICES CHANGES)
  (MULTIPLE-VALUE (DIRECTORY INDICATORS)
    (FS:FILE-PROPERTIES PATHNAME NIL))
  (AND (ERRORP DIRECTORY) (BARF "Err: ~A" DIRECTORY))
  (OR (SETQ PATHNAME (CAR DIRECTORY))
      (BARF "Err:  File not found"))
  (OR INDICATORS (BARF "File has no settable properties."))
  (SETQ VALUES (LOOP FOR IND IN INDICATORS
		     COLLECT (GET DIRECTORY IND)))
  (SETQ CHOICES (LOOP FOR IND IN INDICATORS
		      COLLECT (LIST IND
				    (OR (GET IND 'PRETTY-NAME)
					(PUTPROP IND
						 (STRING-CAPITALIZE-WORDS (STRING-APPEND IND))
						 'PRETTY-NAME))
				    (DO ((L FS:*KNOWN-DIRECTORY-PROPERTIES* (CDR L)))
					((NULL L) ':SEXP)
				      (AND (MEMQ IND (CDAR L))
					   (RETURN (CADDR (CAAR L))))))))
  (LET ((BASE 10.) (IBASE 10.) (*NOPOINT T))
    (*CATCH 'ABORT
      (PROGV INDICATORS VALUES
	(TV:CHOOSE-VARIABLE-VALUES CHOICES
				   ':LABEL (FORMAT NIL "Change properties for ~A" PATHNAME)
				   ':MARGIN-CHOICES '("Do It"
						      ("Abort" (*THROW 'ABORT T))))
	(SETQ CHANGES (LOOP FOR IND IN INDICATORS
			    FOR VAL IN VALUES
			    AS NEW = (SYMEVAL IND)
			    WHEN (NOT (EQUAL NEW VAL))
			    NCONC (LIST IND NEW))))
      (LEXPR-FUNCALL #'FS:CHANGE-FILE-PROPERTIES PATHNAME T CHANGES)
      CHANGES)))
