;;; -*- Mode:LISP; Package:ZWEI; Base:8; Fonts:(CPTFONT HL12B HL12I MEDFNB HL12BI) -*- 

;;;   Uppercase Global Functions Mode
;;;
;;;   This minor mode provides automatic uppercasing of function names that
;;;   are contained in the GLOBAL package.  When any word abbreviation mode
;;;   delimiter except '-' is typed, the editor will check the preceding atom 
;;;   to see if it needs uppercasing.
;;;
;;;   Besides the minor mode, two new commands are added.
;;;	-  com-uppercase-previous-global-function - uppercase the atom under point
;;;	     if it is the name of a function in the global package
;;;	-  com-uppercase-global-functions-region - uppercase global function names
;;;          from point to mark, using the current defun is no region is defined;
;;;	     with an argument, lowercase all non-global function names in the region
;;;
;;;		- rb 2/17/85 (this fix due to lo)


; From file MODES.LISP#> ZWEI; LAM10:
#8R ZWEI#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "ZWEI")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: ZWEI; MODES  "


(defminor com-uppercase-global-functions-mode
	  uppercase-global-functions-mode "Globals" 3
	  "Mode for capitalizing global functions, macros, etc." ()
  (command-hook 'uppercase-global-functions-hook *command-hook*))

(DEFCONST *uppercase-global-function-p* 'uppercase-global-function-p
  "The function called to check whether a symbol should be put in uppercase.
The function should accept three arguments:
1) The starting BP for the symbol
2) The ending BP for the symbol (also the current BP)
3) The actual string, the symbol-name.")

(DEFPROP uppercase-global-functions-hook 10 command-hook-priority)

(DEFUN uppercase-global-functions-hook (char)
  "Uppercases the atom preceding point if it is a function name in the global package."

  (IF (NOT (EQ *interval* (window-interval *mini-buffer-window*)))
      (uppercase-global-function-at-bp (point) char)))

(DEFUN uppercase-global-function-at-bp (bp &optional (char (bp-char bp)) (lowercase nil))
  "Uppercases atom before BP if it is the name of a function in the global package.
Conditions: (a) last command char must have been a WAM delimiter (except '-').
            (b) BP must not be inside a comment, string, or slash context."

    (AND (NOT (EQ (ldb %%ch-char char) #/-))           ; don't expand after dashes

	 (expand-p char)               ; expand after WAM delimiters

	 ; don't expand in comments,comp  strings or slashes
	 (NOT (MULTIPLE-VALUE-BIND (string slash comment)
		  (lisp-bp-syntactic-context bp (forward-defun bp -1 t))
		(or string slash COMMENT)))			     

	 ; okay, check for function name in global package under bp
	 (uppercase-global-function bp lowercase)))


(DEFUN uppercase-global-function (bp &optional (lowercase nil))
  "If the atom under BP is a global function or special form, uppercase it."

  (LET (bp1 str)
    (SETQ bp1  (forward-atom bp -1 t)                      ; beginning of atom
	  str  (STRING-UPCASE (string-interval bp1 bp)))   ; string with atom

    (COND ((FUNCALL *uppercase-global-function-p* bp1 bp str)
	   (UPCASE-INTERVAL BP1 BP T)

	   ; everything after beginning of atom needs redisplay
	   (must-redisplay *window* dis-line (bp-line bp1) (bp-index bp1)))
	  (lowercase
	   (DOWNCASE-INTERVAL BP1 BP T)
	   (must-redisplay *window* dis-line (bp-line bp1) (bp-index bp1))))))


(DEFUN uppercase-global-function-p (start-bp current-bp string)
  "Tests whether or not the current word should be capitalized."
  (DECLARE (RETURN-LIST t-or-nil)
	   (IGNORE start-bp))
  (AND
    ;;Check to see if symbol is function in global package
    (FUNCTIONP (INTERN-SOFT string 'global) t)
    ;;Check to see if we are typing a function name. (after a paren)
;    (%STRING-EQUAL "(" 0
;		   (string-interval (forward-char start-bp -1 t) start-bp) 0 1)
    ;;Check whether symbol is the current function (this is a better, but more costly, method)
    (STRING-EQUAL (RELEVANT-FUNCTION-NAME current-bp t) string)
    ))

(defcom com-uppercase-previous-global-function
	"Uppercase the atom under point if it is the name of a global function." ()
  (uppercase-global-function (point))
  dis-none)

(defcom com-uppercase-global-functions-region
	"Uppercase global function names from point to mark.
If there is no region, the current definition is used.
If run with a numeric parameter, make non-global function names lowercase" ()

  (LET (bp1 bp2)
  
    (IF (window-mark-p *window*)               ; if there is a region
	(SETQ bp1 (point) bp2 (mark))          ; grab point and mark
      (SETQ bp1 (forward-defun (point) -1 t)   ; else use begin and end 
	    bp2 (forward-defun (point) 1 t)))  ; of current DEFUN

    (order-bps bp1 bp2)                        ; put BPs in order
    
    (DO ((bp-temp (forward-char bp1 -1 t)))    ; move forward one char so we don't
                                               ; catch the atom just before BP1
	((OR (bp-< bp2 bp-temp)
	     (bp-= bp2 bp-temp)))              ; stop at BP2
      
      (SETQ bp-temp (forward-atom bp-temp 1 t)) ; move forward one atom
      (uppercase-global-function-at-bp bp-temp (bp-char bp-temp)
				*numeric-arg-p*)))	; and try uppercasing it

  dis-text)

(set-comtab *standard-comtab*
	    '(#\super-c com-uppercase-previous-global-function
	      #\super-k com-uppercase-global-functions-region)
	    (make-command-alist '(com-uppercase-global-functions-mode)))
))