;;; -*- Mode: COMMON-LISP; Package: TV; Base: 10; Fonts: MEDFNT, MEDFNB, HL12B; -*-
;;;===============================================================================
;;;                                    RESTRICTED RIGHTS LEGEND 
;;; Use,  duplication, or  disclosure  by  the  Government is subject to restrictions
;;; as set forth in subdivision (b)(3)(ii) of the Rights in Technical Data and
;;; Computer Software clause at 52.227-7013. 
;;;
;;; TEXAS INSTRUMENTS INCORPORATED, P.O. BOX 2909 AUSTIN, TEXAS 78769  
;;; Copyright (C) 1984, Texas Instruments Incorporated. All rights reserved.
;;;===============================================================================

;1; Blank the screen after BLANK-SCREEN-TIME-DELAY (default 20) seconds*

;;;
;;; Change history:
;;;
;;;  Date	Author	Description
;;; -------------------------------------------------------------------------------------
;1;; 06/08/87*	1LGO*	1Modified to work under Explorer release 2.1*
;1;; 05/12/87*	1LGO*	1Modified to ensure two blackout process aren't started*
;1;; 05/11/87*	1LGO*	1Modified to queue notifications, and print thim on the *
;1;;*			1initial-lisp-listener AFTER un-blanking the screen.*
;1;;*			1This gets around MORE processing problems in the Lisp Listener.*
;1;; 04/30/87*	1LGO*	1Modified to move the who-line documentation back and forth*
;1;; 04/27/87*	1LGO*	1Modified to forward notifications to the initial-lisp-listener*
;1;; 03/24/87*	1LGO*	1Converted for Explorer release 3.0*
;1;; 09/17/84*	1LGO*	1Created*

#+elroy ;; already defined in release 2
(DEFVAR black-screen-time-delay 20)
#+elroy ;; already defined in release 2
(DEFVAR the-screen-is-black nil)

;1; This could run anyplace that executes once a minute.*
;1; We don't have a seperate process, because that's too much overhead...*
(DEFMETHOD (who-line-file-sheet :after :update) ()
  "Makes the screen black after a period if inactivity"
  (WHEN (AND black-screen-time-delay
	     (>= (TRUNCATE
		   (TIME-DIFFERENCE (TIME) tv:kbd-last-activity-time) 3600.)
		 black-screen-time-delay)
	     (NOT the-screen-is-black))
    (SETQ the-screen-is-black t) ;1; Set here to ensure two blackout processes aren't started*
    (PROCESS-RUN-FUNCTION "blackout" 'blackout-screen-hack)))

;1; Full screen window with no save array for covering up the screen*
(DEFFLAVOR blackout-screen-window
	   ((who-line-doc "Press any key or move mouse to continue")
	    (who-line-doc-aux nil)
	    (who-line-doc-len nil)
	    (who-line-doc-position 0)
	    (who-line-doc-direction 1)
	    (who-line-doc-stop)
	    (notification-list nil))
	   (tv:not-externally-selectable-mixin
	    tv:kbd-mouse-buttons-mixin
	    tv:stream-mixin
	    tv:graphics-mixin
	    tv:select-mixin
	    tv:minimum-window)
  (:default-init-plist
    :reverse-video-p nil
    :blinker-p nil
    :save-bits nil
    :superior main-screen)
  (:documentation :combination "Window to use while blanking the screen"))

(DEFUN array-copy (from-array from-start to-array to-start length)
  (si:COPY-ARRAY-PORTION from-array from-start (+ from-start length) to-array to-start (+ to-start length)))

(DEFMETHOD (blackout-screen-window :who-line-documentation-string) ()
  ;1; Setup: First time only*
  (UNLESS who-line-doc-len
    (SETQ who-line-doc-len (+ 4 (LENGTH who-line-doc)))
    (SETQ who-line-doc
	  (FORMAT nil "  ~a  ~v@t" who-line-doc
		  (- (SEND superior :size-in-characters) (LENGTH who-line-doc))
		  ))
    (SETQ who-line-doc-aux (make-string (length who-line-doc)))
    (REPLACE who-line-doc-aux who-line-doc)
    (SETQ who-line-doc-stop (- (LENGTH who-line-doc) who-line-doc-len 1)))
  (ROTATEF who-line-doc who-line-doc-aux)
  ;1; Shift the string*
  (IF (PLUSP who-line-doc-direction)
      ;1; Move string right*
      (PROGN 
	(array-copy who-line-doc-aux who-line-doc-position
		    who-line-doc (1+ who-line-doc-position) who-line-doc-len)
	(IF (>=  who-line-doc-position who-line-doc-stop)
	    (SETQ who-line-doc-direction -1)
	  (INCF who-line-doc-position who-line-doc-direction)))
    ;1; Move string left*
    (array-copy who-line-doc-aux (1+ who-line-doc-position)
		who-line-doc who-line-doc-position who-line-doc-len)
    (IF (ZEROP who-line-doc-position)
	(SETQ who-line-doc-direction 1)
      (INCF who-line-doc-position who-line-doc-direction)))
  who-line-doc)

;1; Forward notifications to the initial-lisp-listener*

;; 1Save notifications for later display.*
(DEFMETHOD (blackout-screen-window :print-notification) (time string window-of-interest)
  (push-end (list time string window-of-interest) notification-list))

(DEFMETHOD (blackout-screen-window :print-delayed-notifications) ()
  (LET ((notifications notification-list))
    (SETQ notification-list nil)    
    (WHEN (AND notifications (NEQ selected-window initial-lisp-listener))
      (notify selected-window "See ~s for ~d delayed notifications."
	      initial-lisp-listener (LENGTH notifications)))
    (LOOP for (time string window-of-interest) in notifications
	  with beep = nil do
	  (IF (SEND initial-lisp-listener :active-p)
	      (SEND initial-lisp-listener :print-notification-on-self time string window-of-interest)
	    (SEND selected-window :print-notification time string window-of-interest)))))

(DEFMETHOD (blackout-screen-window :mouse-moves) (&rest IGNORE)
  (SEND SELF ':FORCE-KBD-INPUT 0))

(COMPILE-FLAVOR-METHODS blackout-screen-window)

(DEFVAR blackout-screen-window (MAKE-INSTANCE 'blackout-screen-window))

(DEFUN blackout-screen-hack (&aux who-line-reverse-p bow-mode)
  "2Black out the screen until a character is typed.*"
  ;1; Disable the abort, system and other such keys*
  (LET ((save-kbd-intercepted-characters kbd-intercepted-characters)
	(save-kbd-global-asynchronous-characters kbd-global-asynchronous-characters))
    ;1; Black out the screen*
    (UNWIND-PROTECT
	(PROGN
	  (SETQ the-screen-is-black t)
	  (SETQ kbd-intercepted-characters nil)
	  (SETQ kbd-global-asynchronous-characters nil)
	  (SETQ who-line-reverse-p
		(SEND who-line-documentation-window ':reverse-video-p))
	  (SETQ bow-mode *current-screen-color*)
	  (SEND who-line-documentation-window ':set-reverse-video-p nil)
	  (white-on-black)		   ;1Make edges black too*
	  (window-call (blackout-screen-window :deactivate)
	    (SETQ mouse-reconsider t)
	    (PROCESS-SLEEP 5)		   ;1Wait for mouse moves*
	    (SEND blackout-screen-window ':clear-input)	   ;1Then clear input*
	    (mouse-set-blinker-definition :character 0 0 t
					  :set-character #+elroy mouse-glyph-hourglass #-elroy 19.)
	    ;1; Wait for typein OR deexposure... Just in case some program*
	    ;1; running out there pops up a window*
	    (PROCESS-WAIT "Keyboard +"	
			  #'(lambda (w width height)
			      (OR (SEND w ':mouse-or-kbd-tyi-no-hang)
				  (NOT (SEND w ':exposed-p))
				  (PROGN (black-screen-hacks w width height) nil)))
			  blackout-screen-window
			  (SEND blackout-screen-window :width)
			  (SEND blackout-screen-window :height))))
      (SETQ kbd-intercepted-characters save-kbd-intercepted-characters
	    kbd-global-asynchronous-characters save-kbd-global-asynchronous-characters)
      (WHEN bow-mode (black-on-white))
      (SEND who-line-documentation-window ':set-reverse-video-p
	    who-line-reverse-p)
      (SETQ tv:kbd-last-activity-time (TIME))
      (SETQ mouse-reconsider t)
      (SETQ the-screen-is-black nil)
      ;1; do this last, because it may hang in *MORE* processing*
      (SEND blackout-screen-window :print-delayed-notifications))))

;(DEFUN black-screen-hacks (window width height)
;  ;1; Do something on the screen when idle*
;  window ;1; not used*
;  (WHEN (>= (INCF mouse-x) width)
;    (SETQ mouse-x 0)
;    (WHEN (>= (INCF mouse-y) height)
;      (SETQ mouse-y 0))))

(DEFUN black-screen-hacks (window width height)
  ;1; Moves the mouse in a rotating heart pattern*
  (DECLARE (SPECIAL bsh-angle bsh-radius bsh-sign)
	   (ignore window))
  (UNLESS (BOUNDP 'bsh-angle)
    (SETQ bsh-angle 0.0s0 bsh-radius 5 bsh-sign 1))
  (SETQ mouse-x (+ (FLOOR width 2) (FLOOR (* bsh-radius (cos bsh-angle)))))
  (SETQ mouse-y (+ (FLOOR height 2) (FLOOR (* bsh-radius (sin bsh-angle)))))
  (INCF bsh-angle #.(COERCE (/ pi 360) 'short-float))
  (WHEN (> bsh-angle #.(COERCE (* pi 2) 'SHORT-FLOAT))
    (DECF bsh-angle #.(COERCE (* pi 2) 'SHORT-FLOAT)))
  (INCF bsh-radius bsh-sign)
  (WHEN (>= bsh-radius (FLOOR height 2))
    (SETQ bsh-sign -1))
  (WHEN (<= bsh-radius 5)
    (SETQ bsh-sign 1)))

;1; Debug - Execute this to invoke the blackout*
;1 *(SETQ tv:kbd-last-activity-time (- (TIME) (* 3600 black-screen-time-delay)))1 
