;;; -*- Base: 10 -*-

;;; This code shows how to write simple programs that use mouse interaction.

;;; Here I make a flavor.  It just includes the flavor TV:WINDOW.  I have to
;;; make a new flavor because I don't want all windows to have the method 
;;; [behavior] that I'm about to define.
;;;
(defflavor twindow ()   ;no instance variables
	   (tv:window)) ;one component flavor, no options.

;;; The :mouse-moves message is sent by the MOUSE PROCESS every time the
;;; mouse hardware moves.  You can define a new method for this message
;;; that  tells the mouse process what to do when the mouse is over a
;;; window of  this flavor.

;;; Here we define an "etch-a-sketch" window.  It draws little  squares
;;; when the control button is down, and erases little squares when  the
;;; hyper button is down.  If neither is down, it just displays the mouse
;;; cursor.
;;;
(defmethod (twindow :mouse-moves) (x y) ;new coordinates
  (tv:mouse-set-blinker-cursorpos)  ;this puts the mouse cursor at the new position
  (cond ((tv:key-state ':hyper) (send self ':draw-rectangle 10 20 x y tv:alu-andca))
	((tv:key-state ':control) (send self ':draw-rectangle 2 2 x y tv:alu-ior))))


;;; Try this: (setq tw (tv:make-window 'twindow ':edges-from ':mouse ':expose-p t))
;;; and move the mouse over the window, sometimes holding the control button down 
;;; sometimes holding the hyper button down.

;;; **********************************************************************



;;; Now we add some new features.  Make a new flavor, building on what we
;;; already have, so that we can specify end points of a line.  The
;;; tv:list-mouse-buttons-mixin allows us to receive MOUSE BLIPS.  When we 
;;; send the :tyi message normally, we can only get a character back.  With 
;;; this flavor mixed in, we can receive a list in the format:

;;;       (MOUSE-BUTTONS encoded-click window x y)

;;; MOUSE-BUTTONS is just the symbol, MOUSE-BUTTONS.  Usually you don't need 
;;; to use this, but it helps to distinguish between different kinds of blips
;;; in complicated user-interfaces.  ENCODED-CLICK is a number represented what 
;;; button was pressed.  Compare this number with #\MOUSE-L-1, #\MOUSE-M-2, etc.
;;; WINDOW is the window you were over, and X and Y are the coordinates relative 
;;; to the upper-left corner of that window.
;;; 
(defflavor twindow-with-click ()
	   (tv:list-mouse-buttons-mixin twindow))


;;; This function lets us use the above feature.
;;; Remember that you can still use the first kind of drawing!
;;;
(defun draw-lines (w)
  (send w ':expose)  ;this makes sure that the window is visible
  (loop for first-blip = (send w :list-tyi)
	for second-blip = (send w :list-tyi)
	do
	(send w ':draw-line (fourth first-blip) (fifth first-blip)
			    (fourth second-blip) (fifth second-blip))))


;;; Try this: (draw-lines (setq twc (tv:make-window 'twindow-with-click
;;;                                                 ':edges-from ':mouse
;;;                                                 ':expose-p t)))

;;; **********************************************************************



;;; Just for fun, we can write a defstruct to use on blips:
(defstruct (blip :conc-name (:type :list))
  type
  button
  window
  x
  y)

;;; Then we don't have to remember which element is the x-coordinate
;;; and which is the y-coordinate, etc.


(defun draw-lines-1 (w)
  (send w ':expose)
  (loop for first-blip = (send w :list-tyi)
	for second-blip = (send w :list-tyi)
	do
	(send w ':draw-line (blip-x first-blip) (blip-y first-blip)
			    (blip-x second-blip) (blip-y second-blip))))


;;; Here is a more complicated function that does generally the same
;;; thing, but allows you to click on the middle button to end the function.

(defun draw-lines-2 (w)
  (send w ':expose)
  (loop with first-blip
	with second-blip
	do
	(setq first-blip (send w :list-tyi))
	(when (= (blip-button first-blip) #\mouse-m-1)
	  (send w ':bury)  ;this makes the window disappear
	  (return nil))    ;the LOOP will return nil
	(setq second-blip (send w :list-tyi))
	(when (= (blip-button second-blip) #\mouse-m-1)
	  (send w ':bury)
	  (return nil))
	(send w ':draw-line (blip-x first-blip) (blip-y first-blip)
			    (blip-x second-blip) (blip-y second-blip))))