/*
 * array.c
 *
 * generic array-handling functions
 */

#include "exploiter.h"
#include "interp.h"
#include "memory.h"
#include "utils.h"

#include <stdio.h>
#include <stdlib.h>

int resolve_aref(lisp_q array, int num_dims, int force, lisp_q dimptr, lisp_q *header, lisp_q *locative, int *offset)
{
    lisp_q displaced_to;
    u32 index;
    u32 i;
    u32 index_length;
    u32 index_offset;
    
    if (DTP(array) != DTP_ARRAY) {
	printf("resolve_aref: array not DTP_ARRAY.\n");
	dump_q(array, 0);
	return 0;
    }

    *header = memread(array);

#if 1
    dump_q(*header, 0);
    dump_array_header(*header);
#endif

    if (DTP(*header) != DTP_ARRAY_HEADER) {
	printf("resolve_aref: array header not DTP_ARRAY_HEADER.\n");
	dump_q(*header, 0);
	return 0;
    }

    if (force) {
	if (num_dims != 1) {
	    printf("resolve_aref: Attempting to force with wrong dimension count %d.\n", num_dims);
	    return 0;
	}
    } else {
	if (num_dims != ARY_DIMS(*header)) {
	    printf("resolve_aref: num_dims %d not equal to array arity %ld, and not forcing.\n", num_dims, ARY_DIMS(*header));
	    return 0;
	}
    }

    index_length = ARY_INDEX(*header);
    
    if (ARY_LL(*header)) {
	printf("resolve_aref: LL set.\n");
	index_length = fixnum_value(memread(array + 1));
    }

#if 0
    if (ARY_NS(*header)) {
	printf("resolve_aref: NS set.\n");
	return 0;
    }
#endif

    index_offset = 0;

    if (ARY_DISPLACED(*header)) {
	printf("resolve_aref: Displaced.\n");

	displaced_to = memread(array + ARY_DIMS(*header));

	index_length = fixnum_value(memread(array + ARY_DIMS(*header) + 1));

	if (DTP(displaced_to) == DTP_ARRAY) {
	    lisp_q displaced_header;
	    
	    printf("resolve_aref: Displaced to array.\n");

	    displaced_header = memread(displaced_to);
	    
	    /* FIXME: Index length adjustment */

	    displaced_to += ARY_DIMS(displaced_header);
	    displaced_to += !!ARY_LL(displaced_header);
	    
/* 	    return 0; */
	}

	if (ARY_INDEX(*header) == 3) {
	    printf("resolve_aref: Displaced index offset.\n");

	    index_offset = fixnum_value(memread(array + ARY_DIMS(*header) + !!ARY_LL(*header) + 2));
	}
    } else {
	/* DTP-Array displaced_to will be special key later */
	displaced_to = ADDRESS(array + ARY_DIMS(*header) + !!ARY_LL(*header));
    }

    if (ARY_PHYSICAL(*header)) {
	printf("resolve_aref: Physical.\n");
	return 0;
    }

    index = 0;
    if (force) {
	i = 1;
    } else {
	for (i = 1; i < ARY_DIMS(*header); i++) {
	    /* FIXME: Compute index here */
/* 	    printf("resolve_aref: arity other than 1 not supported.\n"); */
/* 	    return 0; */

	    index += ADDRESS(memread((dimptr + i) - 1)) *
		ADDRESS(memread(array + i + !!ARY_LL(*header)));
	}
    }
    index += ADDRESS(memread((dimptr + i) - 1));

    index += index_offset;

    /* Damn, but I hate switch statements. --AB */
    switch (ARY_TYPE(*header)) {
    case ART_1B:
	*locative = displaced_to + (index >> 5);
	*offset = index & 31;
	break;
	
    case ART_2B:
	*locative = displaced_to + (index >> 4);
	*offset = index & 15;
	break;
	
    case ART_4B:
	*locative = displaced_to + (index >> 3);
	*offset = index & 7;
	break;
	
    case ART_8B:
    case ART_STRING:
	*locative = displaced_to + (index >> 2);
	*offset = index & 3;
	break;
	
    case ART_16B:
    case ART_HALF_FIX:
    case ART_FAT_STRING:
	*locative = displaced_to + (index >> 1);
	*offset = index & 1;
	break;

    case ART_32B:
    case ART_Q:
    case ART_Q_LIST:
    case ART_STACK_GROUP_HEAD:
    case ART_SPECIAL_PDL:
    case ART_REG_PDL:
    case ART_SINGLE_FLOAT:
    case ART_FIX:
	*locative = displaced_to + index;
	*offset = 0;
	break;

    case ART_DOUBLE_FLOAT:
    case ART_COMPLEX:
    case ART_COMPLEX_SINGLE_FLOAT:
	*locative = displaced_to + (index << 1);
	*offset = 0;
	break;

    case ART_COMPLEX_DOUBLE_FLOAT:
	*locative = displaced_to + (index << 2);
	*offset = 0;
	break;
	
    default:
	printf("resolve_aref: Unhandled array type.\n");
	return 0;
    }
    
    return 1;
}

int read_aref(lisp_q header, lisp_q locative, int offset, lisp_q *destination, int cl_ar)
{
    /* FIXME: Handle physically-displaced arrays. */
    
    /* Damn, but I hate switch statements. --AB */
    switch (ARY_TYPE(header)) {
    case ART_1B:
	*destination = memread(locative);
	*destination >>= offset;
	*destination &= 1;
	*destination |= DTP_FIX;
	break;
	
    case ART_STRING:
	*destination = memread(locative);
	*destination >>= (offset << 3);
	*destination &= 0xff;
	*destination |= (cl_ar)? DTP_CHARACTER: DTP_FIX;
	break;
	
    case ART_16B:
	*destination = memread(locative);
	if (offset) *destination >>= 16;
	*destination &= 0xffff;
	*destination |= DTP_FIX;
	break;

    case ART_Q:
    case ART_Q_LIST:
	*destination = memread(locative);
	break;

    default:
	printf("read_aref: Unhandled array type.\n");
	return 0;
    }

    dump_q(*destination, 0);

    return 1;
}

int write_aref(lisp_q header, lisp_q locative, int offset, lisp_q value)
{
    lisp_q tmp;
    
    /* FIXME: Handle physically-displaced arrays. */
    
    /* Damn, but I hate switch statements. --AB */
    switch (ARY_TYPE(header)) {
    case ART_STRING:
	/* FIXME: Is this the right way around? */
	tmp = memread(locative);
	tmp &= ~(0xff << ((offset) << 3));
	tmp |= (value & 0xff) << ((offset) << 3);
	memwrite(locative, tmp);
	break;
	
    case ART_16B:
	if (DTP(value) != DTP_FIX) {
	    printf("write_aref: array type ART_16B and value not DTP_FIX.\n");
	    exit(-1);
	}
	tmp = memread(locative);
	value &= 0xffff;
	if (offset) {
	    value <<= 16;
	    tmp &= 0xffff;
	} else {
	    tmp &= 0xffff0000;
	}
	tmp |= value;
	memwrite(locative, tmp);
	break;

    case ART_Q:
    case ART_Q_LIST:
	memwrite(locative, value);
	break;
	
    default:
	printf("write_aref: Unhandled array type.\n");
	return 0;
    }

/*     dump_q(*destination, 0); */

    return 1;
}

/* EOF */
