/*
 * dumpraw.c
 *
 * various debugging display functions
 */

#include "utils.h"
#include "exploiter.h"
#include "memory.h"
#include "interp.h"

#include <stdio.h>
#include <stdlib.h>

char *ary_type_names[32] = {
    "ART-ERROR",
    "ART-1B",
    "ART-2B",
    "ART-4B",
    "ART-8B",
    "ART-16B",
    "ART-32B",
    "ART-Q",
    "ART-Q-LIST",
    "ART-STRING",
    "ART-STACK-GROUP-HEAD",
    "ART-SPECIAL-PDL",
    "ART-HALF-FIX",
    "ART-REG-PDL",
    "ART-DOUBLE-FLOAT",
    "ART-SINGLE-FLOAT",
    "ART-FAT-STRING",
    "ART-COMPLEX-DOUBLE-FLOAT",
    "ART-COMPLEX",
    "ART-COMPLEX-SINGLE-FLOAT",
    "ART-FIX",
    "???",
    "???",
    "???",
    "???",
    "???",
    "???",
    "???",
    "???",
    "???",
    "???",
    "???"
};

void dump_array_header(lisp_q ary_header)
{
    int ary_type;

    ary_type = ARY_TYPE(ary_header) >> 19;
    
    printf("AryHeader: Type: %d (%s) P: %d  L: %d  D: %d  S: %d\n"
	   "    Dims: %ld  LL: %d  NS: %d  Index: %ld.\n",
	   ary_type, ary_type_names[ary_type],
	   !!ARY_PHYSICAL(ary_header), !!ARY_LEADER(ary_header),
	   !!ARY_DISPLACED(ary_header), !!ARY_SIMPLE(ary_header),
	   ARY_DIMS(ary_header), !!ARY_LL(ary_header),
	   !!ARY_NS(ary_header), ARY_INDEX(ary_header));
}

void dump_raw_array(lisp_q aryptr)
{
    int arysize;
    lisp_q arystart;
    lisp_q aryheader;
    int i;

    aryheader = memread(aryptr);

    dump_array_header(aryheader);
    
    if (ARY_LEADER(aryheader)) {
	arysize = 3 + ADDRESS(memread(aryptr - 1));
	arystart = 1 + aryptr - arysize;
    } else {
	arystart = aryptr;
	arysize = 1;
    }

    arysize += ARY_INDEX(aryheader);

    for (i = 0; i < arysize; i++) {
	dump_q(memread(arystart+i), arystart+i);
    }
}

void internal_dump_string(lisp_q strptr)
{
    int arysize;
    lisp_q aryheader;
    u32 foo;
    int i;
    char c;

    aryheader = inviz(strptr, &strptr);

    while (DTP(aryheader) == DTP_HEADER_FORWARD) {
	strptr = aryheader;
	aryheader = inviz(strptr, &strptr);
    }

    arysize = (aryheader & 0x1ff);

    if (ARY_LEADER(aryheader)) {
	lisp_q fillpointer;

	fillpointer = memread(strptr - 2);
	if (DTP(fillpointer) == DTP_FIX) {
	    arysize = fixnum_value(fillpointer);
	}
    }

    foo = 0;
    
    for (i = 0; i < arysize; i++) {
	if ((i & 3) == 0) {
	    foo = memread(strptr + (i >> 2) + 1);
	}
	c = foo >> (i << 3);
	printf("%c", c);
    }
}

void dump_string(lisp_q string)
{
    internal_dump_string(string);
    printf("\n");
}

void print_list(lisp_q list);

void print_object(lisp_q object)
{
    if ((DTP(object) == DTP_LIST) ||
	(DTP(object) == DTP_STACK_LIST)){
	print_list(object);
    } else if (DTP(object) == DTP_SYMBOL) {
	internal_dump_string(memread(object));
    } else if (DTP(object) == DTP_FIX) {
	printf("%d", fixnum_value(object));
    } else if (DTP(object) == DTP_ARRAY) {
	lisp_q aryheader;
	aryheader = memread(object);
	if (ARY_NS(aryheader)) {
	    printf("<named structure %08lx>", object);
	} else {
	    printf("<array %08lx>", object);
	}
    } else if (DTP(object) == DTP_FUNCTION) {
	printf("<function %08lx>", object);
    } else {
	printf("\nprint_object: unrecognized object type.\n");
	dump_q(object, 0);
	exit(-1);
    }
}

void print_list(lisp_q list)
{
    lisp_q mycar;
    lisp_q mycdr;
    
    printf("(");

    mycdr = list;

    do {
	mycar = car(mycdr);
	mycdr = cdr(mycdr);

	print_object(mycar);

	/* FIXME: Umm... What?!? */
	if (NOT_CDRCODE(mycdr) != C_NIL) {
	    printf(" ");
	}

	if ((DTP(mycdr) != DTP_LIST) &&
	    (DTP(mycdr) != DTP_STACK_LIST) &&
	    (NOT_CDRCODE(mycdr) != C_NIL)) {
	    printf(". ");
	    print_object(mycdr);
	}
    } while ((DTP(mycdr) == DTP_LIST) || (DTP(mycdr) == DTP_STACK_LIST));

    printf(")");
}

lisp_q dump_function_name(lisp_q fefptr)
{
    lisp_q debug_info;
    lisp_q function_name;

    debug_info = memread_inviz(fefptr+2);
    if (DTP(debug_info) != DTP_ARRAY) {
	printf("function debug info not ARRAY.\n");
	dump_q(debug_info, 1);
	exit(-1);
    }
    function_name = memread_inviz(debug_info+2);

    if (DTP(function_name) == DTP_SYMBOL) {
	dump_string(memread(function_name + SYM_PRINTNAME));
    } else if (DTP(function_name) == DTP_LIST) {
	print_list(function_name);
	printf("\n");
    } else {
	printf("function name has unknown datatype.\n");
	dump_q(function_name, 0);
	exit(-1);
    }

    return function_name;
}

lisp_q dump_function_arglist(lisp_q fefptr)
{
    lisp_q debug_info;
    lisp_q arglist;

    debug_info = memread_inviz(fefptr+2);
    if (DTP(debug_info) != DTP_ARRAY) {
	printf("function debug info not ARRAY.\n");
	dump_q(debug_info, 1);
	exit(-1);
    }
    arglist = memread_inviz(debug_info+3);

    if (NOT_CDRCODE(arglist) == C_NIL) {
	printf("()\n");
    } else {
	print_object(arglist);
	printf("\n");
    }

    return arglist;
}

void dump_local_name(lisp_q fef, int offset)
{
    lisp_q debug_info;
    lisp_q local_map;
    
    debug_info = memread_inviz(fef + 2);
    if (DTP(debug_info) != DTP_ARRAY) {
	printf("function debug info not ARRAY.\n");
	dump_q(debug_info, 1);
	exit(-1);
    }

    local_map = memread_inviz(debug_info+5);

    if (DTP(local_map) != DTP_LIST) {
	printf("function local map not LIST.\n");
	dump_q(local_map, 1);
	exit(-1);
    }

    while (offset--) {
	local_map = cdr(local_map);
    }

    local_map = car(local_map);

    if (NOT_CDRCODE(local_map) == C_NIL) {
	printf("<no mapping>");
    } else {
	print_object(local_map);
    }

    printf("\n");
}

void dump_arg_name(lisp_q fef, int offset)
{
#if 0 /* Doesn't work, the stored arglist is a proper lambda-list. */
    lisp_q debug_info;
    lisp_q arglist;

    /*
     * FIXME: Arg references are only generated for required and
     * optional args. If we could get the number of required args
     * from the FEF header, we could add one to the offset if it
     * were beyond that number so as to skip the &OPTIONAL. This
     * might work well enough for now.
     *
     * Alternatively, we could grovel through the package structs
     * on startup and find the symbol for &OPTIONAL in the keyword
     * package and check for that as we cdr down the arglist.
     */
    
    debug_info = memread_inviz(fef + 2);
    if (DTP(debug_info) != DTP_ARRAY) {
	printf("function debug info not ARRAY.\n");
	dump_q(debug_info, 1);
	exit(-1);
    }

    arglist = memread_inviz(debug_info+3);

    if (DTP(arglist) != DTP_LIST) {
	printf("function arglist not LIST.\n");
	dump_q(arglist, 1);
	exit(-1);
    }

    while (offset--) {
	arglist = cdr(arglist);
    }

    arglist = car(arglist);

    if (NOT_CDRCODE(arglist) == C_NIL) {
	printf("<no mapping>");
    } else {
	print_object(arglist);
    }
#endif
    printf("\n");
}

void dump_raw_fef(lisp_q fefptr)
{
    int codestart;
    int fefsize;
    int i;
    lisp_q tmp;
    lisp_q header;
    lisp_q function_name;
    int skip_instruction;

    header = memread(fefptr);
    codestart = header & 0x3ff;
    fefsize = ADDRESS(memread(fefptr+1));

    function_name = dump_function_name(fefptr);

    printf("FEF: %ld req, %ld opt, %ld loc, calltype %ld.\n",
	   (header >> 14) & 15, (header >> 18) & 7,
	   (header >> 10) & 15, (header >> 21) & 7);

    dump_function_arglist(fefptr);
    
    for (i = 0; i < codestart; i++) {
	dump_q(memread(fefptr+i), fefptr+i);
    }

    skip_instruction = 0;
    
    for (i = codestart; i < fefsize; i++) {
	tmp = memread(fefptr+i);
	
	if (--skip_instruction) {
	    skip_instruction = disassemble_instr(fefptr, (i << 1) + 0,
						 tmp & 0xffff);
	}
	if (--skip_instruction) {
	    skip_instruction = disassemble_instr(fefptr, (i << 1) + 1,
						 tmp >> 16);
	}
    }

    dump_q(function_name, -1);
}

void dump_symbol_table(lisp_q table)
{
    lisp_q header;
    lisp_q size;
    int num_entries;
    int i;
    
    header = memread(table);
    dump_array_header(header);
    size = memread(table + 1);
    dump_q(size, 0);

    num_entries = fixnum_value(size);
    
    for (i = 0; i < num_entries; i += 2) {
	lisp_q word0;
	lisp_q word1;

	word0 = memread(table + i + 3);
	word1 = memread(table + i + 4);

	if (DTP(word0) == DTP_FIX) {
	    printf("hash: %06lx, sym: ", word0 & 0x00ffffff);
	    dump_string(memread(word1));
	}
    }
}

/* EOF */
