/*
 * mainop.c
 *
 * mainop instruction handlers
 */

#include "exploiter.h"
#include "context.h"
#include "memory.h"
#include "interp.h"
#include "utils.h"
#include "funcall.h"
#include <stdio.h>

typedef int (*mainop_handler)(u16 opcode);

#define MAINOP(foo) int mainop_##foo(u16 opcode)


MAINOP(003) { /* EQ-IMMED */
    lisp_q foo;

    foo = (DTP_FIX | (opcode & 0xff) | ((opcode & 0x100)? 0x01ffff00: 0));
    context.indicators = (NOT_CDRCODE(pop()) == foo)? C_T: C_NIL;

    return 1;
}

MAINOP(004) { /* =-IMMED */
    lisp_q foo;
    lisp_q bar;
	
    foo = opcode & 0xff;
    if (opcode & 0x100) foo |= 0x01ffff00;
    foo |= DTP_FIX;

    bar = NOT_CDRCODE(pop());

    context.indicators = (foo == bar)? C_T: C_NIL;
    
    return 1;
}

MAINOP(005) { /* >-IMMED */
    lisp_q foo;
    lisp_q bar;
	
    foo = NOT_CDRCODE(pop());
    
    bar = opcode & 0xff;
    if (opcode & 0x100) bar |= 0x01ffff00;
    bar |= DTP_FIX;

    context.indicators = (foo > bar)? C_T: C_NIL;

    return 1;
}

MAINOP(006) { /* <-IMMED */
    lisp_q foo;
    lisp_q bar;
	
    foo = NOT_CDRCODE(pop());

    bar = opcode & 0xff;
    if (opcode & 0x100) bar |= 0x01ffff00;
    bar |= DTP_FIX;

    context.indicators = (foo < bar)? C_T: C_NIL;

    return 1;
}

MAINOP(010) { /* TEST */
    context.indicators = load_target(opcode);

    return 1;
}

MAINOP(016) { /* TEST-MEMQ */
    lisp_q foo;
    lisp_q list;
    
    list = load_target(opcode);
    foo = NOT_CDRCODE(pop());
    
    while((NOT_CDRCODE(list) != (C_NIL)) &&
	  (NOT_CDRCODE(car(list)) != foo)) {
	list = cdr(list);
    }
    
    context.indicators = list;

    return 1;
}

MAINOP(017) { /* RETURN */
    return_1(load_target(opcode));

    return 1;
}

MAINOP(020) { /* = */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    foo = NOT_CDRCODE(load_target(opcode));
    bar = NOT_CDRCODE(pop());
    context.indicators = (foo == bar)? C_T: C_NIL;
    
    return 1;
}

MAINOP(021) { /* > */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = NOT_CDRCODE(load_target(opcode));
    foo = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);
    context.indicators = (foo > bar)? C_T: C_NIL;

    return 1;
}

MAINOP(022) { /* < */
    lisp_q foo;
    lisp_q bar;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */

    bar = NOT_CDRCODE(load_target(opcode));
    foo = NOT_CDRCODE(pop());
    dump_q(foo, 0);
    dump_q(bar, 1);
    context.indicators = (foo < bar)? C_T: C_NIL;

    return 1;
}

MAINOP(023) { /* EQ */
    lisp_q foo;
    lisp_q bar;

    foo = NOT_CDRCODE(pop());
    bar = NOT_CDRCODE(load_target(opcode));
    context.indicators = (foo == bar)? C_T: C_NIL;

    return 1;
}

MAINOP(031) { /* ARRAYP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = (DTP(foo) == DTP_ARRAY) ? C_T: C_NIL;

    return 1;
}

MAINOP(033) { /* STRINGP */
    lisp_q foo;
    lisp_q bar;

    context.indicators = C_NIL;
    foo = NOT_CDRCODE(load_target(opcode));
    if (DTP(foo) == DTP_ARRAY) {
	bar = memread(foo);
	if (DTP(bar) == DTP_ARRAY_HEADER) {
	    if ((ARY_TYPE(bar) == ART_STRING) ||
		(ARY_TYPE(bar) == ART_FAT_STRING))
		context.indicators = C_T;
	}
    }

    return 1;
}

MAINOP(034) { /* FIXNUMP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = (DTP(foo) == DTP_FIX) ? C_T: C_NIL;

    return 1;
}

MAINOP(035) { /* INTEGERP */
    lisp_q foo;

    foo = NOT_CDRCODE(load_target(opcode));
    context.indicators = (DTP(foo) == DTP_FIX) ? C_T: C_NIL;

    return 1;
}

MAINOP(036) { /* PLUSP */
    lisp_q foo;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */
    
    foo = ADDRESS(load_target(opcode));
    if (foo & 0x01000000) foo |= 0xfe000000;
    context.indicators = (foo > 0) ? C_T : C_NIL;

    return 1;
}

MAINOP(037) { /* MINUSP */
    lisp_q foo;

    /* FIXME: Check datatypes. Make work on non-FIXNUM types */
    
    foo = ADDRESS(load_target(opcode));
    if (foo & 0x01000000) foo |= 0xfe000000;
    context.indicators = (0 > foo) ? C_T : C_NIL;

    return 1;
}

MAINOP(044) { /* LDB-IMMED */
    lisp_q foo;

    foo = pop();
    if (DTP(foo) != DTP_FIX) {
	printf("LDB-IMMED: Not FIXNUM.\n");
	return 0;
    }

    foo >>= (opcode >> 4) & 0x1f;
    foo &= (1 << (opcode & 15)) - 1;
    foo |= DTP_FIX;
    
    push_cdrnext(foo);

    return 1;
}

MAINOP(045) { /* PUSH-NUMBER */
    push_cdrnext(DTP_FIX | (opcode & 0x1ff));

    return 1;
}

MAINOP(046) { /* PUSH-NEG-NUMBER */
    push_cdrnext(DTP_FIX | (opcode & 0x1ff) | 0x01fffe00);

    return 1;
}

MAINOP(050) { /* PUSH */
    push_cdrnext(load_target(opcode));

    return 1;
}

MAINOP(051) { /* PUSH-CAR */
    push_cdrnext(car(load_target(opcode)));

    return 1;
}

MAINOP(057) { /* PUSH-GET */
    lisp_q src1;
    lisp_q src2;

    src1 = pop();
    src2 = load_target(opcode);

    /* FIXME: what is the layout of a SYMBOL ? */

    push_cdrnext(C_NIL);

    return 1;
}

MAINOP(060) { /* + */
    lisp_q src1;
    lisp_q src2;

    src1 = pop();
    src2 = load_target(opcode);

    return math_plus(src1, src2);
}

MAINOP(061) { /* - */
    lisp_q src1;
    lisp_q src2;

    src1 = pop();
    src2 = load_target(opcode);

    return math_minus(src1, src2);
}

MAINOP(062) { /* * */
    lisp_q src1;
    lisp_q src2;

    src1 = pop();
    src2 = load_target(opcode);

    return math_times(src1, src2);
}

MAINOP(063) { /* LOGAND */
    lisp_q src1;
    lisp_q src2;
    lisp_q result;

    /* The description of this op in SSDN2 is rather unilluminating */
	
    src1 = pop();
    src2 = load_target(opcode);
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to LOGAND is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }

    result = DTP_FIX | ADDRESS(src1 & src2);
    push_cdrnext(result);

    return 1;
}

MAINOP(064) { /* LOGXOR */
    lisp_q src1;
    lisp_q src2;
    lisp_q result;

    /* The description of this op in SSDN2 is rather unilluminating */
	
    src1 = pop();
    src2 = load_target(opcode);
    
    if ((DTP(src1) != DTP_FIX) || (DTP(src2) != DTP_FIX)) {
	printf("One or both args to LOGXOR is not DTP_FIX, failing.\n");
	dump_q(src1, 1);
	dump_q(src2, 2);
	return 0;
    }

    result = DTP_FIX | ADDRESS(src1 ^ src2);
    push_cdrnext(result);

    return 1;
}

MAINOP(065) { /* 1+ */
    lisp_q foo;
	
    /* FIXME: What about numbers other than FIXNUMs? */

    foo = load_target(opcode);
    foo = ADDRESS(foo);
    /* FIXME: Check for FIXNUM overflow */
    push_cdrnext(DTP_FIX | (foo + 1));

    return 1;
}

MAINOP(066) { /* 1- */
    lisp_q foo;

    foo = load_target(opcode);
    foo--;
    push_cdrnext(DTP_FIX | ADDRESS(foo));

    return 1;
}

MAINOP(140) { /* POP */
    store_target(opcode, pop());

    return 1;
}

MAINOP(141) { /* MOVEM */
    lisp_q foo;

    foo = pop();
    push_cdrnext(foo);
    store_target(opcode, foo);

    return 1;
}

MAINOP(144) { /* SETE-1+ */
    lisp_q foo;

    foo = load_target(opcode);
    foo++;
    store_target(opcode, foo);

    return 1;
}

MAINOP(147) { /* PUSH-CDR-Store-CAR-IF-CONS */
    lisp_q foo;

    foo = pop();

    /* FIXME: Typecheck. */
    
    /* FIXME: If this is called for target PDL, which gets pushed first? */
    /* FIXME: Is this really the correct logic for when it isn't a cons? */
    if (DTP(foo) == DTP_LIST) { /* May also want DTP_STACK_LIST */
	push_cdrnext(cdr(foo));
	store_target(opcode, car(foo));
	context.indicators = C_T;
    } else {
	context.indicators = C_NIL;
    }

    return 1;
}

MAINOP(150) { /* PUSH-LOC */
    push_cdrnext(locative_target(opcode));

    return 1;
}


MAINOP(151) { /* BIND-NIL */
    bind(locative_target(opcode), C_NIL);

    return 1;
}

MAINOP(152) { /* BIND-T */
    bind(locative_target(opcode), C_T);

    return 1;
}

MAINOP(153) { /* BIND-POP */
    lisp_q oval, nval;

    oval = load_target(opcode);

    /* FIXME: save oval on bind stack */

    nval = pop();
    store_target(opcode, nval);

    /* FIXME: Change to use bind() and locative_target() */
    
    printf("BIND-POP: Not implemented.\n");
    
    return 0;
}

MAINOP(155) { /* SET-NIL */
    store_target(opcode, C_NIL);

    return 1;
}

MAINOP(156) { /* SET-T */
    store_target(opcode, C_T);

    return 1;
}

MAINOP(157) { /* SET-ZERO */
    store_target(opcode, DTP_FIX);

    return 1;
}

MAINOP(160) { /* BR-NIL-ELSE-POP */
    if (NOT_CDRCODE(context.indicators) == C_NIL) {
	do_branch(opcode);
    } else {
	pop();
    }

    return 1;
}

MAINOP(161) { /* BR-NOT-NIL-ELSE-POP */
    if (NOT_CDRCODE(context.indicators) != C_NIL) {
	do_branch(opcode);
    } else {
	pop();
    }

    return 1;
}

MAINOP(162) { /* BR-NIL */
    if (NOT_CDRCODE(context.indicators) == C_NIL) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(163) { /* BR-NOT-NIL */
    if (NOT_CDRCODE(context.indicators) != C_NIL) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(166) { /* BR-ZEROP */
    int x;

    x = context.indicators;

    /* FIXME: What about numbers other than FIXNUMs? */
    if (NOT_CDRCODE(x) == DTP_FIX) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(167) { /* BR-NOT-ZEROP */
    int x;

    x = context.indicators;

    /* FIXME: What about numbers other than FIXNUMs? */
    if ((DTP(x) == DTP_FIX) && ADDRESS(x)) {
	do_branch(opcode);
    }

    return 1;
}

MAINOP(176) { /* BR */
    do_branch(opcode);

    return 1;
}

int callop(u16 opcode) { /* CALL-* */
    lisp_q function;
    int num_args;

    num_args = (opcode >> 11) & 7;

    function = load_target(opcode);

    if (num_args == 7) {
	/* FIXME: check data type? */
	num_args = ADDRESS(pop());

	printf("callop(): CALL-N case.\n");
    }
    
    funcall(function, DTP_FIX | 0x004000 |
	    num_args | (opcode & 0x600));

    return 1;
}

mainop_handler mainop_dispatch[0x80] = {
    /* 000 - 017 */
    step_auxop, step_miscop,NULL,       mainop_003,
    mainop_004, mainop_005, mainop_006, NULL,
    mainop_010, NULL,       NULL,       NULL,
    NULL,       NULL,       mainop_016, mainop_017,

    /* 020 - 037 */
    mainop_020, mainop_021, mainop_022, mainop_023,
    NULL,       NULL,       NULL,       NULL,
    NULL,       mainop_031, NULL,       mainop_033,
    mainop_034, mainop_035, mainop_036, mainop_037,

    /* 040 - 057 */
    NULL,       step_miscop,NULL,       NULL,
    mainop_044, mainop_045, mainop_046, NULL,
    mainop_050, mainop_051, NULL,       NULL,
    NULL,       NULL,       NULL,       mainop_057,

    /* 060 - 077 */
    mainop_060, mainop_061, mainop_062, mainop_063,
    mainop_064, mainop_065, mainop_066, NULL,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       NULL,       NULL,


    /* 100 - 117 */
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,

    /* 120 - 137 */
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,
    callop,     callop,     callop,     callop,

    /* 140 - 157 */
    mainop_140, mainop_141, NULL,       NULL,
    mainop_144, NULL,       NULL,       mainop_147,
    mainop_150, mainop_151, mainop_152, mainop_153,
    NULL,       mainop_155, mainop_156, mainop_157,

    /* 160 - 177 */
    mainop_160, mainop_161, mainop_162, mainop_163,
    NULL,       NULL,       mainop_166, mainop_167,
    NULL,       NULL,       NULL,       NULL,
    NULL,       NULL,       mainop_176, NULL,
};

int step(void)
{
    u16 opcode;
    u16 opnum;

    opcode = get_opcode();
    opnum = opcode >> 9;

    disassemble_instr(context.location_counter, opcode);

    context.location_counter++;

    if (mainop_dispatch[opnum]) {
	return mainop_dispatch[opnum](opcode);



    } else if ((opcode & 0xffc0) == 0x6e00) { /* PUSH-AR-1 FEF|xx */
	lisp_q index;

	index = pop();

	dump_q(index, 0);
	dump_q(memread_inviz(context.function + (opcode & 0x3f)), 1);

	dump_q(memread(01000 + ADDRESS(index)), 2);

	/* FIXME: Just Plain Wrong */
	
	push_cdrnext(memread(01000 + ADDRESS(index)));
    } else {
	printf("Unknown opcode (#o%o).\n", opnum);
	return 0;
    }

    return 1;
}

/* EOF */
