********************************************************************
#!/bin/csh -f
# This script is used after upgrading the DV uVAX.  It is also used to
# restore a MicroVAX disk, or to re-make the CM filesystem (to re-size DV).
# It must be run as root.
#

set path=(. /bin /usr/bin /etc /usr/etc /usr/ucb /usr/local /usr/local/etc)

if (`whoami` != "root") then
  echo "You must be root to run this script."
  exit 1
endif

echo -n "\
1) Finish up new installation (initialize new CM filesystems)\
2) Finish upgrade procedure (restore CM filesystem, config files from tape)\
3) Restore MicroVAX disk drive (restore CM filesystem from good disk)\
4) Re-make CM filesystems (use current CM filesystems)\
\
Enter choice: "
set mode = $<

switch ($mode)
case 1:
case 2:
case 3:
  echo -n "Which disk drive is being restored, ra0 or ra1? "
  set newdisk = $<
  if ($newdisk == ra0) then
    set olddisk = ra1
    set old_dvfs = /dv2
    set new_dvfs = /dv1
  else if ($newdisk == ra1) then
    set olddisk = ra0
    set old_dvfs = /dv1
    set new_dvfs = /dv2
  else
    echo "Invalid response, start over."
    exit 1
  endif
  breaksw
case 4:
  set old_dvfs = '/dv1 /dv2'
  breaksw
default:
  exit 1
endsw

set fss_pid = `ps ax | egrep "[f]sserver" | awk '{ print $1 }'`

if ($fss_pid == "") then
  echo "No running fsserver process found."
else
  echo "Shutting down fsserver process."
  kill -TERM $fss_pid
  if ($status) then
    echo "Unable to shutdown fsserver; aborting..."
    exit 1
  endif
  sleep 3
  set fss_pid = `ps ax | egrep "[f]sserver" | awk '{ print $1 }'`
  if ($fss_pid != "") then
    echo -n "Waiting for fsserver shutdown..."
    sleep 25
    set fss_pid = `ps ax | egrep "[f]sserver" | awk '{ print $1 }'`
    if ($fss_pid != "") then
      echo "timeout.  Killing fsserver"
      kill -KILL $fss_pid
      if ($status) then
        echo "Unable to kill fsserver; aborting..."
        exit 1
      endif
    else
      echo "done."
    endif
  endif
endif

if ($mode == 2) then
  pushd $old_dvfs > /dev/null
  echo -n "Install tape made previously with cmfs_backup.csh.\
Hit return when ready. "
  echo $<
  mt rew
  tar xvpf /dev/nrmt0h
  if ($status) then
    echo "Restore of $old_dvfs failed.  Aborting..."
    exit 1
  endif
  cd /etc
  tar tvf /dev/nrmt0h >& /dev/null  #position tape
  tar xvpf /dev/nrmt0h
  if ($status) then
    echo "Restore of system files failed.  Aborting..."
    exit 1
  endif
# restore rc.local hostname info
  if (-f ifconfig.qe0) then
    head -1 rc.local > rc.local.new
    cat ifconfig.qe0 >> rc.local.new
    tail +4 rc.local >> rc.local.new
    mv rc.local.new rc.local
    sh ifconfig.qe0
    rm -f ifconfig.qe0
  endif
  cd /usr/local/etc/diag
  tar tvf /dev/nrmt0h >& /dev/null  #position tape
  tar xvpf /dev/nrmt0h
  if ($status) then
    echo "Restore of DV config file failed.  Configure DV manually."
  endif
  mt rew
  popd > /dev/null
endif

if ($mode =~ [34]) then
  echo "Running cmfsck on $old_dvfs."
  cmfsck -p $old_dvfs
  if ($status) then
    echo "Cmfsck failed!  Aborting..."
    exit 1
  endif
endif

#
# Make sure hostname is correct and matches config file name
#
set mastername=DVMFG9
set myname=`hostname | tr a-z A-Z`
if (-f /sys/conf/vax/$mastername) then
  mv /sys/conf/vax/$mastername /sys/conf/vax/$myname
  ed /sys/conf/vax/$myname <<xxEOFxx > /dev/null
/$mastername/s//$myname/
w
q
xxEOFxx
  mv /sys/VAX/$mastername /sys/VAX/$myname
endif

#
# Now copy everything from one disk to the other
#

if ($mode != 4) then

  set RAINFO=`devinfo /dev/r${newdisk}a`
  if ($status) then
    echo "Can't get device info for ${newdisk}.  Aborting..."
    exit 1
  endif
  
  set OURDISK=`devinfo /dev/r${olddisk}a`
  if ($status) then
    echo "Can't get device info for ${olddisk}.  Aborting..."
    exit 1
  endif
  
  egrep -s ^/dev/${newdisk}a:/: /etc/fstab && \
  echo "Cannot overwrite $newdisk, as it is the current system disk." && \
  echo "Re-boot off the other disk and run this script again." && \
  exit 1

  egrep -s ^/dev/${newdisk}b:swap: /etc/fstab && \
  echo "The ${newdisk}b partition is currently in use as a swap area." && \
  echo "Delete the line in /etc/fstab which looks like:" && \
  grep swap /etc/fstab && \
  echo "Then re-boot the system and run this script again.  The deleted" && \
  echo "line will be automatically restored when this script is re-run." && \
  exit 1

  umount /dev/${newdisk}a
  umount /dev/${newdisk}g
  umount /dev/${newdisk}h

  unset ans_ok
  while (! $?ans_ok)
    echo -n "\
Is it OK to erase the disk $newdisk $RAINFO? "
  
    set resp = $<
    set ans_ok
    if ($resp !~ [yY]*) unset ans_ok
  end
  
#If current disk is rd53 and new one is rd54, treat new one as rd53:
  if ("$OURDISK" == "RD53" && "$RAINFO" == "RD54") then
    echo "Note: using RD53 partitions on an RD54 drive."
  endif
  
  echo "Copying $olddisk to $newdisk"
  dd if=/dev/r${olddisk}a of=/dev/r${newdisk}a bs=64k conv=noerror,sync && \
  dd if=/dev/r${olddisk}g of=/dev/r${newdisk}g bs=64k conv=noerror,sync && \
  dd if=/dev/r${olddisk}h of=/dev/r${newdisk}h bs=64k conv=noerror,sync
  if ($status) then
    echo "Unable to copy disk; aborting"
    exit 1
  endif
  
  echo "Checking filesystems on $newdisk"
  fsck -p /dev/r${newdisk}a && \
  fsck -p /dev/r${newdisk}g && \
  fsck -p /dev/r${newdisk}h
  if ($status) then
    echo "fsck failed; aborting"
    exit 1
  endif

#
# now make customizations to be able to run off either disk
#

  mount /dev/${newdisk}a /mnt
  mount /dev/${newdisk}g /mnt/usr
  mount /dev/${newdisk}h $new_dvfs

  echo "Updating fstab files"

  grep -s $new_dvfs /etc/fstab || echo "/dev/${newdisk}h:${new_dvfs}:rw:1:4:ufs::" >> /etc/fstab
  grep -s swap /etc/fstab || echo "/dev/${newdisk}b:swap:sw:0:0:ufs::" >> /etc/fstab

  ed /mnt/etc/fstab <<xxEOFxx > /dev/null
/${olddisk}a/s/$olddisk/$newdisk/
/${olddisk}g/s/$olddisk/$newdisk/
w
q
xxEOFxx

  grep -s $new_dvfs /mnt/etc/fstab || echo "/dev/${newdisk}h:${new_dvfs}:rw:1:4:ufs::" >> /mnt/etc/fstab
  grep -s swap /mnt/etc/fstab || echo "/dev/${olddisk}b:swap:sw:0:0:ufs::" >> /mnt/etc/fstab

  echo "Making the OS"
  cd /mnt/sys/VAX/$myname
  if ("$newdisk" == "ra1") then
    make vmunixb
    mv vmunixb vmunix
  else
    make vmunixa
    mv vmunixa vmunix
  endif
  
  echo "Moving new OS to /vmunix"
  mv vmunix /mnt

endif

echo "Coldbooting DV:"
(cd /usr/local/etc/diag; ./dvcoldboot)

rm -f /dv1/raw-disk-special-file /dv1/.raw-disk-special-file.attr /dv2/raw-disk-special-file /dv2/.raw-disk-special-file.attr

cmmkfs 0 /dv1 /dv2

echo "Running cmfsck on /dv1 & /dv2"

cmfsck -s /dv1 /dv2
if ($status) then
  echo "Cmfsck failed - help\!"
  exit 1
endif

echo "Script complete.  System will be fully operational upon reboot."
echo "System rebooting..."
/etc/reboot
