
(in-package '*lisp-i)

;;; *PROCLAIM does some *Lisp bookkeeping and then calls Lisp PROCLAIM to let
;;; the underlying system know about the type.  The problem is that *Lisp types
;;; are a superset of Lisp types. E.G. (array string-char (cFrameNameLength))
;;; is not a valid Lisp type.  The *PROCLAIM code tries to strip away information
;;; from invalid (Lisp) type declarations.  This code fixes the problem where
;;; 
;;; (*proclaim '(type (array string-char (cFrameNameLength)) cBlankFrameName))
;;;
;;; would not evaluate properly since an improper type specification was passed
;;; to Lisp PROCLAIM.

;;; Andy Wilson 3/6/92 (Patch by Jeff Mincy)

(defun invalid-cl-type-expression (object)
  (not (or (eq object '*) (integerp object))))

(defun fixup-type (type)
  (if (and (consp type) (eq (car type) 'pvar)) 
      'pvar
      (if (and (consp type) (member (car type) '(unsigned-byte signed-byte defined-float))
	       (some #'invalid-cl-type-expression (cdr type)))
	  (car type)
	  (if (and (consp type) (eq (car type) 'array))
	      `(array ,(fixup-type (cadr type))
		      ,(if (listp (caddr type))
			   (mapcar #'(lambda (x) (if (invalid-cl-type-expression x) '* x)) (caddr type))
			   (if (invalid-cl-type-expression (caddr type)) '* (caddr type))))
	      type))))

(eval-when (compile load eval)
  (defun starlisp-proclaim (decl-spec)
    (declare (special *old-proclaim-function*))
    (case (car decl-spec)
      (special
	(dolist (variable (cdr decl-spec))
	  (setf (get variable 'special) t)))
      (ftype
	;; Syntax like, (ftype (function (...) type) &rest foos)
	(let ((ftype (cadr decl-spec)))
	  (check-ftype-proclamation decl-spec)
	  (let* ((argument-types (mapcar #'(lambda (type)
					     (if (member type lambda-list-keywords) type (canonical-type type nil t)))
					 (cadr ftype)))
		 (rtype (caddr ftype))
		 (return-type (if (and (consp rtype) (member (car rtype) '(satisfies and or not member values)))
				  rtype
				  (canonical-type rtype nil t))))
	    (dolist (function (cddr decl-spec))
	      (set-function-type function `(function ,argument-types ,return-type)))
	    #+lucid  ;; used to be #+sun4, caused warnings in VAX lucid
	    ;; dont proclaim ftype for *defuns in lucid
	    (if (some #'(lambda (function)
			  ;(get-*defun-function function)
			  (get function '*lisp-defun))
		      (cddr decl-spec))
		(return-from starlisp-proclaim ()))
	    (setq decl-spec
		  `(ftype (function ,(mapcar #'fixup-type argument-types) ,(fixup-type return-type))
			  ,@(cddr decl-spec))))))
      (function
	(if (and (cddr decl-spec) (listp (caddr decl-spec)))
	    (let ((ftype `(function ,(caddr decl-spec) ,@(cdddr decl-spec))))
	      (set-function-type (cadr decl-spec) ftype))
	    (dolist (variable (cdr decl-spec))
	      (setf (get variable 'type) 'function))))
      (inline)
      (notinline)
      ((optimize *optimize)
       (dolist (quality-value (cdr decl-spec))
	 (let ((quality (if (consp quality-value) (car quality-value) quality-value))
	       (value (if (consp quality-value) (cadr quality-value) 3)))
	   (case quality
	     (speed (setq *speed* value))
	     (safety (setq *safety* value))
	     (space (setq *space* value))
	     (compilation-speed (setq *compilation-speed* value))
	     (t (cerror "Ignore declaration." "Invalid quality ~S for optimize proclamation." quality))))))
      (declaration
	(dolist (declaration (cdr decl-spec))
	  (setf (get declaration 'declaration) t)))
      (ignore
	(dolist (variable (cdr decl-spec))
	  (setf (get variable 'ignore) t)))
      (type
	(let ((type (canonical-type (cadr decl-spec) nil t)))
	  (if (and (null type) (cadr decl-spec))
	      (cerror "Ignore declaration." "~S is not a valid type in proclaim decl-spec ~S." (cadr decl-spec) decl-spec)
	      (dolist (variable (cddr decl-spec))
		;; Put the type in an easy to find place.
		(setf (get variable 'type) type)
		;; If type is a pvar type, then put a slc-pvar descriptor on the descriptor property.  
		;; If the type is not a pvar type, then add a vanilla descriptor to the descriptor property.
		(if (and (consp type) (eq (car type) 'pvar))
		    ;; Type is a pvar type.
		    (progn
		      (set-variable-descriptor variable (make-pvar-for-compiler :type type :reference variable :read-only t))
		      (funcall *old-proclaim-function* `(type pvar ,variable)))
		    ;; Type is not a pvar, add vanilla descriptor.
		    (progn
		      (set-variable-descriptor variable (make-descriptor-for-compiler :type type :reference variable))
		      (funcall *old-proclaim-function* 
			       `(type ,(fixup-type type) ,variable))))))))
      (*defun
	(dolist (function (cdr decl-spec))
	  (funcall 'proclaim-*defun-1 function)))
      (*option
	(let* ((keyword (cadr decl-spec)) (value (caddr decl-spec))
	       (option (find-compiler-keyword keyword)))
	  (if option
	      (deal-with-compiler-option-value option value keyword)
	      (cerror "Ignore declaration." "Unknown starlisp compiler option ~S." keyword))))
      #+symbolics
      (#.lt::*symbolics-common-lisp-declarations* ())
      (t (cond ((not (symbolp (car decl-spec)))
		#| (cerror "Try appending TYPE onto decl-spec" "Invalid decl-spec ~S to proclaim." decl-spec) |#
		(warn "Invalid decl-spec ~S to proclaim, retrying with (TYPE ~{~S~^ ~})." decl-spec decl-spec)
		(return-from starlisp-proclaim (proclaim (cons 'type decl-spec))))
	       ((standard-type-p (car decl-spec))
		(let ((type (car (canonical-type decl-spec))))
		  (dolist (variable (cdr decl-spec))
		    ;; If variable was previously proclaimed to be a pvar, then get rid of pvar declaration.
		    (setf (get variable 'type) type)
		    (set-variable-descriptor variable (make-descriptor-for-compiler :type type :reference variable)))))
	       ((or (proclaimed-declaration-p (car decl-spec))
		    (member (car decl-spec) 
			    #+symbolics lt::*user-defined-declarations* 
			    #+lucid lucid::*user-declarations*
			    #-(or symbolics lucid) nil)))
	       (t #| (cerror "Try appending TYPE onto decl-spec" "Invalid decl-spec ~S to proclaim." decl-spec) |#
		  (warn "Invalid decl-spec ~S to proclaim, retrying with (TYPE ~{~S~^ ~})." decl-spec decl-spec)
		  (return-from starlisp-proclaim (proclaim (cons 'type decl-spec)))))))
    (unless (or (eq (car decl-spec) 'type) (eq (car decl-spec) '*defun))
      (funcall *old-proclaim-function* decl-spec))
    nil))

(*lisp-i::increment-patch-level 15)
