{ Dining Philosphers  J.R. Cordy (after C. Lewis)  Dec 1981 }
var Diners : module
    { Random number generator }
    include '%RANDOM'

    pervasive type XCoord = 0 .. 79
    pervasive type YCoord = 0 .. 23

    pervasive const numberOfDiners := 5

    { Position of each philosopher at the table }
    pervasive type Philosopher = 0 .. numberOfDiners - 1
    pervasive const faceXPosition : array Philosopher of XCoord :=
	(36, 56, 48, 24, 16)
    pervasive const faceYPosition : array Philosopher of YCoord :=
	(2, 9, 18, 18, 9)

    { Positions of forks to left and right of each philosopher }
    pervasive type Fork = 0 .. numberOfDiners
    pervasive const forkXPosition : array Fork of XCoord :=
	(25, 49, 57, 37, 19, 25)
    pervasive const forkYPosition : array Fork of YCoord:=
	(5, 5, 14, 21, 14, 5)

    { Meals eaten by each philosopher }
    pervasive const maxMeals := 50
    var meals : array Philosopher of 0 .. maxMeals

    { Maximum eating and thinking times }
    pervasive const maxEatingTime := 10
    pervasive const maxThinkingTime := 10

    { Philospher's expressions }
    pervasive const contemplating := 0
    pervasive const smiling := 1
    pervasive const chewing := 2
    pervasive const frowning := 3

    { Philosopher's activities }
    pervasive const thinking := 0
    pervasive const eating := 1

    { Sides }
    pervasive const left := 0
    pervasive const right := 1

    var Draw : module
	exports (ChangeExpression, MoveFork, Finalize, Interrupt)

	{ Terminal cursor handling library }
	include '%CURSES'

	procedure DrawFace (diner : Philosopher) =
	    imports (var Curses)
	    begin
		const x : XCoord := faceXPosition (diner)
		const y : YCoord := faceYPosition (diner)
		Curses.Move (y, x)
		Curses.AddStr ('MMMMMMM$E')
		Curses.Move (y + 1, x)
		Curses.AddStr ('| o,o |$E')
		Curses.Move (y + 2, x)
		Curses.AddStr ('| --- |$E')
		Curses.Move (y + 3, x)
		Curses.AddStr (' """""$E')
		Curses.Refresh
	    end DrawFace

	procedure ChangeExpression (diner : Philosopher, newExpression :
	    contemplating .. frowning) =
	    imports (var Curses)
	    begin
		const x : XCoord := faceXPosition (diner)
		const y : YCoord := faceYPosition (diner)
		Curses.Move (y + 2, x)
		if newExpression = contemplating then
		    Curses.AddStr ('| --- |$E')
		elseif newExpression = smiling then
		    Curses.AddStr ('| \_/ |$E')
		elseif newExpression = chewing then
		    Curses.AddStr ('| (_) |$E')
		else
		    Curses.AddStr ('| /-\ |$E')
		end if
		Curses.Refresh
	    end ChangeExpression

	procedure DrawFork (diner : Philosopher, side : left .. right, activity
	    : thinking .. eating) =
	    imports (var Curses)
	    begin
		var x : XCoord
		var y : YCoord
		if activity = thinking then
		    x := forkXPosition (diner + side)
		    y := forkYPosition (diner + side)
		else
		    if side = left then
			x := faceXPosition (diner) - 7
		    else
			x := faceXPosition (diner) + 9
		    end if
		    y := faceYPosition (diner) + 1
		end if
		Curses.Move (y, x)
		Curses.AddStr ('|_|_|$E')
		Curses.Move (y + 1, x + 2)
		Curses.AddCh ($|)
		Curses.Move (y + 2, x + 2)
		Curses.AddCh ($|)
		Curses.Refresh
	    end DrawFork

	procedure EraseFork (diner : Philosopher, side : left .. right, activity
	    : thinking .. eating) =
	    imports (var Curses)
	    begin
		var x : XCoord
		var y : YCoord
		if activity = thinking then
		    x := forkXPosition (diner + side)
		    y := forkYPosition (diner + side)
		else
		    if side = left then
			x := faceXPosition (diner) - 7
		    else
			x := faceXPosition (diner) + 9
		    end if
		    y := faceYPosition (diner) + 1
		end if
		Curses.Move (y, x)
		Curses.AddStr ('     $E')
		Curses.Move (y + 1, x + 2)
		Curses.AddCh ($$S)
		Curses.Move (Y + 2, x + 2)
		Curses.AddCh ($$S)
		Curses.Refresh
	    end EraseFork

	procedure MoveFork (diner : Philosopher, side : left .. right,
	    newActivity : thinking .. eating) =
	    imports (DrawFork, EraseFork)
	    begin
		if newActivity = thinking then
		    EraseFork (diner, side, eating)
		else
		    EraseFork (diner, side, thinking)
		end if
		DrawFork (diner, side, newActivity)
	    end MoveFork

	procedure Finalize =
	    imports (var Curses)
	    begin
		var c : Char
		Curses.Move (23, 0)
		Curses.CursorOn
		Curses.AddStr ('Hit return to continue.$E')
		loop
		    exit when not Curses.HasCh
		    Curses.GetCh (c)
		end loop
		Curses.GetCh (c)
		Curses.Clear
		Curses.Refresh
		Curses.Endwin
	    end Finalize

	function Interrupt returns result : Boolean =
	    imports (Curses)
	    begin
		return (Curses.HasCh)
	    end Interrupt

	initially
	    imports (DrawFace, DrawFork, var Curses)
	    begin
		var diner : Philosopher
		Curses.Clear
		Curses.Move (0, 3)
		Curses.CursorOff
		Curses.AddStr ('THE DINING PHILOSOPHERS$E')
		Curses.Move (1, 0)
		Curses.AddStr ('Processes in Concurrent Euclid$E')
		diner := 0
		loop
		    DrawFace (diner)
		    DrawFork (diner, left, thinking)
		    DrawFork (diner, right, thinking)
		    exit when diner = numberOfDiners - 1
		    diner := diner + 1
		end loop
	    end
    end module

    var Forks :
	monitor
	imports (var meals, var Draw)
	exports (PickUp, PutDown, AllDone)

	function LeftDiner (diner : Philosopher) returns dinerToLeft :
	    Philosopher =
	    begin
		return ((diner + 1) mod numberOfDiners)
	    end LeftDiner

	function RightDiner (diner : Philosopher) returns dinerToRight :
	    Philosopher =
	    begin
		return ((diner + numberOfDiners - 1) mod numberOfDiners)
	    end RightDiner

	var forksAvail : array Philosopher of 0 .. 2
	var okayToEat : array Philosopher of Condition
	var done : condition

	procedure PickUp (diner : Philosopher) =
	    imports (var forksAvail, var okayToEat, meals, LeftDiner, RightDiner
		    , var Draw)
	    begin
		if forksAvail (diner) not= 2 then
		    wait (okayToEat (diner))
		end if
		forksAvail (LeftDiner (diner)) := forksAvail (LeftDiner (diner))
		    - 1
		Draw.MoveFork (diner, left, eating)
		forksAvail (RightDiner (diner)) := forksAvail (RightDiner (diner
		    )) - 1
		Draw.MoveFork (diner, right, eating)
		Draw.ChangeExpression (diner, chewing)
	    end PickUp

	procedure PutDown (diner : Philosopher) =
	    imports (var forksAvail, var okayToEat, var meals, var done,
		    LeftDiner, RightDiner, var Draw)
	    begin
		Draw.ChangeExpression (diner, contemplating)
		Draw.MoveFork (diner, left, thinking)
		forksAvail (LeftDiner (diner)) := forksAvail (LeftDiner (diner))
		    + 1
		Draw.MoveFork (diner, right, thinking)
		forksAvail (RightDiner (diner)) := forksAvail (RightDiner (diner
		    )) + 1
		if forksAvail (LeftDiner (diner)) = 2 then
		    signal (okayToEat (LeftDiner (diner)))
		end if
		if forksAvail (RightDiner (diner)) = 2 then
		    signal (okayToEat (RightDiner (diner)))
		end if
		meals (diner) := meals (diner) + 1
		if meals (diner) = maxMeals then
		    Draw.ChangeExpression (diner, smiling)
		    signal (done)
		elseif Draw.Interrupt then
		    meals (diner) := maxmeals
		    Draw.ChangeExpression (diner, frowning)
		    signal (done)
		end if
	    end PutDown

	procedure AllDone =
	    imports (var done)
	    begin
		var numberDone : 0 .. numberOfDiners := 0
		loop
		    wait (done)
		    numberDone := numberDone + 1
		    exit when numberDone = numberOfDiners
		end loop
	    end AllDone

	initially
	    imports (var forksAvail, var meals)
	    begin
		var i : Philosopher
		i := 0
		loop
		    forksAvail (i) := 2
		    meals (i) := 0
		    i := i + 1
		    exit when i = numberOfDiners
		end loop
	    end
    end monitor

    procedure Delay (time : SignedInt) =
	begin
	    var i : SignedInt := time * 100
	    loop
		exit when i <= 0
		i := i - 1
	    end loop
	end Delay

    procedure CommonPhilosopher (diner : Philosopher) =
	imports (var meals, var Forks, var Random, Delay)
	begin
	    loop
		Forks.PickUp (diner)
		Random.Next
		busy (Random.Number mod maxEatingTime)
		Delay (Random.Number mod maxEatingTime)
		Forks.PutDown (diner)
		Random.Next
		busy (Random.Number mod maxThinkingTime)
		Delay (Random.Number mod maxThinkingTime)
		exit when meals (diner) = maxMeals
	    end loop
	end CommonPhilosopher

    process Philosopher_0 (3000)
	imports (CommonPhilosopher)
	begin
	    CommonPhilosopher (0)
	end Philosopher_0

    process Philosopher_1 (3000)
	imports (CommonPhilosopher)
	begin
	    CommonPhilosopher (1)
	end Philosopher_1

    process Philosopher_2 (3000)
	imports (CommonPhilosopher)
	begin
	    CommonPhilosopher (2)
	end Philosopher_2

    process Philosopher_3 (3000)
	imports (CommonPhilosopher)
	begin
	    CommonPhilosopher (3)
	end Philosopher_3

    process Philosopher_4 (3000)
	imports (CommonPhilosopher)
	begin
	    CommonPhilosopher (4)
	end Philosopher_4

    process Cleanup (3000)
	imports (var Forks, var Draw)
	begin
	    Forks.AllDone
	    Draw.Finalize
	end Cleanup
end Diners
