#
/*
Name:
	WEEKLYAPPT Calendar System Command

Function:
	Make a weekly appointment for n consecutive weeks

Algorithm:
	Ask the user for the appointment and then add the appointment for
	n consecutive weeks, where n is supplied as a start option.
	Default value of n is 4 weeks.

Parameters:
	date		the starting date for the weekly appointments
	start-time	the start time of the appointment
	end-time	the end time of the appointment
	n               the number of weeks for appointment.

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or exec/appointments directory
	EFORM	Invalid date or time format
	ESPIPE	Bad seek in suspense file

Files and Programs:
	home-dir/exec/appointments/(file)	appointments files
	home-dir/exec/suspenses/(file)		suspenses file

Installation Instructions:
	Compile as follows: cc -o $EXECDIR/weeklyappt weeklyappt.c library
	Change owner to root
	Deny write permissions excepy to root

History:
	Jul 80,	Release 3 Version 1, M. Smith, coded in shell
	Nov 80, Release 5 Version 1, G. Maltagliati, coded in C for efficency
	Aug 82, Release 7 Version 2, J. Neyer...re-write exec support

*/
#include <stdio.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define APPOINT "exec/appointments"
#define DATEERR "appoint: invalid date format\n"
#define TIMEERR "appoint: invalid time format\n"
#define SUSPENSE "../suspenses"
#define REENTER "****:Re-enter your weekly appointment:%s:::::%s\n"
#define USAGE "You must enter: weeklyappt <date> <start-time> <end-time> <num_weeks>\n"

char id[] = "%W% %H%";

/*

Name:
	main

Function:
	Validate the date and the start and end times.

Algorithm:
	Check and standardize the date, the start time, and the end time (if one
	is provided). The call week_appt to add the weekly appointment for 
	nine consecutive weeks and add a suspense the last week to remind the 
	user to re-enter his weekly appointment.

Parameters:
	date		start date of the weekly appointment
	start-time	start time of the appointment
	end-time	end time of the appointment
	n               number of weeks

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or appointment directory
	EFORM	Bad date or time format or wrong number of arguments

Files and Programs:
	None


*/
extern int atoi();
extern char *itoa();
extern char *gettime();
main (argc, argv, envp)
char   *argv[];						/* Pointer to list of argument pointers */
int     argc;						/* Count of command line arguments */
char **envp;						/* environment parameters */
{
    char    appoint[15],				/* Holds the start and end time of the appointment */
           *appt_ptr = appoint,				/* Pointer to appoint */
            date[7],					/* Holds the standardized date */
           *month,					/* Number of the month of the date */
          **userinfo;					/* Pointer to user information (home directory) */
    int     n_weeks = 0;
    char    store_stm[5],
	    store_etm[5],
	    *time_ptr,
	    temp;
    int     i,
	    time_flag,
	    ival,
	    temp_stm,
	    temp_etm;

    if (chdir (findenv("HOME", envp)))                  /* move to $HOME */
    {
	printf ("bad home directory\n");
	exit (ENOENT);
    }
    if (chdir (APPOINT) == -1)				/* Move to exec/appointments directory */
    {
	printf ("bad directory--%s\n", APPOINT);
	exit (ENOENT);
    }
    if (argc >= 3)					/* Check the number of arguments */
	if (month = valid_date (argv[1], date, 0))      /* Correct number of arguments, check the date */
	{						/* Valid date */
	    time_flag = 1;
	    time_ptr = gettime(argv[2], time_flag);      /* check time validity */
	    for (i=0; *(time_ptr + i) != '\0'; i++) {    /* copy time to an array */
		store_stm[i] = *(time_ptr + i);
		if (store_stm[i] == 'a' || store_stm[i] == 'p')
		   temp = store_stm[i];
	    }
	    temp_stm = atoi(store_stm);

	    if (atoi(store_stm) == 1200 && temp == 'a')
	       temp_stm = 0;
	    else
	       if (atoi(store_stm) == 1200 && temp == 'p')
		  temp_stm = 1200;
	    if (temp_stm >= 0 && temp_stm <= 2400)    {
	       ival = atoi(store_stm);                  /* check whether start time is out-of-bound */
	       if (temp == 'p' && ival >= 1300)
		  ival -=1200;
		appt_ptr = copy (itoa (ival,4), appt_ptr);
		*appt_ptr++ = temp;
	    }
	    else					/* Invalid time format */
	    {
		fprintf(stderr, TIMEERR);
		exit (EFORM);
	    }
	    if (argc >= 4){                             /* Check to see if an end time was entered */
		time_flag = 0;
		time_ptr = gettime(argv[3], time_flag);   /* check time validity  */
		for (i=0; *(time_ptr + i) != '\0';i++) { /* copy time            */
		    store_etm[i] = *(time_ptr + i);
		    if (store_etm[i] == 'a' || store_etm[i] == 'p')
		       temp = store_etm[i];
		}
		temp_etm = atoi(store_etm);
		if (atoi(store_etm) == 1200 && temp == 'a')
		   temp_etm = 0;
		else
		   if (atoi(store_etm) == 1200 && temp == 'p')
		      temp_etm = 1200;
		if (temp_etm >= 0 && temp_etm <= 2400 && temp_etm > temp_stm)
							/* Standardize end time */
		{
		    ival = atoi(store_etm);
		    if (temp == 'p' && ival >= 1300)
		       ival -= 1200;
		    *appt_ptr++ = '-';
		     appt_ptr = copy (itoa (ival,4), appt_ptr);
		    *appt_ptr++ = temp;
		}
		else					/* Invalid time format */
		{
		    fprintf(stderr, TIMEERR);
		    exit (EFORM);
		}
	     }
	    *appt_ptr++ = ':';				/* Finish building appointment record (excluding the actual appointment */
	    *appt_ptr++ = ' ';
	    *appt_ptr++ = ' ';
	    *appt_ptr++ = '\0';
	    if (argc == 5)
		n_weeks = atoi(argv[4]);
	    week_appt (date, atoi (month), appoint, n_weeks, envp);    /* Make the appointment for 9 weeks */
	}
	else						/* Invalid date format */
	{
	    fprintf(stderr, DATEERR);
	    exit (EFORM);
	}
    else						/* Not enough arguments were entered */
    {
	fprintf(stderr,  USAGE);
	exit (EFORM);
    }
    exit (ENOERR);
}
/*

Name:
	week_appt

Function:
	Add an appointment for n weeks in a row and also a suspense on the
	last day

Algorithm:
	Prompt the user for the appointment and add it to the end of the record
	already created (the time of the appointment). Determine the number of
	days in the year. Do the following n times:  Write the appointment to
	the appropriate file. Add 7 to the julian date. If the end of the year
	has been reached increment the year and find the new julian date. Now
	standardized the new julian date to the format ddmmm. After the appoint-
	ment has been added to the nine days, then add a suspense to the date
	of the last appointment reminding the user to re-enter the weekly
	appointment.

Parameters:
	date	the starting date of the weekly appointment
	month	the number of the month of the date
	appoint	the start and end time of the appointments

Returns:
	None

Files and Programs:
	None

*/
week_appt (date, month, appoint, n_weeks, envp)
char   *date,						/* Start date for weekly appointments */
       *appoint;					/* Pointer to time of the appointment */
int     month;						/* The number of the month in the starting date */
int n_weeks;
char **envp;
{
    char    outbuf[256],				/* Holds the appointment record */
            file[12],					/* Holds the name of the file */
           *file_ptr,					/* Pointer into file */
	    appt[256],
	   *user_appt = appt,                           /* Pointer to the appointment entered by user */
           *end_ptr,					/* Pointer to outbuf */
            jdate[6],					/* Julian date in ascii */
            newdate[6];					/* Holds new date after julian date has been incremented */
    register int    num_days,				/* Counter for number of the week */
                    julian;				/* Julian date (integer) */
    int     year, n,                                    /* Year (yy) */
	    days;                                       /* Number of days in the year */

    n = n_weeks;
    if (!n)
	n = 4;                                          /* Default to 4 weeks */
    end_ptr = copy (appoint, outbuf);
    do                                                 /* Get appointment */
	printf ("Enter appointment:\n");
    while (!fgets(user_appt, sizeof(appt), stdin) || *user_appt == '\0');
    copy (user_appt, end_ptr);				/* Add appointment to the end of the record */
    file_ptr = copy ("appts.", file);
    year = atoi (get_year ());				/* Get the current year */
    julian = atoi (julian_date (date, month, year));    /* Get the julian date */
    if ((year / 4) * 4 == year && year != 0)		/* Find the number of days in the year */
	days = 366;
    else
	days = 365;
    for (num_days = 1; num_days <= n; num_days++)       /* Add appointment for 9 weeks */
    {
	copy (date, file_ptr);				/* Create filename of appointment file */
	writeappoint (file, outbuf);			/* Write appointment to the file */
	set_alarm(appoint, date, outbuf);
	if (num_days != 10)
	{
	    julian + = 7;
	    if (julian > days)				/* Check for end of year and increment julian date */
	    {						/* New year */
		julian - = days;
		if (++year == 100)			/* Only occurs if it is the year 2000 */
		    year = 0;
	    }
/* Create the julian date in the form yyjjj */
	    copy (itoa (year,2), jdate);
	    copy (itoa (julian,3), &jdate[2]);
	    if (!valid_date (jdate, newdate, 1)) {             /* Get the new date, format ddmmm */
	       printf(stderr, "appoint: date conversion error\n");
	       exit(EFORM);
	    }
	    date = newdate;
	}
    }
    writesus (&file[6], envp);
}
/*

Name:
	writesus

Function:
	Make a suspense to remind the user to use this program again

Algorithm:
	Create the name of the file using the date of the last weekly 
	appointment made. If that file does not exist, then create it giving
	the file read and write permissions to everyone. Now add the message
	to re-enter the weekly appointment to the end of the file.

Parameters:
	date	date to add the suspense

Returns:
	EACCES	Cannot open suspense file

Files and Programs:
	home-dir/exec/suspenses/(file)		suspense files

*/
writesus (date, envp)
char   *date;						/* Date of the suspense */
char **envp;
{
    char    file[15];					/* Name of the suspense file */
    FILE    *fp;                                        /* File descriptor for suspense file */
    int     tvec[2];

    char do_file[256],
	*dofile = do_file;

    time(tvec);
    copy ("sus.", file);				/* Create file name */
    copy (date, &file[4]);
    if (chdir (SUSPENSE) == -1)				/* Move to the suspenses directory */
    {
	printf ("bad directory-- exec/suspenses\n");
	exit (ENOENT);
    }
    if ((fp = fopen (file, "a")) == NULL)       {             /* Check to see if the suspense file already exists */
	    fprintf (stderr, "weeklyappt: cannot open %s\n", file);
	    exit (EACCES);
    }
    strcat(dofile, findenv("HOME", envp));
    strcat(dofile, "/exec/dolist/do");
    fprintf(fp, REENTER, cnv_time(tvec), dofile);        /* Add the suspense */
    fclose (file);
}

