#
/*
Name:
	ADDLIST Calendar System Command

Function:
	Add items to a user's dolist.

Algorithm:
	Read lines input by the user until a "q" is encountered.
	Write each line to the specified list file. Sort and redisplay
	the list by calling "dolist".

Parameters:
	user specified list file name
	user id

Returns:
	ENOERR	Normal exit
	EACCES  Unable to open user specified list file

Files and Programs:
	user specified list file        place to store new items

Installation Instructions:
	Compile:  cc -o $EXECDIR/addlist addlist.c ../library
	Change owner to root
	Deny write permissions except for owner

History:
	Oct 79,  Release 1 Version 1,G. Maltagliati, coded in shell
	Nov 80,  Release 5 Version 1,G. Maltagliati, coded in C for
		 efficiency.
	Jul 82, Release 7  Version 1, L. Reed for Perkin-Elmer
	Aug 82, Release 7  Version 2, J. Neyer...rewrite exec support

*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#define ITEM "Item: "
#define ENTERINFO "Enter dolist items. Use q to quit\n"
#define NEWLINE '\n'
#define USAGE "Usage: addlist <list_file> <user id>\n"

char id[] = "%W% %H%";

/*

Name:
	main

Function:
	Add entered items to the end of a calendar user's list

Algorithm:
	Parse input arguments. Open the user supplied list file or the
	appropriate default list file.
	Call add_items to add to the dolist file.
	Close the file. Call dolist to sort and display
	the file.

Parameters:
	User supplied list file
	User id
Returns:
	ENOERR	Normal exit
	EACCES  Unable to open list file
	EFORM   Parameter format error

Files and Programs:
	user supplied list file         place to store list items
	default list file               holds name of defaul list


*/
main( argc, argv, envp)

int argc;
char  *argv[], **envp;
{
    char listfile[80],                                  /* Holds default list file name */
	 *lfile, *date;                                 /* Pointer to full list file path name  and dummy date*/
    struct passwd *user;
    FILE *fp;                                           /* List file pointer */
    char *line;
    char answer = '\0';
    int ok = 0;

    parse(argc, argv, envp, &date, &user, &lfile);
    if (argc > 3 || (argc == 3 && !user))     {
	fprintf(stderr, USAGE);
	exit(EFORM);
    }
    if (argc == 3 && user && !lfile)      {
	do                                              /* If the passed file does not exist */
	    switch (answer) {                           /* see if he wants to create a new list */
		case 'y':
		    ok += 1;
		    break;
		case 'n':
		    exit(ENOERR);
		    break;
		default:
		    printf("Do you want to create new list file %s (y or n)\n? ", argv[1] == user->pw_name ? argv[2] : argv[1]);
		    answer = getchar();
		    break;
	    }
	while (!ok);
	lfile = listfile;
	strcpy(lfile, user->pw_dir);
	strcat(lfile, "/exec/dolist/");
	argv[1] == user->pw_name ? strcat(lfile, argv[2]) : strcat(lfile, argv[1]);
    }
    else if (argc == 2 && !user && !lfile)      {       /* No valid user or lfile...see if he wants to make new lfile */
	do      {                                       /* If the passed file does not exist */
	    ok = 0;
	    switch (answer) {                           /* see if he wants to create a new list */
		case 'y':
		    ok += 1;
		    break;
		case 'n':
		    exit(ENOERR);
		    break;
		default:
		    printf("Do you want to create new list file %s (y or n)\n? ", argv[1]);
		    answer = getchar();
		    break;
	    }
	}
	while (!ok);
	lfile = listfile;
	strcpy(lfile, findenv("HOME", envp));
	strcat(lfile, "/exec/dolist/");
	strcat(lfile, argv[1]);
    }
    else if (argc == 2 && user && !lfile)     {         /* User but no lfile */
	lfile = listfile;
	strcat(lfile, user->pw_dir);
	strcat(lfile, "/");                          /* go for the default list file */
	strcat (lfile, DEFAULT);
	if ((fp = fopen(lfile, "r")) == NULL)   {    /* Try to open default */
	    fprintf (stderr, "addlist: cannot open default file %s\n", lfile);
	    exit (EACCES);
	}
	line = getline(fp);
	strcpy(lfile, line);
	fclose(fp);
    }
    if (argc == 1)      {
	lfile = listfile;
	strcat(lfile, findenv("HOME", envp));
	strcat(lfile, "/");                          /* go for the default list file */
	strcat (lfile, DEFAULT);
	if ((fp = fopen(lfile, "r")) == NULL)   {    /* Try to open default */
	    fprintf (stderr, "addlist: cannot open default file %s\n", lfile);
	    exit (EACCES);
	}
	line = getline(fp);
	strcpy(lfile, line);
	fclose(fp);
    }

    if ((fp = fopen(lfile, "a")) == NULL)   {         /* Open the list file */
	fprintf (stderr, "addlist: cannot open list file %s\n", lfile);
	exit (EACCES);
    }
    if (!user)
	user = getpwnam(findenv("USER", envp));
    add_items(fp, lfile);
    fclose(fp);
    dolist_call(lfile, user->pw_name);
    exit(ENOERR);
}

/*

Name:
	add_items

Function:
	Add entered items to the end of a calendar user's list

Algorithm:
	Read the user terminal. Append new lines to the end of the file,
	until a "q" is input.

Parameters:
	list file pointer

Returns:
	ENOERR	Normal exit

Files and Programs:
	list file                 place to store list items


*/
add_items (fp, lfile)
FILE *fp;
char *lfile;
#define BUFSIZE 128
{

    int num_chars,                      /* Number of characters read */
	index,
	tvec[2];                        /* Time system call return info */
    char buffer[BUFSIZE];               /* Item buffer */

    printf(ENTERINFO);
    printf(ITEM);
    while ((num_chars = read (0, buffer, BUFSIZE)))     /* Get one to-do item */
	if (num_chars > 1)      {                       /* Not a carriage return */
	    if (num_chars == 2 && buffer[0] == 'q')     /* Look for quit */
		break;
	    for (index = 0; buffer[index] == '*'; index++)
		fputc(buffer[index], fp);
	    fputc (FIELDCHAR, fp);
	    for (; buffer[index] != '\n'; index++)
		fputc (buffer[index], fp);                /* Put in item */
	    fputc (FIELDCHAR, fp);
	    time(tvec);                                   /* Get current time */
	    fprintf(fp , "%s", (cnv_time(tvec)));         /* Put converted time */
	    fputs ("::", fp);
	    printf ("Due to: ");
	    num_char = read(0, buffer, BUFSIZE);          /* Get to whom the suspense is due */
	    for (index = 0; buffer[index] != '\n'; index++)
		fputc(buffer[index], fp);
	    fputs(":::", fp);
	    fprintf(fp, "%s\n", lfile);
	    printf(ITEM);
	}
    return(ENOERR);
}

