/*
Name:
	DEDO Calendar System Command

Function:
	Delete a dolist item.

Algorithm:
	Scan the user supplied list file for the requested item and extract
	it from the list file. If no user file is supplied, look in the
	users $HOME/exec directory to find the default file. Then use the
	current default dolist. Then copy all but the specified item to
	the temp file. Copy the temp file back to the list file. Unlink
	the temp file.

Parameters:
	item number             the line number of the list item in the list file
	user supplied list file
		or
	default file in $HOME/exec/default

Returns:
	ENOERR	Normal exit
	EFORM	Bad date format or incorrect number of command line arguments

Files and Programs:
	user supplied list file
	$SUSPENSE/sus.ddmmm                             the target suspense file
	TEMPFILE                                        temporary file for extraction

Installation Instructions:
	Compile as follows: cc -o $EXEC.DIR/dedo dedo.c ../library
	Change owner to root
	Deny write permissions except for owner

History:
	Dec 79,	Release 2 Version 1, G. Maltagliati, coded in shell
	Nov 80,	Release 5 Version 1, G. Maltagliati, coded in C for efficiency
	Jul 82, Release 7 Version 1, L. Reed, for Perkin-Elmer
	Aug 82, Release 7 Version 2, J. Neyer...rewrite exec support


*/
#include <stdio.h>
#include "../includes/error.h"
#include "../includes/getdef.h"
#include <sys/types.h>
#include <sys/stat.h>
#define USAGE "You must enter:  dedo item-number <list-file>\n"

char id[] = "%W% %H%";

/*

Name:
	main

Function:
	Validate parameters. Open the list file.
	Delete the item. List the dolist.

Algorithm:
	Insure that the correct number of arguments were entered on the command
	line. Determine if a user supplied list file was given and open it if
	so. Otherwise, find the default dolist file in $HOME/exec/default
	and open it. Call dedo to delete the item. Call dolist_call to
	sort and re-display the new do list.

Parameters:
	item number		the line number of the dolist item in the dolist file
	user supplied list file or default list file

Returns:
	ENOERR	Normal exit
	EFORM	Bad date format or incorrect number of command line arguments

Files and Programs:
	user supplied list file or default list file


*/
main (argc, argv, envp)
int     argc;						/* Count of command line arguments */
char   *argv[];						/* Pointer to list of argument pointers */
char   **envp;                  /* the environment */
{


    char listfile[80],
	 *lfile,                             /* Pointer to list file name */
	 *date,
	 *line;

    struct passwd *user;
    struct stat *stat_buf;
    FILE *fp;

    extern char *getline();
    extern char *mktemp();
    extern char *sprintf();
    extern char *strcat();
    extern char *findenv();

    if (argc < 2 || argc > 3)       {
	fprintf (stderr, USAGE);
	exit(EFORM);
    }
    parse(argc, argv, envp, &date, &user, &lfile);
    if (!lfile)      {
	lfile = listfile;
	strcat (lfile, (findenv("HOME", envp)));        /* Get the HOME directory */
	strcat (lfile, "/");
	strcat (lfile, DEFAULT);
	if ((fp = fopen(lfile, "r")) == NULL)   {       /* Try to open default */
	    fprintf (stderr, "dedo: cannot open default file %s\n", lfile);
	    exit (EACCES);
	}
	if ((line = getline(fp)) == NULL)    {         /* Get the default dolist */
	    fprintf (stderr, "dedo: default file empty\n");
	    exit (EACCES);
	}
	strcpy(lfile, line);
	if ((fclose(fp)) == EOF)                             {
	    fprintf (stderr,"dedo: cannot close file %s\n", lfile);
	    exit (EACCES);
	}
    }
    delist(lfile, argv[1]);
    dolist_call(lfile, findenv("USER", envp));
    exit(ENOERR);
}
