#
/*
 * lcheck
 *
 *
 * Call:
 *	% lcheck filesystem ...
 *
 *
 * Description:
 *	lcheck checks the internal linking consistency of the filesystem and
 * reports errors.  Specifically, the '.' and '..' entries are checked.
 * lcheck also reports directories which aren't executable, and therefore
 * filesystem subtrees which aren't accessable.
 *	lcheck finds directories which cannot be referenced by traversing
 * the filesystem tree starting at inode 1, ie, directories which have no
 * external links, and are therefore somewhat useless.
 *	lcheck assumes that the filesystem checks out both with dcheck, and
 * icheck.
 *
 *
 * CSL	v12.31.7(0) - r12.31.7(0)	Hannes Beinert
 *	v01.02.7(1) - r01.02.7(0)
 */


#include	"/usr/sys/ino.h"	/* system inode structure	*/

#define	EXBITS	0111			/* execute bits for file	*/
#define	DIRSIZ	16			/* directory entry is 16 bytes	*/
#define	BLKSIZ	512			/* length of a block		*/
#define	IBLOCK	16			/* 16 inodes in an ilist block	*/
#define	NINODE	606*16			/* number of inodes per filesys	*/
#define	NDIRS	1000			/* size of directory structure	*/
#define	NAMLEN	14			/* length of a fs node name	*/
#define	ALLOC	01			/* alloc bit in i_ref array	*/
#define	TRVERS	02			/* traverse flag in i_ref	*/


char	*dargv[]	{		/* default arguments		*/
	0,
	"/dev/rp0",
	"/dev/rp2",
	"/dev/rp3",
	-1,
};

struct	sbuf	{			/* used for stat syscall	*/
	int	devn;			/* device number		*/
	int	inumber;		/* inumber of file		*/
	int	flags;			/* file flags			*/
	int	fill[15];		/* unnecessary 			*/
}	sbuf;

struct	d_ent	{			/* directory entry		*/
	int	inum;
	char	dnam[NAMLEN];
};

struct	d_node	{			/* representation of directory	*/
	int	dirmod;			/* mode of directory		*/
	int	dirino;			/* directory inumber		*/
	char	*size0;			/* top 8 bits of 24 bit size	*/
	char	*size1;			/* lower 16 bits...		*/
	int	addr[8];		/* block index list		*/
}	d_node[NDIRS];

char	i_ref[NINODE];			/* traverse flag for each inode	*/

struct	inode	i_block[IBLOCK];	/* block of inodes		*/

int	dev;				/* fildes of input device	*/
int	ndirfs;				/* number of directories in fs	*/
int	ninofs;				/* number of inodes in fs	*/
int	ndirdev;			/* number of directories on dev	*/
int	ninodev;			/* number of inodes on device	*/
int	nerrino;			/* number of bad inodes		*/
int	nerrdir;			/* number of bad directories	*/
int	ino;				/* inumber of read inode	*/
int	d_top;				/* top of d_node array		*/
int	i_size;				/* i-size of current device	*/
char	**argp;				/* argv pointer			*/


main(argc, argv)
int	argc;
char	**argv;
{
	/*
	 * process every specified arg...
	 *
	 * if none specified, use defaults...
	 */

	if (argc <= 1)	argp = dargv;
	else		argp = argv;

	while (*++argp != -1) {
		printf("%s:\n",*argp);
		setup(*argp);
	}

	exit(0);
}


/*
 * setup	checks the specified file name to be a device, etc.
 */

setup(devnam)
char	*devnam;
{
	/*
	 * specified argument must be
	 * either a character or
	 * special device...
	 */

	if (stat(devnam,&sbuf) < 0) {
		printf("Can't find %s\n",devnam);
		return;
	}

	if ((sbuf.flags & IFMT) != IFBLK && (sbuf.flags & IFMT) != IFCHR) {
		printf("Not block or char device.\n");
		return;
	}

	/*
	 * open device...
	 */

	if ((dev = open(devnam,0)) < 0) {
		printf("Can't open %s\n",devnam);
		return;
	}

	/*
	 * initialize...
	 */

	ndirfs  = 0;
	ndirdev = 0;
	nerrdir = 0;
	ninofs  = 0;
	ninodev = 0;
	nerrino = 0;

	d_top = 0;

	sync();

	/*
	 * read in isize of device,
	 * then read in all inodes,
	 * entering all directories
	 * into d_node...
	 */

	intdev();

	/*
	 * start traversing fs tree
	 * with inumber 1, ie / of
	 * the current device...
	 */

	check(1,1);

	/*
	 * scan through entire i_ref
	 * table, and print inumbers
	 * of 'forgotten' inodes...
	 */

	forgot();

	/*
	 * print statistics...
	 */

	printf("fs  direc\t%d\n",ndirfs);
	printf("dev direc\t%d\n",ndirdev);
	printf("err direc\t%d\n",nerrdir);
	printf("fs  inode\t%d\n",ninofs);
	printf("dev inode\t%d\n",ninodev);
	printf("err inode\t%d\n",nerrino);
	printf("max inode\t%d\n",ino);
	printf("\n");

	return;
}


/*
 * intdev	initializes the d_node table, filling in all interesting
 *		things.  It also counts the number of device directories,
 *		ndirdev.
 */

intdev()
{
	register int	i;
	register struct	inode	*iptr;

	/*
	 * read isize...
	 *
	 * also check if i_ref table
	 * is big enough...
	 */

	bread(1,&i_size,2);

	if ((i_size * IBLOCK) > NINODE) {
		printf("Too many ilist blocks.\n");
		return;
	}

	/*
	 * start reading ilist blocks...
	 */

	ino = 0;

	for (i = 0;i < i_size;i++) {
		bread(i+2,i_block,BLKSIZ);
		for (iptr = i_block;iptr < &i_block[IBLOCK];iptr++) {
			++ino;
			if ((iptr->i_mode & IALLOC) != IALLOC) continue;
			i_ref[ino] =| ALLOC;
			ninodev++;
			if ((iptr->i_mode & IFMT) == IFDIR) {
				enterd(iptr);
				ndirdev++;
			}
		}
	}
}


/*
 * enterd	enters a directory into the d_node table, reading the point
 *		and point-point directory entries.
 */

enterd(iptr)
struct	inode	*iptr;
{
	register int	i;
	register struct	d_node	*dptr;
	register int	*ptr1;
	int	 *ptr2;

	if (d_top >= NDIRS) {
		printf("Too many directories.\n");
		return;
	}

	dptr = &d_node[d_top];

	dptr->dirino = ino;
	dptr->dirmod = iptr->i_mode;

	dptr->size0 = iptr->i_size0;
	dptr->size1 = iptr->i_size1;

	ptr1 = iptr->i_addr;
	ptr2 = dptr->addr;
	for (i = 0;i < 8;i++)
		*ptr2++ = *ptr1++;

	d_top++;
	return;
}


/*
 * check(parent,current)
 *
 *	check will look at the current slot in the d_node table.  It will
 *	print an error if either the '.' or the '..' don't exit.  It will
 *	also give an error if they aren't linked properly.  Finally, check
 *	makes sure that the specified directory is executable by anyone.
 *
 *	Note that parent is an actual inode number, however current is an
 *	index into the d_node table.
 */

check(par,cur)
int	par;
int	cur;
{
	register struct	d_node	*dptr;
	register int	dd;
	register int	i;
	int	 rptr;
	struct	 d_ent	*d_ent;

	rptr = 0;

	if (cur == 0) return;			/* deleted file		*/

	if ((i_ref[cur] & TRVERS) == TRVERS) return;	/* linked inode	*/
	i_ref[cur] =| TRVERS;

	ninofs++;
	if ((dptr = find(cur)) == 0) return;	/* not a directory	*/
	ndirfs++;

	if (cur != dptr->dirino)
		printf("check:  error\n");

	if ((dptr->dirmod & EXBITS) == 0)
		printf("%5l:  bad dir mode (0%o)\n",dptr->dirino,dptr->dirmod);

	dd = 0;

	while (d_ent = rentry(dptr,rptr++)) {
		if (dd != 3) {
			i = lchk(par,cur,d_ent);
			dd =| i;
			if (i) continue;
		}
		check(dptr->dirino,d_ent->inum);
	}

	if ((dd & 1) != 1)
		printf("%5l:  Link entry missing (.)\n",cur);

	if ((dd & 2) != 2)
		printf("%5l:  Link entry missing (..)\n",cur);
}


/*
 * find		takes its argument (an inode number) and returns the address
 *		of this inode in d_node if it's a directory.  If it isn't
 *		find will return 0.
 */

find (inumber)
int	inumber;
{
	register struct	d_node	*dptr;
	register int	j;

	for (dptr = d_node;dptr < &d_node[d_top];dptr++) {
		if ((j = dptr->dirino) == inumber)
			return(dptr);

		if (j > inumber) break;
	}

	return(0);
}


/*
 * lchk		is passed the directory entry of what should be either
 *		a '.' or a '..'.  If it is one of these, their inumbers are
 *		checked, and an error is printed if they aren't correct.
 *		If it is neither, no error is printed.
 */

lchk(par,cur,d_ent)
int	par;
int	cur;
struct	d_ent	*d_ent;
{
	register char	*cptr;

	cptr = d_ent->dnam;

	if (cptr[0] == '.' && cptr[1] == '\0') {
		if (cur != d_ent->inum)
			printf("%5l:  Link err (.)\n",cur);

		return(1);
	}

	if (cptr[0] == '.' && cptr[1] == '.' && cptr[2] == '\0') {
		if (par != d_ent->inum)
			printf("%5l:  Link err (..)\n",cur);

		return(2);
	}

	return(0);
}


/*
 * rentry	reads the specified entry from the specified inode.
 *		The address of the directory entry is returned if it
 *		was found, else zero.
 */

rentry(dnod,rptr)
struct	d_node	*dnod;
int	rptr;
{
	register int	blk,off;
	register struct	d_node	*dptr;
	int	 *ibp;
	static	 char	buf[BLKSIZ];

	off = rptr * DIRSIZ;
	dptr = dnod;
	ibp = buf;
	if (off == 0177760) {
		printf("%5l:  Dir too large\n",dptr->dirino);
		return(0);
	}
	if ((dptr->dirmod & ILARG) == 0) {
		if (off >= 010000 || (blk = dptr->addr[off>>9]) == 0)
			return(0);
		bread(blk, buf, BLKSIZ);
	}
	else {
		if (dptr->addr[0] == 0)
			return(0);
		bread(dptr->addr[0], buf, BLKSIZ);
		if ((blk = ibp[(off>>9) & 0177]) == 0)
			return(0);
		bread(blk, buf, BLKSIZ);
	}

	return(&buf[off&0777]);
}


/*
 * bread	reads a block from the current device, and reports any errors.
 */

bread(bno, buf, cnt)
{
	seek(dev, bno, 3);
	if(read(dev, buf, cnt) != cnt) {
		printf("read error %d\n", bno);
		exit(1);
	}
}


/*
 * forgot	scans through the i_ref table, and prints the indexes of those
 *		elements which were never referenced, yet allocated.
 */

forgot()
{
	register int	i;

	for (i = 0;i < NINODE;i++) {
		if ((i_ref[i] & ALLOC) != ALLOC) continue;

		if ((i_ref[i] & TRVERS) != TRVERS) {
			nerrino++;
			printf("%5l:  Never referenced",i);
			if (find(i)) {
				nerrdir++;
				printf(" (dir)");
			}
			printf("\n");
		}
	}
}
