/*
 * pr
 *   print file with headings
 *  2+head+2+page[56]+5
 * modified from an enhanced v6 pr to be mostly compatible with
 * a v7 library (using stdio), by reference to v7 pr.
 * further enhanced with various features including multiline headings
 * by David E. Miran
 *
 * version of 5/3/82
 */

/* available options include:
-f file = file contains multiline heading
-d      = double space mode
-p      = page numbers for multiline heading
-v      = form feed at end of page, not blank lines
-x      = suppress date and time in regular headings
-y      = suppress page number in regular headings
-t or -r = raw mode - no top or bottom margins
-l#     = page length is #
-w#     = page width is #
-m      = print all files simultaneously
-sx     = column separator character is x
-b#     = border (top and bottom margin + heading) size is #.
-#      = # columns per page where # is a number
+#      = start printing with page number # where # is a number
*/

#include	<stdio.h>
#define	BUFS	8976	/* 136 x 66 */
#define	HBS	4096	/* maximum size of multiline headings */
#define	FF	014

int	ncol	1;
char	*header;
int	col;
int	icol;
FILE	*file;
char	*bufp;
char	buffer[BUFS];
char	obuf[BUFSIZ];
int	line;
char	*colp[72];
int	nofile;
char	isclosed[10];
FILE	*ifile[10];
char	**lastarg;
int	peekc;
int	fpage;
int	page;
int	colw;
int	nspace;
int	width	72;
int	length	66;
int	plength 61;
int	margin	10;
int	ntflg;
int	mflg;
int	nflg;
int	tabc;
char	*tty;
int	mode;
int	linenum;

char	hbuf[HBS];	/* multiline heading buffer */
int	xflag	0;
int	dflag	0;
int	yflag	0;
int	hdsize	0;		/* number of chars in heading */
int	hdpgloc	0;	/* loc in heading to insert page number */
int	hdlcnt	0;		/* number of lines in heading */
int	vflag	0;		/* do form feed at end of page instead of blank lines */
int	bdsize	0;		/* size of border (default is hdlcnt + 3) */
int	pflag	0;		/* set if page numbers should be added to end
			 * of first non blank line of heading */
int	hdflag	0;		/* set if multiline heading is available */
int	xlength;
char	*ctime();

struct inode {
	int dev;
	int inum;
	int flags;
	char nlink;
	char uid;
	char gid;
	char siz0;
	int size;
	int ptr[8];
	int atime[2];
	int mtime[2];
};

main(argc, argv)
char **argv;
{
	int nfdone;
	int onintr();
	setbuf(stdout,obuf);

	tty = "/dev/ttyx";
	if ((signal(2, 1) & 01) == 0)
		signal(2, onintr);
	lastarg = &argv[argc - 1];
	fixtty();
	for (nfdone=0; argc>1; argc--) {
		argv++;
		if (**argv == '-') {
			switch (*++*argv) {
			case 'h':
				if (argc>=2) {
					header = *++argv;
					argc--;
				}
				continue;

			case 't':
			case 'r':
				ntflg++;
				continue;

			case 'x':
				xflag++;
				continue;

			case 'y':
				yflag++;
				continue;

			case 'l':
				length = getn(++*argv);
				continue;

			case 'w':
				width = getn(++*argv);
				continue;

			case 's':
				if (*++*argv)
					tabc = **argv;
				else
					tabc = '\t';
				continue;

			case 'm':
				mflg++;
				continue;

			case 'n':
				nflg++;
				continue;

			case 'p':
				pflag++;
				continue;

			case 'v':
				vflag++;
				continue;

			case 'd':
				dflag++;
				continue;

			case 'b':
				bdsize = getn(++*argv);
				if (bdsize < hdlcnt) bdsize = hdlcnt;
				continue;

			case 'f':
				gethdg(*++argv);	/* multiline heading file */
				argc--;
				continue;

			default:
				ncol = getn(*argv);
				continue;
			}
		} else if (**argv == '+') {
			fpage = getn(++*argv);
		} else {
			print(*argv, argv);
			nfdone++;
			if (mflg)
				break;
		}
	}
	if (nfdone==0)
		print(0);
	onintr();
}

onintr()
{

	chmod(tty, mode);
	exit(0);
}

fixtty()
{
	struct inode sbuf;

	tty[8] = ttyn(1);
	fstat(1, &sbuf);
	mode = sbuf.flags&0777;
	chmod(tty, 0600);
}

print(fp, argp)
char *fp;
char **argp;
{
	struct inode sbuf;
	register int sncol, sheader;
	register char *cbuf;

	if (ntflg || hdflag)
		margin = 0;
	else
		margin = 10;
	if (length <= margin)
		length = 66;
	if (width <= 0)
		width = 72;
	if (ncol>72 || ncol>width) {
		write(2, "Too many columns.\n", 18);
		exit();
	}
	if (mflg) {
		mopen(argp);
		ncol = nofile;
	}
	colw = width/ncol;
	sncol = ncol;
	sheader = header;
	if (ntflg) goto setb;
	if (hdflag) {
		plength = length - (bdsize - hdlcnt);
	}  else  {
		if (bdsize == 0) bdsize = 10;
		if (bdsize < 5) bdsize = 5;
		plength = length - (bdsize - 5);
	}
setb:
	if (ntflg)
		plength = length;
	margin = bdsize;
	xlength = length - margin;
	if (dflag) xlength = xlength/2;
	if (--ncol<0)
		ncol = 0;
	if (mflg)
		fp = 0;
	if (fp) {
		file = fopen(fp, "r");
		if (file == NULL)
			return;
		stat(fp, &sbuf);
	} else {
		file = stdin;
		time(sbuf.mtime);
	}
	if (header == 0)
		header = fp;
	cbuf = ctime(sbuf.mtime);
	cbuf[16] = '\0';
	cbuf[24] = '\0';
	page = 1;
	linenum = 1;
	icol = 0;
	colp[ncol] = bufp = buffer;
	if (mflg==0)
		nexbuf();
	while (mflg&&nofile || (!mflg)&&tpgetc(ncol)>0) {
		if (mflg==0) {
			colp[ncol]--;
			if (colp[ncol] < buffer)
				colp[ncol] = &buffer[BUFS];
		}
		line = 0;
		if ((ntflg == 0) && (hdflag == 0)) {
			puts("\n\n");
			if (!xflag) {
				puts(cbuf+4);
				puts(" ");
				puts(cbuf+20);
				puts("  ");
			}
			puts(header);
			if (!yflag) {
				puts(" Page ");
				putd(page);
			}
			puts("\n\n\n");
		} else
			if (hdflag != 0) puthdg();
		putpage();
		if (vflag && (ntflg == 0))
			put(FF);
		else
			if (ntflg==0)
				while(line<length)
					put('\n');
		page++;
	}
	fclose(file);
	ncol = sncol;
	header = sheader;
}

mopen(ap)
char **ap;
{
	register char **p, *p1;

	p = ap;
	while((p1 = *p) && p++ <= lastarg) {
		if((ifile[nofile]=fopen(p1, "r")) == NULL){
			isclosed[nofile] = 1;
			nofile--;
		}
		else
			isclosed[nofile] = 0;
		if(++nofile>=10) {
			fprintf(stderr, "pr: Too many args\n");
			onintr();
		}
	}
}

/* prints the CONTENTS of a page (not the page eject) */
putpage()
{
	register int lastcol, i, c;
	int j;

	if (ncol==0) {
		while (line<plength) {
			if ((c = tpgetc(0)) && c != FF) {
				if (nflg) {
					printfd(linenum++,6);
					putcp(':'); putcp(' ');
				}
				if (c != '\n') {
					putcp(c);
					while((c = tpgetc(0)) && c!='\n' && c!=FF)
						putcp(c);
					if (c==FF)
						linenum--;
				}
			}
			putcp('\n');
			line++;
			if (dflag) {
				putcp('\n');
				line++;
			}
			if (c==FF)
				break;
		}
		return;
	}
	colp[0] = colp[ncol];
	if (mflg==0) for (i=1; i<=ncol; i++) {
		colp[i] = colp[i-1];
		for (j = 0; j<xlength; j++)
			while((c=tpgetc(i))!='\n')
				if (c==0)
					break;
	}
	while (line<plength) {
		lastcol = colw;
		for (i=0; i<ncol; i++) {
			while ((c=pgetc(i)) && c!='\n')
				if (col<lastcol || tabc!=0)
					put(c);
			if (tabc)
				put(tabc);
			else while (col<lastcol)
				put(' ');
			lastcol =+ colw;
		}
		while ((c = pgetc(ncol)) && c!='\n')
			put(c);
		put('\n');
		if (dflag) put('\n');
	}
}

nexbuf()
{
	register int n;
	register char *rbufp;

	rbufp = bufp;
	n = &buffer[BUFS] - rbufp;
	if (n>512)
		n = 512;
	if (feof(file) ||
		(n = fread(rbufp,1,n,file)) <= 0) {
		fclose(file);
		*rbufp = 0376;
	}  else {
		rbufp =+ n;
		if (rbufp >= &buffer[BUFS])
			rbufp = buffer;
		*rbufp = 0375;
	}
	bufp = rbufp;
}

tpgetc(ai)
{
	register char **p;
	register int c, i;

	i = ai;
	if (mflg) {
		if ((c = getc(ifile[i])) == EOF) {
			if (isclosed[i]==0) {
				isclosed[i] = 1;
				if (--nofile <= 0)
					return(0);
			}
			return('\n');
		}
		if (c==FF && ncol>0)
			c = '\n';
		return(c);
	}
loop:
	c = **(p = &colp[i]) & 0377;
	if (c == 0375) {
		nexbuf();
		c = **p & 0377;
	}
	if (c == 0376)
		return(0);
	(*p)++;
	if (*p >= &buffer[BUFS])
		*p = buffer;
	if (c==0)
		goto loop;
	return(c);
}

pgetc(i)
{
	register int c;

	if (peekc) {
		c = peekc;
		peekc = 0;
	} else
		c = tpgetc(i);
	if (tabc)
		return(c);
	switch (c) {

	case '\t':
		icol++;
		if ((icol&07) != 0)
			peekc = '\t';
		return(' ');

	case '\n':
		icol = 0;
		break;

	case 010:
	case 033:
		icol--;
		break;
	}
	if (c >= ' ')
		icol++;
	return(c);
}

puts(as)
char *as;
{
	register int c;
	register char *s;

	if ((s=as)==0)
		return;
	while (c = *s++)
		put(c);
}

putd(an)
{
	register int a, n;

	n = an;
	if (a = n/10)
		putd(a);
	put(n%10 + '0');
}

put(ac)
{
	register int ns, c;

	c = ac;
	if (tabc) {
		putcp(c);
		if (c=='\n')
			line++;
		return;
	}
	switch (c) {

	case ' ':
		nspace++;
		col++;
		return;

	case '\n':
		col = 0;
		nspace = 0;
		line++;
		break;

	case 010:
	case 033:
		if (--col<0)
			col = 0;
		if (--nspace<0)
			nspace = 0;

	}
	while(nspace) {
		if (nspace>2 && col > (ns=((col-nspace)|07))) {
			nspace = col-ns-1;
			putcp('\t');
		} else {
			nspace--;
			putcp(' ');
		}
	}
	if (c >= ' ')
		col++;
	putcp(c);
}

getn(ap)
char *ap;
{
	register int n, c;
	register char *p;

	p = ap;
	n = 0;
	while ((c = *p++) >= '0' && c <= '9')
		n = n*10 + c - '0';
	return(n);
}

putcp(c)
{
	if (page >= fpage)
		putchar(c);
}

/* Print n in decimal; n must be non-negative
   s is the size of the field, 0 means as big as needed.
*/
printfd(n,s)
int n,s;
{
	int minus;
	char buf[10];
	char *p;
	if (n < 0) {
		n  = -n;
		minus = -1;
	} else if (n == 0) minus = 0;
	  else		   minus = 1;
	for(p=buf; p<buf+9; p++) *p=' ';
	*p = '0';
	for(; n; p--) {
		*p = n%10 + '0';
		n =/ 10;
	}
	switch(minus) {
		case -1: *p = '-'; break;
		case 0: break;
		case 1: p++; break;
	}
	if (p > buf+10-s) p = buf+10-s;
	for (; p <= buf+9; p++) putcp(*p);
}

/* get a multiline heading and analyze it */

gethdg(file)
char *file;
{
int fid, cseen;
register int i, ln;
register char c;

	fid = open(file, 0);
	if (fid < 0) return;
	hdsize = read(fid, hbuf, HBS);
	close(fid);
	ln = cseen = 0;
	for (i=0; i<hdsize; i++) {
		c = hbuf[i];
		if (c != '\n') {
			cseen = 1;
			continue;
		}
		ln++;
		if (cseen && !hdpgloc)  hdpgloc = i;
	}
	if (bdsize == 0) bdsize = ln+3;
	hdlcnt = ln;
	if (bdsize < hdlcnt) bdsize = hdlcnt;
	hdflag = 1;
}

puthdg()	/* print multiline heading */
{
register int i;

	if (!pflag) puts(hbuf);
	else {
		for (i=0; i<hdpgloc; i++)  put(hbuf[i]);
		putd(page);
		for (i=hdpgloc; i<hdsize; i++) put(hbuf[i]);
	}
}
