/*
 * tty.c
 * general TTY subroutines
 *
 * Patch by Bob Beck, 2-MAY-77, to keep <rept><tabs> from
 * crashing system.
 * Modified by Paul Pierce, Sep 10 77, to add scrolling mode (screen stop)
 * Modified by Paul Pierce, Jan 12 78, to do 8 bit output in raw mode
 * Modified by Paul Pierce, Mar 13 79, to send hangup signal on stty 0
 * Modified by David Miran, Mar 80, for BLOCK RAW mode, and remove LCASE code
 * modified 9/2/80 to put a null in the queue if input lost due to ttyhog limit
 * modified 11/20/81 to add XON/XOFF capability (on top of scrolling)
 * modified 2/5/82 to put v7 clist handlers in (getc, putc, cinit)
 * and to move clists, etc outside of kernel space
 * version of 5/5/82
 */
#include "../hd/param.h"
#include "../hd/systm.h"
#include "../hd/user.h"
#include "../hd/tty.h"
#include "../hd/proc.h"
#include "../hd/inode.h"
#include "../hd/reg.h"
#include "../hd/conf.h"
#include "../hd/seg.h"
#include "../hd/file.h"

#define	BURST	40  /* Longest string of characters processed before */
			/* allowing interrupts again in canon and ttread */

/*
 * Input mapping table-- if an entry is non-zero, when the
 * corresponding character is typed preceded by "\" the escape
 * sequence is replaced by the table value.  Mostly used for
 * upper-case only terminals.  ***  disabled at state hygiene lab
char	maptab[]
{
	000,000,000,000,004,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,'|',000,'#',000,000,000,'`',
	'{','}',000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	'@',000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,'~',000,
	000,'A','B','C','D','E','F','G',
	'H','I','J','K','L','M','N','O',
	'P','Q','R','S','T','U','V','W',
	'X','Y','Z',000,000,000,000,000,
};   */

/*
 * The actual structure of a clist block manipulated by
 * getc and putc (mch.s)
 */
struct cblock {
	struct cblock *c_next;
	char c_info[CBSIZE];
};

/* The character lists-- space for 14*NCLIST characters */
/* the clists themselves and the space for canonb are mapped out */


struct bufpt clseg;
char *claddr, *canaddr;  /* the memory addresses of the mapped out spaces */
char *dhmem, *dhrmem;

/* note - one large chunk of memory is allocated in cinit for
    a)  DH11 DMA space
    b)  canonical buffer
    c)  clist space
 * in that order.   *** total size must not exceed 8192 bytes
 * clseg is the segmentation pointer for any of these.
 * This grouping reduces the amount of mapping in and out needed.
 */
/* NOTE - this must be the first memory allocation initialization
 * routine called so that the DH space will be entirely under 65K.
 * Otherwise dhstart will need to be modified to set up the dh
 * control registers for the proper extended address bits.
 */
/* List head for unused character blocks. */
struct cblock *cfreelist;

/*
 * structure of device registers for KL, DL, and DC
 * interfaces-- more particularly, those for which the
 * SSTART bit is off and can be treated by general routines
 * (that is, not DH).
 */
struct {
	int ttrcsr;
	int ttrbuf;
	int tttcsr;
	int tttbuf;
};

/*
 * The routine implementing the gtty system call.
 * Just call lower level routine and pass back values.
 */
gtty()
{
	int v[3];
	register *up, *vp;

	vp = v;
	sgtty(vp);
	if (u.u_error)
		return;
	up = u.u_arg[0];
	suword(up, *vp++);
	suword(++up, *vp++);
	suword(++up, *vp++);
}

/*
 * The routine implementing the stty system call.
 * Read in values and call lower level.
 */
stty()
{
	register int *up;

	up = u.u_arg[0];
	u.u_arg[0] = fuword(up);
	u.u_arg[1] = fuword(++up);
	u.u_arg[2] = fuword(++up);
	sgtty(0);
}

/*
 * Stuff common to stty and gtty.
 * Check legality and switch out to individual
 * device routine.
 * v  is 0 for stty; the parameters are taken from u.u_arg[].
 * c  is non-zero for gtty and is the place in which the device
 * routines place their information.
 */
sgtty(v)
int *v;
{
	register struct file *fp;
	register struct inode *ip;

	if ((fp = getf(u.u_ar0[R0])) == NULL)
		return;
	mapin(&fileseg);
	ip = fp->f_inode;
	mapout();
	if ((ip->i_mode&IFMT) != IFCHR) {
		u.u_error = ENOTTY;
		return;
	}
	(*cdevsw[ip->i_addr[0].d_major].d_sgtty)(ip->i_addr[0], v);
}

/*
 * Wait for output to drain, then flush input waiting.
 */
wflushtty(atp)
struct tty *atp;
{
	register struct tty *tp;

	tp = atp;
	spl5();
	while (tp->t_outq.c_cc) {
		tp->t_state =| ASLEEP;
		sleep(&tp->t_outq, TTOPRI);
	}
	flushtty(tp);
	spl0();
}

/*
 * Character list get/put
 */

getc(p)
{
	register s, c;

	s=spl6();
	mapin(&clseg);
	c=xgetc(p);
	mapout();
	splx(s);
	return(c);
}

xgetc(p)
register struct clist *p;
{
	register struct cblock *bp;
	register int c, s;

	if (p->c_cc <= 0) {
		c = -1;
		p->c_cc = 0;
		p->c_cf = p->c_cl = NULL;
	} else {
		c = *p->c_cf++ & 0377;
		if (--p->c_cc<=0) {
			bp = (struct cblock *)(p->c_cf-1);
			bp = (struct cblock *) ((int)bp & ~CROUND);
			p->c_cf = NULL;
			p->c_cl = NULL;
			bp->c_next = cfreelist;
			cfreelist = bp;
		} else if (((int)p->c_cf & CROUND) == 0){
			bp = (struct cblock *)(p->c_cf);
			bp--;
			p->c_cf = bp->c_next->c_info;
			bp->c_next = cfreelist;
			cfreelist = bp;
		}
	}
	return(c);
}

putc(c,p)
{
	register x, s;

	s=spl6();
	mapin(&clseg);
	x=xputc(c,p);
	mapout();
	splx(s);
	return(x);
}

xputc(c, p)
register struct clist *p;
{
	register struct cblock *bp;
	register char *cp;
	register s;

	if ((cp = p->c_cl) == NULL || p->c_cc < 0 ) {
		if ((bp = cfreelist) == NULL)
			return(-1);
		cfreelist = bp->c_next;
		bp->c_next = NULL;
		p->c_cf = cp = bp->c_info;
	} else if (((int)cp & CROUND) == 0) {
		bp = (struct cblock *)cp - 1;
		if ((bp->c_next = cfreelist) == NULL)
			return(-1);
		bp = bp->c_next;
		cfreelist = bp->c_next;
		bp->c_next = NULL;
		cp = bp->c_info;
	}
	*cp++ = c;
	p->c_cc++;
	p->c_cl = cp;
	return(0);
}

/*
 * Initialize clist by freeing all character blocks, then count
 * number of character devices. (Once-only routine)
 * Also, allocate memory for clists and canonb buffer.
 * And for DH11 DMA output space
 */
cinit()
{
	register int ccp;
	register struct cblock *cp;
	register struct cdevsw *cdp;
	int x, mem;

	mem = getobuf(&claddr, &clseg, NCLIST*CLSIZE + NDH11*DHNCH + CANBSIZ);
	mapin(&clseg);
	dhmem = claddr;
	dhrmem = mem<<6;
	canaddr = claddr + NDH11*DHNCH;
	ccp = claddr + NDH11*DHNCH + CANBSIZ;
	ccp = (ccp+CROUND) & ~CROUND;
	x = NCLIST - 1;
	cp = (struct cblock *) ccp;
	for (mem = 0; mem < x; mem++) {
		cp->c_next = cfreelist;
		cfreelist = cp;
		cp++;
	}
	mapout();
	ccp = 0;
	for(cdp = cdevsw; cdp->d_open; cdp++)
		ccp++;
	nchrdev = ccp;
}

/*
 * flush all TTY queues
 */
flushtty(atp)
struct tty *atp;
{
	register struct tty *tp;
	register int sps;

	tp = atp;
	sps = spl5();
	mapin(&clseg);
	while (xgetc(&tp->t_canq) >= 0);
	while (xgetc(&tp->t_outq) >= 0);
	while (xgetc(&tp->t_rawq) >= 0);
	mapout();
	tp->t_delct = 0;
	splx(sps);
	wakeup(&tp->t_rawq);
	wakeup(&tp->t_outq);
	if(tp->t_state&SCSTOP)
		ttrstrt(tp);
	tp->t_scnt = NSCROLL;
}

/*
 * transfer raw input list to canonical list,
 * doing erase-kill processing and handling escapes.
 * It waits until a full line has been typed in cooked mode,
 * or until any character has been typed in regular raw mode
 * or until (blksiz chars read or blkchr char is read) in block raw mode.
 */
canon(atp)
struct tty *atp;
{
	register char *bp;
	char *bp1;
	register struct tty *tp;
	register int c;
	int sps, i;

	tp = atp;
	sps = spl5();
	while (tp->t_delct==0) {
		if ((tp->t_state&CARR_ON)==0)
			return(0);
		sleep(&tp->t_rawq, TTIPRI);
	}
	splx(sps);
	mapin(&clseg);
loop:
	bp = canaddr + 2;
	sps = spl5();
	while ((c=xgetc(&tp->t_rawq)) >= 0) {
		splx(sps);
		if ((tp->t_flags&RAW)==0) {
			if (c==0377) {
				tp->t_delct--;
				break;
			}
			if (bp[-1]!='\\') {
				if (c==tp->t_erase) {
					if (bp > canaddr +2)
						bp--;
					goto cont;
				}
				if (c==tp->t_kill)
					goto loop;
				if (c==CEOT)
					goto cont;
			}
		} else {   /* raw mode */
			if (tp->t_flags&BLOCK)  {
				if (c == 0377) {
					tp->t_delct--;
					goto cont;
					}
				}
			else   /*  not block mode */
				tp->t_delct--;
		/* modified 1/17/80 by dem so that raw mode will return
		   as many characters as are available (up to CANBSIZ)  */
		}
		*bp++ = c;
		if (bp>=canaddr+CANBSIZ-1)
			break;
cont:
		sps = spl5();
	}
	splx(sps);
	bp1 = bp;
	bp = canaddr +2;
	c = &tp->t_canq;
tlp:
	sps = spl5();
	i = BURST;
	while ((bp<bp1) && (--i != 0))
		xputc(*bp++, c);
	splx(sps);
	if (bp<bp1) goto tlp;
	mapout();
	return(1);
}

/*
 * Place a character on raw TTY input queue, putting in delimiters
 * and waking up top half as needed.
 * Also echo if required.
 * The arguments are the character and the appropriate
 * tty structure.
 */
ttyinput(ac, atp)
struct tty *atp;
{
	register int t_flags, c;
	register struct tty *tp;

	tp = atp;
	c = ac;
	t_flags = tp->t_flags;
	/* if((t_flags&RAW)==0 || t_flags&(CRMOD|LCASE))       removed at slh to fix a block raw mode bug */
	c =& 0177;
	if (c == '\r' && t_flags&CRMOD)
		c = '\n';
	if ((t_flags&RAW) == 0) {
		if ( c == CSTOP) {	/* XOFF */
			tp->t_state |= SCSTOP;
			return;
		}
		if (c == CSTART) {
			if (t_flags & SCROLL)  tp->t_scnt = NSCROLL;
			ttrstrt(tp);
			return;
		}
		if(c==CQUIT || c==CINTR) {
			signal(tp, c==CINTR? SIGINT:SIGQIT);
			flushtty(tp);
			return;
		}
		if(t_flags&SCROLL) {
			if(tp->t_scnt == 0) {
				tp->t_scnt = c==' '? 1:NSCROLL;
				ttrstrt(tp);
				return;
			}
			if(c == CSCROLL) {
				tp->t_scnt =+ NSCROLL;
				return;
			}
		}
	}
	if (tp->t_rawq.c_cc>=TTYHOG) {
		printf("ttyhog-%o\n", tp->t_dev);	/* debug */
		flushtty(tp);
		putc(0, &tp->t_rawq);  /* by dem 9/2/80 to warn when input lost */
		if (t_flags & RAW) wakeup(&tp->t_rawq);
		return;
	}
/*	if (t_flags&LCASE && c>='A' && c<='Z')
		c =+ 'a'-'A';   *** no LCASE at slh   */
	if (putc(c, &tp->t_rawq)) printf("CLIST\n");	/* debug */
/*   3 functional handling modes  -  3/25/80  dem
 * normal - wakeup on newline or EOT if no previous wakeup done,
 * raw - wakeup on each char if no previous wakeup done
 * block raw - wakeup after blksiz chars read and no previous wakeup done
	       or if wakeup char is read
 */
	if (t_flags&RAW || c=='\n' || c==CEOT) {
		tp->t_scnt = NSCROLL;
		if (t_flags&RAW) {
			if (t_flags&BLOCK)   {
				if ((tp->t_blkchr == c) || (tp->t_blksiz == tp->t_rawq.c_cc))  {
					if (putc(0377,&tp->t_rawq) == 0)
						tp->t_delct++; /* block ready */
					if (tp->t_delct < 3) wakeup(&tp->t_rawq);
				}
			}  else  {   /*  not block mode  */
				tp->t_delct++;
				if (tp->t_delct < 3) wakeup(&tp->t_rawq);
			}
		}  else  {    /* not raw  */
			if (putc(0377,&tp->t_rawq) == 0)
				tp->t_delct++;
			if (tp->t_delct < 3) wakeup(&tp->t_rawq);
		}
	}
	if (t_flags&ECHO) {
		ttyoutput(c, tp);
		if (((t_flags&RAW) ==0) & (c == tp->t_kill))  {
			c = '\n';
			ttyoutput(c, tp);
		}
		ttstart(tp);
	}
	if ((c == CEOT) && (tp->t_state & SCSTOP)) ttrstrt(tp);
}

/*
 * put character on TTY output queue, adding delays,
 * expanding tabs, and handling the CR/NL bit.
 * It is called both from the top half for output, and from
 * interrupt level for echoing.
 * The arguments are the character and the tty structure.
 *
 * Note patch to allow at most 8 tries at blanks for tabs.
 * Note also that it could be done more efficiently.
 */
ttyoutput(ac, tp)
struct tty *tp;
{
	register int c;
	register struct tty *rtp;
	register char *colp;
	int ctype;
	int tabcnt;

	rtp = tp;
	c = ac;
	if((rtp->t_flags&RAW)==0)
		c =& 0177;
	/*
	 * Ignore EOT in normal mode to avoid hanging up
	 * certain terminals.
 *** commented out at slh ****
	if (c==CEOT && (rtp->t_flags&RAW)==0)
		return;
	 */
	/*
	 * Turn tabs to spaces as required
	 */
	if (c=='\t' && rtp->t_flags&XTABS) {
		tabcnt = 8;
		do
			ttyoutput(' ', rtp);
		while ((rtp->t_col&07) && --tabcnt);
		return;
	}
	/*
	 * for upper-case-only terminals,
	* generate escapes  ***  disabled at slh
	if (rtp->t_flags&LCASE) {
		colp = "({)}!|^~'`";
		while(*colp++)
			if(c == *colp++) {
				ttyoutput('\\', rtp);
				c = colp[-2];
				break;
			}
		if ('a'<=c && c<='z')
			c =+ 'A' - 'a';
		}   *** end disabled code  */

	/*
	 * turn <nl> to <cr><lf> if desired.
	 */
	if(c=='\n' && rtp->t_flags&CRMOD)
		ttyoutput('\r', rtp);
	if (putc(c, &rtp->t_outq))
		return;
	/*
	 * Calculate delays.
	 * The numbers here represent clock ticks
	 * and are not necessarily optimal for all terminals.
	 * The delays are indicated by characters above 0200,
	 * thus (unfortunately) restricting the transmission
	 * path to 7 bits.
	 * unless in raw mode
	 */
	if((rtp->t_flags&RAW)==0) {
		colp = &rtp->t_col;
		ctype = partab[c];
		c = 0;
		switch (ctype&077) {
	
		/* ordinary */
		case 0:
			(*colp)++;
	
		/* non-printing */
		case 1:
		case 5:
			break;
	
		/* backspace */
		case 2:
			if (*colp)
				(*colp)--;
			break;
	
		/* newline */
		case 3:
			ctype = (rtp->t_flags >> 8) & 03;
			if(ctype == 1) { /* tty 37 */
				if (*colp)
					c = max((*colp>>4) + 3, 6);
			} else
			if(ctype == 2) { /* vt05 */
				c = 6;
			}
			*colp = 0;
			break;
	
		/* tab */
		case 4:
			ctype = (rtp->t_flags >> 10) & 01; /* was "& 03" - prp */
			if(ctype == 1) { /* tty 37 */
				c = 1 - (*colp | ~07);
				if(c < 5)
					c = 0;
			}
			*colp =| 07;
			(*colp)++;
			break;
	
		/* vertical motion */
		/* disabled at slh to save space
		case 5:
			if(rtp->t_flags & VTDELAY) 
				c = 0177;
			break;
		** end of disabled code */
	
		/* carriage return */
		case 6:
			ctype = (rtp->t_flags >> 12) & 03;
			if(ctype == 1) { /* tn 300 */
				c = 5;
			} else
			if(ctype == 2) { /* ti 700 */
				c = 10;
			}
			*colp = 0;
			break;
		/* null for timing - 1 clock tick */
		case 7:
			c = 1;
		}
		if(c)
			putc(c|0200, &rtp->t_outq);
	}
}

/*
 * Restart typewriter output following a delay
 * timeout.
 * The name of the routine is passed to the timeout
 * subroutine and it is called during a clock interrupt.
 */
ttrstrt(tp)
register struct tty *tp;
{

	tp->t_state =& ~(TIMEOUT|SCSTOP);
	ttstart(tp);
}
ttxstrt(tp)
register struct tty *tp;
{
	if (tp->t_state & SCSTOP) return;
	ttrstrt(tp);
}

/*
 * Start output on the typewriter. It is used from the top half
 * after some characters have been put on the output queue,
 * from the interrupt routine to transmit the next
 * character, and after a timeout has finished.
 * If the SSTART bit is off for the tty the work is done here,
 * using the protocol of the single-line interfaces (KL, DL, DC);
 * otherwise the address word of the tty structure is
 * taken to be the name of the device-dependent startup routine.
 */
ttstart(atp)
struct tty *atp;
{
	register int *addr, c;
	register struct tty *tp;
	struct { int (*func)(); };

	tp = atp;
	addr = tp->t_addr;
	if (tp->t_state&SSTART) {
		(*addr.func)(tp);
		return;
	}
	if ((addr->tttcsr&DONE)==0 || tp->t_state&(TIMEOUT|SCSTOP))
		return;
	if ((c=getc(&tp->t_outq)) >= 0) {
		if (tp->t_flags&RAW) {
			addr->tttbuf = c;
			return;
		}
		if (c<=0177) {
			addr->tttbuf = c | (partab[c]&0200);
			/* scroll */
			if(tp->t_flags&SCROLL && c=='\n') {
				if(--(tp->t_scnt) == 0) {
					tp->t_state =| SCSTOP;
					return;
				}
			}
		}
		else {
			if(c =& 0177)
				timeout(ttxstrt, tp, c);
			tp->t_state =| TIMEOUT;
		}
	}
}

/*
 * Called from device's read routine after it has
 * calculated the tty-structure given as argument.
 * The pc is backed up for the duration of this call.
 * In case of a caught interrupt, an RTI will re-execute.
 */
ttread(atp)
struct tty *atp;
{
	register struct tty *tp;
	register sps, i;

	tp = atp;
	if ((tp->t_state&CARR_ON)==0)
		return;
	if (tp->t_canq.c_cc || canon(tp)) {
		mapin(&clseg);
		sps = spl5();
		i = BURST;
		while (tp->t_canq.c_cc && passc(xgetc(&tp->t_canq))>=0)
			if (--i == 0) {
				splx(sps);
				i = BURST;
				sps = spl5();
			}
		mapout();
		splx(sps);
	}
}

/*
 * Called from the device's write routine after it has
 * calculated the tty-structure given as argument.
 */
ttwrite(atp)
struct tty *atp;
{
	register struct tty *tp;
	register int c;

	tp = atp;
	if ((tp->t_state&CARR_ON)==0)
		return;
	while ((c=cpass())>=0) {
		spl5();
		while (tp->t_outq.c_cc > TTHIWAT) {
			ttstart(tp);
			tp->t_state =| ASLEEP;
			sleep(&tp->t_outq, TTOPRI);
		}
		spl0();
		ttyoutput(c, tp);
	}
	ttstart(tp);
}

/*
 * Common code for gtty and stty functions on typewriters.
 * If v is non-zero then gtty is being done and information is
 * passed back therein;
 * if it is zero stty is being done and the input information is in the
 * u_arg array.
 */
ttystty(atp, av)
int *atp, *av;
{
	register  *tp, *v;

	tp = atp;
	if(v = av) {
		*v++ = tp->t_speeds;
		v->lobyte = tp->t_erase;
		v->hibyte = tp->t_kill;
		v[1] = tp->t_flags;
		return(1);
	}
	wflushtty(tp);
	v = u.u_arg;
	tp->t_speeds = *v++;
	tp->t_flags = v[1];
	if (tp->t_flags&BLOCK) {
		tp->t_blksiz = v->lobyte;
		tp->t_blkchr = v->hibyte;
		tp->t_blk = 0;
	}  else  {
		tp->t_erase = v->lobyte;
		tp->t_kill = v->hibyte;
	}
	if(tp->t_flags&RAW)
		tp->t_flags =& ~SCROLL;
	if(tp->t_speeds == 0)		/* hangup - prp,1979 */
		signal(tp, SIGHUP);
	return(0);
}
