Program Scan;

{  SCAN.PAS - A utility to scan binary files for text strings.

Usage:  SCAN [-<offswitches>] [+<onswitches>] [/min] [input file] [output file]
where:    <offswitches> is the list of switches to turn off,
          <onswitches> is the list of switches to turn on, and
          min is the minimum length a string must be to be printed.
          input file and output file are the source and dest. files.
          To specify an output file you must give an input file.
          If either is not specified, SCAN will use stdin/stdout so
          you can also use command-line redirection.

Switches are as follows:
* = Default is ON.

*    $    Display escape as '\$'; escape becomes a printable char.
     H    Treat character values 128-255 as printable.
     ?    Treat high-bit foreign characters as printable (ie, no
          graphics characters or math symbols).
     7    Strip high bits from input.
     *    Strip high bits from output.
*    S    Strip leading spaces from strings.
*    ~    Strip trailing spaces from strings.
     U    Upper-case output.
     B    Display space as '_'.
     E    (For English) String must contain at least one consonant and
          one vowell (a,e,i,o,u,y) to be printed.
     \    Display \ as '\\'.  Useful with $, >, C, L, F.
     C    Display Carriage Return as '\C'.  CR becomes printable.
     L    Display Line Feed as '\L'.  LF becomes printable.
     F    Display Form Feed as '\F'.  FF becomes printable.
*    T    Don't end a string on a tab character (passes tabs to output).
     @    Display tabs as '\T' (implies T).
     0    NULL (ascii 0) becomes a printable character but still ends
          a string.  NULLs are printed as '\0'.
     !    String must end in NULL to be displayed (implies 0).

"Printable" means the character is eligible for inclusion in an output
string.

Note that switches are not case-significant and some may conflict, but no
reality check is performed.
Switches are processed left-to-right both within and between + and - groups.

Caveats:  The default minimum string length is 3 characters.  The maximum
possible string length is 254 and any which run over will be lost.

}
type
     filename = string[80];      {Input/output files if not stdin/stdout}
     switchset = set of char;    {binary switches (could trim this)}
     msgstring = string[37];

var
     ifile, ofile: filename;
     switches: switchset;
     MinLen: byte;
     Infile: file of char;
     Output: text;

Const
     DefSwitches: Switchset = ['T', 'S', '~'];   {default switches} 
     ForeignSet: set of char = [#128..#165, #168, #173, #224..#239];
     DefMin = 4;       { Default minimum line length }
     DefInput = '';    { Default input file (console)}
     DefOutput = '';   { Default output file (console)}

     Copyright1: MsgString = 'SCAN Version 1.0 21-Dec-87 by Kenneth';
     Copyright2: MsgString = ' Herron.  Placed in the public domain.';
{=============================}
procedure SetSwitches(var Ifile, Ofile: Filename; 
                      var S: Switchset; var MinLen: byte);

var
     T: filename;
     I: byte;
     J: integer;

begin
I := 1;
while I <= paramcount do
begin
     T := paramstr(I);
     case T[1] of
     '-': for J := 1 to length(T) do
          S := S - [upcase(T[J])];  { That's set difference }
     '+': for J := 1 to length(T) do
          S := S + [upcase(T[J])];
     '/': begin
               val(copy(T,2,(length(T) -1)), MinLen, J);
               if J <> 0 then MinLen := 3;
          end;
     else { File name }
          if ifile = '' then
               ifile := T
          else
          if ofile = '' then
               ofile := T
     end; { case }
     inc(I)
end;
{ Handle dependent switches here }
if '!' in S then
     S := S + ['0'];
if '@' in S then 
     S := S + ['T']
end;  {procedure SetSwitches}
{=============================}
procedure Process(Switches: Switchset; MinLen: byte);

type
     Statuses = (Valid, Invalid, EndOfInput);
     MaxString = string[255];

var
     status: statuses;
     Str: MaxString;
     ch: char;
     validchar: boolean;

Function Validate(var Str: maxstring): boolean;

var foundc, foundv: boolean;
    I: byte;

begin
     if length(str) < Minlen then
          Validate := false
     else
     if ('!' in switches) and (str[length(str)] <> #0) then
          Validate := false
     else
     if 'E' in switches then 
     begin
     { routine to check the string for >= one consonant
       & >= one vowel }
          foundc := false;
          foundv := false;
          I := 1;
          while (I <= length(str)) and not (foundv and foundc) do
          begin
               foundv := foundv or (upcase(str[I]) in 
                    ['A','E','I','O','U','Y']);
               foundc := foundc or (upcase(str[I]) in 
                    ['B'..'D','F'..'H','J'..'N','P'..'T','V'..'Z']);
               inc(I)
          end;
          Validate := foundc and foundv
     end
     else Validate := true
end;

procedure massage(var str: maxstring);

{perform any changes to the string which can't be done on output. Currently
 only leading- and trailing-string spaces are stripped }

var I: byte;

begin
     if 'S' in switches then  {leading spaces}
     begin
          I := 1;
          while str[I] = ' ' do
               inc(I);
          str := copy(str, I, length(str) + 1 - I);
     end;
     if '~' in switches then  {trailing spaces}
     begin
          I := length(str);
          while str[I] = ' ' do
               dec(I);
          str := copy(str, 1, I)
     end;
     if '*' in switches then {strip high bits}
          for I := 1 to length(str) do
               str[I] := char(byte(str[I]) and $7f)
end;

procedure print(var str: maxstring);

var I: byte;
    ch: char;

begin
for I := 1 to length(Str) do
begin
     ch := str[I];
     case ch of
          '!'..'[',
          ']'..'`',
          '{'..'~',
          #128..#255: write(output, ch);
          'a'..'z': if 'U' in switches then
                         write(output, upcase(ch))
                    else
                         write(output, ch);
          ' ':      if 'B' in switches then
                         write(output, '_')
                    else 
                         write(output, ' ');
          #27:      write(output, '\$');
          #9:       if '@' in switches then
                         write(output, '\T')
                    else
                         write(output, #9);
          '\':      if '\' in switches then
                         write(output, '\\')
                    else
                         write(output, '\');
          #13:      write(output, '\C');
          #10:      write(output, '\L');
          #12:      write(output, '\F');
          #0:       write(output, '\0');
          else write(output, ch)
     end;  { case }
end;
writeln(output)
end;


begin
repeat
     { set up to read one string }
     status := Invalid;
     Str := '';
     repeat
          if eof(infile) then 
               status := EndOfInput
          else
          begin
               read(infile, ch);
               if '7' in switches then
                    ch := char(byte(ch) and $7f);
               case ch of
                ' '..'~': ValidChar := true;
                #128..#255: Validchar := ('H' in switches) or (('?' in 
                            switches) and (ch in ForeignSet));
                #27:     ValidChar := '$' in switches;
                #09:     ValidChar := ('T' in switches) or ('@' in switches);
                #13:     ValidChar := 'C' in switches;
                #10:     ValidChar := 'L' in switches;
                #12:     ValidChar := 'F' in switches;
                #0:      begin
                              ValidChar := false;
                              if '0' in switches then
                                   str := str + ch
                         end
                else ValidChar := false
               end;  { case }
               case status of
               valid:    if validchar then
                              str := str + ch
                         else
                              status := invalid;
               invalid:  if validchar then
                         begin
                              str := ch;
                              status := valid
                         end;
               EndOfInput: {Do nothing}
               end  {case}
          end
     until status <> valid;
     if Validate(str) then
     begin
          massage(str);
          print(str);
     end;
until status = EndOfInput
end;
{=============================}
begin {main}
Ifile := DefInput;
Ofile := DefOutput;
Switches := DefSwitches;
Minlen := DefMin;
SetSwitches(Ifile, Ofile, Switches, MinLen);
assign(infile, ifile);
assign(output, ofile);
FileMode := 0;   {read-only}
reset(infile);
rewrite(output);
Process(Switches, MinLen);
close(infile);
end.

