/* amatch(ts, cs) Version 1.2
 *
 * usage: 
 *	amatch(ts, cs)
 *
 * description:
 *	match returns TRUE if a match has been made between a
 *	compare string (cs) and a test string (ts).
 *
 *	The test string may have UNIX style wild cards including
 *
 *		    * = zero or more arbitrary chars,
 *	            ? = one arbitrary char,
 *		    \ = process the following wild card
 *			char as a char to match,
 *		 [..] = match any char in the bracket and
 *		[!..] = don't match any char in the bracket.
 *
 *	At this time entry of non-printing char using the '\OOO'
 *	notation is NOT allowed. That gives me something to do later.
 *
 *	The .. inside the bracket stands for a list of char which may
 *	include a range of chars denoted by char-char. For example
 *	[ceBg-ks] will match the chars 'c', 'e', 'B', the range 'g'
 *	through 'k' inclusive or 's'. All range checking is done 
 *	assuming the ASCII value of the char for the compare. If you 
 *	give a lower limit > the upper limit the range check will 
 *	never get a match on that range. If you reach the end of the
 *	test string before hitting the ']' you can get a "match" only
 *	if the ! flag is set. A missing ']' after a match means that
 *	the "matched" char must be the last char in the string. Finally 
 *	[-g... ([!-g..) will match (not match) any char <= 'g' 
 *	while ...k-] will match (or not match if the ! flag is set) 
 *	any char >= 'k'.
 *
 *	The test string *?f[io]* will match "afool", "youareafoobar"
 *	and "hifi" but would not match "ife", "fi", "fo", "foo" or
 *	"fum": there must be at least one char before the 'f', it
 *	must be followed by either an 'i' or 'o' and any number of
 *	trailing chars. The test string "a\[[\-+]\][!123]" will 
 *	match "a[-]t" and "a[+]0" but not "a[-]2", "a[]f" or "a[+]".
 *
 *	To enter any of the wild card chars ('*', '?', '\', '[',
 *	'!', '-', ']') as chars to be matched precede the char by
 *	a '\'. For example to evaluate '[' as a char write it as 
 *	'\[' in the test string; '\\' allows evaluation of '\' as
 *	a match char. 
 *
 * peculiarities(bugs?):
 *      A quick review of match shows that use of '\' before all
 *	wild cards to act as matching chars isn't always necessary 
 *	(e.g. you only run into trouble with '-' inside a bracket
 *	and '*' outside a bracket), but it is safest to always use 
 *	this procedure.
 *
 *	For efficiency avoid expressions of the form *?*??* (i.e.
 *	embedded '?' between '*'). Match makes recursive calls
 *	when it comes across a '*' (multiple '*' in a row are
 *	compressed to one '*') and that slows down the evaluation.
 *	Sorry, but I went the easy route there. The expression
 *	above is better evaluated as ***??? or ???*** and even
 *	better as *??? or ???*. P.S. it would match any string 
 *	that had three or more chars.
 *
 *	Please send any information on errors, etc. to
 *			
 *			    Herb Schulz
 *			6008 Forest View Rd.
 *			      Apt. 3C
 *			 Lisle, Il. 60532 
 */
#define TRUE 1
#define FALSE 0

int amatch(ts, cs)
char *ts, *cs;
{
	int low, hi, notf;

	while (1) { /* keep going until done */

		if (*cs == '\0') { /* out of string */
			for (; *ts == '*'; ++ts)
				; /* get rid of extra '*' */
			return(*ts == '\0');
		}

		switch (*ts) {

		case '\0':
			return(FALSE);
		
		case '[': /* the hardest case (see '*' below) */
			  
			   /* is the not flag set? */
			if (notf = (*(ts + 1) == '!'))
				++ts; /* ! flag set */

			   /* loop through the bracket */
			while (*++ts != ']' && *ts != '\0') {

				if (*ts == '-') { /* a range of values */
				
					    /* get lower limit */
					if ((*--ts == '[' || *ts == '!')
     						           && *(ts - 1) != '\\')
						low = '\0';
					else
						low = *ts;

					    /* get upper limit */
				 	if (*(ts += 2) == ']' || *ts == '\0') {
						hi = '\377';
						--ts;
					}
					else {
						if (*ts == '\\' && 
							        *++ts == '\0') {
							--ts;
							hi = '\377';
						}
						else
							hi = *ts;
					}
				
					     /* and compare */				
					if (*cs > low && *cs <= hi)
						goto foo;

					continue; /* in bracket loop */
				}

				    /* process wildcard */
				if (*ts == '\\' && *++ts == '\0')
					break; /* bump and check for \0 */

				/* check if they are the same */
				if (*cs == *ts)
					goto foo;

			} /* while */

			/* get here if no match (out of ts or reached ]) */
			if (!notf)
				return(FALSE);
			if (*ts)
				++ts;
			break;

			/* come here if a match */
		foo:	if (notf)
				return(FALSE);
			++ts;
			while (*ts != '\0' && *ts++ != ']')
				; /* get to end of bracket */
			break;

		case '*': /* a chicken way out! my only recursive part */
			while (*++ts == '*')
				; /* get rid of extra '*' */

			if (!*ts) /* trailing '*' matches anything */
				return(TRUE);

			do
				if (amatch(ts, cs))
					return(TRUE);
			while (*++cs);
			return(*ts == '\0');

		case '?': /* just bump the pointers */
			++ts;
			break;

		case '\\': /* this drops through to next one */
			++ts;

		default: /* if they ain't the same here forget it */
			if (*cs != *ts++)
				return(FALSE);
		} /* switch */

		++cs;

	} /* while (1) */

} /* match */
