/********************************************************************
	SWEEP version 3.6 - July 2, 1982
			by Robert Fisher
			   DePaul University
			   243 S. Wabash
			   Chicago, Illinois 60604

Released to the Public Domain by the author.
************************************************************************/

/*Must be linked with SWEEPIO.REL for the direct i/o and the crc
  routines. SWEEPIO.DCL is the file of declarations for SWEEPIO.

	PLI SWEEP
	LINK SWEEP,SWEEPIO[A]				*/

/*Most of the large data structures do not exist in set locations.  They
are declared as BASED variables, and there location is determined by
setting the corresponding pointer when they are needed.  Several pointers 
are provided by the SWEEPIO (based on DR's PLIDIO) package that are useful 
in this situation. For example, memptr() points to the first free byte of 
memory following the program.  (This determines the location of the directory 
array. Note that the array is declared to be of size (1:1), but, since no 
checking is done, it can be treated as being arbitrarily long.) Similarly, 
dfcb0() determines the default file control block, and is used for searchfile.
(I use searchfile.drive as a general holding place for the logged-in drive.)
Other variables are set-up following the last directory entry as they are
needed by the various procedures.*/

direct:procedure options(main);

	%replace true by '1'b,
		 false by '0'b,
		 buffwds by 64,
		 buffchar by 128,
		 columns by 4,		/*number of columns for directory*/
		 k_per_ext by 16,	/*kilobytes per file extent*/
		 atsign  by 64,		/*ascii @*/
		 ascii_zero by 48,	/*ascii 0*/
		 quest   by 63;		/*ascii question mark*/

	%include 'sweepio.dcl';		/*declarations for swpio*/

	dcl dpb 	pointer;
	dcl 1 disk_parameter_block based(dpb),
		2 spt		fixed(15),	/*sectors per track*/
		2 bsh		fixed(7),	/*block shift factor*/
		2 blm		fixed(7),
		2 exm		fixed(7),	/*extent mask*/
		2 dsm		fixed(15),	/*blocks per disk*/
		2 drm		fixed(15),	/*number of directory entries*/
		2 all		bit(16),	/*directory allocation mask*/
		2 cks		fixed(15),	/*size of directory check vector*/
		2 off		fixed(15);	/*number of reserved tracks*/

	dcl max_user		fixed(15);	/*maximum user number allowed*/
	dcl number_of_drives	fixed(7);

	dcl sel_drv		char(1);	/*ascii version of selected
						     drive. Numeric version
						     is in searcfile.drive*/
	dcl selected_user	fixed(7);	/*current user area*/
	dcl sel_usr		char(2) var;	/*ascii version of selected
							user area*/
	dcl target_user		fixed(7);	/*transfer target user area*/
	dcl asc_target_user	char(2) var;	/*ascii version*/


	dcl continue		bit(1);		/*true for extended functions*/
	dcl choice		char(1);	/*input character*/
	dcl dummy		bit(1);		/*return value for logical fcns*/

	dcl 1 directory(1:1) based (memptr()),		/*directory array*/
		2 fname		char(8),
		2 ftype		char(3),
		2 user		fixed(7),
		2 fexth		fixed(7),
		2 fextl		fixed(7),
		2 recs		fixed(7),
		2 filesize	fixed(15),	/*size on disk*/
		2 actualsize	fixed(15),	/*actual transfer size*/
		2 tag		char(1);

	dcl tag_total		fixed;		/*total size of tagged files*/
	dcl act_total		fixed;		/*same assuming 1k blocks*/
		
	dcl (last,current)	fixed;		/*directory pointers*/

	dcl 1 searchfile based(dfcb0()),	/*dummy file control block*/
		    2 drive fixed(7),  /* drive number */
		    2 fname char(8),   /* file name */
		    2 ftype char(3),   /* file type */
		    2 fext  fixed(7),  /* file extent */
		    2 space bit(8),
		    2 fexth fixed(7),
		    2 space1 bit(8);/* filler */
	
	dcl 1 second_param based(dfcb1()),
		2 drive2 fixed(7),
		2 start_user char(2);

	dcl start_string	char(11);	
	
	dcl verfy bit(1);

/**********************************************************************

			MAIN PROCEDURE

***********************************************************************/

/*The main line, after some initialization, is a standard menu select
construction.*/

	/*patch this to exact number of drives in system*/
	number_of_drives = 4;

	/*patch this to maximum user number you wish to use
          Note that the CCP will only allow users 0-15 but 
          CPM seems to allow 0-31.*/
	max_user = 15;

	if mod(vers(),256) = 0 then do;	/*CP/M1.x or 2.x?*/
		put list('Sorry, CP/M2.2 is required.');
		call reboot();
		end;


	/*if default drive requested, find drive and put it in searchfile*/
	if searchfile.drive = 0 then 
		searchfile.drive = curdsk()+1;
	sel_drv = ascii(searchfile.drive + atsign);

	if start_user = '??' then do;
		selected_user = -1;
		sel_usr = '*';
		end;
	else do;
		sel_usr = substr(start_user,1,
				verify(start_user,'1234567890')-1);
		if sel_usr ^= '' then do;
			selected_user = fixed(sel_usr);
			if selected_user > max_user then
				start_user = '  ';
			end;
		else 
			start_user = '  ';
		if start_user = '  ' then do;
			selected_user = getusr();
			sel_usr = numtstr(selected_user);
			end;
		end;

	start_string = searchfile.fname||searchfile.ftype;

	/*Let's tell 'em who we are*/
	call reset;
	call signon;	
	call menu;
	call setup;
	call inicrc;	

	/*MAIN PROCESSING LOOP*/
	continue = true;
	do while (continue);

		if last = 0 then
			put edit('No files. ')(col(1),a);
		else do;
			put edit(current,'. ')(col(1),f(4),a);
			call putname(current);
			put edit(directory(current).filesize,'k :',
				directory(current).tag)(f(4),2a);
			end;
		choice =upcase(rdcon());
		if (last = 0)&(choice ^= 'L')&(choice ^='S')
					&(choice ^= 'X') then
			put skip list('^g^m');
		else if (choice = ' '|choice = '^M') then 
			call advance(current);
		else if choice = 'A' then
			call retag;
		else if choice = 'B' then 
			call backup(current);
		else if choice = 'C' then do;
			put edit('  To')(a);		
			dummy = copy(current,getdrv(1,''));
			put skip(2);
			end;
		else if choice = 'D' then
			call dodel;
		else if choice = 'E' then
			call mdel;
		else if choice = 'L' then 
			call newdrv;
		else if choice = 'M' then 
			call mass;
		else if choice = 'R' then 
			call renmit;
		else if choice = 'S' then
			call space;
		else if choice = 'T' then do;
			call tagit(current);
			call advance(current);
			end;
		else if choice = 'U' then 
			call untag(current);
		else if choice = 'V' then 
			call view;
		else if choice = '?' then 
			call menu;
		else if choice = 'X' then 
			continue = false;
		end;

	/*do a warmboot to avoid "end of execution" message*/
	call reboot();

/*********************************************************************
       			Procedures and Functions
**********************************************************************/

/*setup new drive, read directory, and get it sorted.*/
setup:procedure;
	dcl a fixed(7);
	dcl 1 buffer(0:3) based (dbuff()),      /*directory buffer for search*/
		2 user 		fixed(7),
		2 fname		char(8),
		2 ftype 	char(3),
		2 fextl		fixed(7),
		2 space		bit(8),
		2 fexth		fixed(7),
		2 recs		fixed(7),
		2 space0(16)	bit(8);
	dcl (i,ubnd,lbnd)	fixed(7);	
	dcl username	char(2) var;

	/*Initialize searchfile to match all files and all extents.
	  Drive should already be set.*/

	if index(start_string,'?') = 0 then do;
		searchfile.fname='????????';
		searchfile.ftype='???';
		end;
	else 
		start_string = '';

	searchfile.fext=quest;
	searchfile.fexth=quest;
	call select(searchfile.drive-1);

	/*determine range of user areas to search*/
	if selected_user < 0 then do;
		lbnd = 0;
		ubnd = max_user;
		end;
	else do;
		lbnd = selected_user;
		ubnd = selected_user;
		end;
	call setdma(dbuff());

	last=0;
	do i = lbnd to ubnd;
		call setusr(i);
		a = sear(dfcb0());	/*get first directory entry*/
		/*move each file name to directory, stripping off any parity
	  	  bits that have been tagged by a call to "peel"*/
		do while (a ^= -1);	
			last=last+1;
			directory(last).fname = peel(buffer(a).fname);
			directory(last).ftype = peel(buffer(a).ftype);
			directory(last).fextl  = buffer(a).fextl;
			directory(last).fexth  = buffer(a).fexth;
			directory(last).user = i;
			directory(last).recs  = buffer(a).recs;
			directory(last).tag   = ' ';
			a = searn();
			end;
		end;

	call sort;		/*sort the directory*/
	call clean;		/*eliminate duplicates*/
	call compute;		/*compute filesizes*/

	tag_total = 0;
	act_total = 0;
	current = setcur(start_string); /*set the cursor */

	end;

/*Eliminate duplicates in the directory file.  Keep last entry since it
has the size information*/	
clean:procedure;
	dcl next fixed;
	dcl i fixed;
	
	if last = 0 then
		return;
	next=1;
	do i = 2 to last;
		if (directory.fname(next) ^= directory.fname(i)) |
		    (directory.ftype(next) ^= directory.ftype(i))|
		    (directory(next).user ^= directory(i).user) then 
			next=next+1;
		directory(next)=directory(i);
		end;
	last=next;
	end;									


/*Compute file size and update appropriate fields in the directory*/
compute:procedure;
	dcl (i,extents,records,extra,bsize) fixed;
	dcl free	fixed;		/*free space on disk*/
	dcl used	fixed;		/*space used by files on disk*/
	

	free = size(searchfile.drive-1);
	bsize = 2**(bsh-3);
	free = bsize*free;

	used = 0;
	do i = 1 to last;
		records = directory(i).recs;
		extents = directory(i).fexth*32+directory(i).fextl;

		/*if records = 80H then call it another extent*/
		if records < 0 then do;
			records = 0;
			extents = extents + 1;
			end;

		/*extents = number of kilobytes in the full extents*/
		extents = extents*k_per_ext;

		/*compute size in 1k blocks*/
		extra = divide(records+7,8,15);
		directory(i).actualsize = extents + extra;

		/*compute size in actual block size of disk*/
		extra = bsize*divide(extra + bsize -1,bsize,15);
		directory(i).filesize = extents + extra;

		used = used + directory(i).filesize;
		end;

	put edit(last,' Files occupying ',used,'k')(f(3),a,f(4),a);
	put edit('(',free,'k remaining)')(x(1),a,f(4),a);
	put skip(2);
	end;

/* function to compute file space remaining on drive*/
space:procedure;
	dcl freespace fixed;
	dcl save_flag bit(1);
	dcl ch char(1);
	dcl a fixed(7);
	
	/*get drive - disable user area check for this one*/
	put edit('  Which')(a);
	a = getdrv(3,ch);

	if a = -1 then 
		return;

	freespace = size(a-1);
	freespace = freespace*(2**(bsh-3));
	put skip edit('******',freespace,'k remaining on Drive ',ch,':')
		(a,f(5),a,a,a);
	put skip(2);
	end;


/*actual computation of free space on drive*/
size:procedure(drive) returns(fixed);
	dcl drive	fixed;

	dcl alp		pointer;
	dcl 1 allocbyte based(alp),
		2 realbyte 	bit(8),
		2 nextbyte	bit(8);	/*for cycling through allocation vec*/

	dcl number_of_bytes fixed;
	dcl free_blocks	fixed;
	dcl (i,j)	fixed;
	
	call select(drive);
	dpb = getdpb();	/*establish disk_parameter_block*/
	alp = allvec(); /*establish allocation vector*/
	number_of_bytes =divide(dsm,8,15)+1;

	free_blocks = dsm+1;
	do i = 1 to number_of_bytes;
		do j = 1 to 8;
			if substr(realbyte,j,1) = '1'b then
				free_blocks = free_blocks-1;
			end;
		alp = addr(nextbyte);
		end;
	return(free_blocks);
	end;

/*Sort directory by (non-recursive) Quicksort. We make our own stack
  to simulate recursion. 
  Quicksort works by splitting the list into 2 pieces, where 1 piece
  contains items <= teststring and the other contains items 
  >= teststring.  The teststring is always chosen as the middle item. 
  (This makes the algorithm easier to write, and also eliminates bad
  effects if the directory is already sorted for some reason.)  The 
  two pieces are then (recursively) sorted by the same method, one piece
  is split immediately, while the other is saved till later on the stack.  
  When the pieces reach length 1, they are sorted, and we go back to the
  stacked pieces that we saved.  We always save the longer segment on the
  stack, and split the shorter immediately.  This keeps the size of the stack
  to a guaranteed <= log2(last)*/

sort:procedure;
	dcl ovlyptr pointer;
	dcl 1 dirovly(1:1) based(ovlyptr),	/*overlay for directory*/
		2 sortfield	char(14),	/*used to make sort easier*/
		2 recs		fixed(7),	/*to code*/
		2 filesize	fixed(15),
		2 actualsize	fixed(15),
		2 tag		char(1);

	dcl saveptr pointer;
	dcl 1 save based(saveptr),
		2 sortfield	char(14),
		2 recs		fixed(7),
		2 filesize	fixed(15),
		2 actualsize	fixed(15),
		2 tag		char(1);

	dcl tempptr pointer;
	dcl 1 temp based(tempptr),
		2 sortfield	char(14),
		2 recs		fixed(7),
		2 filesize	fixed(15),
		2 actualsize	fixed(15),
		2 tag		char(1);

	dcl stackptr pointer;
	dcl 1 stack(1:1) based(stackptr),	/*We only need 10 or 20 but*/
		2 left	fixed,	       		/*we'll move it out in memory*/
		2 right	fixed;			/*to save room in the .com file*/
	dcl top	fixed;

	dcl (mid,i,j) fixed;
	dcl (l,r) fixed;
	
	if last = 0 then
		return;

	/*set pointers to set up various variables following the
	  directory*/
	saveptr = addr(directory(last+1));
	tempptr = addr(directory(last+2));
	stackptr = addr(directory(last+3));

	/*dirovly 'is' the directory, in a different suit*/
	ovlyptr = addr(directory);

	/*PUSH 1,last to prepare for entry into the sort.*/
	top = 1; stack.left(top)=1; stack.right(top) = last;
	do while(top > 0);
		/*POP l,r*/
		l = stack.left(top); 
		r = stack.right(top); 
		top = top - 1;

		do while (r > l);
			i=l;  	j=r;	
			mid=divide(i+j,2,15);	
			save = dirovly(mid);
			
			/*Split the list into those <= save and those >= save*/
			do while(i<=j);
	
				do while (dirovly(i).sortfield < save.sortfield);
					i=i+1;
					end;
	
				do while (dirovly(j).sortfield > save.sortfield);
					j=j-1;
					end;
	
				if i <= j then do;
					temp = dirovly(i); 
					dirovly(i)=dirovly(j);
					dirovly(j)=temp;
					i=i+1;j=j-1;	
					end;
				end;
	
			/*PUSH either l,j or i,r depending on which is 
			  the larger range. (Saves stack space).
			  We then process the other half immediately.*/
			top = top + 1;
			if r-i < j-l then do;
				stack.left(top) = l;
				stack.right(top) = j;
				l = i;
				end;
			else do;
				stack.left(top) = i;
				stack.right(top) = r;
				r = j;
				end;
			end;
		end;
	end;

/*use a (modified) binary search to find first file >= string*/
setcur: procedure(string) returns(fixed);
	dcl	string	char(11);
	dcl	(hi,lo,mid) fixed;

	if string = '           ' then
		return(1);
	hi = last; lo = 1;
	do while(lo <= hi);
		mid = divide(lo + hi,2,15);
		if directory(mid).fname||directory(mid).ftype < string then
			lo = mid + 1;
		else
			hi = mid - 1;
		end;

	if lo > last then
		lo = 1;

	return(lo);
	end;

/*Get new drive and user area and set it up*/
newdrv: procedure;
	dcl a fixed(7);
	dcl ch char(1);

	put edit('  New')(a); 
	a=getdrv(2,ch);
	if a = -1 then 
		return;
	searchfile.drive = a;

	sel_drv = ch;
	selected_user = target_user;
	sel_usr = asc_target_user;
	call menu;
	call setup;
	end;

/*convert user or drive number to ascii version*/
numtstr: procedure(number) returns (char(2) var);
	dcl number fixed(7);
	dcl string char(2) var;
	dcl int_string char(7);
	dcl l fixed(7);

	int_string = character(number);
	if number < 10 then
		string = substr(int_string,7);
	else 
		string = substr(int_string,6);
	return(string);
	end;		


/*remove any set parity bits from input string*/
peel: procedure(s) returns(char(8) var);

	dcl s char(8) var;
	dcl p pointer;
	dcl log_s(0:8) bit(8) based(p);
	dcl i fixed(7);

	p = addr(s);
	do i = 1 to length(s);
		log_s(i) = log_s(i)&'01111111'b;
		end;
	return(s);
	end;

/*strip leading blanks from input string and convert to upper case*/
strip: procedure(s) returns (char(11) var);
	dcl	s 	char(20) var;
	dcl	p	fixed;
	dcl 	(i,l)	fixed;

	p = verify(s,' ');
	if p ^= 0 then 
		return(ucstr(substr(s,p)));
	else 
		return('');
	end;

/*convert string to upper case*/
ucstr: procedure(s) returns (char(12) var);
	dcl s char(12) var;
	dcl r char(12) var;

	r = '';
	dcl i fixed;
	do i = 1 to length(s);
		r = r||upcase(substr(s,i,1));
		end;
	return(r);
	end;

/*convert character to upper-case*/
upcase:procedure(ch) returns(char(1));
	dcl ch char(1);
	dcl ascii_value fixed;
	
	ascii_value = rank(ch);
	if (96 < ascii_value)& (ascii_value < 123) then 
		ch = ascii(ascii_value - 32);
	return(ch);
	end;

/*Header for start of program*/
signon:procedure;
	put edit('SWEEP version 3.6 - July 2, 1982')(col(23),a);
	put skip edit('by Robert Fisher')(col(36),a);
	put edit('DePaul University')(col(39),a);
	put edit('Chicago, Illinois')(col(39),a);
	end;

/*display function menu on screen*/
menu:procedure;

	put skip(2) edit('Commands:')(a);
	put      edit('A:  Again. Retag # files.')(col(15),a);
	put skip edit('B:  Backup to last file.')(col(15),a);
	put skip edit('C:  Copy file with optional verify.')(
				col(15),a);
	put skip edit('D:  Delete file.')(col(15),a);
	put skip edit('E:  Erase all tagged/untagged files.')(col(15),a);
	put skip edit('L:  Login new disk and user and reset system.')(col(15),a);
	put skip edit('M:  Mass copy of tagged files with optional verify.')
				(col(15),a);
	put skip edit('R:  Rename file(s).')(col(15),a);
	put skip edit('S:  Space remaining on disk.')(col(15),a);
	put skip edit('T:  Tag file for transfer.')(col(15),a);
	put skip edit('U:  Untag a file.')(col(15),a);
	put skip edit('V:  View a file at console.')(col(15),a);
	put skip edit('X:  Exit to CP/M.')(col(15),a);
	put skip edit('?:  Redisplay menu.')(col(15),a);
	put skip edit('sp or cr:  Next file.')(col(8),a);
	put skip(2) edit ('Drive ',sel_drv,sel_usr,':')(a);
	end;

/*determines what kind of rename function we want*/
renmit: procedure;
	dcl input char(20) varying;
	dcl p fixed;
	
	call select(searchfile.drive-1);
	put edit('  New name or * : ')(a);
	get edit(input)(a);
	/*if a drive was specified, ignore it*/
	p = index(input,':');
	input = strip(substr(input,p+1));

	if input = '' then 
		return;
	else if (index(input,'*') ^= 0) | (index(input,'?') ^= 0) then
		call brenmit;
	else do;
		put skip;
		call crenmit(current,expand(input),false);
		put skip(2);
		end;
	end;

/*rename current*/
crenmit:procedure(current,newname,ask);
	dcl newname char(12);
	dcl current fixed;
	dcl ask bit(1);
	dcl renptr pointer;
	dcl 1 renfile based(renptr),
		  2 name1,
		    3 drive fixed(7),  /* drive number */
		    3 fname char(8),   /* file name */
		    3 ftype char(3),   /* file type */
		    3 fext  fixed(7),  /* file extent */
		    3 space (3) char(1),/* filler */
		  2 name2,             /* used in rename */
		    3 drive2 fixed(7),
		    3 fname2 char(8),
		    3 ftype2 char(3),
		    3 fext2  fixed(7),
		    3 space2 (3) char(1),
		  2 crec  fixed(7),    /* current record */
		  2 rrec  fixed(15),   /* random record */
		  2 rovf  fixed(7);    /* random rec overflow */
	dcl response char(1);
	dcl p fixed;
	dcl i	   fixed;

	/*setup fcb in 'free space' after directory*/
	renptr=addr(directory(last+1));

	renfile.name1.drive = searchfile.drive;
	renfile.name1.fname = directory(current).fname;
	renfile.name1.ftype = directory(current).ftype;
	renfile.name1.fext  = quest;
	renfile.name1.space(2) = '?';
	renfile.name2.fname2 = substr(newname,1,8);
	renfile.name2.ftype2 = substr(newname,10);
	renfile.name2.drive2 = 0;
	renfile.name2.fext2 = quest;
	renfile.name2.space2(2) = '?';

	call setusr(directory(current).user);
	/*if file exists already, go back*/
	if sear(addr(renfile.name2)) ^= -1 then do;
		put skip edit('Cannot rename. ',sel_drv,
			numtstr(directory(current).user)||':',newname,
			' already exists.')(col(1),2a,a(3),2a);
		return;
		end;

	if readonly(renptr) then do;
		put edit('File ')(col(1),a);
		call putname(current);
		put edit(' is R/O. Okay to rename?(Y/N)')(a);
		response = upcase(rdcon());
		if response = 'Y' then
			call filatt(renptr);
		else
			return;
		end;

	/*screen message*/
	if ask then
		put edit('O.K. to rename ')(col(1),a);
	else 
		put edit('Renaming ')(col(1),a);
	call putname(current);
	put edit(' ====> ')(a);
	put edit(sel_drv,numtstr(directory(current).user)||':',
		renfile.name2.fname2,'.',renfile.name2.ftype2)
		(a,a(3),3a);

	response = 'Y';
	if ask then do;
		put edit(' ? ')(a);
		response = upcase(rdcon());
		end;
	if response = 'Y' then do;
		/*rename it*/
		call rename(addr(renfile));

		/*update directory information*/
		directory(current).fname=renfile.name2.fname2;
		directory(current).ftype=renfile.name2.ftype2;
		end;
	return;
	end;

/*batch rename*/
brenmit: procedure;
	dcl 	input char(20) var;
	dcl	oldname	char(12);
	dcl	newname char(12);
	dcl	i fixed;

	put skip edit('Old name: ')(a);
	get edit(input)(a);
	input = strip(substr(input,1+index(input,':')));
	if input = '' then
		return;
	oldname = expand(input);

	put edit('New name: ')(a);
	get edit(input)(a);
	put skip;
	input = strip(substr(input,1+index(input,':')));
	if input = '' then
		return;
	newname = expand(input);

	do i = 1 to last;
		if match(oldname,i) then
			call crenmit(i,bldname(newname,i),true);
		end;
	put skip(2);
	end;

/*build name from wild card string and ith directory entry*/
bldname: procedure(wild_string,i) returns (char(12));
	dcl wild_string char(12);
	dcl i fixed;
	dcl (j,k) fixed;
	dcl newfname char(12);
	dcl ch char(1);

	newfname = directory(i).fname||'.'||directory(i).ftype;
	do j = 1 to 12;
		ch = substr(wild_string,j,1);
		if ch ^= '?' then
			substr(newfname,j,1) = ch;
		end;
	/* remove extraneous internal blanks here*/
	k = 1;
	do j = 1 to 12;
		ch = substr(newfname,j,1);
		if ch ^= ' ' then do;
			substr(newfname,k,1) = ch;
			k = k+1;
			end;
		end;
	do j = k to 12;
		substr(newfname,j,1) = ' ';
		end;
	return(expand(newfname));
	end;

/*determine whether wild_string matches directory(entry)*/
match: procedure(wild_string,entry) returns(bit(1));
	dcl wild_string char(12);
	dcl entry fixed;
	dcl i fixed;
	dcl string char(12);
	dcl match bit(1);

	string = directory(entry).fname||'.'||directory(entry).ftype;
	do i = 1 to 12;
		if (substr(wild_string,i,1) ^= '?')&
			(substr(wild_string,i,1)^=substr(string,i,1)) then
			return(false);
		end;
	return(true);
	end;

/*Convert name to a good file name, expanding wild card indicators.*/
expand: procedure(name) returns(char(12));
	dcl name char(20) var;
	dcl (p,k) fixed;
	dcl fname char(8);
	dcl ftype char(3);
	dcl got_astrsk	bit(1);

	p = index(name,'.');
	if p ^= 0 then do;
		fname = substr(name,1,p-1);
		ftype = substr(name,p+1);
		end;
	else do;
		fname = name;
		ftype = '   ';
		end;

	got_astrsk = false;
	do k = 1 to 8 while(^got_astrsk);
		if substr(fname,k,1) = '*' then do;
			got_astrsk = true;
			substr(fname,k) = '????????';
			end;
		end;
	got_astrsk = false;
	do k = 1 to 3 while (^got_astrsk);
		if substr(ftype,k,1) = '*' then do;
			got_astrsk = true;
			substr(ftype,k) = '???';
			end;
		end;

	return(fname||'.'||ftype);

	end;	

/*is file named in fcb pointed to by p a read only file?*/
readonly: procedure(p) returns(bit(1));
	dcl p pointer;
	dcl 1 buffer(0:3) based (dbuff()),      /*directory buffer for search*/
		2 user 		fixed(7),
		2 fname		char(8),
		2 ftype 	char(3),
		2 fextl		fixed(7),
		2 space		bit(8),
		2 fexth		fixed(7),
		2 recs		fixed(7),
		2 space0(16)	bit(8);
	dcl a fixed(7);

	call setdma(dbuff());
	a = sear(p);
	if a = -1 then
		return(false);
	if buffer(a).ftype > ascii(127) then
		return(true);
	else return(false);		
	end;

/*delete file*/
delit: procedure(del_file,drv)returns(bit(1));;
	dcl del_file fixed;
	dcl drv fixed(7);
	dcl delptr pointer;
	dcl 1 deletefile based(delptr),
		2 drive 	fixed(7),
		2 fname		char(8),
		2 ftype		char(3),
		2 fext		fixed(7),
		2 space(3)	char(1);
	dcl response char(1);
	delptr = addr(directory(last+1));

	deletefile.drive = drv;
	deletefile.fname = directory(del_file).fname;
	deletefile.ftype = directory(del_file).ftype;	
	deletefile.fext = quest;
	deletefile.space(2) = '?';

	call select(drv-1);
	call setusr(directory(del_file).user);

	if readonly(delptr) then do;
		put skip edit('File ')(a);
		call putname(del_file);
		put edit(' is R/O. Delete anyway?(Y/N) ')(a);
		response = upcase(rdcon());
		if response = 'Y' then
			call filatt(delptr);
		else 
			return(false);
		end;
	call delete(delptr);
	put edit(' Deleted.')(a);
	return(true);
	end;

/*perform delete file menu option*/
dodel:procedure;
	dcl i fixed;

	/*give 'em a chance to back out*/
	put edit('  Delete? (Y/N):')(a);
	if upcase(rdcon()) ^= 'Y' then
		return;

	/*o.k. we go*/
	/*now delete it*/
	if ^delit(current,searchfile.drive) then
		return;

	/*was it tagged?*/
	if directory(current).tag = '*' then do;
		tag_total = tag_total - directory(current).filesize;
		act_total = act_total - directory(current).actualsize;
		end;


	/*update directory*/
	last = last - 1;
	do i = current to last;
		directory(i) = directory(i+1);
		end;


	/*check position in list*/
	if current > last then do;
		put skip(2);
		current = 1;
		end;

	end;

/*erase all untagged files*/
mdel:procedure;
	dcl 	ch	char(1);
	dcl	prompt	bit(1);
	dcl	abort	bit(1);
	dcl	i	fixed;
	dcl	tagged  bit(1);

	put skip(2) edit('^gErase tagged or untagged files? (T/U) : ')(a);
	tagged = (upcase(rdcon()) = 'T');

	put skip(2) edit('^gDo you wish to be prompted? (Y/N) : ')(a);
	prompt = (upcase(rdcon()) ^= 'N');

	if prompt then
		put skip edit('Type A at prompt to abort.')(a);
	put skip;

	abort = false;
	do i = 1 to last while (^abort);
		if directory(i).tag ^= '*' then do;
			if ^tagged then
				call doerase;
			end;
		else if tagged then
			call doerase;
		end;

	/*easiest thing is to reinitialize the drive*/
	call menu;
	call setup;

	doerase: proc;
		if prompt then do;
			put skip edit('Erase   ')(a);
			call putname(i);
			put edit(' ?(Y/N/A)')(a);
			ch = upcase(rdcon());
			if ch ='Y' then
				dummy = delit(i,searchfile.drive);
			else if ch = 'A' then
				abort = true;
			end;
		else do;
			put skip edit('Erasing ')(a);
			call putname(i);
			dummy = delit(i,searchfile.drive);
			if break() then do;
				ch = rdcon();
				abort = true;
				end;
			end;
		end;
	
	end;

/*This procedure copies file i to drive drve. It returns
  a value of true if everything goes o.k., otherwise it returns false.
 If requested, it performs crc verification.*/
copy:procedure(i,drve) returns(bit(1));
	dcl i	fixed;
	dcl ch char(1);
	dcl drve fixed(7);
	dcl return_flg bit(1);
	dcl eof bit(1);
	dcl filename char(16) varying;
	dcl p fixed;
	dcl rec fixed;
	dcl m	fixed;
	dcl nbuffs fixed;
	dcl wrt_drv char(1);
	dcl w_user fixed(7);
	dcl buffsize fixed;
	dcl o_crc bit(16);
	dcl c_crc bit(16);

	dcl memryptr pointer;
	dcl memory(0:0) bit(16) based(memryptr);

	dcl destptr pointer;
	dcl 1 destfile based(destptr),
		  2 name1,
		    3 drive fixed(7),  /* drive number */
		    3 fname char(8),   /* file name */
		    3 ftype char(3),   /* file type */
		    3 fext  fixed(7),  /* file extent */
		    3 space (3) bit(8),/* filler */
		  2 name2,             /* used in rename, just filler here */
		    3 drive2 fixed(7),
		    3 fname2 char(8),
		    3 ftype2 char(3),
		    3 fext2  fixed(7),
		    3 space2 (3) bit(8),
		  2 crec  fixed(7),    /* current record */
		  2 rrec  fixed(15),   /* random record */
		  2 rovf  fixed(7);    /* random rec overflow */


	dcl sourceptr pointer;
	dcl 1 sourcefile based(sourceptr),
		  2 name1,
		    3 drive fixed(7),  /* drive number */
		    3 fname char(8),   /* file name */
		    3 ftype char(3),   /* file type */
		    3 fext  fixed(7),  /* file extent */
		    3 space (3) bit(8),/* filler */
		  2 name2,             /* used in rename, just filler here*/
		    3 drive2 fixed(7),
		    3 fname2 char(8),
		    3 ftype2 char(3),
		    3 fext2  fixed(7),
		    3 space2 (3) bit(8),
		  2 crec  fixed(7),    /* current record */
		  2 rrec  fixed(15),   /* random record */
		  2 rovf  fixed(7);    /* random rec overflow */


	/*Check to make sure a good drive was specified.*/
	if drve = -1 then
		return(false);
	wrt_drv = ascii(drve + 64);

	/*Establish location of file control blocks following directory*/
	destptr	= addr(directory(last+1));
	sourceptr = addr(destfile.rovf);	/*we don't use random access*/
						/*so we can overwrite this*/
	destfile.drive=drve;
	destfile.fname=directory(i).fname;
	destfile.ftype=directory(i).ftype;
	destfile.fext=0;

	sourcefile=destfile;

	sourcefile.drive=searchfile.drive;


	/*Set up alternate name if same drive.*/
	if wrt_drv = sel_drv then 
		substr(destfile.ftype,3,1) = '$';
		/*delete original file name from directory if other user area*/

	if target_user < 0 then
		w_user = directory(i).user;
	else 
		w_user = target_user;
	/*delete any previous file by same name on destination drive*/
	call setusr(w_user);
	if readonly(addr(destfile)) then do;
		put skip edit(wrt_drv,numtstr(w_user)||':',destfile.fname,
			'.',destfile.ftype, 'is R/O. File not copied.')
			(a,a(3),4a);
		return(true);
		end;
	call delete(addr(destfile));

	if (wrt_drv = sel_drv) & (w_user ^= directory(i).user) then do;
			/*In this case we will rename it later, so delete
			  that name too.*/
		if readonly(addr(sourcefile)) then do;
			put skip edit(wrt_drv,numtstr(w_user)||':',
				'.',sourcefile.fname,sourcefile.ftype,
				' is R/O. File not copied.')
				(a,a(3),4a);
			return(true);
			end;
		call delete(addr(sourcefile));
		end;

	destfile.crec = 0;
	sourcefile.crec = 0;
	
	if make(addr(destfile)) = -1 then do;
		put edit('  Out of directory space')(a);
		return(false);
		end;
	call setusr(directory(i).user);
	dummy = open(addr(sourcefile));
			
	/*Set nbuffs according to available memory less space used by
	the directory file and the two file control blocks.*/
	nbuffs = divide(memwds()-10*last-36,buffwds,15);

	if verfy then do;
		o_crc = '0000000000000000'b;
		c_crc = o_crc;
		end;

	/*Give them the word.*/
	put skip edit('Copying ')(a);
	call putname(i);
	put edit(' =====> ',wrt_drv,numtstr(w_user),':')(a);

	if verfy then 
		put edit(' with verification')(a);

	/*set memory array address*/
	memryptr = addr(sourcefile.rovf);

	/*Initialize flags and go.*/
	eof = false; return_flg = true;
	do while(^eof);

		/*Read as much as possible*/
		m=0;
		/*selecting disk improves performance on some systems*/
		call select(sourcefile.drive-1);
		call setusr(directory(i).user);
		do rec = 0  to nbuffs - 1 while (^eof);
			call setdma(addr(memory(m)));
			m = m + buffwds;
			if rdseq(addr(sourcefile)) ^=0 then do;
				eof = true;
				nbuffs = rec;
				end;
			end;

		/*Write all that we have read, if there is room.*/
		m=0;
		call select(destfile.drive-1);
		call setusr(w_user);
		do rec = 0 to nbuffs - 1 while (return_flg);
			call setdma(addr(memory(m)));
			if wrseq(addr(destfile)) ^= 0 then do;
				return_flg = false;
				eof = true;
				end;
			if verfy then
				o_crc = updsec(o_crc,addr(memory(m)));
			m = m + buffwds;
			end;
		end;
	
	dummy = close(addr(destfile));

	if verfy & return_flg then do;
		memryptr = addr(sourcefile.crec);
		/*This is a shift of 3 bytes from previous calculation
		  to catch some possible memory errors.*/
		destfile.crec = 0;
		destfile.fext = 0;
		dummy = open(addr(destfile));
		eof = false;
		do while(^eof);
			m = 0;
			do rec = 0 to nbuffs - 1 while(^eof);
				call setdma(addr(memory(m)));
				if rdseq(addr(destfile)) ^= 0 then
					eof = true;
				else
					c_crc = updsec(c_crc,addr(memory(m)));
				m = m + buffwds;
				end;
			end;
		if o_crc ^= c_crc then do;
			return_flg = false;
			put skip edit('^g CRC check failed.')(a);
			end;
		end;
	
	if ^return_flg then do;
		put edit('  Error. Disk full.')(a);
		destfile.fext=quest;
		call delete(addr(destfile));
		end;
	else if (wrt_drv = sel_drv) then do;
		if w_user = directory(i).user | selected_user<0 then
			put skip edit('  Warning: source = target. Re-log ',
					'drive for an accurate directory.')
					(a);
		if w_user ^= directory(i).user then do;
			destfile.name2 = sourcefile.name1;
			if sear(addr(destfile.name2)) ^= -1 then
				call delete(addr(destfile.name2));
			call rename(destptr);
			end;
		end;

	/*go back and close the sourcefile*/
	dummy = close(addr(sourcefile));

	return(return_flg);
	end;

/*This procedure tries to get a new drive number. It returns the drive
number in the range 1:number_of_drives. It returns -1if it can't get
get legal drive.  It also checks for any start_string specified.*/
getdrv: procedure(message,ch) returns(fixed(7));
	dcl message fixed;
	dcl ch char(1);
	dcl asc fixed;
	dcl i fixed(7);
	dcl p fixed(7);
	dcl digit char(1);
	dcl input char(20) var;
	dcl prompt(3) char(29) varying static 
		init(' and user (and v to verify): ',' and user: ',': ');
		

	verfy = false;
	put edit(' drive')(a);
	put edit(prompt(message))(a);
	if message < 3 then do;
		get edit(input)(a);
		if input = '' then
			return(-1);

		p = index(input,':');
		if p ^= 0 then 
			substr(input,p,1) = ' ';
		p = index(input,' ');
		ch = upcase(substr(input,1,1));
		if p = 0 then do;
			asc_target_user = substr(input,2);
			start_string = '';
			end;
		else do;
			asc_target_user = substr(input,2,p-2);
			input = strip(substr(input,p+1));
			if index(input,'?') + index(input,'*') ^= 0 then
				input = expand(input);
			p = index(input,'.');
			if p = 0 then 
				start_string = input;
			else do;
				substr(start_string,1,8) = substr(input,1,p-1);
				substr(start_string,9,3) = substr(input,p+1);
				end;
			searchfile.fname = substr(start_string,1,8);
			searchfile.ftype = substr(start_string,9,3);
			verfy = (substr(start_string,1,1) = 'V');
			end;

		if asc_target_user = '' then do;
			asc_target_user = sel_usr;
			target_user = selected_user;
			end;
		else if asc_target_user = '*' then
			target_user = -1;
		else if verify(asc_target_user,'0123456789') ^= 0 then
			return(-1);
		else 
			target_user = fixed(asc_target_user);
		if target_user > max_user then
			return(-1);
		end;
	else do;    /*not user_flag*/
		get edit(ch)(a);
		if ch = '' then 
			return(-1);
		ch = upcase(ch);
		end;

	asc = rank(ch)-atsign;
	if (asc < 1)|(asc > number_of_drives) then 
		return(-1);

	call reset;	/*just in case it's read-only*/
	return(asc);
	end;

/*Transfer all tagged files while space remains on target disk*/
mass:procedure;
	dcl i	fixed;
	dcl drive fixed(7);
	dcl copy_ok bit(1);
	dcl ch char(1);
	dcl wild bit(1);

	put edit( '  To')(a);
	drive = getdrv(1,ch);
	if drive = -1 then 
		return;
	if target_user < 0 then
		wild = true;
	else 
		wild = false;

	copy_ok = true;
	do i = 1 to last while (copy_ok);
		if directory(i).tag = '*' then do;
			if wild then do;
				target_user = directory(i).user;
				asc_target_user = numtstr(target_user);
				end;
			copy_ok = copy(i,drive);
		
			/*Update the books*/
			if copy_ok then do;
				directory(i).tag = '#';
				tag_total = tag_total - directory(i).filesize;
				act_total = act_total - directory(i).actualsize;
				end;
			end;

		/*If a character is typed, we quit*/
		if break() then do;
			ch = rdcon();	/*gobble the character*/
			copy_ok = false;
			end;
		end;

	if ^copy_ok then
		put edit('^g****Aborted.')(col(1),a);

	put skip(2);
	end;

/*retag files marked with '#'*/
retag: procedure;
	dcl i 	fixed;
	put skip;
	do i = 1 to last;
		if directory(i).tag = '#' then do;
			put skip edit('Tagging ')(a);
			call putname(i);
			call	tagit(i);
			end;
		end;
		put skip(2);
	end;

/*Print file name in standard form*/
putname: procedure(i);
	dcl i fixed;

	put edit(sel_drv,numtstr(directory(i).user)||':',
		directory(i).fname,'.',directory(i).ftype)
		(a,a(3),3a);
	end;

/*advance cursor*/
advance: procedure(current);
	dcl current fixed;

	/*advance cursor in directory*/
	if current = last then do;
		put skip(2);
		current = 1;
		end;
	else 
		current = current + 1;

	end;

/*back cursor up*/
backup: procedure(current);
	dcl current fixed;

	if current = 1 then do;
		current = last;
		put skip(2);
		end;
	else 
		current = current-1;

	end;


/*Tag files for transfer.*/
tagit:procedure(current);
	dcl current	fixed;

	if directory(current).tag ^= '*' then do;
		directory(current).tag = '*';
		tag_total = tag_total + directory(current).filesize;
		act_total = act_total + directory(current).actualsize;
		end;

	put edit('  Total of tagged files = ', tag_total,'k (',
			act_total,'k)')(a,f(4));

	end;

/*untag a file*/
untag:procedure(current);
	dcl current fixed;
	
	if directory(current).tag = '*' then do;
		tag_total = tag_total - directory(current).filesize;
		act_total = act_total - directory(current).actualsize;
		end;
	directory(current).tag = ' ';

	put edit(' Total of tagged files = ', tag_total,'k (',
			act_total,'k)')(a,f(4));

	call advance(current);
	end;

/*Type current file on console device*/
view:procedure;
	dcl ch char(1);
	dcl rec fixed;
	dcl eof bit(1);
	dcl more bit(1);
	dcl k fixed;
	dcl m	fixed;
	dcl nbuffs fixed;
	dcl wrt_drv char(1);

	dcl memryptr pointer;		
	dcl memory(0:0) bit(16) based(memryptr);
	dcl char_buff(0:0) char(1) based(memryptr);

	dcl sourceptr pointer;
	dcl 1 sourcefile based(sourceptr),
		  2 name1,
		    3 drive fixed(7),  /* drive number */
		    3 fname char(8),   /* file name */
		    3 ftype char(3),   /* file type */
		    3 fext  fixed(7),  /* file extent */
		    3 space (3) bit(8),/* filler */
		  2 name2,             /* used in rename */
		    3 drive2 fixed(7),
		    3 fname2 char(8),
		    3 ftype2 char(3),
		    3 fext2  fixed(7),
		    3 space2 (3) bit(8),
		  2 crec  fixed(7),    /* current record */
		  2 rrec  fixed(15),   /* random record */
		  2 rovf  fixed(7);    /* random rec overflow */
	
	/*set location of sourcefile*/
	sourceptr = addr(directory(last+1));

	/*set up file control block for read*/
	sourcefile.drive = searchfile.drive;
	sourcefile.fname = directory(current).fname;
	sourcefile.ftype = directory(current).ftype;
	sourcefile.fext = 0;
	sourcefile.crec = 0;
	put skip(2);
	call setusr(directory(current).user);

	dummy = open(sourceptr);
	/*set char_buff array address*/
	memryptr = addr(sourcefile.rovf);

	/*set nbuffs according to available memory*/
	nbuffs = divide(memwds()-10*last-18,buffwds,15);
	/*let's not make it too big so we can get a quick sample - 4k is
		enough*/
	nbuffs = min(nbuffs,32);

	eof = false; more = true;
	do while(more&^eof);
		m=0;
		do rec = 0  to nbuffs - 1 while (^eof);
			call setdma(addr(char_buff(m)));
			m = m + buffchar;
			if rdseq(sourceptr) ^=0 then do;
				eof = true;
				nbuffs = rec;
				end;
			end;

		m=0; 

		do rec = 0 to nbuffs - 1 while (more);

			do k = 0 to buffchar-1 while (more);
				ch = char_buff(m + k);
				if ch = '^z' then
					more = false;
				else call wrcon(ch);
				end;



			m = m + buffchar;

			if break() then do;
				ch = rdcon();
				if ch = '^s' then 
					ch=rdcon();
				else more = false;
				end;
			end;
		end;
	
	dummy = close(sourceptr);

	put skip(2);
	end;

end;
/********************************************************************
     SWEEP version 3.6, July 2, 1982
			by Robert Fisher
			   DePaul University
			   243 S. Wabash
			   Chicago, Illinois 60604

Released to the Public Domain by the author.
***********************************************************************
SWEEP is a utility program for maintaining file directories, and 
transferring files between drives and user areas under CP/M2.2. The
latest version will not run under CP/M1.4.

Invoking SWEEP and moving about the directory.
----------------------------------------------
When SWEEP is invoked, you will see a menu of 'commands' and the first
entry in the directory.  You will also be given the total space (in k)
occupied by the files and the remaining space on the disk.  SWEEP also
reports the current user area in the prompt, or an asterisk if you have
selected all user areas.

Typing 'space' or 'return' moves you forward in the directory, typing 
B (or b) moves you backward.  The directory is circular, so typing space
at the end of the directory moves you to the beginning. (A line is skipped.)

When you first invoke SWEEP, you can specify a drive and a starting place
in the directory.  The user area can be specified as a second parameter.

SWEEP B:FOO.COM

will begin with drive B: logged in, and the cursor on the first entry
in the directory which is >= FOO.COM in alphabetical order.  In the latest
version, both file name and file type are recognized.

SWEEP B:FOO.COM 8 

will log in user area 8.  Selecting user area * will cause all user
areas to be "swept" at once.

NEW: If the starting file contains wild card symbols (* or ?), then it
will be treated as a directory wild card specification.  Only those
files in the selected user area will be handled.

At any time, you can recover the menu of functions by typing a
question mark (?).

Simply typing 'x' exits the program.

Note that all characters can be entered in upper or lower case.


Functions that act on one file.
_______________________________

C   This allows you to copy the file to the drive and user area of your
    choice.  You will be prompted for the drive and user area.  Just
    enter 'b3', say, to transfer the file to drive B: user area 3.
    If you omit the user area, the current user area of the file is used.
    Any file of the same name in the target area is first erased, an
    exception being when you are transferring to the current drive and
    user area.  In the latter case, the transferred file is copied with
    its named changed by putting a $ in the last letter of the file type.

    If an existing file in the target area that must be erased is marked
    read-only (R/O), then the copy is simply not made.  You will be told.
    You may use the delete function to delete the file, and then retry
    the copy.

    As an option, sweep will perform a CRC verification of the copy. 
    When you are prompted for the drive and user, add a 'v' or 'V' to
    request verification.  Thus specifying: b3 v  will copy the file
    to drive b:, user area 3, with verification. See below for a 
    description of the verification method.

    BUG:  CP/M does not allow two files of the same name to be open on
    the same drive, even in different user areas.  Thus, when transferring
    between user areas on the same drive, the target file is first named
    with the $ convention and then renamed.  This can result in an
    extra file being erased.

    If there is too little room on the target drive, the transfer is 
    aborted and the directory entry is erased.  Any previous file of
    the same name in the target area is also erased.(Sorry.)

D   The current file is deleted.  You will be asked if you really want
    it erased.  If the file is read-only (R/O), then you will be given
    a second chance to back out.

R   The current file is renamed to the name you specify.  Lower case
    will be converted to upper case, but other 'illegal' characters are
    not trapped.  It is quite possible to give names to files that
    CP/M won't recognize.  (You can also change them back again later,
    however, so no harm is done.)  Responding with "*" to the prompt will 
    put you in the batch wildcard  renaming mode.  See below.

    If the file is read-only, you will be asked if you still want it 
    renamed.  It will no longer be read-only after renaming.

V   The current file is displayed at the terminal.  ^S will stop the display
    until any other key is typed.  Otherwise, any keypress will abort.

T   Tag a file for later transfer.  (See below.)
    After being tagged, the file will be marked with an asterisk,'*'.
    The total of all tagged files is displayed both in units they
    occupy on the current disk and in units they would occupy on
    a single-density disk with a 1k block size.

U   Undo the effect of T (and take away the '*' or '#').


Commands that act on many files.
--------------------------------
M   Mass transfer of all tagged files to a selected drive and user area.
    This is the main reason for the program.  If the target drive fills,
    the operation is aborted, with the untransferred files remaining 
    tagged.  This function invokes the copy function used by C, so 
    everything said above applies here.

    If any key is typed while files are being transferred, the transfer
    is aborted after the current file. The remaining files remain tagged.

    After transferring a tagged file, the tag is changed to '#' so 
    you can tell what has been sent.  This is for display only, the 
    file is otherwise considered untagged, but see the A command. The '#'
    can be removed with the U command, if desired.

    If no user area is specified, or if "*" is specified, the current user
    area of the file is used.

    You may request verification on mass transfers just as in the single
    file copy.  Just follow the drive and user specification with a space
    and a 'v' or 'V'.

A   Retags all the files currently marked with '#'.  This allows you to
    repeat the transfers on another disk.  Useful when the same batch of
    files must be sent to several disks.

E   The E command erases all untagged files or all tagged files. 
    You will be asked whether you want to erase the tagged files or the
    untagged files.  Any answer other than T or t means untagged. 
    
    You will then be asked whether you want to be prompted or not.
    If you ask to be prompted, you will be asked for a Y or N on each
    file.  Otherwise, the untagged files will be deleted.

    Be very careful with this one.  You can wipe out an entire directory
    very quickly.

    You can abort this operation by responding A to the prompt, or typing
    any key in the unprompted mode (if you are quick enough).

    The TAGGED option is useful after a mass transfer.  Use the A command
    to retag all the transferred files, then use the E command to erase 
    them.

    The UNTAGGED option is useful if you want to clean up a disk or user
    area, but retain a few key files.  Just tag the files you want to retain
    and use the E command to erase the others.

R * Batch rename.  You will be prompted for an "Old name:" and a "New name:".
    In each case you may give any "ambiguous file name" in response.  The
    logged-in directory, as currently displayed, is searched for matches
    with "Old name", and the files are renamed, if possible, by the formula
    specified in "New name". A '?' in "New name" means keep that character of
    the current file name, an * is equivalent to filling out the remainder of
    the field (filename or filetype) with "?"'s. Any other character replaces
    the corresponding character in the current file name.  When the 
    substitutions are done, embedded spaces are removed. 

    You will be asked to approve each substitution before it is actually 
    made.

Old name: SWEEP???.*
New name: TEST ???.*

   These responses would result in SWEEP.COM being renamed TEST.COM, and
   SWEEP30.PLI being renamed TEST30.PLI.


Other commands.
---------------

B   Move to previous file. Moves to last file if you are currently on the
    first file.

L   Login new drive and user area. If the user area is ommitted, the 
    current user area is used.  You can specify a starting place in
    the directory. Responding 'b3 foo' will put you in user area 3
    of drive B:, beginning at the first file >= FOO in alphabetical
    order. Selecting "*" for user area, logs in all user areas at once.

    If the starting file is a wild card specification, it is instead used
    to select the files to be used.  Thus:

	l New drive (and user area): b1:*.PLI

    will log you into drive b, user area 1, and select only PLI source
    files. 


S   Calculates space remaining on a drive.

X   Exit to CP/M.

?   Redisplay menu of commands.


Tags.
-----
*   Marks a tagged file.  See M and E commands.

#   Marks a previously tagged file that has been transferred by the
    M command.  It is logically untagged, and the # can be removed
    with the U command if it bothers you. Files marked with '#' can
    be retagged with the A command.

    Unmarked files are untagged.


Remarks.
--------
Whenever a drive is asked for (and on start-up), the disk system is
reset.  This is to prevent a disk from being 'read only' under CP/M,
and to allow you to repeat operations on fresh disks (e.g., finish a
mass transfer after a disk fills by inserting a second disk). It also
means that you should be free to switch density and/or number of sides
of a disk in a particular drive.  This, however, depends on how well
your BIOS is written.It does mean, however that a system disk must be 
present in drive A:.


There is no logical upper limit to directory size, nor to the size of a file
that may be transferred.  The directory size is limited by available
memory, though, as certain information must be kept on each entry. You should
be able to use well over 1500 entries in a 64k system.  Figure 20 bytes
per entry starting at 26k and running up to within 1k, or so,of the BDOS. 
(Note: later versions of SWEEP are about 2k larger.)

The sort is done with quicksort, so it will not deteriorate too badly
with large directories.  (On the order of n*log(n).)  The rest of the
log-on stuff is either fixed - displaying the menu - or linear in the
directory size - reading the directory and compressing it after sorting.
On my system, a directory of about 99 entries takes 3.5 seconds to come up.
Of this, about 1.1 seconds is used in the sort with the rest split about
evenly between the operations before the sort and after the sort. (Disk
access is not figured here, since my system uses the Intersystems CACHE
BIOS with track buffering.  Once the directory is read, it doesn't need
to be read again. Using the standard system raises the total time to 4.9
seconds, the difference being the physical disk access.)

File transfers make use of as much memory as possible to buffer 
data.

The V command buffers only 4k (or what's available, if 4k is too much
for either your file or your RAM).This allows large files to be sampled 
quickly, without reading in the full file.

Invalid or empty answers to any prompt will result in no action being
taken.  Exceptions are specifying a drive you do not have (see patch
points below) or an illegal file name under the R command.

It is legal to include the colon when specifying a drive and user 
area.  Thus responding  "B3:" is equivalent to just "B3", and
"B3:FOO.COM" is the same as "B3 FOO.COM".

Copy verification is done by computing a CRC checksum while writing the
file. SWEEP then reads the copied file back computing it's checksum.  The
two checksum's are compared and you are told if they don't match.
The algorithm used is based on the standard CRC-CCITT polynomial:
	x^16 + x^12 + x^5 + 1.
The algorithm is similar to the one in D. Barker's corrections to CRCK,
and it is now table driven.  It is coded in 8080 assembler for speed, and
is found in the file SWEEPIO.ASM.

The second crc computation is done with a buffer that is offset 3 bytes
from the original buffer.  This is to help catch memory errors. Any CRC 
errors which appear but are not flagged as bad disk sectors are probably
due to bad memory.

The CRC verification takes approximately 2.5 times as long as the straight
copy.

Patch points.
-------------
Two parameters should be set for your system.  If you do not have a
PL/I compiler, they can be set with DDT.

number_of_drives  should be set to the number of drives in your system.
     This will eliminate SELECT errors.  Patch the byte at location 107H.
     It is currently set to 4.

max_user   Set this if you want a number other than 16 (0-15). Patch the
     byte at location 10CH.

In addition, you may want to patch:

CRC_polynomial If you prefer a different choice of polynomial, patch the
     bytes at locations 356CH and 3570H.  If you know enough to have a
     preference, I assume you know what values to put in.
	name	'SWEEPIO'
	title	'Direct CP/M Calls and CRC calculation for SWEEP'
;
; version 1.1 June 24, 1982
; modified to do table driven CRC calculation
;***********************************************************
;*                                                         *
;*	I. cp/m calls from pl/i for direct i/o             *
;*                                                         *
;* mostly from DR's PLIDIO, with help from J.Karras's      *
;* articles in Lifelines for the getpl* modifications.     *
;***********************************************************
	public	memptr	;return pointer to base of free mem
	public	memsiz	;return size of memory in bytes
	public	memwds	;return size of memory in words
	public	dfcb0	;return address of default fcb 0
	public	dfcb1	;return address of default fcb 1
	public	dbuff	;return address of default buffer
	public	reboot	;system reboot (#0)
	public	rdcon	;read console character (#1)
	public	wrcon	;write console character(#2)
	public	rdstat	;read console status (#6c)
	public	break	;get console status (#11)
	public	vers	;get version number (#12)
	public	reset	;reset disk system (#13)
	public	select	;select disk (#14)
	public	open	;open file (#15)
	public	close	;close file (#16)
	public	sear	;search for file (#17)
	public	searn	;search for next (#18)
	public	delete	;delete file (#19)
	public	rdseq	;read file sequential mode (#20)
	public	wrseq	;write file sequential mode (#21)
	public	make	;create file (#22)
	public	rename	;rename file (#23)
	public	curdsk	;return current disk number
	public	setdma	;set DMA address (#26)
	public	allvec	;return address of alloc vector (#27)
	public	filatt	;set file attributes (#30)
	public	getdpb	;get base of disk parm block (#31)
	public	getusr	;get user code (#32a)
	public	setusr	;set user code (#32b)
;
	public	update	;update crcsum from current byte
	public	updsec	;update crcsum from current sector
	public	inicrc	;initialize crc table
;
;
	extrn	?begin	;beginning of free list
	extrn	?boot	;system reboot entry point
	extrn	?bdos	;bdos entry point
	extrn	?dfcb0	;default fcb 0
	extrn	?dfcb1	;default fcb 1
	extrn	?dbuff	;default buffer
;
;***********************************************************
;*                                                         *
;*        equates for interface to cp/m bdos               *
;*                                                         *
;***********************************************************
cr	equ	0dh	;carriage return
lf	equ	0ah	;line feed
eof	equ	1ah	;end of file
;
readc	equ	1	;read character from console
writc	equ	2	;write console character
statf	equ	11	;return console status
versf	equ	12	;get version number
resetf	equ	13	;system reset
seldf	equ	14	;select disk function
openf	equ	15	;open file function
closef	equ	16	;close file
serchf	equ	17	;search for file
serchn	equ	18	;search next
deletf	equ	19	;delete file
readf	equ	20	;read next record
writf	equ	21	;write next record
makef	equ	22	;make file
renamf	equ	23	;rename file
cdiskf	equ	25	;get current disk number
setdmf	equ	26	;set dma function
getalf	equ	27	;get allocation base
setatf	equ	30	;set file attributes
getdpf	equ	31	;get disk parameter block
userf	equ	32	;set/get user code
;
;	utility functions
;***********************************************************
;*                                                         *
;*       general purpose routines used upon entry          *
;*                                                         *
;***********************************************************
;
getp1:	;get single byte parameter to register c
	mov	e,m		;low (addr)
	inx	h
	mov	d,m		;high(addr)
	inx	h
	xchg			;hl = .char
	mov	c,m		;to register e
	xchg
	ret
;
getp2:	;get single word value to DE
getp2i:	;(equivalent to getp2)
	call	getp1
	inx	d
	ldax	d		;get high byte as well
	mov	d,a
	mov	e,c
	ret
;
getver:	;get cp/m or mp/m version number
	push	h		;save possible data adr
	mvi	c,versf
	call	?bdos
	pop	h		;recall data addr
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
memptr:	;return pointer to base of free storage
	lhld	?begin
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
memsiz:	;return size of free memory in bytes
	lhld	?bdos+1		;base of bdos
	xchg			;de = .bdos
	lhld	?begin		;beginning of free storage
	mov	a,e		;low(.bdos)
	sub	l		;-low(begin)
	mov	l,a		;back to l
	mov	a,d		;high(.bdos)
	sbb	h
	mov	h,a		;hl = mem size remaining
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
memwds:	;return size of free memory in words
	call	memsiz		;hl = size in bytes
	mov	a,h		;high(size)
	ora	a		;cy = 0
	rar			;cy = ls bit
	mov	h,a		;back to h
	mov	a,l		;low(size)
	rar			;include ls bit
	mov	l,a		;back to l
	ret			;with wds in hl
;
;***********************************************************
;*                                                         *
;***********************************************************
dfcb0:	;return address of default fcb 0
	lxi	h,?dfcb0
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
dfcb1:	;return address of default fcb 1
	lxi	h,?dfcb1
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
dbuff:	;return address of default buffer
	lxi	h,?dbuff
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
reboot:	;system reboot (#0)
	jmp	?boot
;
;***********************************************************
;*                                                         *
;***********************************************************
rdcon:	;read console character (#1)
	;return character value to stack
	mvi	c,readc
chrin:
	;common code for character input
	call	?bdos		;value returned to A
	pop	h		;return address
	push	psw		;character to stack
	inx	sp		;delete flags
	mvi	a,1		;character length is 1
	pchl			;back to calling routine
;
;***********************************************************
;*                                                         *
;***********************************************************
wrcon:	;write console character(#2)
	;1->char(1)
	call	getp1		;output char to register e
	mov	e,c
	mvi	c,writc		;console write function
	jmp	?bdos		;to write and return
;
;***********************************************************
;*                                                         *
;***********************************************************
rdstat:	;direct console status read
	lxi	h,rdsret	;read status return
	push	h		;return to rdsret
	lhld	?boot+1		;base of jmp vector
	lxi	d,1*3		;offset to .jmp const
	dad	d		;hl = .jmp const
	pchl
;
;***********************************************************
;*                                                         *
;***********************************************************
;***********************************************************
;*                                                         *
;***********************************************************
break:	;get console status (#11)
	mvi	c,statf
	call	?bdos		;return through bdos
;
rdsret:	;return clean true value
	ora	a		;zero?
	rz			;return if so
	mvi	a,0ffh		;clean true value
	ret
;
;***********************************************************
;*                                                         *
;***********************************************************
vers:	;get version number (#12)
	mvi	c,versf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
reset:	;reset disk system (#13)
	mvi	c,resetf
	jmp	?bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
select:	;select disk (#14)
	;1->fixed(7) drive number
	call	getp1		;disk number to C
	mov	e,c
	mvi	c,seldf
	jmp	?bdos		;return through bdos
;***********************************************************
;*                                                         *
;***********************************************************
open:	;open file (#15)
	;1-> addr(fcb)
	call	getp2i		;fcb address to de
	mvi	c,openf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
close:	;close file (#16)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,closef
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
sear:	;search for file (#17)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,serchf
	jmp	?bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
searn:	;search for next (#18)
	mvi	c,serchn	;search next function
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
delete:	;delete file (#19)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,deletf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
rdseq:	;read file sequential mode (#20)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,readf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
wrseq:	;write file sequential mode (#21)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,writf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
make:	;create file (#22)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,makef
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
rename:	;rename file (#23)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,renamf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
curdsk:	;return current disk number (#25)
	mvi	c,cdiskf
	jmp	?bdos		;return value in A
;
;***********************************************************
;*                                                         *
;***********************************************************
setdma:	;set DMA address (#26)
	;1-> pointer (dma address)
	call	getp2		;dma address to DE
	mvi	c,setdmf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
allvec:	;return address of allocation vector (#27)
	mvi	c,getalf
	jmp	?bdos		;return through bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
filatt:	;set file attributes (#30)
	;1-> addr(fcb)
	call	getp2i		;.fcb to DE
	mvi	c,setatf
	jmp	?bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
getdpb:	;get base of current disk parm block (#31)
	mvi	c,getdpf
	jmp	?bdos		;addr returned in HL
;
;***********************************************************
;*                                                         *
;***********************************************************
getusr:	;get user code to register A
	mvi	e,0ffh		;to get user code
	mvi	c,userf
	jmp	?bdos
;
;***********************************************************
;*                                                         *
;***********************************************************
setusr:	;set user code
	call	getp1		;code to E
	mov	e,c
	mvi	c,userf
	jmp	?bdos
;
;***********************************************************
;*							   *
;*  II. Cyclic Redundancy Check algorithms                 *
;* update updates the CRC (first parameter) by the single  *
;* byte second parameter.                                  *
;* updsec is similar, but works at an entire 128 byte      *
;* block pointed at by the second parameter.               *
;* The algorithm is essentially the same as that used in   *
;* the Dave Barker modified versions of CRCK, except that  *
;* I don't use the table lookup, and I use a different     *
;* polynomial : x^16 + x^12 + x^5 + 1.                     *
;* People who care should change the 10h and 21h to the    *
;* appropriate values (0a0h,97h).	                   *
;* Version 1.1 now uses table lookup.			   *
;***********************************************************

update:
	;get the parameters
	call 	getp2
	push	d
	call	getp1
	mov	a,c
	pop	h	
docrc:	;common entry with crc in h,byte in a
	xra	h
	mov	h,a
	mvi	c,8
loop:	dad	h
	jnc	l1
	mov	a,h
	xri	10h
	mov	h,a
	mov	a,l
	xri	21h
	mov	l,a
l1:	dcr	c
	jnz	loop
	ret
;
;
;This builds the table of CRC checksums for 0-255
inicrc:	mvi	b,0
	lxi	d,table
iloop:  mov	a,b
	lxi	h,0
	call	docrc
	xchg
	mov	m,d	;note that high bit is stored first
	inx	h
	mov	m,e
	inx	h
	xchg
	inr	b
	jnz	iloop
	ret
;

updsec:
	call	getp2	;get partial crcsum
	push	d
	call	getp2	;get location of sector
	pop	h
	mvi	b,128
tablp:	ldax	d	;get next byte
	push	d
	lxi	d,table	;do table lookup
	xra	h	; a + h
	add	a	; two byte CRC 
	jnc	taboff
	inr	d	; adjust d 
taboff:
	add	e
	mov	e,a
	jnc	lookup
	inr	d	; another adjustment
lookup:	ldax	d	; work it into the CRC
	xra	l
	mov	h,a
	inx	d
	ldax	d
	mov	l,a
	pop	d	;let's go get another one
	inx	d
	dcr	b
	jnz	tablp
	ret
;
table	ds	512
;
	end
	dcl
		memptr entry         returns (ptr),
		memsiz entry         returns (fixed(15)),
		memwds entry         returns (fixed(15)),
		dfcb0  entry         returns (ptr),
		dfcb1  entry         returns (ptr),
		dbuff  entry         returns (ptr),
		reboot entry,
		rdcon  entry         returns (char(1)),
		wrcon  entry         (char(1)),
		rdstat entry         returns (bit(1)),
		break  entry         returns (bit(1)),
		vers   entry         returns (fixed(15)),
		reset  entry,
		select entry         (fixed(7)),
		open   entry   (ptr) returns (fixed(7)),
		close  entry   (ptr) returns (fixed(7)),
		sear   entry   (ptr) returns (fixed(7)),
		searn  entry         returns (fixed(7)),
		delete entry   (ptr),
		rdseq  entry   (ptr) returns (fixed(7)),
		wrseq  entry   (ptr) returns (fixed(7)),
		make   entry   (ptr) returns (fixed(7)),
		rename entry   (ptr),
		curdsk entry         returns (fixed(7)),
		setdma entry         (ptr),
		allvec entry         returns (ptr),
		filatt entry         (ptr),
		getdpb entry         returns (ptr),
		getusr entry         returns (fixed(7)),
		setusr entry   (fixed(7)),
		update entry (bit(16),bit(8)) returns (bit(16)),
		inicrc entry,
		updsec entry (bit(16),ptr) returns(bit(16));
