/* multiple file printer, version 1.4
 *
 * usage: mfp {[-an21] Uafn | ufn}
 *
 *	[ .. ] means optional and { .. } means one or more.
 *
 *	Uafn is a UNIX style abiguous file name and ufn an unambiguous
 *	file name for a CP/M text file. The possible parameters are 
 *	described below.
 *	
 * description:
 *	Mfp prints out multiple text files on a printer placing the file 
 *	name, date of printing and page number at the top of each page. 
 *	In addition the user can request that a special second header 
 *	line be included on each page. 
 *
 *	The possible parameters are
 *	
 *		-a = alternate char width print mode *
 *		-n = go to normal char width print mode (default) *
 *		-2 = prompt for second header line
 *		-1 = return to single header line (default)
 *	
 *		* = if your printer supports this feature
 *		      (see configuration section below)
 *	
 *	which may be used in combination (e.g. -a2 will print in alternate
 *	character width and prompt for a second header line). All parameters
 *	stay in effect unless explicitely changed in a following parameter 
 *	list. If the -2 option extends over more than one file the program
 *	will prompt for a new second header line before printing each file.
 *	An entry of a NULL line (<cr> only) will abandon the -2 option for
 *	that file but you will continue to be prompted on each succeeding
 *	file until a reset (-1) option is reached. For example
 *	
 *		 mfp zot.c -a2 foo.c -n c:[kb]it.doc -1 b:*c
 *	
 *	will print out zot.c with the defaults, foo.c with alternate width
 *	characters and prompt for a second header line, c:kit.doc and 
 *	c:bit.doc (assuming that they exist) also in normal width character
 *	mode with headers (the program will prompt for the header before 
 *	each file -  a single <cr> skips the header for that file) and 
 *	finally all files on b: that end in 'c' (e.g. *.doc, *.c, *c., 
 *	etc.) in normal width character mode and no header. For more	
 *	information on UNIX wild cards see amatch12.c.
 *
 *	Mfp14.c is actually two programs put together: a custom version of
 *	frnd13.c (all references to dio are eliminated and all uses of 
 *	buffered file i/o functions to get chars out to the console [in
 *	case dio was being used] have been changed to standard i/o
 *	functions) and the multiple file printer program (mainp( ... )).
 * 
 *	Once you have mfp up and running you can always get a usage memory
 *	jog by just typing in mfp with no arguments. Abbreviated insructions,
 *	appropriate for your printer configuration, will be displayed.
 *
 * configuration procedure:
 *
 *	To configure your printer to mfp:
 *
 *		1)Set your printer so that it does NOT send a '\r'
 *	when it gets a '\n' (auto <cr>) or a '\n' when it gets a
 *	'\r' (auto <lf>).
 *
 *		2)If your printer has alternate character widths,
 *	enable the software switching of the character width.
 *
 *	To configure mfp to your printer:
 *
 *		1)Follow the directions for the alternate character
 *	width defines below. (This holds even if your printer doesn't 
 *	have alternate character widths! So you should read that section 
 *	of comments also)
 *
 *		2)You can change the number of lines/page to suit your 
 *	particular taste by changing the LPP define. Be careful not to 
 *	over-run the page; things get very messy.
 *
 *		3)If you prefer fewer/more chars to a tab change the
 *	value of the TAB define.
 *
 *		4)There is a built in safety factor of 2 characters/line
 *	to avoid any auto-wrap on long lines being sent to the printer:
 *	mfp takes care of all wrapping of long lines and interference
 *	from the printer causes the line count to be in error. If your
 *	printer doesn't auto-wrap at the end of a page or you want to
 *	experiment change the value of the SAFETY define.
 *
 *	To compile and run mfp for use with Uafn's:
 *
 *		1)You MUST have a version of bdscio.h and deff.crl which 
 *	have the memory allocation functions (alloc and free) turned ON. 
 *	If this isn't true for your versions, follow the directions in 
 *	bdscio.h to activate them. (Alloc() is used by the argument list 
 *	expander [main()] to get memory for argument storage.)
 *
 *		2)If you don't already have amatch12.c, get it. It can be
 *	found on all the better RCP/M systems.
 *
 *		3)Then		
 *			    cc1 amatch12.c
 *		   	    cc1 mfp14.c
 *			    clink mfp14 amatch12 -o mfp .
 *
 *	To compile and run mfp for use without wild cards:
 *
 *		1)Delete or comment out all code AFTER the usage() function.
 *
 *		2)Change the name of mainp( .. ) to main( .. ).
 *
 *		3)Then
 *				cc1 mfp14.c
 *				clink mfp14 -o mfp .
 *
 * Please send any information about bugs, etc. to
 *
 *				  Herb Schulz
 *			 6008 Forest View Rd., Apt 3C
 *			       Lisle, Il. 60532.
 *
 */
#include <a:bdscio.h>

/*
 * Defines for printers with more than one print width. To set
 * mfp up to use the alternate mode for printing "un-comment"
 * the HASALT define, enter the strings necessary to go into the
 * alternate width and normal width modes for your printer in to the
 * appropriate defines and also the number of characters per line
 * for each of the two print modes. Some common width values 
 * (approximate - assuming 8.5" wide paper [printing width]) are
 *
 *			 5 char/in =  40 char/line
 *		       	10     "   =  80     "
 *			12     "   = 100     "
 *			17     "   = 132     "     .
 *
 * If your printer has only one print width leave the HASALT define
 * commented out and make sure the value of the NORMWIDTH define is
 * correct for your printer.
 */

/*
#define HASALT
*/

#define CONFIGFOR "tty"	/* name of printer for which mfp is configured - used if HASALT is set */
#define ALTWIDTH 100	/* width of alternate pitch in chars/line */
#define NORMWIDTH 80	/* width of normal pitch in chars/line - also if you have no alt width */
#define ALT ""	 	/* string to go into alternate pitch */
#define NORM ""		/* string to go into normal pitch */

/* other user changeable defines */
#define PARMPREFIX '-'	/* prefix to parameter list */
#define LPP 62		/* lines per page */
#define TAB 8		/* tab stops every 8 char */
#define SAFETY 2	/* safety factor: to prevent auto-wrap by the printer */

/* These defines should be left alone */
#define LST 2		/* sending output to LST: device */
#define FF '\014'	/* form feed character */
#define DEFHDR 2	/* default number of lines in header + 1 blank */

/* "external" variables */
char *p,		/* file name pointer - deletes the disk directive */
     date[MAXLINE],	/* buffer to hold present date (other stuff also) */
     hdr[MAXLINE];	/* buffer to hold possible second header line */
int  lc,		/* line counter */
     lcoffset,		/* number of lines in header (DEFHDR or DEFHDR+1) */
     pgc,		/* page counter */
     altflg;		/* alternate mode flag */

/*
 * This is the actual file lister program. The main(), found towards the
 * end of this file, is the argument expander. By doing things in this 
 * way it is relatively easy to delete the wild card features of the 
 * program: mainp() expects ONLY ufn's and parameter lists and can't 
 * cope with wild cards.
 */
mainp(argc, argv)
int argc;
char **argv;
{
	char inbuf[BUFSIZ],	/* input buffer */
	     ch,		/* single char from buffer */
	     nextc;		/* look ahead char for line overflow */
	int  cc,		/* character counter */
	     hdrflg;		/* second header line flag */
     
	altflg = hdrflg = FALSE; /* set defaults */

	/* oops */
	if (argc < 2)
		usage();

	/* get date */
	puts("\nEnter today's date (Month/Day/Year): ");
	gets(date); /* NOTE: this will be in the header for ALL files! */

	puts("\nPress any key when your printer is ready: ");
	getchar();
	puts("\nPressing any key suspends printing.\n");

#ifdef HASALT
	fputs(NORM, LST); /* initialize printer to normal mode */
#endif

	while (--argc) {
		
		p = *(++argv);

		if (*p == PARMPREFIX) {
			while (*(++p)) {
				switch (tolower(*p)) {
#ifdef HASALT
				case 'a':
					altflg = TRUE;
					break;

				case 'n':
					altflg = FALSE;
					fputs(NORM, LST);
					break;
#endif
				case '2':
					hdrflg = TRUE;
					break;

				case '1':
					hdrflg = FALSE;
					break;

				default:
					printf("\n-%c isn't an available option", tolower(*p));
				} /* switch */
			} /* while (*(++p)) */
			continue;
		} /* if (*p == PARMPREFIX) */

		/* have a file name */
		if (fopen(p, inbuf) == ERROR) {
			printf("\nCannot open %s. Continuing ...\n", p);
			continue;
		}

		printf("\nPrinting %s ", p);

#ifdef HASALT
		printf(": %d char/line", altflg ? ALTWIDTH : NORMWIDTH);
#endif

		if (*(p + 1) == ':')
			p += 2; /* now delete disk directive */

		lcoffset = DEFHDR;
		if (hdrflg) {
			puts("\nSecond header line (<cr> alone deletes second line):\n");
			gets(hdr);
			if (*hdr)
				++lcoffset;
		}

		cc = pgc = lc = 0;
		while ((ch = getc(inbuf)) != EOF && (ch &= '\177') != CPMEOF) {

			if (kbhit()) {
				getchar(); /* empty buffer */
				puts("\nPrinting halted (^C exits, <ESC> skips, anything else continues): ");
				if (getchar() == '\033') {
					if (argc > 1) {
						puts("\nGet the printer ready and press any key: ");
						getchar();
					}
					goto nextfile; 
				}
			}

			if (!lc) /* put in header */
				prhdr();
	
			switch (ch) {

			case '\r':
				break; /* '\r' generated when '\n' encountered */

			case '\n':
				cc = 0;
				crlf();
				break;

			case '\t':
				do {
					putc(' ', LST);
#ifdef HASALT
					if (!(++cc % (altflg ? (ALTWIDTH - SAFETY) : (NORMWIDTH - SAFETY)))) {
#else
					if (!(++cc % (NORMWIDTH - SAFETY))) {
#endif
						nextc = getc(inbuf);
						if(!(!(cc % TAB) && eol(nextc)) && !crlf()) 
							prhdr();
						ungetc(nextc, inbuf);
					}
				} while (cc % TAB);
				break;

			default:
				putc(ch, LST);
#ifdef HASALT
				if (!(++cc % (altflg ? (ALTWIDTH - SAFETY) : (NORMWIDTH - SAFETY)))) {
#else
				if (!(++cc % (NORMWIDTH - SAFETY))) {
#endif
					nextc = getc(inbuf);
					if (!eol(nextc))
						crlf();
					ungetc(nextc, inbuf);
				}
			} /* switch (ch) */

		} /* while ((ch ...) */
		if (lc)
			putc(FF, LST);
nextfile:	fclose(inbuf);
	} /* while (--argc) */

#ifdef HASALT
	if (altflg)
		fputs(NORM, LST);
#endif

} /* mainp() */

/* sends a '\r', and either a FF if at the bottom of the page or a '\n' */
int crlf()
{
	++lc;
	putc('\r', LST); /* putc(.., LST) doesn't auto send '\r' with '\n' */
	if (!(lc %= (LPP - lcoffset))) 
		putc(FF, LST);
	else
		putc('\n', LST);
	return(lc);
}

/* 
 * print a header on the top of each page and increment 
 * the line and page counts.
 */
prhdr()
{
#ifdef HASALT
	if (altflg)
		fputs(NORM, LST);
#endif

	++lc;
	fprintf(LST, "%s        %s        Page %d\n", p, date, ++pgc);
	if (lcoffset != DEFHDR) /* second header present */
		fprintf(LST, "%s\n", hdr);
	putc('\n', LST);

#ifdef HASALT
	if (altflg)
		fputs(ALT, LST);
#endif
}

int eol(c)
char c;
{
	char ch;
	ch = c; /* need a variable */

	return(ch == EOF || (ch &= '\177') == CPMEOF || ch == '\r' || ch == '\n');
}

usage()
{
	puts("\nMultiple File Printer by Herb Schulz, Jan. 1982\n");
#ifdef HASALT
	printf("Configured for %s\n", CONFIGFOR);
#endif
	puts("\nusage: mfp {[-");
#ifdef HASALT
	puts("an");
#endif
	puts("21] Uafn | ufn}\n");
	puts("\nWhere Uafn is UNIX style ambiguous file name, ufn");
	puts("\nis an unambiguous file name and the possible");
	puts("\nparameters ");
#ifdef HASALT
	puts("(any combination of one or more may\nbe used - read left to right) ");
#endif
	puts("are\n");
#ifdef HASALT
	printf("\n\t-a = print using %d characters/line", ALTWIDTH);
	printf("\n\t-n = print using %d characters/line (default)", NORMWIDTH);
#endif
	puts("\n\t-2 = prompt for a second header line");
	puts("\n\t-1 = reset to single line header (default)");
	puts("\n\n { ... } = 1 or more, [ ... ] = optional\n");
	puts("\nAll parameters stay in effect until the next");
	puts("\nparameter list is encountered. ");
#ifdef HASALT
	puts("Even then, only\nthe specified parameters change.");
#endif
	exit();
}

/* Comment all of the rest of this program out if you don't want to use wild cards */

/* BIOS calls and related defines */
#define SELDSK 9
#define MAXDSKN 16

/* BDOS calls and related defines */
#define SRCH1 17
#define SRCHN 18
#define CURDSK 25
#define NOMORE 255

/* Other defines */
#define FCBSIZ 36
#define DIRNTRYSZ 32
#define BUFBASE 0x80
#define DSKSIZ 3	/* # chars in disk designator (D:\0) */
#define UFNSIZ 15	/* max # chars in File Name (D:FFFFFFFF.EEE\0) */

/* Change these if necessary */
#define MAXARG 100

/* main() is just frnd13.c with the unused dio references deleted */
main(argc, argv)
int argc;
char **argv;
{
	char *alloc(), 		/* uses memory allocation */
	     afcb[FCBSIZ], 	/* fcb for getting directory */
	     *argxv[MAXARG], 	/* new argument vector */
	     dsk[DSKSIZ],	/* for disk designator */
	     fname[UFNSIZ], 	/* store ufn here for compare */
	     maxdisk;		/* highest disk present */
	int  offset, 		/* offset in default buffer to dir entry */
	     argxc, 		/* new argument counter */
	     i; 		/* general counter */
	unsigned _biosh();	/* bios call returning HL instead of A */	

	/* initialization */
	_allocp = NULL;     /* mandatory initialization for mem. allocation */
	argxv[0] = argv[0]; /* always at least ...*/
	argxc = 1;	    /*  ... one parameter  */

	/* get highest disk present */
	offset = bdos(CURDSK, 0); /* Get Logged in Disk */
	for (i = 0; (i < MAXDSKN) && _biosh(SELDSK, i); ++i)
		; /* keep cycling until NULL pointer to dpb */
	maxdisk = 'A' + i - 1; /* overcount by 1 */
	_biosh(SELDSK, offset); /* restore old Logged in disk for Default */

	if (argc >= 2)
		puts("Processing:");
	
	for (i=1; i < argc; ++i) {

		printf("\n\t%s", argv[i]);

		if (*argv[i] == PARMPREFIX) { /* param list is taken straight */
			argxv[argxc++] = argv[i];
			if (argc >= MAXARG)
				goto oops; /* print message & break out */
			continue;
		}

		/* Must be ufn or UNIX afn to get here */

		/* Test for legal disk designator */
		if (argv[i][1] == ':') {
			sprintf(dsk, "%c:", *argv[i]);
			if ((*dsk < 'A') || (*dsk > maxdisk)) {
				puts(": Illegal disk reference");
				continue;
			}
		}
		else
			*dsk = '\0';

		if (_ufn(argv[i])) {
			argxv[argxc++] = argv[i];
			if (argxc >= MAXARG) 
				goto oops; /* print message & break out */
			continue;
		}
					
		/* UNIX afn to get here */

		/* set disk and set up fcb */
		strcpy(fname, dsk);
		setfcb(afcb, strcat(fname, "????????.???"));
			
		if ((offset = bdos(SRCH1, afcb)) == NOMORE) {
			puts(": Empty directory");
			continue;
		}
		do {
			strcpy(fname, dsk);
			_getfname(fname, offset);
			if (amatch(argv[i], fname)) {
				if ((argxv[argxc] = alloc(strlen(fname) + 1)) == NULL) { /* + 1 needed for '\0' */
					puts("\nOut of memory\007\n");
					exit();
				}
				strcpy(argxv[argxc++], fname);
				if (argxc >= MAXARG) {
	oops:				printf("\nToo many arguments.\n\tpassing first %d args.\n", MAXARG);
					break;
				}
				printf("\n\t\t%s ", fname);
			}
		}
		while ((offset = bdos(SRCHN, afcb)) != NOMORE);
	}
	if (argxc >= 2)
		putchar('\n');
	mainp(argxc, argxv);

}

/* returns TRUE if fname is unambiguous (in the UNIX sense) */
int _ufn(fname)
char *fname;
{
	return(!amatch("*[*?[]*", fname)); /* I cheat a bit here */
}

/* concatenates the file name from the directory in the 
 * default buffer to the end of fname. Returns pointer to fname.
 */
int _getfname(fname, offset)
char *fname;
int offset; /* DIRNTRYSZ byte offset in sector buffer */
{
	char *chp, *temp;
	int i;

	temp = fname;
	for (; *fname; ++fname)
		; /* get to end of of fname - may already have "D:" */

	chp = offset * DIRNTRYSZ + BUFBASE + 1;
	for (i = 1; (i <= 8) && !isspace(*chp); ++i)
		*fname++ = *chp++;

	chp = offset * DIRNTRYSZ + BUFBASE + 9;
	if (!isspace(*chp)) { /* if there is an extension */
		*fname++ = '.'; /* put in '.' */
		for (i = 1; (i <= 3) && !isspace(*chp & '\177'); ++i)
			*fname++ = *chp++ & '\177'; /* get extension high bit off */
	}
	*fname = '\0'; /* terminate it */
	return(temp);
}

/* does bios call n with BC set to c. Returns HL. */
unsigned _biosh(n, c)
int n, c;
{
	int *ip;

	ip = BASE + 1;
	return(call(*ip + 3 * (n - 1), 0, 0, c, 0));
}
