/*==========================================================================
  ATTR.C

  Functions to set MACH 64 draw attributes such mix and color.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "atim64.h"
#include "sample.h"

// Color codes for 4 & 8 bpp modes - palette entry based
static unsigned long color_table_8[17] =
{
    0x00000000,
    0x00000001,     //DARKBLUE8
    0x00000002,     //DARKGREEN8
    0x00000003,     //DARKCYAN8
    0x00000004,     //DARKRED8
    0x00000005,     //DARKMAGENTA8
    0x00000006,     //BROWN8
    0x00000007,     //LIGHTGRAY8
    0x00000008,     //DARKGRAY8
    0x00000009,     //LIGHTBLUE8
    0x0000000A,     //LIGHTGREEN8
    0x0000000B,     //LIGHTCYAN8
    0x0000000C,     //LIGHTRED8
    0x0000000D,     //LIGHTMAGENTA8
    0x0000000E,     //YELLOW8
    0x0000000F,     //WHITE8
    0x000000FF      //HIWHITE8
};

// Color codes for 15 bpp (16 bpp 555 color weight) modes - direct mapped
static unsigned long color_table_15[16] =
{
    0x00000000,
    0x00000014,     //DARKBLUE15
    0x00000280,     //DARKGREEN15
    0x00000294,     //DARKCYAN15
    0x00005000,     //DARKRED15
    0x00005014,     //DARKMAGENTA15
    0x00005280,     //BROWN15
    0x00005294,     //LIGHTGRAY15
    0x0000294A,     //DARKGRAY15
    0x0000001F,     //LIGHTBLUE15
    0x000003E0,     //LIGHTGREEN15
    0x000003FF,     //LIGHTCYAN15
    0x00007C00,     //LIGHTRED15
    0x00007C1F,     //LIGHTMAGENTA15
    0x00007FE0,     //YELLOW15
    0x0000FFFF      //WHITE15
};

// Color codes for 16 bpp (16 bpp 565 color weight) modes - direct mapped
static unsigned long color_table_16[16] =
{
    0x00000000,
    0x00000014,     //DARKBLUE16
    0x000004C0,     //DARKGREEN16
    0x000004D4,     //DARKCYAN16
    0x0000A000,     //DARKRED16
    0x0000A014,     //DARKMAGENTA16
    0x0000A4C0,     //BROWN16
    0x0000A4D4,     //LIGHTGRAY16
    0x0000528A,     //DARKGRAY16
    0x0000001F,     //LIGHTBLUE16
    0x000007E0,     //LIGHTGREEN16
    0x000007FF,     //LIGHTCYAN16
    0x0000F800,     //LIGHTRED16
    0x0000F81F,     //LIGHTMAGENTA16
    0x0000FFE0,     //YELLOW16
    0x0000FFFF      //WHITE16
};

// Color codes for 24 bpp RGB color weight modes - direct mapped
static unsigned long color_table_ARGB[16] =
{
    0x00000000,
    0x0000009E,
    0x00009E00,
    0x00009E9E,
    0x009E0000,
    0x009E009E,
    0x009E9E00,
    0x009E9E9E,
    0x00555555,
    0x000000FF,
    0x0000FF00,
    0x0000FFFF,
    0x00FF0000,
    0x00FF00FF,
    0x00FFFF00,
    0x00FFFFFF
};

// Color codes for 24 bpp BGR color weight modes - direct mapped
static unsigned long color_table_ABGR[16] =
{
    0x00000000,
    0x009E0000,
    0x00009E00,
    0x009E9E00,
    0x0000009E,
    0x009E009E,
    0x00009E9E,
    0x009E9E9E,
    0x00555555,
    0x00FF0000,
    0x0000FF00,
    0x00FFFF00,
    0x000000FF,
    0x00FF00FF,
    0x0000FFFF,
    0x00FFFFFF
};

// Color codes for 32 bpp RGBA color weight modes - direct mapped
static unsigned long color_table_RGBA[16] =
{
    0x00000000,
    0x00009E00,
    0x009E0000,
    0x009E9E00,
    0x9E000000,
    0x9E009E00,
    0x9E9E0000,
    0x9E9E9E00,
    0x55555500,
    0x0000FF00,
    0x00FF0000,
    0x00FFFF00,
    0xFF000000,
    0xFF00FF00,
    0xFFFF0000,
    0xFFFFFF00
};

// Color codes for 32 bpp BGRA color weight modes - direct mapped
static unsigned long color_table_BGRA[16] =
{
    0x00000000,
    0x9E000000,
    0x009E0000,
    0x9E9E0000,
    0x00009E00,
    0x9E009E00,
    0x009E9E00,
    0x9E9E9E00,
    0x55555500,
    0xFF000000,
    0x00FF0000,
    0xFFFF0000,
    0x0000FF00,
    0xFF00FF00,
    0x00FFFF00,
    0xFFFFFF00
};

/* --------------------------------------------------------------------------
  GET_COLOR_CODE - return the color code for the given generic color.

  A "generic" color value is converted into a real color value for sample
  functions set_fg_color () and get_fg_color () to make color setting
  video mode independant. Typically, the returned value is used as the
  input value to set_fg_color () or set_bg_color ().
-------------------------------------------------------------------------- */
unsigned long get_color_code (int color_index)
{
    switch (modeinfo.bpp)
    {
        case 4:
        case 8:
            return (color_table_8[color_index]);

        case 16:
            switch (modeinfo.depth)
            {
                case DEPTH_SUPPORT_555:
                    return (color_table_15[color_index & 0xF]);

                case DEPTH_SUPPORT_565:
                    return (color_table_16[color_index & 0xF]);
            }
            break;

        case 24:
            switch (modeinfo.depth)
            {
                case DEPTH_SUPPORT_RGB:
                    return (color_table_ARGB[color_index & 0xF]);

                case DEPTH_SUPPORT_BGR:
                    return (color_table_ABGR[color_index & 0xF]);
            }
            break;

        case 32:
            switch (modeinfo.depth)
            {
                case DEPTH_SUPPORT_RGBA:
                    return (color_table_RGBA[color_index & 0xF]);

                case DEPTH_SUPPORT_ARGB:
                    return (color_table_ARGB[color_index & 0xF]);

                case DEPTH_SUPPORT_BGRA:
                    return (color_table_BGRA[color_index & 0xF]);

                case DEPTH_SUPPORT_ABGR:
                    return (color_table_ABGR[color_index & 0xF]);
            }
            break;

        default:
            return ((unsigned long) BLACK);
    } // end switch modeinfo.bpp
}

/* --------------------------------------------------------------------------
  SET_FG_COLOR - set foreground color
-------------------------------------------------------------------------- */
void set_fg_color (unsigned long color)
{
    wait_for_fifo (1);
    regw (DP_FRGD_CLR, color);
}

/* --------------------------------------------------------------------------
  GET_FG_COLOR - get foreground color
-------------------------------------------------------------------------- */
unsigned long get_fg_color (void)
{
    wait_for_idle ();   // insure engine is idle before reading a GUI register
    return (regr (DP_FRGD_CLR));
}

/* --------------------------------------------------------------------------
  SET_BG_COLOR - set background color
-------------------------------------------------------------------------- */
void set_bg_color (unsigned long color)
{
    wait_for_fifo (1);
    regw (DP_BKGD_CLR, color);
}

/* --------------------------------------------------------------------------
  GET_BG_COLOR - get background color
-------------------------------------------------------------------------- */
unsigned long get_bg_color (void)
{
    wait_for_idle ();   // insure engine is idle before reading a GUI register
    return (regr (DP_BKGD_CLR));
}

/* --------------------------------------------------------------------------
  SET_FG_MIX - set the current foreground mix
-------------------------------------------------------------------------- */
void set_fg_mix (int mix)
{
    unsigned long temp;

    wait_for_idle ();   // insure engine is idle before reading a GUI register
    temp = (unsigned long) (mix);
    temp = temp << 16;
    temp = (unsigned long) (temp | (regr (DP_MIX) & 0xFFFF));
    regw (DP_MIX, temp);
}

/* --------------------------------------------------------------------------
  GET_FG_MIX - get the current foreground mix
-------------------------------------------------------------------------- */
int get_fg_mix (void)
{
    wait_for_idle();  // insure engine is idle before reading a GUI register
    return ((int)(regr(DP_MIX) >> 16));
}

/* --------------------------------------------------------------------------
  SET_BG_MIX - set the current background mix
-------------------------------------------------------------------------- */
void set_bg_mix (int mix)
{
    unsigned long temp;

    wait_for_idle ();   // insure engine is idle before reading a GUI register
    temp = regr (DP_MIX) & 0xFFFF0000;
    regw (DP_MIX, (unsigned long) (temp | mix));
}

/* --------------------------------------------------------------------------
  GET_BG_MIX - get the current background mix
-------------------------------------------------------------------------- */
int get_bg_mix (void)
{
    wait_for_idle ();   // insure engine is idle before reading a GUI register
    return ((int) (regr (DP_MIX) & 0xFFFF));
}

/* --------------------------------------------------------------------------
  get_primary_color - separate the primary color component from a given
                      color (i.e. separate the RED, GREEN, or BLUE
                      component value

  This function is only useful for non-palettized modes such as 15, 16, 24,
  and 32 bpp color depths. If this function is called for 4 or 8 bpp modes,
  the index value will be returned without modification.
-------------------------------------------------------------------------- */
unsigned long get_primary_color (PRIMARYCOLOR primarycolor,
                                 unsigned long color)
{
    unsigned long primary;

    switch (modeinfo.bpp)
    {
        case 4:
        case 8:
            primary = color;
            break;

        case 16:
            if (modeinfo.depth == 555)
            {
                switch (primarycolor)
                {
                    case RED:
                        primary = (color >> 10) & 0x1F;
                        break;
                    case GREEN:
                        primary = (color >> 5) & 0x1F;
                        break;
                    case BLUE:
                        primary = color & 0x1F;
                        break;
                }
            }
            else    // 565 weight
            {
                switch (primarycolor)
                {
                    case RED:
                        primary = (color >> 11) & 0x1F;
                        break;
                    case GREEN:
                        primary = (color >> 5) & 0x3F;
                        break;
                    case BLUE:
                        primary = color & 0x1F;
                        break;
                }
            }
            break;

        case 24:
            switch (primarycolor)
            {
                case RED:
                    primary = (color >> 16) & 0xFF;
                    break;
                case GREEN:
                    primary = (color >> 8) & 0xFF;
                    break;
                case BLUE:
                    primary = color & 0xFF;
                    break;
            }
            break;

        case 32:
            switch (primarycolor)
            {
                case RED:
                    primary = (color >> 24) & 0xFF;
                    break;
                case GREEN:
                    primary = (color >> 16) & 0xFF;
                    break;
                case BLUE:
                    primary = (color >> 8) & 0xFF;
                    break;
            }
            break;
    }

    return (primary);
}

/* --------------------------------------------------------------------------
  GET_XY_OFFSET - calculate the byte offset from a given point (x, y)
-------------------------------------------------------------------------- */
unsigned long get_xy_offset (int x, int y)
{
    unsigned long offset;

    offset = (unsigned long) y;
    offset = (unsigned long) (offset * modeinfo.pitch);
    if (modeinfo.bpp == 4)
    {
        offset = (unsigned long) (offset / 2);
        offset = (unsigned long) (offset + (x / 2));
    }
    else
    {
        offset = (unsigned long) (offset * (modeinfo.bpp / 8));
        offset = (unsigned long) (offset + ((modeinfo.bpp / 8) * x));
    }

    return (offset);
}

